/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "irendertool.h"


#include "ivtk.h"
#include "iactor.h"
#include "icamera.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "ilightkit.h"
#include "imagnifier.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "istereoimage.h"
#include "istereoimagearray.h"
#include "iviewmodule.h"

#include <vtkActor2D.h>
#include <vtkCommand.h>
#include <vtkCuller.h>
#include <vtkCullerCollection.h>
#include <vtkLight.h>
#include <vtkLightCollection.h>
#include <vtkLineSource.h>
#include <vtkOpenGLRenderWindow.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkPropAssembly.h>
#include <vtkPropCollection.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowCollection.h>
#include <vtkRenderWindowInteractor.h>

//
//  Templates
//
#include "iarraytemplate.h"


//
//  Helper classes
//
class iDualWindowObserver : public vtkCommand
{

public:

	static iDualWindowObserver* New(iRenderTool *rv)
	{
		return new iDualWindowObserver(rv);
	}

	void Block(bool s)
	{
		mIsBlocked = s;
	}

	virtual void Execute(vtkObject * /*caller*/, unsigned long event, void * /*callData*/)
	{
		if(mIsBlocked || (event!=StartEvent && event!=EndEvent && event!=ModifiedEvent)) return;
		
		if(event == StartEvent)
		{
			if(mRenderTool->mMainWin->GetStereoRender() == 0)
			{
				if(mRenderTool->mDualWin != 0) mRenderTool->ShowDualWindow(false); 
			}
			else
			{
				if(mRenderTool->mDualWin == 0) mRenderTool->ShowDualWindow(true); 
			}
		}

		if(event == EndEvent)
		{
			if(mRenderTool->mDualWin != 0) mRenderTool->mDualWin->Render();
		}

		if(event == ModifiedEvent)
		{
			if(mRenderTool->mDualWin != 0) mRenderTool->mDualWin->SetSize(mRenderTool->mMainWin->GetSize());
		}
	}

protected:

	iDualWindowObserver(iRenderTool *rv)
	{
		mRenderTool = rv;
		mIsBlocked = false;
	}

	bool mIsBlocked;
	iRenderTool *mRenderTool;
};


class iRendererObserver : public vtkCommand
{

public:

	static iRendererObserver* New(iRenderTool *rv)
	{
		return new iRendererObserver(rv);
	}

	void Block(bool s)
	{
		mIsBlocked = s;
	}

	virtual void Execute(vtkObject * /*caller*/, unsigned long event, void * /*callData*/)
	{
		if(mIsBlocked) return;

		switch(event)
		{
		case ModifiedEvent:
			{
				mRenderTool->GetViewModule()->ClearCache(); // our size is cached by VM
				break;
			}
		case ResetCameraClippingRangeEvent:
			{
				double *cr = mRenderTool->mMainRen->GetActiveCamera()->GetClippingRange();
				if(mRenderTool->mAutoClippingRange)
				{
					mRenderTool->mClippingRange[0] = cr[0];
					mRenderTool->mClippingRange[1] = cr[1];
				}
				else
				{
					cr[0] = mRenderTool->mClippingRange[0];
					cr[1] = mRenderTool->mClippingRange[1];
				}
				break;
			}
		}
	}

protected:

	iRendererObserver(iRenderTool *rv)
	{
		mRenderTool = rv;
		mIsBlocked = false;
	}

	bool mIsBlocked;
	iRenderTool *mRenderTool;
};


//
//  Main class
//
iRenderTool* iRenderTool::New(iViewModule *vm)
{
	iRenderTool *tmp =  iExtensionFactory::CreateRenderTool(vm);
	IERROR_ASSERT(tmp);
	return tmp;
}


iRenderTool::iRenderTool(iViewModule *vm, vtkRenderer *ren, iMagnifier *mag) : iViewModuleComponent(vm), mObservers(Observer())
{
	//
	//  Main Window stuff
	//
	mMainWin = iShellFactory::CreateRenderWindow(this); IERROR_ASSERT(mMainWin);
	if(ren == 0)
	{
		mMainRen = vtkRenderer::New(); IERROR_ASSERT(mMainRen);
		mMainInt = iShellFactory::CreateRenderWindowInteractor(this); IERROR_ASSERT(mMainInt);
		mMainCam = iCamera::New(this); IERROR_ASSERT(mMainCam);
		mMainRen->SetActiveCamera(mMainCam);
		mMainCam->Delete();
		//
		//  Configure Renderer. It is important to switch the backing store off for the
		//  stereo mode to work properly.
		//
		mMainRen->BackingStoreOff();
		mMainRen->TwoSidedLightingOff();
		mMainRen->LightFollowCameraOn();
		mMainRen->GetActiveCamera()->SetEyeAngle(2.0);
		//
		//  Configure Interactor
		//
		mMainInt->SetRenderWindow(mMainWin);
		mMainInt->LightFollowCameraOff();
		//
		//  Create and initiate lights
		//
		mLights = iLightKit::New(); IERROR_ASSERT(mLights);	
		mLights->AddLightsToRenderer(mMainRen);
		//
		//  Image magnifier
		//
		mMagnifier = iMagnifier::New(); IERROR_ASSERT(mMagnifier);
	}
	else
	{
		mMainRen = ren;
		mMainInt = 0;
		mMainCam = 0;
		mLights = 0;
		mMagnifier = mag; IERROR_ASSERT(mMagnifier);
		mMagnifier->Register(ren);
	}

	//
	//  Configure Window
	//
	mMainWin->SetSize(640,480);
	mMainWin->StereoCapableWindowOn();
	mMainWin->AddRenderer(mMainRen);
	mMainWin->SetWindowName("IFrIT - Visualization Window");

	//
	//  Dual window renderer and observer
	//
	mDualWin = 0;
	mDualRen = this->CreateRenderer();

	mDualRen->SetActiveCamera(mMainRen->GetActiveCamera());
	mDualRen->LightFollowCameraOff();

	//
	//  Dual window observer
	//
	mDualWindowObserver = iDualWindowObserver::New(this); IERROR_ASSERT(mDualWindowObserver);

	//
	//  Dual window alignment markers (only needed for the primary window)
	//
	if(ren == 0)
	{
		mStereoAlignmentMarkersActor = vtkPropAssembly::New(); IERROR_ASSERT(mStereoAlignmentMarkersActor);
		mStereoAlignmentMarkersActor->VisibilityOff();
		mStereoAlignmentMarkersActor->PickableOff();
		vtkPolyDataMapper2D *mapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mapper2D);
		vtkLineSource *ls;
		vtkActor2D *actor2D[8];

		float aml = 0.15;
		float amo = 0.1;
		int i;
		for(i=0; i<8; i++)
		{
			actor2D[i] = vtkActor2D::New(); IERROR_ASSERT(actor2D[i]);
			mapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mapper2D);
			ls = vtkLineSource::New(); IERROR_ASSERT(ls);
			ls->SetResolution(1);
			if(i%2 == 0)
			{
				ls->SetPoint1(-aml,0.0,0.0);
				ls->SetPoint2(aml,0.0,0.0);
			}
			else
			{
				ls->SetPoint1(0.0,-aml,0.0);
				ls->SetPoint2(0.0,aml,0.0);
			}
			ls->Update();
			mapper2D->SetInput(ls->GetOutput());
			ls->Delete();
			vtkCoordinate *c = vtkCoordinate::New(); IERROR_ASSERT(c);
			c->SetCoordinateSystemToView();
			mapper2D->SetTransformCoordinate(c);
			c->Delete();
			actor2D[i]->SetMapper(mapper2D);
			mapper2D->Delete();
			actor2D[i]->GetProperty()->SetColor(0.0,0.0,0.0);
			actor2D[i]->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
			actor2D[i]->PickableOff();	
			actor2D[i]->SetPosition((0.5-amo)*(2*((i/2)%2)-1),(0.5-amo)*(2*((i/4)%2)-1));
			actor2D[i]->SetLayerNumber(1);
			mStereoAlignmentMarkersActor->AddPart(actor2D[i]);
			actor2D[i]->Delete();
		}

		this->AddObject(mStereoAlignmentMarkersActor);
		mStereoAlignmentMarkersActor->Delete();
	}
	else
	{
		mStereoAlignmentMarkersActor = 0;
	}

	//
	//  Clipping range controls
	//
	mRendererObserver = iRendererObserver::New(this); IERROR_ASSERT(mRendererObserver);
	mAutoClippingRange = true;
	mMainRen->AddObserver(vtkCommand::ResetCameraClippingRangeEvent,mRendererObserver);
	mMainRen->AddObserver(vtkCommand::ModifiedEvent,mRendererObserver);

	//
	//  RenderWindow collection
	//
	mWindowCollection = vtkRenderWindowCollection::New(); IERROR_ASSERT(mWindowCollection);
	mWindowCollectionUpToDate = false;

	//
	//  Properties
	//
	mStereoType = -1;
	mAntialiasing = false;
	mStereoAlignmentMarkersOn = true;

	//
	//  Set the properties
	//
	this->SetStereoType(VTK_STEREO_RED_BLUE);
	this->SetAntialiasing(true);
}


iRenderTool::~iRenderTool()
{
	this->ShowDualWindow(false);

	mWindowCollection->RemoveAllItems();
	mWindowCollection->Delete();
	mDualWindowObserver->Delete();

	mMagnifier->Delete();

	mMainRen->RemoveObserver(mRendererObserver);
	mRendererObserver->Delete();

	if(mLights != 0) mLights->Delete();
	if(mMainInt != 0) mMainInt->Delete();

	mMainRen->SetRenderWindow(0);
	mMainWin->RemoveRenderer(mMainRen);

	mDualRen->Delete();

	mMainRen->Delete();
	mMainWin->Delete();
}


int iRenderTool::GetRenderingMagnification() const
{
	return mMagnifier->GetMagnification();
}


int iRenderTool::GetNumberOfActiveViews() const
{
	return (mDualWin == 0) ? 1 : 2;
}


void iRenderTool::Render()
{
	mMainWin->InvokeEvent(vtkCommand::StartEvent,NULL);
	this->ResetCameraClippingRange();
	mMainWin->Render();
	mMainWin->InvokeEvent(vtkCommand::EndEvent,NULL);
}


void iRenderTool::ResetCamera()
{
	double *dp = mMainRen->GetActiveCamera()->GetDirectionOfProjection();
	int idir = 1; if(dp[0] > 0.0) idir = -1;
	double dmax = fabs(dp[0]);
	if(fabs(dp[1]) > dmax)
	{
		dmax = fabs(dp[1]);
		idir = 2; if(dp[1] > 0.0) idir = -2;
	}
	if(fabs(dp[2]) > dmax)
	{
		idir = 3; if(dp[2] > 0.0) idir = -3;
	}

	mMainRen->GetActiveCamera()->SetFocalPoint(0.0,0.0,0.0);
	mMainRen->GetActiveCamera()->SetPosition(0.0,0.0,7.0);
	mMainRen->GetActiveCamera()->SetViewUp(0.0,1.0,0.0);
	mMainRen->GetActiveCamera()->SetParallelScale(1.6);

	switch (idir)
	{
	case  1: { mMainRen->GetActiveCamera()->Azimuth(90.0); break; }
	case -1: { mMainRen->GetActiveCamera()->Azimuth(-90.0); break; }
	case  2: { mMainRen->GetActiveCamera()->Elevation(90.0); mMainRen->GetActiveCamera()->SetViewUp(0.0,0.0,-1.0); break; }
	case -2: { mMainRen->GetActiveCamera()->Elevation(-90.0); mMainRen->GetActiveCamera()->SetViewUp(0.0,0.0,1.0); break; }
	case  3: { break; }
	case -3: { mMainRen->GetActiveCamera()->Azimuth(180.0); break; }
	}

	mMainRen->ResetCameraClippingRange();
}

void iRenderTool::AddObserver(unsigned long event, vtkCommand *command)
{
	Observer tmp;

	tmp.Event = event;
	tmp.Command = command;
	mObservers.Add(tmp);

	mMainWin->AddObserver(event,command);
	if(mDualWin != 0) mDualWin->AddObserver(event,command);
}


void iRenderTool::UpdateWindowName(const iString &base)
{
	if(mDualWin == 0)
	{
		mMainWin->SetWindowName(base.ToCharPointer());
	}
	else
	{
		mMainWin->SetWindowName((base+": Left Eye").ToCharPointer());
		mDualWin->SetWindowName((base+": Right Eye").ToCharPointer());
	}
}


vtkRenderWindowCollection* iRenderTool::GetRenderWindowCollection()
{
	if(!mWindowCollectionUpToDate)
	{
		this->UpdateWindowCollection();
		mWindowCollectionUpToDate = true;
	}
	return mWindowCollection;
}


void iRenderTool::UpdateWindowCollection()
{
	mWindowCollection->RemoveAllItems();
	mWindowCollection->AddItem(mMainWin);
	if(mDualWin != 0) mWindowCollection->AddItem(mDualWin);
}


void iRenderTool::WindowsModified()
{
	mWindowCollectionUpToDate = false;
}


void iRenderTool::ResetCameraClippingRange()
{
	if(mAutoClippingRange) mMainRen->ResetCameraClippingRange();
}


void iRenderTool::SetBackground(iColor &color)
{
	mMainRen->SetBackground(color.ToVTK());
	mDualRen->SetBackground(color.ToVTK());
}


void iRenderTool::AddObject(vtkProp* p)
{
	if(p != 0)
	{
#ifdef IVTK_4
		mMainRen->AddProp(p);
		mDualRen->AddProp(p);
#else
		mMainRen->AddViewProp(p);
		mDualRen->AddViewProp(p);
#endif
	}
}


void iRenderTool::RemoveObject(vtkProp* p)
{
	if(p != 0)
	{
#ifdef IVTK_4
		mMainRen->RemoveProp(p);
		mDualRen->RemoveProp(p);
#else
		mMainRen->RemoveViewProp(p);
		mDualRen->RemoveViewProp(p);
#endif
	}
}


const int* iRenderTool::GetRenderWindowSize() const
{ 
	return mMainWin->GetSize(); 
}


void iRenderTool::SetRenderWindowSize(int w, int h)
{ 
	mMainWin->SetSize(w,h);
	if(mDualWin != 0) mDualWin->SetSize(w,h);
}


const int* iRenderTool::GetRenderWindowPosition() const
{ 
	return mMainWin->GetPosition(); 
}


void iRenderTool::SetRenderWindowPosition(int x, int y)
{ 
	mMainWin->SetPosition(x,y);
}


void iRenderTool::SetAntialiasing(bool s)
{
	mAntialiasing = s;
	if(s)
	{
		mMainWin->SetLineSmoothing(1);
		mMainWin->SetPointSmoothing(1);
		if(mDualWin != 0)
		{
			mDualWin->SetLineSmoothing(1);
			mDualWin->SetPointSmoothing(1);
		}
    } 
	else 
	{
		mMainWin->SetLineSmoothing(0);
		mMainWin->SetPointSmoothing(0);
		if(mDualWin != 0)
		{
			mDualWin->SetLineSmoothing(0);
			mDualWin->SetPointSmoothing(0);
		}
 	}
	vtkOpenGLRenderWindow *w = dynamic_cast<vtkOpenGLRenderWindow*>(mMainWin);
	if(w != 0)
	{
		w->MakeCurrent();
		w->OpenGLInit();
	}
	if(mDualWin != 0)
	{
		w = dynamic_cast<vtkOpenGLRenderWindow*>(mDualWin);
		if(w != 0)
		{
			w->MakeCurrent();
			w->OpenGLInit();
		}
	}
}


void iRenderTool::SetAdjustCameraClippingRangeAutomatically(bool s)
{
	mAutoClippingRange = s;
}


void iRenderTool::SetCameraClippingRange(double cr[2])
{
	//
	//  This is a dirty trick to avoid the minimum bound on the nearest clipping range of 0.0001
	//
	if(!mAutoClippingRange)
	{
		double *d = mMainRen->GetActiveCamera()->GetClippingRange();
		mClippingRange[0] = d[0] = cr[0];
		mClippingRange[1] = d[1] = cr[1];
	}
}


void iRenderTool::RenderImages(int mag, iStereoImageArray &images)
{
	iStereoImage tmp;
	
	images.Clear();
	this->RenderStereoImage(mag,tmp);
	images.Add(tmp);
}


void iRenderTool::RenderStereoImage(int mag, iStereoImage &image)
{
	int v = 0;

	mDualWindowObserver->Block(true);
	if(mStereoAlignmentMarkersActor != 0)
	{
		v = mStereoAlignmentMarkersActor->GetVisibility();
		mStereoAlignmentMarkersActor->VisibilityOff();
	}

	mMagnifier->SetMagnification(mag);

	//
	//  Render main window
	//
	int lfc = mMainRen->GetLightFollowCamera();
	mMainRen->SetLightFollowCamera(0);  //  Maintain correct lighting under magnification
	mMagnifier->SetInput(mMainRen);
	mMagnifier->Modified();
	mMagnifier->UpdateWholeExtent();
	mMagnifier->Update();
	image.ReplaceData(0,mMagnifier->GetOutput());
	mMainRen->SetLightFollowCamera(lfc);

	//
	//  Render dual window
	//
	if(mDualWin != 0)
	{
		mMagnifier->SetInput(mDualRen);
		mMagnifier->Modified();
		mMagnifier->UpdateWholeExtent();
		mMagnifier->Update();
		image.ReplaceData(1,mMagnifier->GetOutput());
	}

	mMagnifier->SetMagnification(1);

	if(mStereoAlignmentMarkersActor != 0)
	{
		mStereoAlignmentMarkersActor->SetVisibility(v);
	}
	mDualWindowObserver->Block(false);
}


float iRenderTool::GetLastRenderTimeInSeconds() const
{
	float s = mMainRen->GetLastRenderTimeInSeconds();
	if(mDualWin != 0) s += mDualRen->GetLastRenderTimeInSeconds();
	return s;
}


void iRenderTool::GetAspectRatio(double ar[2]) const
{
	mMainRen->GetAspect(ar);
}


//
//  Stereo operations
//
void iRenderTool::ShowStereoAlignmentMarkers(bool s)
{
	mStereoAlignmentMarkersOn = s;
	mStereoAlignmentMarkersActor->SetVisibility((s && mDualWin!=0)?1:0);
}


bool iRenderTool::GetStereo() const
{
	return mMainWin->GetStereoRender() != 0;
}


void iRenderTool::SetStereo(bool s)
{
	if(s) mMainWin->StereoRenderOn(); else mMainWin->StereoRenderOff();
}


void iRenderTool::SetStereoType(int n)
{
	if(n<0 || mMainWin->GetStereoCapableWindow()==0 || n==mStereoType) return;

	if(n > 0)
	{
		mMainWin->SetStereoType(n);
		mStereoType = mMainWin->GetStereoType();
		if(mDualWin != 0)
		{
			this->ShowDualWindow(false);
		}
		mMainWin->RemoveObserver(mDualWindowObserver);
	}
	else
	{
		mStereoType = 0;
		mMainWin->SetStereoTypeToLeft();
		mMainWin->AddObserver(vtkCommand::StartEvent,mDualWindowObserver);
		mMainWin->AddObserver(vtkCommand::EndEvent,mDualWindowObserver);
		mMainWin->AddObserver(vtkCommand::ModifiedEvent,mDualWindowObserver);
	}
}

	
void iRenderTool::ShowDualWindow(bool s)
{
	if(s)
	{
		if(mDualWin == 0) // DualWindow is not shown yet
		{
			if(mMainWin->GetStereoRender() == 0)
			{
				IERROR_REPORT("Creating DualWindow in the non-stereo mode.");
			}
			mDualWin = iShellFactory::CreateRenderWindow(this); IERROR_ASSERT(mDualWin);
			int *s = mMainWin->GetSize();
			mDualWin->SetSize(s[0],s[1]);
			mDualWin->StereoCapableWindowOn();
			mDualWin->SetStereoTypeToRight();
			mDualWin->StereoRenderOn();
			this->WindowsModified();

			int i;
			for(i=0; i<mObservers.Size(); i++)
			{
				mDualWin->AddObserver(mObservers[i].Event,mObservers[i].Command);
			}
			//
			//  Copy RenderWindows
			//
			mDualWin->SetLineSmoothing(mMainWin->GetLineSmoothing());
			mDualWin->SetPointSmoothing(mMainWin->GetPointSmoothing());

			mDualWin->AddRenderer(mDualRen);
	
			if(mStereoAlignmentMarkersActor!=0 && mStereoAlignmentMarkersOn) mStereoAlignmentMarkersActor->VisibilityOn();
			this->GetViewModule()->UpdateWindowNumber();
		}
	}
	else
	{
		if(mDualWin != 0)
		{
			this->WindowsModified();
			mDualRen->SetRenderWindow(0);
			mDualWin->RemoveRenderer(mDualRen);
			mDualWin->Delete();
			mDualWin = 0;

			if(mStereoAlignmentMarkersActor != 0) mStereoAlignmentMarkersActor->VisibilityOff();
			this->GetViewModule()->UpdateWindowNumber();
		}
	}
}


iRenderTool* iRenderTool::CreateInstance(vtkRenderer *ren) const
{
	return new iRenderTool(this->GetViewModule(),ren,mMagnifier);
}


vtkRenderer* iRenderTool::CreateRenderer() const
{
	vtkRenderer *ren = vtkRenderer::New();
	if(ren == 0) return 0;

	//
	//  Copy the state of the primary renderer
	//
	ren->SetBackingStore(mMainRen->GetBackingStore());
	ren->SetTwoSidedLighting(mMainRen->GetTwoSidedLighting());
	ren->SetBackground(mMainRen->GetBackground());
	ren->LightFollowCameraOff(); // only the primary renderer can control lights

	//
	//  Copy props. The trick is that we need to register the new renderer with every iActor
	//  so that iActors can create special mappers for this renderer. Otherwise, vtkPolyDataMappers 
	//  remember the last window they rendered into, and will reset at switching to a different window.
	//
#ifdef IVTK_4
	vtkPropCollection *pc = mMainRen->GetProps();
#else
	vtkPropCollection *pc = mMainRen->GetViewProps();
#endif
	vtkProp *p;
	pc->InitTraversal();
	while((p = pc->GetNextProp()) != 0)
	{
#ifdef IVTK_4
		ren->AddProp(p);
#else
		ren->AddViewProp(p);
#endif
	}

	//
	//  Copy lights
	//
	vtkLight *l;
	vtkLightCollection *lc = mMainRen->GetLights();
	lc->InitTraversal();
	while((l = lc->GetNextItem()) != 0) ren->AddLight(l);

	//
	//  Copy cullers
	//
	vtkCuller *c;
	vtkCullerCollection *cc = mMainRen->GetCullers();
	cc->InitTraversal();
	ren->GetCullers()->RemoveAllItems();
	while((c = cc->GetNextItem()) != 0) ren->AddCuller(c);

	return ren;
}

