/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogscriptdebugger.h"


#include "ianimator.h"
#include "ianimatorscript.h"
#include "icontrolmodule.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iimagefactory.h"
#include "iscriptkit.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdialog.h"
#include "iggdialoghelp.h"
#include "iggframetopparent.h"
#include "iggmainwindow.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgettext.h"

#include "ibgmenuwindowsubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgettexteditorsubject.h"

#include <vtkTimerLog.h>

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


using namespace iParameter;
using namespace iParameter;


//
//  Helper class
//
class iggDialogScriptDebuggerEditor : public iggWidgetTextEditor
{

public:

	iggDialogScriptDebuggerEditor(iggDialogScriptDebugger *d, iggFrame *parent) : iggWidgetTextEditor(0U,parent)
	{
		mDebugger = d; 
		mBreakPointButtons[0] = mBreakPointButtons[1] = 0;

		mMarkBefore = true;
		mBreakPointColor = iColor(210,180,120);

		//
		//  Syntax highlighting
		//
		mSubject->HighlightSyntax(true);
		mSubject->HighlightComments("#",iColor(0,120,0));
		mSubject->HighlightStrings(iColor(120,0,0));

		mSubject->AddCategory(iColor(0,0,200));
		mSubject->AddCategory(iColor(170,0,170));

		iAnimatorScript *s = this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetScript();
		int i, n = s->GetNumberOfReservedWords();
		for(i=0; i<n; i++)
		{
			if(s->IsConstantName(s->GetReservedWord(i)))
			{
				mSubject->AddKeyword(s->GetReservedWord(i),1);
			}
			else
			{
				mSubject->AddKeyword(s->GetReservedWord(i),0);
			}
		}

		//
		//  Search capabilities
		//
		mFindPara = mFindIndex = 0;
	}

	void AttachBreakPointButton(int n, iggWidget *w)
	{
		if(n>=0 && n<2) mBreakPointButtons[n] = w;
	}

	void SetMarkBefore(bool s)
	{
		mMarkBefore = s;
	}

	void FindNext(const iString &text, bool cs, bool wo, bool forward)
	{
		if(!mSubject->Find(text,cs,wo,forward,mFindPara,mFindIndex))
		{
			mFindPara = mFindIndex = 0;
		}
		else mFindIndex++;
	}

	virtual void SetText(const iString &text)
	{
		this->iggWidgetTextEditor::SetText(text);
		this->OnTextChanged();
	}

	void ShowError(iAnimatorScript *s)
	{
		mSubject->Select(s->GetThisLine(),s->GetErrorPosition(),s->GetThisLine(),mSubject->GetLine(s->GetThisLine()).Length());
	}

	void MarkLine(iAnimatorScript *s)
	{
		if(mMarkBefore)
		{
			mSubject->Select(s->GetNextLine(),0,s->GetNextLine(),mSubject->GetLine(s->GetNextLine()).Length());
		}
		else
		{
            mSubject->Select(s->GetThisLine(),0,s->GetThisLine(),mSubject->GetLine(s->GetThisLine()).Length());
		}
	}

	void ClearSelection()
	{
		mSubject->Select(-1,0,0,0);
	}

	void SetBreakPoint(bool s)
	{
		int line, index;
		mSubject->GetCursorPosition(line,index);
		iString ws = mSubject->GetLine(line);
		ws.ReduceWhiteSpace();

		bool mod = this->TextModified();
		if(s && ws.Length()>0 && ws[0]!='#') mSubject->SetLineColor(line,mBreakPointColor); else mSubject->UnSetLineColor(line);
		this->SetModified(mod);

		this->OnCursorPositionChanged(line,index);
	}

	bool IsBreakPoint(int line = -1) const
	{
		if(line < 0)
		{
			int index;
			mSubject->GetCursorPosition(line,index);
		}
		return mSubject->GetLineColor(line) == mBreakPointColor;
	}

protected:

	virtual void UpdateWidgetBody()
	{
		const iString &text(mDebugger->GetAnimator()->GetScript()->GetText());
		if(mSubject->GetText() != text)
		{
			mDebugger->Debug(false);
			mSubject->SetText(text);
		}
		this->iggWidgetTextEditor::UpdateWidgetBody();
	}

	virtual void OnTextChanged()
	{
		const iString &text(mDebugger->GetAnimator()->GetScript()->GetText());
		if(mSubject->GetText() != text)
		{
			mDebugger->GetAnimator()->GetScript()->SetText(mSubject->GetText());
		}
		mDebugger->Debug(false);
	}

	virtual void OnCursorPositionChanged(int line, int /*index*/)
	{
		bool b = this->IsBreakPoint(line);

		if(mBreakPointButtons[0] != 0) mBreakPointButtons[0]->Enable(!b);
		if(mBreakPointButtons[1] != 0) mBreakPointButtons[1]->Enable(b);
	}

	bool mMarkBefore;
	iggWidget *mBreakPointButtons[2];
	int mFindPara, mFindIndex;
	iggDialogScriptDebugger *mDebugger;
	iColor mBreakPointColor;
};


namespace iggDialogScriptDebugger_Private
{
	//
	//  Menu ids
	//
	const int _MenuFileNew = 1;
	const int _MenuFileOpen = 2;
	const int _MenuFileSave = 3;
	const int _MenuFileSaveAs = 4;
	const int _MenuFileClose = 5;
	const int _MenuFileMax = 100;

	const int _MenuEditUndo = 101;
	const int _MenuEditRedo = 102;
	const int _MenuEditCut = 103;
	const int _MenuEditCopy = 104;
	const int _MenuEditPaste = 105;
	const int _MenuEditFind = 106;
	const int _MenuEditMax = 200;

	const int _MenuDebugSyntax = 201;
	const int _MenuDebugStart = 202;
	const int _MenuDebugStop = 203;
	const int _MenuDebugToggleBreakPoint = 204;
	const int _MenuDebugMax = 300;

	const int _MenuHelpUsingDebugger = 301;
	const int _MenuHelpScriptSyntax = 302;
	const int _MenuHelpMax = 400;

	//
	//  Run states
	//
	const int _RunStatePaused = 0;
	const int _RunStateExecute = 1;
	const int _RunStateLineStep = 2;
	const int _RunStateFrameStep = 3;
	const int __NumRunStates = 4;

	//
	//  Find dialog and its classes
	//
	class FindText : public iggWidget
	{

	public:

		FindText(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,false,0,"Find:",0);
			this->SetBaloonHelp("Type a string to find");
		}

		const iString GetText() const
		{
			return mSubject->GetText();
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		ibgWidgetEntrySubject *mSubject;
	};


	class FindCheckBox : public iggWidgetSimpleCheckBox
	{

	public:

		FindCheckBox(const iString &title, iggFrame *parent) : iggWidgetSimpleCheckBox(title,parent)
		{
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void OnChecked(bool)
		{
		}
	};


	class FindButton : public iggWidgetSimpleButton
	{

	public:

		FindButton(iggDialogScriptDebuggerEditor *ed, FindText *t, FindCheckBox *cs, FindCheckBox *wo, FindCheckBox *bs, iggFrame *parent) : iggWidgetSimpleButton("Next",parent)
		{
			mEditor = ed;
			mText = t;
			mCaseCheckBox = cs;
			mWordCheckBox = wo;
			mBackCheckBox = bs;
			this->SetBaloonHelp("Find the next occurence of the highlighted string");
		}

	protected:

		virtual void Execute()
		{
			mEditor->FindNext(mText->GetText(),mCaseCheckBox->IsChecked(),mWordCheckBox->IsChecked(),!mBackCheckBox->IsChecked());
		}

		iggDialogScriptDebuggerEditor *mEditor;
		FindText *mText;
		FindCheckBox *mCaseCheckBox, *mWordCheckBox, *mBackCheckBox;
	};


	class FindDialog : public iggDialog
	{
		
	public:

		FindDialog(iggDialogScriptDebuggerEditor *ed, iggMenuWindow *mw) : iggDialog(mw,DialogFlag::Modal,iImageFactory::FindIcon("searchfind.png"),"Find",0,2)
		{
			FindText *t = new FindText(mFrame);
			FindCheckBox *cs = new FindCheckBox("Case sensitive",mFrame);
			FindCheckBox *wo = new FindCheckBox("Whole word only",mFrame);
			FindCheckBox *bs = new FindCheckBox("Search backwards",mFrame);
			mFrame->AddLine(t,new FindButton(ed,t,cs,wo,bs,mFrame));
			mFrame->AddLine(cs);
			mFrame->AddLine(wo);
			mFrame->AddLine(bs);
		}
	};


	//
	//  Debugging buttons
	//
	class DebugControlButton : public iggWidgetSimpleButton
	{

	public:

		DebugControlButton(int state, iggDialogScriptDebugger *d, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mDebugger = d; 
			mRunState = state;

			switch(state)
			{
			case _RunStatePaused:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("er_stop.png"));
					this->SetBaloonHelp("Pause the script");
					break;
				}
			case _RunStateExecute:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("moveright.png"));
					this->SetBaloonHelp("Run the script");
					break;
				}
			case _RunStateLineStep:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("debug_linestep.png"));
					this->SetBaloonHelp("Step the script one line at-a-time");
					break;
				}
			case _RunStateFrameStep:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("debug_framestep.png"));
					this->SetBaloonHelp("Step the script one frame at-a-time","This button steps the script one line or one frame at a time. The only difference from the step-one-line-at-a-time button is in executing operators <b>render</b>, <b>render-all</b>, and <b>render-set</b>. With these operators this button executes one frame at a time, while the step-one-line-at-a-time button executes the whole line at once.");
					break;
				}
			default:
				{
					IERROR_FATAL("Incorrectly configured Script Debugger.");
				}
			}
		}

	protected:

		virtual void Execute()
		{
			mDebugger->SetRunState(mRunState);
		}

		int mRunState;
		iggDialogScriptDebugger *mDebugger;
	};


	class DelaySlider : public iggWidget
	{

	public:

		DelaySlider(iggDialogScriptDebugger *d, iggFrame *parent) : iggWidget(parent)
		{
			mDebugger = d; 

			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,true,0,"Delay",0);
			mSubject->SetRange(0,10);
			mSubject->SetValue(3);

			this->SetBaloonHelp("Adjust the time the current line is highlighted");
		}

	protected:

		virtual void OnInt1Body(int v)
		{
			mDebugger->SetDelayTime(0.01f*v*v);
		}

		virtual void UpdateWidgetBody()
		{
		}

		ibgWidgetEntrySubject *mSubject;
		iggDialogScriptDebugger *mDebugger;
	};


	class DummySceneCheckBox : public iggWidgetSimpleCheckBox
	{

	public:

		DummySceneCheckBox(iggDialogScriptDebugger *d, iggFrame *parent) : iggWidgetSimpleCheckBox("Use dummy scene",parent)
		{
			mDebugger = d; 
			this->SetBaloonHelp("Use a dummy object instead of the real scene","If this box is checked, a dummy scene is displayed instead of the real one (to speed-up rendering), and subsequent data file are not loaded but only checked for existence.");
		}

	protected:

		virtual void OnChecked(bool s)
		{
			mDebugger->SetDebugFlag(s?2:1);
		}

		iggDialogScriptDebugger *mDebugger;
	};


	class MarkLineCheckBox : public iggWidgetSimpleCheckBox
	{

	public:

		MarkLineCheckBox(iggDialogScriptDebuggerEditor *ed, iggFrame *parent) : iggWidgetSimpleCheckBox("Mark line after it is executed.",parent)
		{
			mEditor = ed; 
			this->SetBaloonHelp("Toggle whether a current line is marked before or after it is executed","Usually, debuggers mark a line of code to be executed next. If this box is checked, Script Debugger will mark the current line <b>after</b> rather than <b>before</b> it is executed.");
		}

	protected:

		virtual void OnChecked(bool s)
		{
			mEditor->SetMarkBefore(!s);
		}

		iggDialogScriptDebuggerEditor *mEditor;
	};


	class SetBreakPointButton : public iggWidgetSimpleButton
	{

	public:

		SetBreakPointButton(bool set, iggDialogScriptDebuggerEditor *ed, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mIsSet = set;
			mEditor = ed;

			if(mIsSet)
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("debug_setbrpnt.png"));
				this->SetBaloonHelp("Set a breakpoint at cursor's position");
			}
			else
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("debug_delbrpnt.png"));
				this->SetBaloonHelp("Set a breakpoint at cursor's position");
			}
		}

	protected:

		virtual void Execute()
		{
			mEditor->SetBreakPoint(mIsSet);
		}

		bool mIsSet;
		iggDialogScriptDebuggerEditor *mEditor;
	};
}


using namespace iggDialogScriptDebugger_Private;


//
//  Main class
//
iggDialogScriptDebugger::iggDialogScriptDebugger(iggMainWindow *parent) : iggMenuWindow(parent), iScriptObserver(0)
{
	int i;

	//
	//  Create MenuSubject
	//
	this->AttachSubject(iggSubjectFactory::CreateMenuWindowSubject(this,iImageFactory::FindIcon("debug.png"),"Ionization Front Interactive Tool"),1);

	mDebugging = mInsideALine = false;
	mCurrentAnimator = this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator();
	mDelayTime = 0.09f;
	mCurIter = -1;
	mNumIter = 0;
	mDebugFlag = 1;

	mFirstShow = true;

	//
	//  Top frame
	//
	mGlobalFrame->SetPadding(true);

	mEditor = new iggDialogScriptDebuggerEditor(this,mGlobalFrame);
	mGlobalFrame->AddLine(mEditor);

	mDebugFrame = new iggFrame("Debugging tools",mGlobalFrame,2);
	mGlobalFrame->AddLine(mDebugFrame);
	mDebugFrame->Show(false);

	iggFrame *tmp = new iggFrame(mDebugFrame);
	iggFrame *tmp1 = new iggFrame("Controls",tmp,4);

	for(i=0; i<__NumRunStates; i++)
	{
		mControlButtons[i] = new DebugControlButton(i,this,tmp1);
	}
	tmp1->AddLine(mControlButtons[1],mControlButtons[0],mControlButtons[2],mControlButtons[3]);
	tmp->AddLine(tmp1);

//	tmp1 = new iggFrame("Breakpoints",tmp,3);
//	iggWidget *b1 = new SetBreakPointButton(true,ed,tmp1);
//	iggWidget *b2 = new SetBreakPointButton(false,ed,tmp1);
//	b2->Enable(false);
//	ed->AttachBreakPointButton(0,b1);
//	ed->AttachBreakPointButton(1,b2);
//	tmp1->AddLine(b1,b2);
//	tmp1->SetColStretch(2,10);
//	tmp->AddLine(tmp1);

	tmp->AddSpace(10);
	tmp->AddLine(new DelaySlider(this,tmp));
	tmp->AddLine(new DummySceneCheckBox(this,tmp));
#ifdef I_DEBUG
	tmp->AddLine(new MarkLineCheckBox(mEditor,tmp));
#endif

	tmp1 = new iggFrame("Variables",mDebugFrame);
	mVarView = new iggWidgetListView(true,tmp1);
	tmp1->AddLine(mVarView);
	mDebugFrame->AddLine(tmp,tmp1);
	mDebugFrame->SetColStretch(1,10);

	//
	//  Menus
	//
	this->BuildMenus();

	//
	//  Helper dialogs
	//
	mFindDialog = new FindDialog(mEditor,this);

	mTimer = vtkTimerLog::New();

	//
	//  Resize window
	//
	int wg[4];
	this->GetSubject()->GetWindowGeometry(wg);
	wg[2] = 600;
	wg[3] = 600;
	this->GetSubject()->SetWindowGeometry(wg);

#if defined(I_DEBUG) && defined(_WIN32)
//	this->LoadFromFile("C:\\Users\\Nick\\IFRIT\\Base\\test3.ias");
#endif
}


iggDialogScriptDebugger::~iggDialogScriptDebugger()
{
	mTimer->Delete();
	this->CheckModified();
	delete mFindDialog;
}


void iggDialogScriptDebugger::Show(bool s)
{
	if(s && mFirstShow)
	{
		mFirstShow = false;
		int wg1[4], wg2[4];
		this->GetMainWindow()->GetSubject()->GetWindowGeometry(wg1);
		this->GetSubject()->GetWindowGeometry(wg2);
		wg2[0] = wg1[0] + 50;
		wg2[1] = wg1[1] + 100;
		this->GetSubject()->SetWindowGeometry(wg2);
	}
	this->iggMenuWindow::Show(s);
}

inline iAnimator* iggDialogScriptDebugger::GetAnimator()
{
	if(mCurrentAnimator != this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator())
	{
		//
		//  Animator changed, reset
		//
		this->Debug(false);
		mCurrentAnimator = this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator();
		mEditor->iggWidgetTextEditor::SetText(mCurrentAnimator->GetScript()->GetText());
	}
	return mCurrentAnimator;
}


void iggDialogScriptDebugger::Debug(bool s)
{
	if(s == mDebugging) return;

	if(s)
	{
		if(!this->GetAnimator()->GetViewModule()->GetReader()->IsFileAnimatable())
		{
			this->GetMainWindow()->PopupWindow(this->GetGlobalFrame(),"Unable to start debugging:\nan animatable data file must be loaded first.");
			return;
		}
		if(!this->CheckSyntax(true)) return;
	}

	mDebugging = s;
	mDebugFrame->Show(s);

	if(mDebugging)
	{
		mInsideALine = mAllowFrameStep = false;
		this->GetAnimator()->SetDebugFlag(mDebugFlag); // make sure we are current
		this->SetScript(this->GetAnimator()->GetScript());
		mRunState = _RunStatePaused;
		this->GetAnimator()->SetUseScript(true);
		this->GetAnimator()->Start();
		this->UpdateVariables();
		mEditor->MarkLine(this->GetAnimator()->GetScript());
	}
	else
	{
		this->GetAnimator()->Stop();
		this->GetAnimator()->SetDebugFlag(0); // to avoid a loop
		this->SetScript(0);
	}

	this->UpdateAll();
}


void iggDialogScriptDebugger::OnScriptStartBody()
{
}


void iggDialogScriptDebugger::OnScriptStopBody(const iString &error)
{
	if(!error.IsEmpty() && error!="Animation finished.")
	{
		this->GetMainWindow()->PopupWindow(mGlobalFrame,iString("Run time error:\n")+error,PopupWindow::Warning);
	}
	mEditor->ClearSelection();
}


void iggDialogScriptDebugger::OnScriptBeginLineBody(int , const iString &)
{
	mInsideALine = true;
}


void iggDialogScriptDebugger::OnScriptEndLineBody(int , const iString &)
{
	mEditor->MarkLine(this->GetAnimator()->GetScript());
	mInsideALine = false;
}


bool iggDialogScriptDebugger::OnScriptCheckAbortBody(int cur, int num, int lev)
{
	mCurIter = cur;
	mNumIter = num;

	this->UpdateVariables();

	while(mRunState==_RunStateFrameStep && !mAllowFrameStep) this->GetMainWindow()->ProcessEvents();
	mAllowFrameStep = false;

	return false;
}


void iggDialogScriptDebugger::SetRunState(int n)
{
	if(n>=0 && n<__NumRunStates)
	{
		mRunState = n;

		this->UpdateButtons();

		switch(mRunState)
		{
		case _RunStatePaused:
			{
				//
				//  This is automatic
				//
				break;
			}
		case _RunStateExecute:
			{
				while(mDebugging && mRunState==_RunStateExecute && this->GetAnimator()->Continue())
				{
					if(mEditor->IsBreakPoint(this->GetAnimator()->GetScript()->GetThisLine()))
					{
						mRunState = _RunStatePaused;
						this->UpdateButtons();
					}
					if(mDelayTime > 0.0f)
					{
						mTimer->StartTimer();
						do
						{
							this->GetMainWindow()->ProcessEvents();
							mTimer->StopTimer();
						}
						while(mTimer->GetElapsedTime() < mDelayTime);
					}
					this->GetMainWindow()->ProcessEvents();
				}
				if(mRunState==_RunStateExecute || this->GetAnimator()->GetErrorStatus()->IsError()) this->Debug(false);
				break;
			}
		case _RunStateFrameStep:
			{
				mAllowFrameStep = true;
			}
		case _RunStateLineStep:
			{
				if(!mInsideALine)
				{
					if(!this->GetAnimator()->Continue() || this->GetAnimator()->GetErrorStatus()->IsError()) this->Debug(false);
				}
				break;
			}
		}
	}
}


void iggDialogScriptDebugger::SetDebugFlag(int f)
{
	mDebugFlag = f;
	if(mDebugging) this->GetAnimator()->SetDebugFlag(mDebugFlag);
}


bool iggDialogScriptDebugger::CheckSyntax(bool silent)
{
	iAnimatorScript *s = this->GetAnimator()->GetScript();
	if(!s->Compile())
	{
		this->GetMainWindow()->PopupWindow(mGlobalFrame,iString("Syntax error: line ")+iString::FromNumber(s->GetThisLine()+1)+"\n"+s->GetErrorStatus()->Message(),PopupWindow::Error);
		mEditor->ShowError(s);
		return false;
	}
	else
	{
		if(!silent) this->GetMainWindow()->PopupWindow(mGlobalFrame,"Congratulations: script contains no errors!",PopupWindow::Message);
		return true;
	}
}


void iggDialogScriptDebugger::UpdateVariables()
{
	iAnimatorScript *s = this->GetAnimator()->GetScript();

	int i;
	iString s1, w;
	int maxNameLen = 0, maxTypeLen = 0, maxDataLen = 0;

	iArray<const iScriptKit::Value*> list;
	s->QueryVariables(list);

	for(i=0; i<list.Size(); i++)
	{
		if(maxNameLen < list[i]->Name().Length()) maxNameLen = list[i]->Name().Length();
		if(maxTypeLen < list[i]->GetTypeAsText().Length()) maxTypeLen = list[i]->GetTypeAsText().Length();
		if(maxDataLen < list[i]->GetValueAsText().Length()) maxDataLen = list[i]->GetValueAsText().Length();
	}
	while(w.Length() < maxNameLen) w += "..........";
	while(w.Length() < maxTypeLen) w += "..........";
	while(w.Length() < maxDataLen) w += "..........";

	mVarView->Clear();
	for(i=0; i<list.Size(); i++)
	{
		s1 = "<b>" + list[i]->Name() + "</b>" + w.Part(0,maxNameLen-list[i]->Name().Length()) + "..:  ";
		s1 += "<font color=blue>" + list[i]->GetTypeAsText() + "</font>" + w.Part(0,maxTypeLen-list[i]->GetTypeAsText().Length()) + "  ";
		s1 += list[i]->GetValueAsText();
		mVarView->InsertItem(s1);
	}
	if(mNumIter > 0)
	{
		if(mCurIter > 0)
		{
			s1 = "<b><font color=red>Loop count:</font> " + iString::FromNumber(mCurIter) + "</b> (out of <b>" + iString::FromNumber(mNumIter) + "</b>)";
		}
		else
		{
			s1 = "<b><font color=red>Loop is done.</font></b>";
		}
		mVarView->InsertItem(s1);
	}
}


void iggDialogScriptDebugger::UpdateButtons()
{
	iAnimatorScript *s = this->GetAnimator()->GetScript();

	mControlButtons[_RunStatePaused]->Enable(mRunState==_RunStateExecute || (mRunState==_RunStateLineStep && s->IsInRender()));
	mControlButtons[_RunStateExecute]->Enable(mRunState!=_RunStateExecute);
	mControlButtons[_RunStateLineStep]->Enable(mRunState!=_RunStateExecute);
	mControlButtons[_RunStateFrameStep]->Enable(mRunState!=_RunStateExecute);
}


void iggDialogScriptDebugger::UpdateContents()
{
	mGlobalFrame->UpdateWidget();
	this->UpdateButtons();
}


void iggDialogScriptDebugger::CheckModified()
{
	if(mEditor->TextModified())
	{
		switch(mMainWindow->PopupWindow(mGlobalFrame,"The current script is not saved.",PopupWindow::Warning,"Save","Ignore")) 
		{
		case 0: // The user clicked the Save button or pressed Enter
			{ 
				this->SaveToFile(mCurrentFileName);
				break;
			}
		case 1: // The user clicked the Ignore or pressed Escape
			{
				break;
			}
		}
	}
}


void iggDialogScriptDebugger::SetContents(const iString &text)
{
	if(mDebugging) this->Debug(false);

	this->CheckModified();

	mEditor->SetText(text);
	mEditor->SetModified(false);
}


void iggDialogScriptDebugger::LoadFromFile(const iString &filename)
{
	iString fn;

	if(filename.IsEmpty())
	{
		if(mCurrentFileName.IsEmpty())
		{
			fn = this->GetFileName("Open script file",this->GetShell()->GetEnvironment(Environment::Script),"Animation script file (*.ias)");
		}
		else
		{
			fn = this->GetFileName("Open script file",mCurrentFileName,"Animation script file (*.ias)");
		}
	}
	else
	{
		fn = filename;
	}

	if(fn.IsEmpty()) return; 
	if(fn.Part(-4).Lower() != ".ias") fn += ".ias";

	iFile f(fn);
	if(!f.Open(iFile::_Read,iFile::_Text))
	{
		mMainWindow->PopupWindow(mGlobalFrame,"Failed to open a script file.",PopupWindow::Error);
		return;
	}

	iString text, line;
	while(f.ReadLine(line))
	{
		text += line +"\n";
	}
	f.Close();

	this->SetContents(text);
	mCurrentFileName = fn;

	this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->SetScriptFileName(fn);
}


void iggDialogScriptDebugger::SaveToFile(const iString &filename)
{
	iString fn;

	if(filename.IsEmpty())
	{
		if(mCurrentFileName.IsEmpty())
		{
			fn = this->GetFileName("Save script file",this->GetShell()->GetEnvironment(Environment::Script),"Animation script file (*.ias)",false);
		}
		else
		{
			fn = this->GetFileName("Save script file",mCurrentFileName,"Animation script file (*.ias)",false);
		}
	}
	else
	{
		fn = filename;
	}

	if(fn.IsEmpty()) return; 
	if(fn.Part(-4).Lower() != ".ias") fn += ".ias";

	iFile f(fn);
	if(!f.Open(iFile::_Write,iFile::_Text))
	{
		mMainWindow->PopupWindow(mGlobalFrame,"Failed to open a script file.",PopupWindow::Error);
		return;
	}

	iString line;
	int i, n = mEditor->GetNumberOfLines();
	for(i=0; i<n; i++)
	{
		line = mEditor->GetLine(i);
		if(!line.IsEmpty() && line.IsWhiteSpace(line.Length()-1)) line = line.Part(0,line.Length()-1); // weird Qt bug
		if(i<n-1 || !line.IsEmpty()) //  Qt bug
		{
			if(!f.WriteLine(line))
			{
				mMainWindow->PopupWindow(mGlobalFrame,"Error in writing a script file.",PopupWindow::Error);
				f.Close();
				return;
			}
		}
	}
	f.Close();

	mCurrentFileName = fn;
	mEditor->SetModified(false);
}


void iggDialogScriptDebugger::BuildMenus()
{
	//
	//  1. File menu
	//
	this->GetSubject()->BeginMenu("&File",false);
	{
		this->GetSubject()->AddMenuItem(_MenuFileNew,"&New",iImageFactory::FindIcon("filenew.png"),"Ctrl+N",false,false);
		this->GetSubject()->AddMenuItem(_MenuFileOpen,"&Open...",iImageFactory::FindIcon("fileopen.png"),"Ctrl+O",false,false);
		this->GetSubject()->AddMenuItem(_MenuFileSave,"&Save",iImageFactory::FindIcon("filesave.png"),"Ctrl+S",false,false);
		this->GetSubject()->AddMenuItem(_MenuFileSaveAs,"Save &As...",0,"",false,false);
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuFileClose,"&Close",0,"Ctrl+C",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  2. Edit menu
	//
	this->GetSubject()->BeginMenu("&Edit",false);
	{
		if(mEditor->HasEditingFunction(TextEditorFunction::Undo)) this->GetSubject()->AddMenuItem(_MenuEditUndo,"&Undo",iImageFactory::FindIcon("undo.png"),"Ctrl+U",false,false);
		if(mEditor->HasEditingFunction(TextEditorFunction::Redo)) this->GetSubject()->AddMenuItem(_MenuEditRedo,"&Redo",iImageFactory::FindIcon("redo.png"),"Ctrl+R",false,false);
		if(mEditor->HasEditingFunction(TextEditorFunction::Cut)) this->GetSubject()->AddMenuItem(_MenuEditCut,"Cu&t",iImageFactory::FindIcon("editcut.png"),"Ctrl+X",false,false);
		if(mEditor->HasEditingFunction(TextEditorFunction::Copy)) this->GetSubject()->AddMenuItem(_MenuEditCopy,"&Copy",iImageFactory::FindIcon("editcopy.png"),"Ctrl+C",false,false);
		if(mEditor->HasEditingFunction(TextEditorFunction::Paste)) this->GetSubject()->AddMenuItem(_MenuEditPaste,"&Paste",iImageFactory::FindIcon("editpaste.png"),"Ctrl+V",false,false);
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuEditFind,"&Find",iImageFactory::FindIcon("searchfind.png"),"Ctrl+F",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  3. Debug menu
	//
	this->GetSubject()->BeginMenu("&Debug",false);
	{
		this->GetSubject()->AddMenuItem(_MenuDebugSyntax,"&Check syntax",iImageFactory::FindIcon("script_check.png"),"F7",false,false,&iAnimator::KeyDebugging(),true);
		this->GetSubject()->AddMenuItem(_MenuDebugStart,"Start &debugging",iImageFactory::FindIcon("script_run.png"),"F5",false,false,&iAnimator::KeyDebugging(),true);
		this->GetSubject()->AddMenuItem(_MenuDebugStop,"&Stop debugging",iImageFactory::FindIcon("close.png"),"Esc",false,false,&iAnimator::KeyDebugging());
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuDebugToggleBreakPoint,"&Toggle &breakpoint",iImageFactory::FindIcon("debug_setbrpnt.png"),"Ctrl+B",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  4. Help menu
	//
	this->GetSubject()->BeginMenu("&Help",false);
	{
		this->GetSubject()->AddMenuItem(_MenuHelpUsingDebugger,"&Using debugger",0,"",false,false);
		this->GetSubject()->AddMenuItem(_MenuHelpScriptSyntax,"&Script syntax",0,"F1",false,false);
	}
	this->GetSubject()->EndMenu();

	//
	//  Build a tool bar
	//
	this->GetSubject()->AddToolBarButton(_MenuFileNew,"Create a new script");
	this->GetSubject()->AddToolBarButton(_MenuFileOpen,"Open an existing script from a file");
	this->GetSubject()->AddToolBarButton(_MenuFileSave,"Save the current script into a file");
	this->GetSubject()->AddToolBarSeparator();
	if(mEditor->HasEditingFunction(TextEditorFunction::Undo)) this->GetSubject()->AddToolBarButton(_MenuEditUndo,"Undo");
	if(mEditor->HasEditingFunction(TextEditorFunction::Redo)) this->GetSubject()->AddToolBarButton(_MenuEditRedo,"Redo");
	if(mEditor->HasEditingFunction(TextEditorFunction::Cut)) this->GetSubject()->AddToolBarButton(_MenuEditCut,"Cut the current selection");
	if(mEditor->HasEditingFunction(TextEditorFunction::Copy)) this->GetSubject()->AddToolBarButton(_MenuEditCopy,"Copy the current selection to the clipboard");
	if(mEditor->HasEditingFunction(TextEditorFunction::Paste)) this->GetSubject()->AddToolBarButton(_MenuEditPaste,"Paste the contents of the clipboard");
	this->GetSubject()->AddToolBarButton(_MenuEditFind,"Search the script text");
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_MenuDebugSyntax,"Check the script syntax");
	this->GetSubject()->AddToolBarButton(_MenuDebugStart,"Start debugging");
	this->GetSubject()->AddToolBarButton(_MenuDebugStop,"Stop debugging");
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_MenuDebugToggleBreakPoint,"Toggle a breakpoint");
}


void iggDialogScriptDebugger::OnMenuBody(int id, bool on)
{
	static const iString none;
	iString fname;

	if(id < 0)
	{
		IERROR_LOW("Invalid menu item id.");
	}

	if(id <= _MenuFileMax)
	{
		switch(id)
		{
		case _MenuFileNew:
			{
				this->SetContents(none);
			 	mCurrentFileName.Clear();
				break;
			}
		case _MenuFileOpen:
			{
				this->LoadFromFile(none);	
				break;
			}
		case _MenuFileSave:
			{
				this->SaveToFile(mCurrentFileName);
				break;
			}
		case _MenuFileSaveAs:
			{
				this->SaveToFile(none);
				break;
			}
		case _MenuFileClose:
			{
				this->CheckModified();
				this->Show(false);
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuEditMax)
	{
		switch(id)
		{
		case _MenuEditUndo:
		case _MenuEditRedo:
		case _MenuEditCut:
		case _MenuEditCopy:
		case _MenuEditPaste:
			{
				mEditor->UseEditingFunction(id-_MenuEditUndo+TextEditorFunction::Undo);
				break;
			}
		case _MenuEditFind:
			{
				mFindDialog->Show(true);	
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuDebugMax)
	{
		switch(id)
		{
		case _MenuDebugSyntax:
			{
				this->CheckSyntax(false);
				break;
			}
		case _MenuDebugStart:
			{
				this->Debug(true);	
				break;
			}
		case _MenuDebugStop:
			{
				this->Debug(false);	
				break;
			}
		case _MenuDebugToggleBreakPoint:
			{
				mEditor->SetBreakPoint(!mEditor->IsBreakPoint());
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuHelpMax)
	{
		switch(id)
		{
		case _MenuHelpUsingDebugger:
			{
				mMainWindow->GetDialogHelp()->Open("sr.gg.ds");	
				break;
			}
		case _MenuHelpScriptSyntax:
			{
				mMainWindow->GetDialogHelp()->Open("ug.sc.i");	
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	IERROR_LOW("Invalid menu item id.");
}

#endif
