/*==================================================================
 * sfload.c - sound font loading functions
 * Based on the awesfx utility Copyright (C) 1996-1999 Takashi Iwai
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2000 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdarg.h>
#include "smurf.h"
#include "iiwusynth_priv.h"

/*=================================sfload.c========================*/

/*
   functions for loading data from sfont files, with appropriate byte swapping
   on big endian machines. Sfont IDs are not swapped because the ID read is
   equivalent to the matching ID list in memory regardless of LE/BE machine
*/

#define READCHUNK(var,fd)	G_STMT_START {		\
    if (!safe_fread(var, 8, fd))			\
	return(FAIL);					\
    ((SFChunk *)(var))->size = GUINT32_FROM_LE(((SFChunk *)(var))->size);  \
} G_STMT_END
#define READID(var,fd)		G_STMT_START {		\
    if (!safe_fread(var, 4, fd))			\
	return(FAIL);					\
} G_STMT_END
#define READSTR(var,fd)		G_STMT_START {		\
    if (!safe_fread(var, 20, fd))			\
	return(FAIL);					\
    (*var)[20] = '\0';					\
} G_STMT_END
#define READD(var,fd)		G_STMT_START {		\
    guint32 _temp;					\
    if (!safe_fread(&_temp, 4, fd))			\
	return(FAIL);					\
    var = GINT32_FROM_LE(_temp);			\
} G_STMT_END
#define READW(var,fd)		G_STMT_START {		\
    guint16 _temp;					\
    if (!safe_fread(&_temp, 2, fd))			\
	return(FAIL);					\
    var = GINT16_FROM_LE(_temp);			\
} G_STMT_END
#define READB(var,fd)		G_STMT_START {		\
    if (!safe_fread(&var, 1, fd))			\
	return(FAIL);					\
} G_STMT_END
#define FSKIP(size,fd)		G_STMT_START {		\
    if (!safe_fseek(fd, size, SEEK_CUR))		\
	return(FAIL);					\
} G_STMT_END
#define FSKIPW(fd)		G_STMT_START {		\
    if (!safe_fseek(fd, 2, SEEK_CUR))			\
	return(FAIL);					\
} G_STMT_END

/* removes and advances a GSList pointer */
#define SLADVREM(list, item)	G_STMT_START {		\
    GSList *_temp = item;				\
    item = g_slist_next(item);				\
    list = g_slist_remove_link(list, _temp);		\
    g_slist_free_1(_temp);				\
} G_STMT_END

static gint chunkid (guint32 id);
static gint load_body (gint32 size, SFData * sf, FILE * fd);
static gint read_listchunk (SFChunk * chunk, FILE * fd);
static gint process_info (gint32 size, SFData * sf, FILE * fd);
static gint process_sdta (gint32 size, SFData * sf, FILE * fd);
static gint pdtahelper (guint32 expid, guint32 reclen, SFChunk * chunk,
  gint32 * size, FILE * fd);
static gint process_pdta (gint32 size, SFData * sf, FILE * fd);
static gint load_phdr (gint32 size, SFData * sf, FILE * fd);
static gint load_pbag (gint32 size, SFData * sf, FILE * fd);
static gint load_pmod (gint32 size, SFData * sf, FILE * fd);
static gint load_pgen (gint32 size, SFData * sf, FILE * fd);
static gint load_ihdr (gint32 size, SFData * sf, FILE * fd);
static gint load_ibag (gint32 size, SFData * sf, FILE * fd);
static gint load_imod (gint32 size, SFData * sf, FILE * fd);
static gint load_igen (gint32 size, SFData * sf, FILE * fd);
static gint load_shdr (gint32 size, SFData * sf, FILE * fd);
static gint fixup_pgen (SFData * sf);
static gint fixup_igen (SFData * sf);
static gint fixup_sample (SFData * sf);

gchar idlist[] = {
  "RIFFLISTsfbkINFOsdtapdtaifilisngINAMiromiverICRDIENGIPRD"
    "ICOPICMTISFTsnamsmplphdrpbagpmodpgeninstibagimodigenshdr"
};

static gint sdtachunk_size;

/* sound font file load functions */
static gint
chunkid (guint32 id)
{
  gint i;
  guint32 *p;

  p = (guint32 *) & idlist;

  for (i = 0; i < sizeof (idlist) / sizeof (gint); i++, p += 1)
    if (*p == id)
      return (i + 1);

  return (UNKN_ID);
}

SFData *
sfload_file (gchar * fname)
{
  SFData *sf = NULL;
  FILE *fd;
  guint32 fsize;
  gint err = FALSE;

  if (!(fd = fopen (fname, "rb")))
    {
      gerr (Errno, _("Unable to open file \"%s\""), fname);
      return (NULL);
    }

  if (!(sf = safe_malloc (sizeof (SFData))))
    err = TRUE;
  if (!err)
    {
      memset (sf, 0, sizeof (SFData));	/* zero sfdata */
      sf->fname = g_strdup (fname);	/* copy file name */
      sf->sffd = fd;
    }

  /* get size of file */
  if (!err && fseek (fd, 0L, SEEK_END) == -1)
    {				/* seek to end of file */
      err = TRUE;
      gerr (Errno, _("Seek to end of file failed"));
    }
  if (!err && (fsize = ftell (fd)) == -1)
    {				/* position = size */
      err = TRUE;
      gerr (Errno, _("Get end of file position failed"));
    }
  if (!err)
    rewind (fd);

  if (!load_body (fsize, sf, fd))
    err = TRUE;			/* load the sfont */

  if (err)
    {
      if (sf)
	sfont_close (sf);
      return (NULL);
    }

  /* do this last, as most sfont_* routines set it to FALSE */
  sf->up2date = TRUE;		/* sfont in memory is in sync with file */
  sf->beensaved = FALSE;	/* has not been saved yet */

  return (sf);
}

static gint
load_body (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  READCHUNK (&chunk, fd);	/* load RIFF chunk */
  if (chunkid (chunk.id) != RIFF_ID)	/* error if not RIFF */
    return (gerr (ErrFatal, _("Not a RIFF file")));

  READID (&chunk.id, fd);	/* load file ID */
  if (chunkid (chunk.id) != SFBK_ID)	/* error if not SFBK_ID */
    return (gerr (ErrFatal, _("Not a sound font file")));

  if (chunk.size != size - 8)
    return (gerr (ErrCorr, _("Sound font file size mismatch")));

  /* Process INFO block */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != INFO_ID)
    return (gerr (ErrCorr, _("Invalid ID found when expecting INFO chunk")));
  if (!process_info (chunk.size, sf, fd))
    return (FAIL);

  /* Process sample chunk */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != SDTA_ID)
    return (gerr (ErrCorr,
	_("Invalid ID found when expecting SAMPLE chunk")));
  if (!process_sdta (chunk.size, sf, fd))
    return (FAIL);

  /* process HYDRA chunk */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != PDTA_ID)
    return (gerr (ErrCorr, _("Invalid ID found when expecting HYDRA chunk")));
  if (!process_pdta (chunk.size, sf, fd))
    return (FAIL);

  if (!fixup_pgen (sf))
    return (FAIL);
  if (!fixup_igen (sf))
    return (FAIL);
  if (!fixup_sample (sf))
    return (FAIL);

  /* sort preset list by bank, preset # */
  sf->preset = g_slist_sort (sf->preset,
    (GCompareFunc) sfont_preset_compare_func);

  return (OK);
}

static gint
read_listchunk (SFChunk * chunk, FILE * fd)
{
  READCHUNK (chunk, fd);	/* read list chunk */
  if (chunkid (chunk->id) != LIST_ID)	/* error if ! list chunk */
    return (gerr (ErrCorr, _("Invalid chunk id in level 0 parse")));
  READID (&chunk->id, fd);	/* read id string */
  chunk->size -= 4;
  return (OK);
}

static gint
process_info (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;
  guint8 id;
  gchar *item;
  guint16 ver;

  while (size > 0)
    {
      READCHUNK (&chunk, fd);
      size -= 8;

      id = chunkid (chunk.id);

      if (id == IFIL_ID)
	{			/* sound font version chunk? */
	  if (chunk.size != 4)
	    return (gerr (ErrCorr,
		_("Sound font version info chunk has invalid size")));

	  READW (ver, fd);
	  sf->version.major = ver;
	  READW (ver, fd);
	  sf->version.minor = ver;

	  if (sf->version.major < 2)
	    return (gerr (ErrFatal,
		_("Sound font version is %d.%d which is not"
		  " supported, convert to version 2.0x"), sf->version.major,
		sf->version.minor));
	  if (sf->version.major > 2)
	    gerr (ErrWarn, _("Sound font version is %d.%d which is newer than"
		" what this version of Smurf was designed for (v2.0x)"));
	}
      else if (id == IVER_ID)
	{			/* ROM version chunk? */
	  if (chunk.size != 4)
	    return (gerr (ErrCorr,
		_("ROM version info chunk has invalid size")));

	  READW (ver, fd);
	  sf->romver.major = ver;
	  READW (ver, fd);
	  sf->romver.minor = ver;
	}
      else if (id != UNKN_ID)
	{
	  if ((id != ICMT_ID && chunk.size > 256) || (chunk.size > 65536)
	    || (chunk.size % 2))
	    return (gerr (ErrCorr,
		_("INFO sub chunk %.4s has invalid chunk size"
		  " of %d bytes"), &chunk.id, chunk.size));

	  /* alloc for chunk id and da chunk */
	  if (!(item = safe_malloc (chunk.size + 1)))
	    return (FAIL);

	  /* attach to INFO list, sfont_close will cleanup if FAIL occurs */
	  sf->info = g_slist_append (sf->info, item);

	  *(guint8 *) item = id;
	  if (!safe_fread (&item[1], chunk.size, fd))
	    return (FAIL);

	  /* force terminate info item (don't forget uint8 info ID) */
	  *(item + chunk.size) = '\0';
	}
      else
	return (gerr (ErrCorr, _("Invalid chunk id in INFO chunk")));
      size -= chunk.size;
    }

  if (size < 0)
    return (gerr (ErrCorr, _("INFO chunk size mismatch")));

  return (OK);
}

static gint
process_sdta (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  if (size == 0)
    return (OK);		/* no sample data? */

  /* read sub chunk */
  READCHUNK (&chunk, fd);
  size -= 8;

  if (chunkid (chunk.id) != SMPL_ID)
    return (gerr (ErrCorr,
	_("Expected SMPL chunk found invalid id instead")));

  if ((size - chunk.size) != 0)
    return (gerr (ErrCorr, _("SDTA chunk size mismatch")));

  /* sample data follows */
  sf->samplepos = ftell (fd);

  /* used in fixup_sample() to check validity of sample headers */
  sdtachunk_size = chunk.size;
  sf->samplesize = chunk.size;

  FSKIP (chunk.size, fd);

  return (OK);
}

static gint
pdtahelper (guint32 expid, guint32 reclen, SFChunk * chunk,
  gint32 * size, FILE * fd)
{
  guint32 id;
  gchar *expstr;

  expstr = CHNKIDSTR (expid);	/* in case we need it */

  READCHUNK (chunk, fd);
  *size -= 8;

  if ((id = chunkid (chunk->id)) != expid)
    return (gerr (ErrCorr, _("Expected"
	  " PDTA sub-chunk \"%.4s\" found invalid id instead"), expstr));

  if (chunk->size % reclen)	/* valid chunk size? */
    return (gerr (ErrCorr,
	_("\"%.4s\" chunk size is not a multiple of %d bytes"), expstr,
	reclen));
  if ((*size -= chunk->size) < 0)
    return (gerr (ErrCorr,
	_("\"%.4s\" chunk size exceeds remaining PDTA chunk size"), expstr));
  return (OK);
}

static gint
process_pdta (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  if (!pdtahelper (PHDR_ID, SFPHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_phdr (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PBAG_ID, SFBAGSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pbag (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PMOD_ID, SFMODSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pmod (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PGEN_ID, SFGENSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pgen (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IHDR_ID, SFIHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_ihdr (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IBAG_ID, SFBAGSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_ibag (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IMOD_ID, SFMODSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_imod (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IGEN_ID, SFGENSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_igen (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (SHDR_ID, SFSHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_shdr (chunk.size, sf, fd))
    return (FAIL);

  return (OK);
}

/* preset header loader */
static gint
load_phdr (gint32 size, SFData * sf, FILE * fd)
{
  gint i, i2;
  SFPreset *p, *pr = NULL;	/* ptr to current & previous preset */
  guint16 zndx, pzndx;

  if (size % SFPHDRSIZE || size == 0)
    return (gerr (ErrCorr, _("Preset header chunk size is invalid")));

  i = size / SFPHDRSIZE - 1;
  if (i == 0)
    {				/* at least one preset + term record */
      gerr (ErrWarn, _("File contains no presets"));
      FSKIP (SFPHDRSIZE, fd);
      return (OK);
    }

  for (; i > 0; i--)
    {				/* load all preset headers */
      p = g_chunk_new (SFPreset, chunk_preset);
      sf->preset = g_slist_append (sf->preset, p);
      p->zone = NULL;		/* In case of failure, sfont_close can cleanup */
      READSTR (&p->name, fd);	/* possible read failure ^ */
      READW (p->prenum, fd);
      READW (p->bank, fd);
      READW (zndx, fd);
      READD (p->libr, fd);
      READD (p->genre, fd);
      READD (p->morph, fd);

      if (pr)
	{			/* not first preset? */
	  if (zndx < pzndx)
	    return (gerr (ErrCorr, _("Preset header indices not monotonic")));
	  i2 = zndx - pzndx;
	  while (i2--)
	    {
	      pr->zone = g_slist_prepend (pr->zone, NULL);
	    }
	}
      else if (zndx > 0)	/* 1st preset, warn if ofs >0 */
	gerr (ErrWarn, _("%d preset zones not referenced, discarding"), zndx);
      pr = p;			/* update preset ptr */
      pzndx = zndx;
    }

  FSKIP (24, fd);
  READW (zndx, fd);		/* Read terminal generator index */
  FSKIP (12, fd);

  if (zndx < pzndx)
    return (gerr (ErrCorr, _("Preset header indices not monotonic")));
  i2 = zndx - pzndx;
  while (i2--)
    {
      pr->zone = g_slist_prepend (pr->zone, NULL);
    }

  return (OK);
}

/* preset bag loader */
static gint
load_pbag (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2;
  SFZone *z, *pz = NULL;
  guint16 genndx, modndx;
  guint16 pgenndx, pmodndx;
  guint16 i;

  if (size % SFBAGSIZE || size == 0)	/* size is multiple of SFBAGSIZE? */
    return (gerr (ErrCorr, _("Preset bag chunk size is invalid")));

  p = sf->preset;
  while (p)
    {				/* traverse through presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse preset's zones */
	  if ((size -= SFBAGSIZE) < 0)
	    return (gerr (ErrCorr, _("Preset bag chunk size mismatch")));
	  z = g_chunk_new (SFZone, chunk_zone);
	  p2->data = z;
	  z->gen = NULL;	/* Init gen and mod before possible failure, */
	  z->mod = NULL;	/* to ensure proper cleanup (sfont_close) */
	  READW (genndx, fd);	/* possible read failure ^ */
	  READW (modndx, fd);
	  z->instsamp = NULL;

	  if (pz)
	    {			/* if not first zone */
	      if (genndx < pgenndx)
		return (gerr (ErrCorr,
		    _("Preset bag generator indices not monotonic")));
	      if (modndx < pmodndx)
		return (gerr (ErrCorr,
		    _("Preset bag modulator indices not monotonic")));
	      i = genndx - pgenndx;
	      while (i--)
		pz->gen = g_slist_prepend (pz->gen, NULL);
	      i = modndx - pmodndx;
	      while (i--)
		pz->mod = g_slist_prepend (pz->mod, NULL);
	    }
	  pz = z;		/* update previous zone ptr */
	  pgenndx = genndx;	/* update previous zone gen index */
	  pmodndx = modndx;	/* update previous zone mod index */
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  size -= SFBAGSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("Preset bag chunk size mismatch")));

  READW (genndx, fd);
  READW (modndx, fd);

  if (!pz)
    {
      if (genndx > 0)
	gerr (ErrWarn, _("No preset generators and terminal index not 0"));
      if (modndx > 0)
	gerr (ErrWarn, _("No preset modulators and terminal index not 0"));
      return (OK);
    }

  if (genndx < pgenndx)
    return (gerr (ErrCorr, _("Preset bag generator indices not monotonic")));
  if (modndx < pmodndx)
    return (gerr (ErrCorr, _("Preset bag modulator indices not monotonic")));
  i = genndx - pgenndx;
  while (i--)
    pz->gen = g_slist_prepend (pz->gen, NULL);
  i = modndx - pmodndx;
  while (i--)
    pz->mod = g_slist_prepend (pz->mod, NULL);

  return (OK);
}

/* preset modulator loader */
static gint
load_pmod (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3;
  SFMod *m;

  p = sf->preset;
  while (p)
    {				/* traverse through all presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse this preset's zones */
	  p3 = ((SFZone *) (p2->data))->mod;
	  while (p3)
	    {			/* load zone's modulators */
	      if ((size -= SFMODSIZE) < 0)
		return (gerr (ErrCorr,
		    _("Preset modulator chunk size mismatch")));
	      m = g_chunk_new (SFMod, chunk_mod);
	      p3->data = m;
	      READW (m->src, fd);
	      READW (m->dest, fd);
	      READW (m->amount, fd);
	      READW (m->amtsrc, fd);
	      READW (m->trans, fd);
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  /*
     If there isn't even a terminal record
     Hmmm, the specs say there should be one, but..
   */
  if (size == 0)
    return (OK);

  size -= SFMODSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("Preset modulator chunk size mismatch")));
  FSKIP (SFMODSIZE, fd);	/* terminal mod */

  return (OK);
}

/* -------------------------------------------------------------------
 * preset generator loader
 * generator (per preset) loading rules:
 * Zones with no generators or modulators shall be annihilated
 * Global zone must be 1st zone, discard additional ones (instrumentless zones)
 *
 * generator (per zone) loading rules (in order of decreasing precedence):
 * KeyRange is 1st in list (if exists), else discard
 * if a VelRange exists only preceded by a KeyRange, else discard
 * if a generator follows an instrument discard it
 * if a duplicate generator exists replace previous one
 * ------------------------------------------------------------------- */
static gint
load_pgen (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3, *dup, **hz;
  SFZone *z;
  SFGen *g;
  SFGenAmount genval;
  guint16 genid;
  gint level, skip, drop, gzone, discarded;

  p = sf->preset;
  while (p)
    {				/* traverse through all presets */
      gzone = FALSE;
      discarded = FALSE;
      p2 = ((SFPreset *) (p->data))->zone;
      if (p2)
	hz = &p2;
      while (p2)
	{			/* traverse preset's zones */
	  level = 0;
	  z = (SFZone *) (p2->data);
	  p3 = z->gen;
	  while (p3)
	    {			/* load zone's generators */
	      dup = NULL;
	      skip = FALSE;
	      drop = FALSE;
	      if ((size -= SFGENSIZE) < 0)
		return (gerr (ErrCorr,
		    _("Preset generator chunk size mismatch")));

	      READW (genid, fd);

	      if (genid == Gen_KeyRange)
		{		/* nothing precedes */
		  if (level == 0)
		    {
		      level = 1;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_VelRange)
		{		/* only KeyRange precedes */
		  if (level <= 1)
		    {
		      level = 2;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_Instrument)
		{		/* inst is last gen */
		  level = 3;
		  READW (genval.uword, fd);
		  GPOINTER_TO_INT (((SFZone *) (p2->data))->instsamp) =
		    genval.uword + 1;
		  break;	/* break out of generator loop */
		}
	      else
		{
		  level = 2;
		  if (gen_validp (genid))
		    {		/* generator valid? */
		      READW (genval.sword, fd);
		      dup = gen_inlist (genid, z->gen);
		    }
		  else
		    skip = TRUE;
		}

	      if (!skip)
		{
		  if (!dup)
		    {		/* if gen ! dup alloc new */
		      g = g_chunk_new (SFGen, chunk_gen);
		      p3->data = g;
		      g->id = genid;
		    }
		  else
		    {
		      g = (SFGen *) (dup->data);	/* ptr to orig gen */
		      drop = TRUE;
		    }
		  g->amount = genval;
		}
	      else
		{		/* Skip this generator */
		  discarded = TRUE;
		  drop = TRUE;
		  FSKIPW (fd);
		}

	      if (!drop)
		p3 = g_slist_next (p3);	/* next gen */
	      else
		SLADVREM (z->gen, p3);	/* drop place holder */

	    }			/* generator loop */

	  if (level == 3)
	    SLADVREM (z->gen, p3);	/* zone has inst? */
	  else
	    {			/* congratulations its a global zone */
	      if (!gzone)
		{		/* Prior global zones? */
		  gzone = TRUE;

		  /* if global zone is not 1st zone, relocate */
		  if (*hz != p2)
		    {
		      gpointer save = p2->data;
		      gerr (ErrWarn,
			_("Preset \"%s\": Global zone is not first zone"),
			((SFPreset *) (p->data))->name);
		      SLADVREM (*hz, p2);
		      *hz = g_slist_prepend (*hz, save);
		      continue;
		    }
		}
	      else
		{		/* previous global zone exists, discard */
		  gerr (ErrWarn,
		    _("Preset \"%s\": Discarding invalid global zone"),
		    ((SFPreset *) (p->data))->name);
		  sfont_zone_delete (sf, hz, (SFZone *) (p2->data));
		}
	    }

	  while (p3)
	    {			/* Kill any zones following an instrument */
	      discarded = TRUE;
	      if ((size -= SFGENSIZE) < 0)
		return (gerr (ErrCorr,
		    _("Preset generator chunk size mismatch")));
	      FSKIP (SFGENSIZE, fd);
	      SLADVREM (z->gen, p3);
	    }

	  p2 = g_slist_next (p2);	/* next zone */
	}
      if (discarded)
	gerr (ErrWarn,
	  _("Preset \"%s\": Some invalid generators were discarded"),
	  ((SFPreset *) (p->data))->name);
      p = g_slist_next (p);
    }

  /* in case there isn't a terminal record */
  if (size == 0)
    return (OK);

  size -= SFGENSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("Preset generator chunk size mismatch")));
  FSKIP (SFGENSIZE, fd);	/* terminal gen */

  return (OK);
}

/* instrument header loader */
static gint
load_ihdr (gint32 size, SFData * sf, FILE * fd)
{
  gint i, i2;
  SFInst *p, *pr = NULL;	/* ptr to current & previous instrument */
  guint16 zndx, pzndx;

  if (size % SFIHDRSIZE || size == 0)	/* chunk size is valid? */
    return (gerr (ErrCorr, _("Instrument header has invalid size")));

  size = size / SFIHDRSIZE - 1;
  if (size == 0)
    {				/* at least one preset + term record */
      gerr (ErrWarn, _("File contains no instruments"));
      FSKIP (SFIHDRSIZE, fd);
      return (OK);
    }

  for (i = 0; i < size; i++)
    {				/* load all instrument headers */
      p = g_chunk_new (SFInst, chunk_inst);
      sf->inst = g_slist_append (sf->inst, p);
      p->zone = NULL;		/* For proper cleanup if fail (sfont_close) */
      READSTR (&p->name, fd);	/* Possible read failure ^ */
      READW (zndx, fd);

      if (pr)
	{			/* not first instrument? */
	  if (zndx < pzndx)
	    return (gerr (ErrCorr,
		_("Instrument header indices not monotonic")));
	  i2 = zndx - pzndx;
	  while (i2--)
	    pr->zone = g_slist_prepend (pr->zone, NULL);
	}
      else if (zndx > 0)	/* 1st inst, warn if ofs >0 */
	gerr (ErrWarn, _("%d instrument zones not referenced, discarding"),
	  zndx);
      pzndx = zndx;
      pr = p;			/* update instrument ptr */
    }

  FSKIP (20, fd);
  READW (zndx, fd);

  if (zndx < pzndx)
    return (gerr (ErrCorr, _("Instrument header indices not monotonic")));
  i2 = zndx - pzndx;
  while (i2--)
    pr->zone = g_slist_prepend (pr->zone, NULL);

  return (OK);
}

/* instrument bag loader */
static gint
load_ibag (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2;
  SFZone *z, *pz = NULL;
  guint16 genndx, modndx, pgenndx, pmodndx;
  gint i;

  if (size % SFBAGSIZE || size == 0)	/* size is multiple of SFBAGSIZE? */
    return (gerr (ErrCorr, _("Instrument bag chunk size is invalid")));

  p = sf->inst;
  while (p)
    {				/* traverse through inst */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* load this inst's zones */
	  if ((size -= SFBAGSIZE) < 0)
	    return (gerr (ErrCorr, _("Instrument bag chunk size mismatch")));
	  z = g_chunk_new (SFZone, chunk_zone);
	  p2->data = z;
	  z->gen = NULL;	/* In case of failure, */
	  z->mod = NULL;	/* sfont_close can clean up */
	  READW (genndx, fd);	/* READW = possible read failure */
	  READW (modndx, fd);
	  z->instsamp = NULL;

	  if (pz)
	    {			/* if not first zone */
	      if (genndx < pgenndx)
		return (gerr (ErrCorr,
		    _("Instrument generator indices not monotonic")));
	      if (modndx < pmodndx)
		return (gerr (ErrCorr,
		    _("Instrument modulator indices not monotonic")));
	      i = genndx - pgenndx;
	      while (i--)
		pz->gen = g_slist_prepend (pz->gen, NULL);
	      i = modndx - pmodndx;
	      while (i--)
		pz->mod = g_slist_prepend (pz->mod, NULL);
	    }
	  pz = z;		/* update previous zone ptr */
	  pgenndx = genndx;
	  pmodndx = modndx;
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  size -= SFBAGSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("Instrument chunk size mismatch")));

  READW (genndx, fd);
  READW (modndx, fd);

  if (!pz)
    {				/* in case that all are no zoners */
      if (genndx > 0)
	gerr (ErrWarn,
	  _("No instrument generators and terminal index not 0"));
      if (modndx > 0)
	gerr (ErrWarn,
	  _("No instrument modulators and terminal index not 0"));
      return (OK);
    }

  if (genndx < pgenndx)
    return (gerr (ErrCorr, _("Instrument generator indices not monotonic")));
  if (modndx < pmodndx)
    return (gerr (ErrCorr, _("Instrument modulator indices not monotonic")));
  i = genndx - pgenndx;
  while (i--)
    pz->gen = g_slist_prepend (pz->gen, NULL);
  i = modndx - pmodndx;
  while (i--)
    pz->mod = g_slist_prepend (pz->mod, NULL);

  return (OK);
}

/* instrument modulator loader */
static gint
load_imod (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3;
  SFMod *m;

  p = sf->inst;
  while (p)
    {				/* traverse through all inst */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* traverse this inst's zones */
	  p3 = ((SFZone *) (p2->data))->mod;
	  while (p3)
	    {			/* load zone's modulators */
	      if ((size -= SFMODSIZE) < 0)
		return (gerr (ErrCorr,
		    _("Instrument modulator chunk size mismatch")));
	      m = g_chunk_new (SFMod, chunk_mod);
	      p3->data = m;
	      READW (m->src, fd);
	      READW (m->dest, fd);
	      READW (m->amount, fd);
	      READW (m->amtsrc, fd);
	      READW (m->trans, fd);
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  /*
     If there isn't even a terminal record
     Hmmm, the specs say there should be one, but..
   */
  if (size == 0)
    return (OK);

  size -= SFMODSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("Instrument modulator chunk size mismatch")));
  FSKIP (SFMODSIZE, fd);	/* terminal mod */

  return (OK);
}

/* load instrument generators (see load_pgen for loading rules) */
static gint
load_igen (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3, *dup, **hz;
  SFZone *z;
  SFGen *g;
  SFGenAmount genval;
  guint16 genid;
  gint level, skip, drop, gzone, discarded;

  p = sf->inst;
  while (p)
    {				/* traverse through all instruments */
      gzone = FALSE;
      discarded = FALSE;
      p2 = ((SFInst *) (p->data))->zone;
      if (p2)
	hz = &p2;
      while (p2)
	{			/* traverse this instrument's zones */
	  level = 0;
	  z = (SFZone *) (p2->data);
	  p3 = z->gen;
	  while (p3)
	    {			/* load zone's generators */
	      dup = NULL;
	      skip = FALSE;
	      drop = FALSE;
	      if ((size -= SFGENSIZE) < 0)
		return (gerr (ErrCorr, _("IGEN chunk size mismatch")));

	      READW (genid, fd);

	      if (genid == Gen_KeyRange)
		{		/* nothing precedes */
		  if (level == 0)
		    {
		      level = 1;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_VelRange)
		{		/* only KeyRange precedes */
		  if (level <= 1)
		    {
		      level = 2;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_SampleId)
		{		/* sample is last gen */
		  level = 3;
		  READW (genval.uword, fd);
		  GPOINTER_TO_INT (((SFZone *) (p2->data))->instsamp) =
		    genval.uword + 1;
		  break;	/* break out of generator loop */
		}
	      else
		{
		  level = 2;
		  if (gen_valid (genid))
		    {		/* gen valid? */
		      READW (genval.sword, fd);
		      dup = gen_inlist (genid, z->gen);
		    }
		  else
		    skip = TRUE;
		}

	      if (!skip)
		{
		  if (!dup)
		    {		/* if gen ! dup alloc new */
		      g = g_chunk_new (SFGen, chunk_gen);
		      p3->data = g;
		      g->id = genid;
		    }
		  else
		    {
		      g = (SFGen *) (dup->data);
		      drop = TRUE;
		    }
		  g->amount = genval;
		}
	      else
		{		/* skip this generator */
		  discarded = TRUE;
		  drop = TRUE;
		  FSKIPW (fd);
		}

	      if (!drop)
		p3 = g_slist_next (p3);	/* next gen */
	      else
		SLADVREM (z->gen, p3);

	    }			/* generator loop */

	  if (level == 3)
	    SLADVREM (z->gen, p3);	/* zone has sample? */
	  else
	    {			/* its a global zone */
	      if (!gzone)
		{
		  gzone = TRUE;

		  /* if global zone is not 1st zone, relocate */
		  if (*hz != p2)
		    {
		      gpointer save = p2->data;
		      gerr (ErrWarn,
			_("Instrument \"%s\": Global zone is not first zone"),
			((SFPreset *) (p->data))->name);
		      SLADVREM (*hz, p2);
		      *hz = g_slist_prepend (*hz, save);
		      continue;
		    }
		}
	      else
		{		/* previous global zone exists, discard */
		  gerr (ErrWarn,
		    _("Instrument \"%s\": Discarding invalid global zone"),
		    ((SFInst *) (p->data))->name);
		  sfont_zone_delete (sf, hz, (SFZone *) (p2->data));
		}
	    }

	  while (p3)
	    {			/* Kill any zones following a sample */
	      discarded = TRUE;
	      if ((size -= SFGENSIZE) < 0)
		return (gerr (ErrCorr,
		    _("Instrument generator chunk size mismatch")));
	      FSKIP (SFGENSIZE, fd);
	      SLADVREM (z->gen, p3);
	    }

	  p2 = g_slist_next (p2);	/* next zone */
	}
      if (discarded)
	gerr (ErrWarn,
	  _("Instrument \"%s\": Some invalid generators were discarded"),
	  ((SFInst *) (p->data))->name);
      p = g_slist_next (p);
    }

  /* for those non-terminal record cases, grr! */
  if (size == 0)
    return (OK);

  size -= SFGENSIZE;
  if (size != 0)
    return (gerr (ErrCorr, _("IGEN chunk size mismatch")));
  FSKIP (SFGENSIZE, fd);	/* terminal gen */

  return (OK);
}

/* sample header loader */
static gint
load_shdr (gint32 size, SFData * sf, FILE * fd)
{
  guint32 i;
  SFSample *p;

  if (size % SFSHDRSIZE || size == 0)	/* size is multiple of SHDR size? */
    return (gerr (ErrCorr, _("Sample header has invalid size")));

  size = size / SFSHDRSIZE - 1;
  if (size == 0)
    {				/* at least one sample + term record? */
      gerr (ErrWarn, _("File contains no samples"));
      FSKIP (SFSHDRSIZE, fd);
      return (OK);
    }

  /* load all sample headers */
  for (i = 0; i < size; i++)
    {
      p = g_chunk_new (SFSample, chunk_sample);
      sf->sample = g_slist_append (sf->sample, p);
      READSTR (&p->name, fd);
      READD (p->start, fd);
      READD (p->end, fd);	/* - end, loopstart and loopend */
      READD (p->loopstart, fd);	/* - will be checked and turned into */
      READD (p->loopend, fd);	/* - offsets in fixup_sample() */
      READD (p->samplerate, fd);
      READB (p->origpitch, fd);
      READB (p->pitchadj, fd);
      FSKIPW (fd);		/* skip sample link */
      READW (p->sampletype, fd);
      p->samfile = 0;
    }

  FSKIP (SFSHDRSIZE, fd);	/* skip terminal shdr */

  return (OK);
}

/* "fixup" (inst # -> inst ptr) instrument references in preset list */
static gint
fixup_pgen (SFData * sf)
{
  GSList *p, *p2, *p3;
  SFZone *z;
  gint i;

  p = sf->preset;
  while (p)
    {
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse this preset's zones */
	  z = (SFZone *) (p2->data);
	  if ((i = GPOINTER_TO_INT (z->instsamp)))
	    {			/* load instrument # */
	      p3 = g_slist_nth (sf->inst, i - 1);
	      if (!p3)
		return (gerr (ErrCorr,
		    _("Preset %03d %03d: Invalid instrument reference"),
		    ((SFPreset *) (p->data))->bank,
		    ((SFPreset *) (p->data))->prenum));
	      z->instsamp = p3;
	    }
	  else
	    z->instsamp = NULL;
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  return (OK);
}

/* "fixup" (sample # -> sample ptr) sample references in instrument list */
static gint
fixup_igen (SFData * sf)
{
  GSList *p, *p2, *p3;
  SFZone *z;
  gint i;

  p = sf->inst;
  while (p)
    {
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* traverse instrument's zones */
	  z = (SFZone *) (p2->data);
	  if ((i = GPOINTER_TO_INT (z->instsamp)))
	    {			/* load sample # */
	      p3 = g_slist_nth (sf->sample, i - 1);
	      if (!p3)
		return (gerr (ErrCorr,
		    _("Instrument \"%s\": Invalid sample reference"),
		    ((SFInst *) (p->data))->name));
	      z->instsamp = p3;
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  return (OK);
}

/* convert sample end, loopstart and loopend to offsets and check if valid */
static gint
fixup_sample (SFData * sf)
{
  GSList *p;
  SFSample *sam;

  p = sf->sample;
  while (p)
    {
      sam = (SFSample *) (p->data);

      /* if sample is not a ROM sample and end is over the sample data chunk
         or sam start is greater than 4 less than the end (at least 4 samples) */
      if ((!(sam->sampletype & SF_SAMPLETYPE_ROM)
	  && sam->end > sdtachunk_size) || sam->start > (sam->end - 4))
	{
	  gerr (ErrWarn, _("Sample '%s' start/end file positions are invalid,"
	      " disabling and will not be saved"), sam->name);

	  /* disable sample by setting all sample markers to 0 */
	  sam->start = sam->end = sam->loopstart = sam->loopend = 0;

	  return (OK);
	}
      else if (sam->loopend > sam->end || sam->loopstart >= sam->loopend
	|| sam->loopstart <= sam->start)
	{			/* loop is fowled?? (cluck cluck :) */
	  /* can pad loop by 8 samples and ensure at least 4 for loop (2*8+4) */
	  if ((sam->end - sam->start) >= 20)
	    {
	      sam->loopstart = sam->start + 8;
	      sam->loopend = sam->end - 8;
	    }
	  else
	    {			/* loop is fowled, sample is tiny (can't pad 8 samples) */
	      sam->loopstart = sam->start + 1;
	      sam->loopend = sam->end - 1;
	    }
	}

      /* convert sample end, loopstart, loopend to offsets from sam->start */
      sam->end -= sam->start + 1;	/* marks last sample, contrary to SF spec. */
      sam->loopstart -= sam->start;
      sam->loopend -= sam->start;

      p = g_slist_next (p);
    }

  return (OK);
}

/*=================================sfont.c========================*/


/* optimum chunk area sizes (could be more optimum) */
#define PRESET_CHUNK_OPTIMUM_AREA	256
#define INST_CHUNK_OPTIMUM_AREA		256
#define SAMPLE_CHUNK_OPTIMUM_AREA	256
#define ZONE_CHUNK_OPTIMUM_AREA		256
#define MOD_CHUNK_OPTIMUM_AREA		256
#define GEN_CHUNK_OPTIMUM_AREA		256

/* Memory chunk pointers */
GMemChunk *chunk_preset;
GMemChunk *chunk_inst;
GMemChunk *chunk_sample;
GMemChunk *chunk_zone;
GMemChunk *chunk_mod;
GMemChunk *chunk_gen;

static gboolean sfont_chunks_inited = FALSE;	/* Have chunks been initialized? */


guint16 badgen[] = { Gen_Unused1, Gen_Unused2, Gen_Unused3, Gen_Unused4,
  Gen_Reserved1, Gen_Reserved2, Gen_Reserved3, 0
};

guint16 badpgen[] = { Gen_StartAddrOfs, Gen_EndAddrOfs, Gen_StartLoopAddrOfs,
  Gen_EndLoopAddrOfs, Gen_StartAddrCoarseOfs, Gen_EndAddrCoarseOfs,
  Gen_StartLoopAddrCoarseOfs, Gen_Keynum, Gen_Velocity,
  Gen_EndLoopAddrCoarseOfs, Gen_SampleModes, Gen_ExclusiveClass,
  Gen_OverrideRootKey, 0
};

/*----- General sfont routines -----*/

/* initialize sound font mem chunks */
void
sfont_init_chunks (void)
{
  if (sfont_chunks_inited)
    return;
  chunk_preset =
    g_mem_chunk_create (SFPreset, PRESET_CHUNK_OPTIMUM_AREA,
    G_ALLOC_AND_FREE);
  chunk_inst =
    g_mem_chunk_create (SFInst, INST_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_sample =
    g_mem_chunk_create (SFSample, SAMPLE_CHUNK_OPTIMUM_AREA,
    G_ALLOC_AND_FREE);
  chunk_zone =
    g_mem_chunk_create (SFZone, ZONE_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_mod =
    g_mem_chunk_create (SFMod, MOD_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_gen =
    g_mem_chunk_create (SFGen, GEN_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);

  sfont_chunks_inited = TRUE;
}

void
sfont_close (SFData * sf)
{
  if (sf->sffd)
    fclose (sf->sffd);

  sfont_free_data (sf);
}

/* delete a sound font structure */
void
sfont_free_data (SFData * sf)
{
  GSList *p, *p2;

  if (sf->fname)
    free (sf->fname);

  p = sf->info;
  while (p)
    {
      free (p->data);
      p = g_slist_next (p);
    }
  g_slist_free (sf->info);
  sf->info = NULL;

  p = sf->preset;
  while (p)
    {				/* loop over presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* loop over preset's zones */
	  sfont_free_zone (p2->data);
	  p2 = g_slist_next (p2);
	}			/* free preset's zone list */
      g_slist_free (((SFPreset *) (p->data))->zone);
      g_mem_chunk_free (chunk_preset, p->data);	/* free preset chunk */
      p = g_slist_next (p);
    }
  g_slist_free (sf->preset);
  sf->preset = NULL;

  p = sf->inst;
  while (p)
    {				/* loop over instruments */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* loop over inst's zones */
	  sfont_free_zone (p2->data);
	  p2 = g_slist_next (p2);
	}			/* free inst's zone list */
      g_slist_free (((SFInst *) (p->data))->zone);
      g_mem_chunk_free (chunk_inst, p->data);	/* free inst chunk */
      p = g_slist_next (p);
    }
  g_slist_free (sf->inst);
  sf->inst = NULL;

  p = sf->sample;
  while (p)
    {
      g_mem_chunk_free (chunk_sample, p->data);
      p = g_slist_next (p);
    }
  g_slist_free (sf->sample);
  sf->sample = NULL;
}

/* free all elements of a zone (Preset or Instrument) */
void
sfont_free_zone (SFZone * zone)
{
  GSList *p;

  if (!zone)
    return;

  p = zone->gen;
  while (p)
    {				/* Free gen chunks for this zone */
      if (p->data)
	g_mem_chunk_free (chunk_gen, p->data);
      p = g_slist_next (p);
    }
  g_slist_free (zone->gen);	/* free genlist */

  p = zone->mod;
  while (p)
    {				/* Free mod chunks for this zone */
      if (p->data)
	g_mem_chunk_free (chunk_mod, p->data);
      p = g_slist_next (p);
    }
  g_slist_free (zone->mod);	/* free modlist */

  g_mem_chunk_free (chunk_zone, zone);	/* free zone chunk */
}

/* preset sort function, first by bank, then by preset # */
gint
sfont_preset_compare_func (gconstpointer a, gconstpointer b)
{
  gint32 aval, bval;

  aval = (gint32) (((SFPreset *) a)->bank) << 16 | ((SFPreset *) a)->prenum;
  bval = (gint32) (((SFPreset *) b)->bank) << 16 | ((SFPreset *) b)->prenum;

  return (aval - bval);
}

/* delete zone from zone list */
void
sfont_zone_delete (SFData * sf, GSList ** zlist, SFZone * zone)
{
  *zlist = g_slist_remove (*zlist, (gpointer) zone);
  sfont_free_zone (zone);

  sf->up2date = FALSE;
}

/* Find generator in gen list */
GSList *
gen_inlist (gint gen, GSList * genlist)
{				/* is generator in gen list? */
  GSList *p;

  p = genlist;
  while (p)
    {
      if (p->data == NULL)
	return (NULL);
      if (gen == ((SFGen *) p->data)->id)
	break;
      p = g_slist_next (p);
    }
  return (p);
}

/* check validity of instrument generator */
gint
gen_valid (gint gen)
{				/* is generator id valid? */
  gint i = 0;

  if (gen > Gen_MaxValid)
    return (FALSE);
  while (badgen[i] && badgen[i] != gen)
    i++;
  return (badgen[i] == 0);
}

/* check validity of preset generator */
gint
gen_validp (gint gen)
{				/* is preset generator valid? */
  gint i = 0;

  if (!gen_valid (gen))
    return (FALSE);
  while (badpgen[i] && badpgen[i] != (guint16) gen)
    i++;
  return (badpgen[i] == 0);
}

/*================================util.c===========================*/

/* Logging function, returns FAIL to use as a return value in calling funcs */
gint
gerr (gint ev, gchar * fmt, ...)
{
  va_list args;

  va_start (args, fmt);
  vprintf(fmt, args);
  va_end (args);

  printf("\n");

  return (FAIL);
}

gint
safe_fread (void *buf, gint count, FILE * fd)
{
  if (fread (buf, count, 1, fd) != 1)
    {				// size_t = count, nmemb = 1
      if (feof (fd))
	gerr (ErrEof, _("EOF while attemping to read %d bytes"), count);
      else
	gerr (Errno, _("File read failed"));
      return (FAIL);
    }
  return (OK);
}

gint
safe_fseek (FILE * fd, long ofs, gint whence)
{
  if (fseek (fd, ofs, whence) == -1)
    return (gerr (Errno, _("File seek failed with offset = %ld and"
	  " whence = %d"), ofs, whence));
  return (OK);
}

void *
safe_malloc (size_t size)
{
  void *ptr;

  if (!(ptr = malloc (size)))
    gerr (ErrMem, _("Attempted to allocate %d bytes"), (gint) size);
  return (ptr);
}



/*=================================glib========================*/

/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GLib Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GLib Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GLib at ftp://ftp.gtk.org/pub/gtk/. 
 */

/*--------------------------------gmem.c--------------------------*/

typedef struct _GRealMemChunk  GRealMemChunk;

struct _GRealMemChunk
{
  guint atom_size;           /* the size of an atom */
};


gpointer
g_mem_chunk_alloc (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  rmem_chunk = (GRealMemChunk*) mem_chunk;
  return IIWU_MALLOC(rmem_chunk->atom_size);
}


void
g_mem_chunk_free (GMemChunk *mem_chunk,
		  gpointer   mem)
{
  IIWU_FREE(mem);
}

GMemChunk*
g_mem_chunk_new (gchar  *name,
		 gint    atom_size,
		 gulong  area_size,
		 gint    type)
{
  GRealMemChunk *mem_chunk;
  mem_chunk = g_new (struct _GRealMemChunk, 1);
  mem_chunk->atom_size = atom_size;
  return ((GMemChunk*) mem_chunk);
}

/*--------------------------------gslist.c--------------------------*/

GSList*
g_slist_alloc (void)
{
  GSList* list;
  list = (GSList*) IIWU_MALLOC(sizeof(GSList));
  list->data = NULL;
  list->next = NULL;
  return list;
}

void
g_slist_free (GSList *list)
{
  GSList *next;
  while (list)
    {
      next = list->next;
      IIWU_FREE(list);
      list = next;
    }
}

void
g_slist_free_1 (GSList *list)
{
  if (list)
    {
      IIWU_FREE(list);
    }
}

GSList*
g_slist_append (GSList   *list,
		gpointer  data)
{
  GSList *new_list;
  GSList *last;

  new_list = g_slist_alloc ();
  new_list->data = data;

  if (list)
    {
      last = g_slist_last (list);
      /* g_assert (last != NULL); */
      last->next = new_list;

      return list;
    }
  else
      return new_list;
}

GSList*
g_slist_prepend (GSList   *list,
		 gpointer  data)
{
  GSList *new_list;

  new_list = g_slist_alloc ();
  new_list->data = data;
  new_list->next = list;

  return new_list;
}

GSList*
g_slist_nth (GSList *list,
	     guint   n)
{
  while ((n-- > 0) && list)
    list = list->next;

  return list;
}

GSList*
g_slist_remove (GSList   *list,
		gpointer  data)
{
  GSList *tmp;
  GSList *prev;

  prev = NULL;
  tmp = list;

  while (tmp)
    {
      if (tmp->data == data)
	{
	  if (prev)
	    prev->next = tmp->next;
	  if (list == tmp)
	    list = list->next;

	  tmp->next = NULL;
	  g_slist_free (tmp);

	  break;
	}

      prev = tmp;
      tmp = tmp->next;
    }

  return list;
}

GSList*
g_slist_remove_link (GSList *list,
		     GSList *link)
{
  GSList *tmp;
  GSList *prev;

  prev = NULL;
  tmp = list;

  while (tmp)
    {
      if (tmp == link)
	{
	  if (prev)
	    prev->next = tmp->next;
	  if (list == tmp)
	    list = list->next;

	  tmp->next = NULL;
	  break;
	}

      prev = tmp;
      tmp = tmp->next;
    }

  return list;
}

static GSList* 
g_slist_sort_merge  (GSList      *l1, 
		     GSList      *l2,
		     GCompareFunc compare_func)
{
  GSList list, *l;

  l=&list;

  while (l1 && l2)
    {
      if (compare_func(l1->data,l2->data) < 0)
        {
	  l=l->next=l1;
	  l1=l1->next;
        } 
      else 
	{
	  l=l->next=l2;
	  l2=l2->next;
        }
    }
  l->next= l1 ? l1 : l2;
  
  return list.next;
}

GSList* 
g_slist_sort (GSList       *list,
	      GCompareFunc compare_func)
{
  GSList *l1, *l2;

  if (!list) 
    return NULL;
  if (!list->next) 
    return list;

  l1 = list; 
  l2 = list->next;

  while ((l2 = l2->next) != NULL)
    {
      if ((l2 = l2->next) == NULL) 
	break;
      l1=l1->next;
    }
  l2 = l1->next; 
  l1->next = NULL;

  return g_slist_sort_merge (g_slist_sort (list, compare_func),
			     g_slist_sort (l2,   compare_func),
			     compare_func);
}


GSList*
g_slist_last (GSList *list)
{
  if (list)
    {
      while (list->next)
	list = list->next;
    }

  return list;
}


/*--------------------------------gstrfuncs.c--------------------------*/

gchar*
g_strdup (const gchar *str)
{
  gchar *new_str;

  if (str)
    {
      new_str = g_new (char, strlen (str) + 1);
      strcpy (new_str, str);
    }
  else
    new_str = NULL;

  return new_str;
}
