/***************************************************************************
                         camera.h  -  description
                            -------------------
   begin                : Thu Nov 2 2000
   copyright            : (C) 2000 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CAMERA_H
#define CAMERA_H

#include "entity.h"

/**
  *@author Jon Anderson
  */

class Camera : public Entity
{
   public:
      Camera( const Vector4 &pos, const Vector4 &look_at, const Vector3 &up,
              double near, double far, double fov, double right = 0 );
      Camera( const Vector4 &pos, const Vector3 &dir, const Vector3 &updir,
              double near, double far, double fov, double right = 0 );

      /**
       * Sets up the Modelview matrix using the camera's position and orientation
       */
      void setupModelview();

      /**
       * Sets up the Projection matrix using camera's configuration
       */
      void setupProjection();

      /**
       * Change the camera's configuration to reflect the shape of the viewport
       */
      void setViewport( int x, int y, int width, int height );

      /**
       * Zoom the camera by 'amount'. This involves narowing/wideing of the
       * camera's field of view.
       */
      void zoom( double amount );

      /**
       * Advances the camera by the specified amount along the axes specified
       * by the camera's current orientation.
       */
      void advance( double x, double y, double z );

      /**
       * Orbit around the point (x,y,z). The camera will move on the surface of
       * a sphere given by the point (x,y,z) and the radius given by the distance
       * from this point to the camera's position. This method will advance
       * the camera by 'up' along the current longitude line and by 'right'
       * along the current latitude line. Note that the latitude circles are
       * always perpendicular to the (0,1,0) vector.
       */
      void orbit( double up, double right, double x, double y, double z );

      /**
       * Reverse the camera's transformation
       */
      Vector4 unProjectPoint( int x, int y, float z = 0 );

      /**
       * Apply the camera's transformation to a point
       */
      Vector3 projectPoint( const Vector4 &pt );

   private:
      // Private convenience initialization method (should this be public?)
      void setOrientation( const Vector3 &dir, const Vector3 &up );

      // Field of view of the camera. Zero means an orthographic projection.
      double fov;

      // Aspect ratio of the current viewport.
      double aspect;

      // For orthographic projection: right coordinate of the viewing volume.
      // (left = -right, top = aspect/right, bottom = -top)
      double right;

      // Near and far clipping planes.
      double near, far;

      // Information for (un)projecting points
      bool matrices_dirty;
      GLdouble modelview_matrix[ 16 ], projection_matrix[ 16 ];
      GLint viewport[ 4 ];
};

#endif
