/*
 commands.c : IRC bot plugin for irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "bot.h"

static gboolean event_privmsg(gchar *data, SERVER_REC *server, gchar *nick, gchar *address, gpointer kludge)
{
    gchar *params, *target, *msg, *args, *str;

    g_return_val_if_fail(data != NULL, FALSE);

    if (kludge) return TRUE;

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (nick == NULL) nick = "!server!";

    if (*msg == 1 || ischannel(*target))
    {
	g_free(params);
	return TRUE;
    }

    /* private message for us */
    str = g_strconcat("bot command ", msg, NULL);
    args = strchr(str+12, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";

    if (!signal_emit(str, 4, args, server, nick, address))
	signal_emit("default bot command", 4, msg, server, nick, address);
    g_free(str);
    return FALSE;
}

static gboolean botcmd_op(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    CHANNEL_REC *channel;
    USER_REC *user;
    USER_CHAN_REC *userchan;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == '\0')
    {
	/* no password given? .. */
	return TRUE;
    }

    user = botuser_find(nick, address);
    if (user == NULL || (user->not_flags & USER_OP) || !botuser_verify_password(user, data))
    {
	/* not found, can't op with this mask or failed password */
	return TRUE;
    }

    /* find the channels where to op.. */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	channel = tmp->data;

	if (channel->type != CHANNEL_TYPE_CHANNEL)
	    continue;

	userchan = g_hash_table_lookup(user->channels, channel->name);
	if ((user->flags & USER_OP) || (userchan->flags & USER_OP))
	    signal_emit("command op", 3, nick, server, channel);
    }

    return TRUE;
}

static gboolean botcmd_ident(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    USER_REC *user;
    gchar *mask, *str;

    g_return_val_if_fail(data != NULL, FALSE);

    user = botuser_find(nick, address);
    if (user != NULL)
    {
	/* Already know this host */
	return TRUE;
    }

    user = botuser_find(nick, NULL);
    if (user == NULL || !botuser_verify_password(user, data))
    {
	/* failed password */
	return TRUE;
    }

    /* add the new mask */
    mask = irc_get_mask(nick, address, IRC_MASK_USER | IRC_MASK_DOMAIN);
    botuser_add_mask(user, mask);

    str = g_strdup_printf("NOTICE %s :Added new mask %s", nick, mask);
    irc_send_cmd(server, str);
    g_free(str);
    g_free(mask);

    return TRUE;
}

static gboolean botcmd_pass(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    USER_REC *user;
    gchar *params, *str, *pass, *newpass;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, &pass, &newpass);

    user = botuser_find(nick, address);
    if (user == NULL || *pass == '\0')
    {
	g_free(params);
	return TRUE;
    }

    if (user->password != NULL && (*newpass == '\0' || !botuser_verify_password(user, pass)))
    {
	g_free(params);
	return TRUE;
    }

    /* change the password */
    botuser_set_password(user, user->password == NULL ? pass : newpass);

    str = g_strdup_printf("NOTICE %s :Password changed", nick);
    irc_send_cmd(server, str);
    g_free(str);

    return TRUE;
}

static gboolean botcmd_default(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    data = g_strdup_printf("%s :%s", server->nick, data);
    signal_emit("event privmsg", 5, data, server, nick, address, (gpointer) 1);
    g_free(data);
    return TRUE;
}

void plugin_bot_commands_init(PLUGIN_REC *plugin)
{
    plugin_bind(plugin, "event privmsg", (SIGNAL_FUNC) event_privmsg, FALSE);
    plugin_bind(plugin, "bot command op", (SIGNAL_FUNC) botcmd_op, TRUE);
    plugin_bind(plugin, "bot command ident", (SIGNAL_FUNC) botcmd_ident, TRUE);
    plugin_bind(plugin, "bot command pass", (SIGNAL_FUNC) botcmd_pass, TRUE);
    plugin_bind(plugin, "default bot command", (SIGNAL_FUNC) botcmd_default, TRUE);
}

void plugin_bot_commands_deinit(PLUGIN_REC *plugin)
{
}
