/*
 dialog-channels.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#include "dialog-join.h"

static GtkWidget *channel_dialog, *clist = NULL;

static void save_channels(void)
{
    proplist_t lprop, chprop, pkey;
    GList *tmp;

    /* First clear the channels.. */
    pkey = PLMakeString("channels");
    PLRemoveDictionaryEntry(cprop, pkey);

    lprop = NULL;
    if (g_list_length(setupchannels) > 0)
    {
	/* Create an array of channels */
	lprop = PLMakeArrayFromElements(NULL);
	cprop = PLInsertDictionaryEntry(cprop, pkey, lprop);
    }
    PLRelease(pkey);

    for (tmp = g_list_first(setupchannels); tmp != NULL; tmp = tmp->next)
    {
        SETUP_CHANNEL_REC *rec = tmp->data;

	/* create dictionary for channel and add it to array */
	chprop = PLMakeDictionaryFromEntries(NULL, NULL);
	lprop = PLAppendArrayElement(lprop, chprop);

	config_set_str(chprop, "name", rec->name);
	config_set_str(chprop, "ircnet", rec->ircnet);
	if (rec->autojoin) config_set_bool(chprop, "autojoin", rec->autojoin);
	if (rec->password != NULL) config_set_str(chprop, "password", rec->password);
	if (rec->botmasks != NULL) config_set_str(chprop, "botmasks", rec->botmasks);
	if (rec->autosendcmd != NULL) config_set_str(chprop, "autosendcmd", rec->autosendcmd);
	if (rec->background != NULL) config_set_str(chprop, "background", rec->background);
	if (rec->font != NULL) config_set_str(chprop, "font", rec->font);
    }
    if (!readonly) PLSave(cprop, TRUE);
}

static void sig_join(GtkWidget *clist)
{
    SETUP_CHANNEL_REC *channel;
    SERVER_REC *server;
    gint selection;
    gchar *str;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    channel = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    server = server_find_ircnet(channel->ircnet);
    if (server == NULL) server = cur_channel->server;
    if (server != NULL)
    {
	str = g_strdup_printf(channel->password == NULL ? "%s" : "%s %s",
			      channel->name, channel->password);
	signal_emit("command join", 3, str, server, cur_channel);
	g_free(str);
    }
}

static void sig_part(GtkWidget *clist)
{
    SETUP_CHANNEL_REC *channel;
    SERVER_REC *server;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    channel = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    server = server_find_ircnet(channel->ircnet);
    if (server == NULL) server = cur_channel->server;
    if (server != NULL)
        signal_emit("command part", 3, channel->name, server, cur_channel);
}

static void sig_fontsel_ok(GtkFontSelectionDialog *fontsel)
{
    GtkWidget *dialog;

    g_return_if_fail(fontsel != NULL);

    dialog = gtk_object_get_data(GTK_OBJECT(fontsel), "dialog");
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "font")),
		       gtk_font_selection_dialog_get_font_name(fontsel));
    gtk_widget_destroy(GTK_WIDGET(fontsel));
}

static void sig_fontsel_clicked(GtkWidget *button, GtkWidget *dialog)
{
    GtkWidget *fontsel;

    g_return_if_fail(button != NULL);

    fontsel = gtk_font_selection_dialog_new(_("Select new default font"));
    gtk_object_set_data(GTK_OBJECT(fontsel), "dialog", dialog);
    gui_widget_depends(dialog, fontsel);
    gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fontsel),
                                            gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "font"))));

    gtk_signal_connect_object(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG (fontsel)->ok_button), "clicked",
                              GTK_SIGNAL_FUNC(sig_fontsel_ok), GTK_OBJECT(fontsel));
    gtk_signal_connect_object(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG (fontsel)->cancel_button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fontsel));

    gtk_widget_show(fontsel);
}

static GtkWidget *channel_properties_dialog(GtkWidget *clist, SETUP_CHANNEL_REC *channel, GtkSignalFunc func)
{
  GtkWidget *dialog;
  GtkWidget *table, *hbox, *button, *label;
  GtkWidget *cboIRCNet;
  GtkWidget *lblChannel;
  GtkWidget *lblIRCNet;
  GtkWidget *chkAutoJoin;
  GtkWidget *channame_entry, *ircnet_entry, *password_entry, *botmasks_entry, *autosendcmd_entry, *background_entry, *font_entry;

  g_return_val_if_fail(clist != NULL, NULL);
  g_return_val_if_fail(func != NULL, NULL);
  
  dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL, NULL);
  gui_widget_depends(channel_dialog, dialog);
  gtk_signal_connect_object(GTK_OBJECT(dialog), "delete_event",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
  gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);

  table = gtk_table_new (3, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);

  gtk_container_border_width (GTK_CONTAINER (table), 7);
  gtk_table_set_row_spacings (GTK_TABLE (table), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table), 7);

  /* autojoin toggle */
  chkAutoJoin = gtk_check_button_new_with_label (_("Autojoin channel"));
  gtk_table_attach (GTK_TABLE (table), chkAutoJoin, 1, 2, 0, 1,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_object_set_data(GTK_OBJECT(dialog), "autojoin", chkAutoJoin);

  /* channel name */
  lblChannel = gtk_label_new (_("Channel"));
  gtk_table_attach (GTK_TABLE (table), lblChannel, 0, 1, 1, 2,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblChannel), 1, 0.5);

  channame_entry = gtk_entry_new ();
  gtk_object_set_data(GTK_OBJECT(dialog), "name", channame_entry);
  gtk_table_attach (GTK_TABLE (table), channame_entry, 1, 2, 1, 2,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_EXPAND | GTK_FILL, 0, 0);

  /* irc network */
  lblIRCNet = gtk_label_new (_("IRC Network"));
  gtk_table_attach (GTK_TABLE (table), lblIRCNet, 0, 1, 2, 3,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblIRCNet), 1, 0.5);

  cboIRCNet = gtk_combo_new ();
  gtk_table_attach (GTK_TABLE (table), cboIRCNet, 1, 2, 2, 3,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_EXPAND | GTK_FILL, 0, 0);

  if (ircnets != NULL)
  {
      GList *tmp, *list;

      list = NULL;
      for (tmp = ircnets; tmp != NULL; tmp = tmp->next)
      {
          IRCNET_REC *ircnet = tmp->data;

          list = g_list_append(list, ircnet->name);
      }
      gtk_combo_set_popdown_strings(GTK_COMBO(cboIRCNet), list);
      g_list_free(list);
  }

  ircnet_entry = GTK_WIDGET(GTK_COMBO(cboIRCNet)->entry);
  gtk_object_set_data(GTK_OBJECT(dialog), "ircnet", ircnet_entry);

  /* bot masks, bot command */
  password_entry = gui_create_tablelabelentry(table, 0, 3, _("Password"), NULL);
  gtk_object_set_data(GTK_OBJECT(dialog), "password", password_entry);
  botmasks_entry = gui_create_tablelabelentry(table, 0, 4, _("Bot masks"), NULL);
  gtk_object_set_data(GTK_OBJECT(dialog), "botmasks", botmasks_entry);
  autosendcmd_entry = gui_create_tablelabelentry(table, 0, 5, _("Autosend cmd to bot"), NULL);
  gtk_object_set_data(GTK_OBJECT(dialog), "autosendcmd", autosendcmd_entry);
  background_entry = gui_create_tablelabelentry(table, 0, 6, _("Background pixmap"), NULL);
  gtk_object_set_data(GTK_OBJECT(dialog), "background", background_entry);

  label = gtk_label_new(_("Font"));
  gtk_misc_set_alignment(GTK_MISC(label), 1, .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 7, 8);

  hbox = gtk_hbox_new(FALSE, 7);
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 7, 8);

  font_entry = gtk_entry_new();
  gtk_object_set_data(GTK_OBJECT(dialog), "font", font_entry);
  gtk_box_pack_start(GTK_BOX(hbox), font_entry, TRUE, TRUE, 0);

  button = gtk_button_new_with_label(_("Change..."));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(sig_fontsel_clicked), dialog);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  if (channel != NULL)
  {
      if (channel->autojoin) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkAutoJoin), TRUE);
      gtk_entry_set_text(GTK_ENTRY(channame_entry), channel->name);
      gtk_entry_set_text(GTK_ENTRY(ircnet_entry), channel->ircnet);
      if (channel->password != NULL) gtk_entry_set_text(GTK_ENTRY(password_entry), channel->password);
      if (channel->botmasks != NULL) gtk_entry_set_text(GTK_ENTRY(botmasks_entry), channel->botmasks);
      if (channel->autosendcmd != NULL) gtk_entry_set_text(GTK_ENTRY(autosendcmd_entry), channel->autosendcmd);
      if (channel->background != NULL) gtk_entry_set_text(GTK_ENTRY(background_entry), channel->background);
      if (channel->font != NULL) gtk_entry_set_text(GTK_ENTRY(font_entry), channel->font);
  }

  gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, func, GTK_OBJECT(dialog));
  gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

  gtk_widget_show_all(dialog);
  return dialog;
}

static void clist_insert_channel(GtkWidget *clist, SETUP_CHANNEL_REC *channel, gint pos, gboolean checkjoin)
{
    gchar *titles[4];
    SERVER_REC *server;

    g_return_if_fail(clist != NULL);
    g_return_if_fail(channel != NULL);

    server = server_find_ircnet(channel->ircnet);
    titles[0] = checkjoin && channel_find(server != NULL ? server : cur_channel->server , channel->name) ? _("Yes") : _("No");
    titles[1] = channel->autojoin ? _("Yes") : _("No");
    titles[2] = channel->ircnet;
    titles[3] = channel->name;

    if (pos < 0)
        pos = gtk_clist_append(GTK_CLIST(clist), titles);
    else
        pos = gtk_clist_insert(GTK_CLIST(clist), pos, titles);

    gtk_clist_set_row_data(GTK_CLIST(clist), pos, channel);
}

static gint clist_remove_channel(GtkWidget *clist, SETUP_CHANNEL_REC *channel)
{
    gint row;

    g_return_val_if_fail(clist != NULL, -1);
    g_return_val_if_fail(channel != NULL, -1);

    row = gtk_clist_find_row_from_data(GTK_CLIST(clist), channel);
    if (row != -1) gtk_clist_remove(GTK_CLIST(clist), row);
    return row;
}

static SETUP_CHANNEL_REC *channel_new_entry(GtkWidget *dialog, gint pos)
{
    gchar *name, *ircnet, *password, *botmasks, *autosendcmd, *background, *font;
    gboolean autojoin;
    SETUP_CHANNEL_REC *rec;

    name = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "name")));
    ircnet = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "ircnet")));
    password = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "password")));
    botmasks = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "botmasks")));
    autosendcmd = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "autosendcmd")));
    background = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "background")));
    font = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "font")));
    autojoin = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gtk_object_get_data(GTK_OBJECT(dialog), "autojoin")));

    rec = g_new(SETUP_CHANNEL_REC, 1);
    rec->autojoin = autojoin;
    rec->name = g_strdup(name);
    rec->ircnet = g_strdup(ircnet);
    rec->password = (password == NULL || *password == '\0') ? NULL : g_strdup(password);
    rec->botmasks = (botmasks == NULL || *botmasks == '\0') ? NULL : g_strdup(botmasks);
    rec->autosendcmd = (autosendcmd == NULL || *autosendcmd == '\0') ? NULL : g_strdup(autosendcmd);
    rec->background = (background == NULL || *background == '\0') ? NULL : g_strdup(background);
    rec->font = (font == NULL || *font == '\0') ? NULL : g_strdup(font);

    setupchannels = pos < 0 ?
	g_list_append(setupchannels, rec) :
	g_list_insert(setupchannels, rec, pos);
    return rec;
}

static void sig_add_ok(GtkWidget *dialog)
{
    GtkWidget *clist;

    g_return_if_fail(dialog != NULL);

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    g_return_if_fail(clist != NULL);

    clist_insert_channel(clist, channel_new_entry(dialog, -1), -1, TRUE);
    gtk_widget_destroy(dialog);
}

static void sig_add_channel(GtkWidget *clist)
{
    SETUP_CHANNEL_REC *channel;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    channel = selection == -1 ? NULL : gtk_clist_get_row_data(GTK_CLIST(clist), selection);

    channel_properties_dialog(clist, channel, GTK_SIGNAL_FUNC(sig_add_ok));
}

static void sig_delete_channel(GtkWidget *clist)
{
    SETUP_CHANNEL_REC *channel;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    channel = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    clist_remove_channel(clist, channel);
    channels_setup_destroy(channel);
}

static void sig_edit_ok(GtkWidget *dialog)
{
    SETUP_CHANNEL_REC *channel;
    GtkWidget *clist;
    gint selection;

    g_return_if_fail(dialog != NULL);

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    g_return_if_fail(clist != NULL);

    /* destroy old one */
    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
    clist_remove_channel(clist, channel);
    channels_setup_destroy(channel);

    /* insert new one */
    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dialog), "selection"));
    clist_insert_channel(clist, channel_new_entry(dialog, selection), selection, TRUE);
    gtk_widget_destroy(dialog);
}

static void sig_edit_channel(GtkWidget *clist)
{
    SETUP_CHANNEL_REC *channel;
    GtkWidget *dialog;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    channel = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    dialog = channel_properties_dialog(clist, channel, GTK_SIGNAL_FUNC(sig_edit_ok));
    gtk_object_set_data(GTK_OBJECT(dialog), "channel", channel);
    gtk_object_set_data(GTK_OBJECT(dialog), "selection", GINT_TO_POINTER(selection));
}

static void sig_move_row(GtkWidget *widget, gint source, gint dest)
{
    gpointer data;

    data = g_list_nth(setupchannels, source)->data;
    setupchannels = g_list_remove(setupchannels, data);
    setupchannels = g_list_insert(setupchannels, data, dest);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
}

static void sig_unselect_row(void)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
}

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    save_channels();
    *dialog = NULL;
    clist = NULL;
}

static void redraw_channels(void)
{
    GList *tmp;

    gtk_clist_freeze(GTK_CLIST(clist));
    gtk_clist_clear(GTK_CLIST(clist));
    for (tmp = g_list_first(setupchannels); tmp != NULL; tmp = tmp->next)
        clist_insert_channel(clist, tmp->data, -1, TRUE);
    gtk_clist_thaw(GTK_CLIST(clist));
}

void dialog_channels(void)
{
    GtkWidget *buttonbox, *button, *sep;
    GtkWidget *scrollbox, *hbox;
    gchar *titles[4];

    if (channel_dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(channel_dialog->window);
        return;
    }

    titles[0] = _("Joined"); titles[1] = _("Autojoin");
    titles[2] = _("IRC network"); titles[3] = _("Channel");

    channel_dialog = gnome_dialog_new(PACKAGE, _("Join"), _("Part"), GNOME_STOCK_BUTTON_CLOSE, NULL);
#ifndef HAVE_GNOME
    gtk_widget_set_usize(channel_dialog, 300, -1);
#endif
    gtk_window_set_policy(GTK_WINDOW(channel_dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(channel_dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &channel_dialog);
    gtk_signal_connect(GTK_OBJECT(channel_dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    hbox = gtk_hbox_new(FALSE, 7);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(channel_dialog)->vbox), hbox, TRUE, TRUE, 0);

    /* Create server clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(hbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(4, titles);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(clist), TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(clist), "row_move",
		       GTK_SIGNAL_FUNC(sig_move_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);

    buttonbox = gtk_vbutton_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), buttonbox, FALSE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 7);

    button = gtk_button_new_with_label(_("Add..."));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_add_channel), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Edit..."));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_edit_channel), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Delete"));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_delete_channel), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    sep = gtk_hseparator_new();
    gtk_container_add(GTK_CONTAINER(buttonbox), sep);

    button = gtk_button_new_with_label(_("Quick join..."));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(dialog_join), NULL);
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    redraw_channels();

    gnome_dialog_button_connect_object(GNOME_DIALOG(channel_dialog), 0, GTK_SIGNAL_FUNC(sig_join), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(channel_dialog), 1, GTK_SIGNAL_FUNC(sig_part), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(channel_dialog), 2, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(channel_dialog));

    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 3, TRUE);
    gtk_widget_show_all(channel_dialog);
}

static gboolean event_join(gchar *data, SERVER_REC *server, gchar *nick)
{
    SETUP_CHANNEL_REC *rec;
    gchar *params, *channel, *ptr;
    gint row;

    g_return_val_if_fail(data != NULL, FALSE);

    if (clist == NULL || g_strcasecmp(nick, server->nick) != 0)
        return TRUE;

    params = event_get_params(data, 1, &channel);
    ptr = strchr(channel, 7); /* ^G does something weird.. */
    if (ptr != NULL) *ptr = '\0';

    rec = channels_setup_find(channel, server);
    if (rec != NULL)
    {
        row = clist_remove_channel(clist, rec);
        if (row != -1) clist_insert_channel(clist, rec, row, TRUE);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_part(gchar *data, SERVER_REC *server, gchar *nick)
{
    SETUP_CHANNEL_REC *rec;
    gchar *params, *channel;
    gint row;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);

    if (clist == NULL || g_strcasecmp(nick, server->nick) != 0)
        return TRUE;

    params = event_get_params(data, 1, &channel);

    rec = channels_setup_find(channel, server);
    if (rec != NULL)
    {
        row = clist_remove_channel(clist, rec);
        if (row != -1) clist_insert_channel(clist, rec, row, FALSE);
    }

    g_free(params);
    return TRUE;
}

/* irc network's name changed from setup dialog, change it to channel setup too */
static gboolean sig_ircnet_changed(gchar *oldname, gchar *newname)
{
    GList *tmp;

    g_return_val_if_fail(oldname != NULL, FALSE);
    g_return_val_if_fail(newname != NULL, FALSE);

    if (*oldname == '\0') return TRUE;

    for (tmp = g_list_first(setupchannels); tmp != NULL; tmp = tmp->next)
    {
        SETUP_CHANNEL_REC *rec = tmp->data;

        if (g_strcasecmp(rec->ircnet, oldname) == 0)
        {
            g_free(rec->ircnet);
            rec->ircnet = g_strdup(newname);
        }
    }

    if (clist != NULL)
        redraw_channels();

    save_channels();
    return TRUE;
}

/* channel changed in window / new channel created - check for channel
   specific backgrounds/fonts */
static gboolean sig_channel_focused(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *gui;
    SETUP_CHANNEL_REC *rec;
    gchar *path, *normal, *bold;

    if (channel == NULL || channel->server == NULL) return TRUE;

    rec = channels_setup_find(channel->name, channel->server);
    if (rec == NULL) return TRUE;

    if (rec->background != NULL)
    {
        /* channel specific background pixmap */
        path = convert_home(rec->background);
        gui_window_set_background_pixmap(CHANNEL_PARENT(channel), path);
        g_free(path);
    }

    if (rec->font != NULL)
    {
        gui = CHANNEL_GUI(channel);

        gui_printtext_get_fonts(rec->font, &normal, &bold);
        gui->font_normal = gdk_font_load(normal);
        gui->font_bold = gdk_font_load(bold);
        if (gui->font_bold == NULL && gui->font_normal != NULL)
            gui->font_bold = gui->font_normal;
        g_free(normal); g_free(bold);

        if (WINDOW_VIEW(CHANNEL_PARENT(channel))->itext)
        {
	    gtk_itext_set_fonts(GTK_ITEXT(WINDOW_VIEW(CHANNEL_PARENT(channel))->text),
				gui->font_normal, gui->font_bold);
	}
    }

    return TRUE;
}

void dialog_channels_init(void)
{
    channel_dialog = NULL;
    signal_add("event join", (SIGNAL_FUNC) event_join);
    signal_add("event part", (SIGNAL_FUNC) event_part);
    signal_add("setup ircnet changed", (SIGNAL_FUNC) sig_ircnet_changed);
    signal_add("channel focused", (SIGNAL_FUNC) sig_channel_focused);
}

void dialog_channels_deinit(void)
{
    signal_remove("event join", (SIGNAL_FUNC) event_join);
    signal_remove("event part", (SIGNAL_FUNC) event_part);
    signal_remove("setup ircnet changed", (SIGNAL_FUNC) sig_ircnet_changed);
    signal_remove("channel focused", (SIGNAL_FUNC) sig_channel_focused);
}
