/*
 gui-events-who.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-nicklist-popup.h"

typedef struct
{
    gchar *channel;
    gchar *nick;
    gchar *host;
    gchar *status;
    gchar *hops;
    gchar *realname;
}
WHO_REC;

static void sig_select_row(GtkCList *clist, gint row)
{
    GList *list;

    g_return_if_fail(clist != NULL);

    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
    list = g_list_append(list, GINT_TO_POINTER(row));
    gtk_object_set_data(GTK_OBJECT(clist), "selections", list);
}

static void sig_unselect_row(GtkCList *clist, gint row)
{
    GList *list;

    g_return_if_fail(clist != NULL);

    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
    list = g_list_remove(list, GINT_TO_POINTER(row));
    gtk_object_set_data(GTK_OBJECT(clist), "selections", list);
}

static void sig_button_press(GtkCList *clist, GdkEventButton *event)
{
    CHANNEL_REC *chanrec;
    SERVER_REC *server;
    GList *list;
    gint row, col;
    gchar *channel, *nick, *servertag;

    g_return_if_fail(clist != NULL);
    g_return_if_fail(event != NULL);

    if (event->button != 3) return;

    list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
    if (list == NULL)
    {
        if (gtk_clist_get_selection_info(clist, event->x, event->y, &row, &col) < 0)
            return;
        gtk_clist_select_row(clist, row, col);

        list = gtk_object_get_data(GTK_OBJECT(clist), "selections");
        if (list == NULL)
            return;
    }

    servertag = gtk_object_get_data(GTK_OBJECT(clist), "server");
    server = server_find_tag(servertag);
    if (server == NULL) return;

    gtk_clist_get_text(clist, GPOINTER_TO_INT(list->data), 0, &channel);
    chanrec = channel_find_closest(server, channel, 0);
    if (chanrec == NULL) chanrec = cur_channel;

    /* show popup menu */
    gtk_clist_get_text(clist, GPOINTER_TO_INT(list->data), 1, &nick);
    if (nick != NULL)
        gui_nicklist_popup(NULL, chanrec, nick, event);

    gtk_clist_unselect_all(clist);
}

static gboolean event_who(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *channel, *user, *host, *status, *hops, *realname;
    WHO_REC *rec;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 8, NULL, &channel, &user, &host, NULL, &nick, &status, &realname);

    /* split hops/realname */
    hops = realname;
    while (*realname != '\0' && *realname != ' ') realname++;
    while (*realname == ' ') realname++;
    if (realname > hops) realname[-1] = '\0';

    rec = g_new(WHO_REC, 1);
    SERVER_GUI(server)->wholist = g_list_append(SERVER_GUI(server)->wholist, rec);

    rec->channel = g_strdup(channel);
    rec->nick = g_strdup(nick);
    rec->host = g_strdup_printf("%s@%s", user, host);
    rec->status = g_strdup(status);
    rec->hops = g_strdup(hops);
    rec->realname = g_strdup(realname);

    g_free(params);
    return TRUE;
}

static void list_free(WHO_REC *rec)
{
    g_free(rec->channel);
    g_free(rec->nick);
    g_free(rec->host);
    g_free(rec->status);
    g_free(rec->hops);
    g_free(rec->realname);
    g_free(rec);
}

static void sig_destroy(GtkWidget *dialog)
{
    GList *list;

    list = gtk_object_get_data(GTK_OBJECT(dialog), "list");
    g_list_foreach(list, (GFunc) list_free, NULL);
    g_list_free(list);
}

static void sig_search(GtkEntry *entry, GtkWidget *dialog)
{
    gchar *text, *texts[5];
    GtkCList *clist;
    GList *list;

    text = gtk_entry_get_text(entry);

    /* redraw the list */
    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    list = gtk_object_get_data(GTK_OBJECT(dialog), "list");
    for (; list != NULL; list = list->next)
    {
	WHO_REC *rec = list->data;

	if (*text != '\0' &&
	    stristr(rec->channel, text) == NULL &&
	    stristr(rec->nick, text) == NULL &&
	    stristr(rec->host, text) == NULL &&
	    stristr(rec->status, text) == NULL &&
	    stristr(rec->hops, text) == NULL &&
	    stristr(rec->realname, text) == NULL)
	    continue;

	/* add record to list */
	texts[0] = rec->channel;
	texts[1] = rec->nick;
	texts[2] = rec->status;
	texts[3] = rec->host;
	texts[4] = rec->realname;

	gtk_clist_append(clist, texts);

    }
    gtk_clist_thaw(clist);
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

static gboolean event_end_of_who(gchar *data, SERVER_REC *server)
{
    GtkWidget *dialog, *clist, *scrollbox, *entry, *label;
    gchar *texts[5], *servertag, *str, *params, *channel;
    GUI_SERVER_REC *gui;
    GList *chans, *queue, *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);

    if (strchr(channel, ',') != NULL)
    {
	/* instead of multiple End of WHO replies we get only this one... */
	server->one_endofwho = TRUE;

	chans = str2list(channel, ',');
	for (tmp = chans; tmp != NULL; tmp = tmp->next)
	{
	    /* remove other expected end of who messages from queue */
	    queue = server_redirect_getqueue(server, "event 315", data);
	    if (queue != NULL)
		server_redirect_remove_next(server, "event 315", queue);

	}
	if (chans != NULL)
	{
	    g_free(chans->data);
	    g_list_free(chans);
	}
    }
    g_free(params);

    gui = SERVER_GUI(server);
    if (gui->wholist == NULL)
    {
	gui_dialog(DIALOG_OK, _("/WHO request didn't find anything"));
	return TRUE;
    }

    servertag = g_strdup(server->tag);
    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_widget_set_usize(dialog, 600, 300);

    /* Create server clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    texts[0] = _("Channel"); texts[1] = _("Nick"); texts[2] = _("Status");
    texts[3] = _("Host"); texts[4] = _("Real name");

    clist = gtk_clist_new_with_titles(5, texts);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_signal_connect_object(GTK_OBJECT(clist), "destroy",
			      GTK_SIGNAL_FUNC(g_free), (GtkObject *) servertag);
    gtk_signal_connect_object(GTK_OBJECT(clist), "destroy",
			      GTK_SIGNAL_FUNC(sig_destroy), GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
		       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
		       GTK_SIGNAL_FUNC(sig_button_press), NULL);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
    gtk_object_set_data(GTK_OBJECT(dialog), "list", gui->wholist);
    gtk_object_set_data(GTK_OBJECT(clist), "server", servertag);

    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_EXTENDED);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 4, TRUE);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_clist_freeze(GTK_CLIST(clist));
    for (tmp = gui->wholist; tmp != NULL; tmp = tmp->next)
    {
	WHO_REC *rec = tmp->data;

	texts[0] = rec->channel;
	texts[1] = rec->nick;
	texts[2] = rec->status;
	texts[3] = rec->host;
	texts[4] = rec->realname;

	gtk_clist_append(GTK_CLIST(clist), texts);
    }
    gtk_clist_thaw(GTK_CLIST(clist));

    /* Create the search widget */
    entry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox,
				  _("Search"), NULL, FALSE);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(sig_search), dialog);
    gtk_widget_grab_focus(entry);

    /* Display total number of who request */
    str = g_strdup_printf(_("Found total of %d matches"), GTK_CLIST(clist)->rows);
    label = gtk_label_new(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0);
    g_free(str);

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 3, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 4, TRUE);
    gtk_widget_show_all(dialog);

    gui->wholist = NULL;
    return TRUE;
}

static gboolean cmd_who(gchar *data, SERVER_REC *server)
{
    gchar *params, *args, *channel, *chlist, *rest, *str;
    GList *list, *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_OPTARGS | PARAM_FLAG_GETREST, &args, &channel, &rest);

    if (stristr(args, "-nogui") != NULL)
    {
	g_free(params);
	return TRUE;
    }

    /* Use GUI list */
    if (strcmp(channel, "*") == 0 || *channel == '\0')
    {
	if (cur_channel->type != CHANNEL_TYPE_CHANNEL)
	    cmd_param_error(CMDERR_NOT_JOINED);

	channel = cur_channel->name;
    }
    if (strcmp(channel, "**") == 0)
    {
	/* ** displays all nicks.. */
	*channel = '\0';
    }

    str = g_strdup_printf(*rest == '\0' ? "WHO %s" : "WHO %s %s",
			  channel, rest);
    irc_send_cmd(server, str);
    g_free(str);

    /* /WHO a,b,c may respond with either one "a,b,c End of WHO" message or
       three different "a End of WHO", "b End of WHO", .. messages */
    chlist = g_strdup(channel);
    list = str2list(channel, ',');

    for (tmp = list; tmp != NULL; tmp = tmp->next)
    {
	if (strcmp((gchar *) tmp->data, "0") == 0)
	{
	    /* /who 0 displays everyone */
	    ((gchar *) tmp->data)[0] = '*';
	}

	channel = g_strdup_printf("%s %s", chlist, (gchar *) tmp->data);
	server_redirect_event(server, channel, 2,
			      "event 401", "event 401", 1,
			      "event 315", "gui event 315", 1,
			      "event 352", "gui event 352", -1, NULL);
	g_free(channel);
    }

    if (list != NULL)
    {
	g_free(list->data);
	g_list_free(list);
    }

    g_free(chlist);
    g_free(params);
    return FALSE;
}

void gui_event_who_init(void)
{
    command_bind("who", "GUI commands", (SIGNAL_FUNC) cmd_who);
    signal_add("gui event 352", (SIGNAL_FUNC) event_who);
    signal_add("gui event 315", (SIGNAL_FUNC) event_end_of_who);
}

void gui_event_who_deinit(void)
{
    command_unbind("who", (SIGNAL_FUNC) cmd_who);
    signal_remove("gui event 352", (SIGNAL_FUNC) event_who);
    signal_remove("gui event 315", (SIGNAL_FUNC) event_end_of_who);
}
