/*
 setup-servers.c : Servers notebook entry for preferences

    Copyright (C) 1999 Timo Sirainen
                       Kari Lavikka (tuner@bdb.fi)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"
#include "setup-int.h"
#include "setup-ircnets.h"

GList *setupservers; /* list of selected servers */
GList *temp_setupservers; /* to make cancel work.. */

static GtkWidget *setup_dialog;
static gpointer clistData;



/* remove server from server list */
static void remove_server_from_list(GList **servers, SETUP_SERVER_REC *rec)
{
  g_return_if_fail(servers != NULL);
  g_return_if_fail(rec != NULL);
  
  *servers = g_list_remove(*servers, rec);
  
  g_free(rec->ircnet);
  g_free(rec->server);
  g_free(rec->password);
  g_free(rec);
}

/* add new server to server list. Last SETUP_SERVER_REC is used if we
   want to replace a record instead of adding a new one. Use NULL
   if we are just adding a new record. */
static SETUP_SERVER_REC *add_server_to_list(GList **servers, gchar *ircnet, gchar *server, gint port, gchar *password, gboolean autoconnect, gboolean cmd_queue_speed, SETUP_SERVER_REC *replacerec)
{
  SETUP_SERVER_REC *rec;
  GList *tmp;
  
  g_return_val_if_fail(servers != NULL, NULL);
  g_return_val_if_fail(ircnet != NULL, NULL);
  g_return_val_if_fail(server != NULL, NULL);
  g_return_val_if_fail(password != NULL, NULL);
  
  if(replacerec != NULL) {
    if((tmp = g_list_find(*servers, replacerec)) != NULL) {
      rec = tmp->data;
      g_free(rec->ircnet);
      g_free(rec->server);
      g_free(rec->password);

      rec->ircnet = g_strdup(ircnet);
      rec->server = g_strdup(server);
      rec->password = g_strdup(password);
      rec->port = port;
      rec->autoconnect = autoconnect;
      rec->cmd_queue_speed = cmd_queue_speed;
    } else
      rec = NULL;
  } else {
    for (tmp = *servers; tmp != NULL; tmp = tmp->next)
    {
	rec = tmp->data;

	if (g_strcasecmp(rec->server, server) == 0 &&
	    g_strcasecmp(rec->ircnet, ircnet) == 0 && rec->port == port)
	    return NULL;
    }

    rec = g_new(SETUP_SERVER_REC, 1);
    rec->ircnet = g_strdup(ircnet);
    rec->server = g_strdup(server);
    rec->password = g_strdup(password);
    rec->port = port;
    rec->autoconnect = autoconnect;
    rec->cmd_queue_speed = cmd_queue_speed;
    
    *servers = g_list_append(*servers, rec);
  }
  return rec;
}


static gboolean sig_setup_apply(GtkWidget *dialog)
{
  proplist_t lprop, iprop, pkey;
  SETUP_SERVER_REC *rec;
  GList *tmp;

  g_return_val_if_fail(dialog != NULL, FALSE);

  setup_write("server", server_options);

  while (setupservers != NULL)
    remove_server_from_list(&setupservers, setupservers->data);
  for (tmp = g_list_first(temp_setupservers); tmp != NULL; tmp = tmp->next)
  {
    SETUP_SERVER_REC *rec = tmp->data;

    add_server_to_list(&setupservers, rec->ircnet, rec->server,
                       rec->port, rec->password, rec->autoconnect, rec->cmd_queue_speed, NULL);
  }

  /* Save servers */
  pkey = PLMakeString("setupservers");
  PLRemoveDictionaryEntry(cprop, pkey);

  lprop = NULL;
  if (g_list_length(setupservers) > 0)
  {
    lprop = PLMakeArrayFromElements(NULL);
    cprop = PLInsertDictionaryEntry(cprop, pkey, lprop);
  }
  PLRelease(pkey);

  for (tmp = g_list_first(setupservers); tmp != NULL; tmp = tmp->next)
  {
    rec = tmp->data;

    iprop = PLMakeDictionaryFromEntries(NULL, NULL);
    lprop = PLAppendArrayElement(lprop, iprop);

    config_set_str(iprop, "server", rec->server);
    config_set_str(iprop, "ircnet", rec->ircnet);
    config_set_int(iprop, "port", rec->port);
    if (*rec->password != '\0') config_set_str(iprop, "password", rec->password);
    config_set_bool(iprop, "autoconnect", rec->autoconnect);
    config_set_int(iprop, "cmd_queue_speed", rec->cmd_queue_speed);
  }

  if (!readonly) PLSave(cprop, TRUE);

  return TRUE;
}



static void sig_serverlist_move_row(GtkWidget *widget, gint source, gint dest)
{
    gpointer data;

    data = g_list_nth(temp_setupservers, source)->data;
    temp_setupservers = g_list_remove(temp_setupservers, data);
    temp_setupservers = g_list_insert(temp_setupservers, data, dest);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_serverlist_select_row(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer data)
{
  clistData = gtk_clist_get_row_data(GTK_CLIST((GtkWidget*)gtk_object_get_data(GTK_OBJECT(setup_dialog), "clistServers")), row);
}



static void sig_serverlist_unselect_row(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer data)
{
  clistData = NULL;
}



/* Add server to GtkCList */
static void add_server_to_clist(GtkWidget *list, SETUP_SERVER_REC *server)
{
  gchar *columns[3];
  gint row;

  columns[0] = server->autoconnect ? _("y") : NULL;
  columns[1] = server->ircnet;
  columns[2] = server->server;

  row = gtk_clist_append(GTK_CLIST(list), columns);
  gtk_clist_set_row_data(GTK_CLIST(list), row, server);

}


/* make a list of servers */
static void fill_clist_with_servers(GList *servers)
{
  GList *counter;
  GtkWidget *list;

  list = (GtkWidget*)gtk_object_get_data(GTK_OBJECT(setup_dialog), "clistServers");

  gtk_clist_freeze(GTK_CLIST(list));
  gtk_clist_clear(GTK_CLIST(list));
  
  for (counter = g_list_first(servers); counter != NULL; counter = counter->next)
    add_server_to_clist(list, (SETUP_SERVER_REC *) counter->data);
  
  gtk_clist_thaw(GTK_CLIST(list));
  clistData = NULL;
}


static void sig_clear_dialog(GtkWidget *dialog)
{
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "cboIRCNet"))->entry), "");
  gtk_entry_set_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtServer")), "");
  gtk_entry_set_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtPassWord")), "");
  gtk_entry_set_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtPort")), "6667");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "chkAutoConnect")), FALSE);
  gtk_entry_set_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtQueueSpeed")), "");
}



/* Display server setup dialog. If *server is NULL we are displaying
   an empy dialog. In other case we are probably editing the record
   *server is pointing to, and we shall fill entries with appropriate
   information */
static void server_dialog(SETUP_SERVER_REC *server, GtkSignalFunc func)
{
  GtkWidget *dialog;
  GtkWidget *table2;
  GtkWidget *lblIRCNet;
  GtkWidget *lblServer;
  GtkWidget *lblPort;
  GtkWidget *lblPassword;
  GtkWidget *lblQueueSpeed;
  GtkWidget *txtPassWord;
  GtkWidget *txtPort;
  GtkWidget *txtServer;
  GtkWidget *cboIRCNet;
  GtkWidget *chkAutoConnect;
  GtkWidget *txtQueueSpeed;
  gchar *str;

  g_return_if_fail(func != NULL);

  dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL, _("Clear"), NULL);
  gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
  gui_widget_depends(setup_dialog, dialog);

  table2 = gtk_table_new (5, 2, FALSE);

  gtk_container_border_width (GTK_CONTAINER (table2), 7);
  gtk_table_set_row_spacings (GTK_TABLE (table2), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table2), 7);

  lblIRCNet = gtk_label_new (_("IRC Net"));
  gtk_table_attach (GTK_TABLE (table2), lblIRCNet, 0, 1, 0, 1,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblIRCNet), 1, 0.5);

  lblServer = gtk_label_new (_("Server"));
  gtk_table_attach (GTK_TABLE (table2), lblServer, 0, 1, 1, 2,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblServer), 1, 0.5);

  lblPort = gtk_label_new (_("Port"));
  gtk_table_attach (GTK_TABLE (table2), lblPort, 0, 1, 2, 3,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblPort), 1, 0.5);

  lblPassword = gtk_label_new (_("Password"));
  gtk_table_attach (GTK_TABLE (table2), lblPassword, 0, 1, 3, 4,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblPassword), 1, 0.5);

  lblQueueSpeed = gtk_label_new (_("Timeout between sending commands to server (0 = default)"));
  gtk_table_attach (GTK_TABLE (table2), lblQueueSpeed, 0, 2, 4, 5,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (lblPassword), 1, 0.5);

  txtPassWord = gtk_entry_new ();
  gtk_object_set_data (GTK_OBJECT (dialog), "txtPassWord", txtPassWord);
  gtk_table_attach (GTK_TABLE (table2), txtPassWord, 1, 2, 3, 4,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);

  txtPort = gtk_entry_new ();
  gtk_object_set_data (GTK_OBJECT (dialog), "txtPort", txtPort);
  gtk_table_attach (GTK_TABLE (table2), txtPort, 1, 2, 2, 3,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);

  txtServer = gtk_entry_new ();
  gtk_object_set_data (GTK_OBJECT (dialog), "txtServer", txtServer);
  gtk_table_attach (GTK_TABLE (table2), txtServer, 1, 2, 1, 2,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);

  cboIRCNet = gtk_combo_new ();
  gtk_object_set_data (GTK_OBJECT (dialog), "cboIRCNet", cboIRCNet);
  gtk_table_attach (GTK_TABLE (table2), cboIRCNet, 1, 2, 0, 1,
                    (GtkAttachOptions) GTK_EXPAND | GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 0);

  txtQueueSpeed = gtk_entry_new ();
  gtk_object_set_data (GTK_OBJECT (dialog), "txtQueueSpeed", txtQueueSpeed);
  gtk_table_attach (GTK_TABLE (table2), txtQueueSpeed, 1, 2, 5, 6,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_EXPAND | GTK_FILL, 0, 0);

  chkAutoConnect = gtk_check_button_new_with_label (_("Autoconnect at startup"));
  gtk_object_set_data (GTK_OBJECT (dialog), "chkAutoConnect", chkAutoConnect);
  gtk_table_attach (GTK_TABLE (table2), chkAutoConnect, 0, 2, 6, 7,
                    (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_EXPAND | GTK_FILL, 0, 0);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table2, TRUE, TRUE, 0);

  if (temp_ircnets != NULL)
  {
      GList *tmp, *list;

      list = NULL;
      for (tmp = temp_ircnets; tmp != NULL; tmp = tmp->next)
      {
          IRCNET_REC *ircnet = tmp->data;

          list = g_list_append(list, ircnet->name);
      }
      gtk_combo_set_popdown_strings(GTK_COMBO(cboIRCNet), list);
      g_list_free(list);
  }

  if(server != NULL) {
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(cboIRCNet)->entry), server->ircnet);
    gtk_entry_set_text(GTK_ENTRY(txtServer), server->server);
    gtk_entry_set_text(GTK_ENTRY(txtPassWord), server->password);
    
    str = g_strdup_printf("%d", server->port);
    gtk_entry_set_text(GTK_ENTRY(txtPort), str);
    g_free(str);
    
    str = g_strdup_printf("%d", server->cmd_queue_speed);
    gtk_entry_set_text(GTK_ENTRY(txtQueueSpeed), str);
    g_free(str);
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkAutoConnect), server->autoconnect);
  } else {
    sig_clear_dialog(dialog);
  }
  
  gtk_object_set_data(GTK_OBJECT(dialog), "server_data", server);
  
  gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, func, GTK_OBJECT(dialog));
  gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
  gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(sig_clear_dialog), GTK_OBJECT(dialog));

  gtk_widget_show_all(dialog);
}



/* signal: OK button pressed in add global server dialog */
static void sig_add_server_ok(GtkWidget *dialog)
{
  SETUP_SERVER_REC *rec;
  IRCNET_REC *irec;
  gchar *ircnet, *server, *password, *tmp, *tmp2;
  gint port, cmd_queue_speed;
  gboolean autoconnect;

  rec = (SETUP_SERVER_REC *)gtk_object_get_data(GTK_OBJECT(dialog), "server_data");

  ircnet = g_strstrip(g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "cboIRCNet"))->entry))));
  server = g_strstrip(g_strdup(gtk_entry_get_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtServer")))));
  tmp = g_strstrip(g_strdup(gtk_entry_get_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtPort")))));
  password = g_strstrip(g_strdup(gtk_entry_get_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtPassWord")))));
  autoconnect = GTK_TOGGLE_BUTTON((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "chkAutoConnect"))->active;
  tmp2 = g_strstrip(g_strdup(gtk_entry_get_text(GTK_ENTRY((GtkWidget*)gtk_object_get_data(GTK_OBJECT(dialog), "txtQueueSpeed")))));

  /* let's check if user has forgot to fill something */
  if(!strlen(ircnet) || !strlen(server) || !strlen(tmp))
    gui_dialog(DIALOG_WARNING, _("You have empty entries!\nAt least IRC Net, Server and Port\nmust be filled."));
  else
  {
    if (sscanf(tmp, "%d", &port) != 1)
      port = 6667;
    if (sscanf(tmp2, "%d", &cmd_queue_speed) != 1)
      cmd_queue_speed = 0;
    if (!ircnet_find(ircnet))
    {
      irec = g_new0(IRCNET_REC, 1);
      irec->name = g_strdup(ircnet);
      ircnet_add(irec);
    }
  
    add_server_to_list(&temp_setupservers, ircnet, server, port, password, autoconnect, cmd_queue_speed, rec);
    fill_clist_with_servers(temp_setupservers);

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
    gtk_widget_destroy(dialog);
  }

  g_free(tmp);
  g_free(tmp2);
  g_free(ircnet);
  g_free(password);
  g_free(server);
}



/* signal: add server button pressed */
static void sig_add_server(void)
{
  server_dialog(NULL, GTK_SIGNAL_FUNC(sig_add_server_ok));
}



static void sig_edit_server(void)
{
  if(clistData == NULL) return;

  server_dialog((SETUP_SERVER_REC *)clistData, GTK_SIGNAL_FUNC(sig_add_server_ok));
}



/* signal: delete server button pressed */
static void sig_delete_server(GtkWidget *widget, GtkList *list)
{
  if(clistData == NULL) return;
  remove_server_from_list(&temp_setupservers, (SETUP_SERVER_REC *)clistData);
  fill_clist_with_servers(temp_setupservers);
  
  gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}


static gboolean sig_setup_destroyed(void)
{
    while (temp_setupservers != NULL)
        remove_server_from_list(&temp_setupservers, temp_setupservers->data);
    setup_dialog = NULL;

    return TRUE;
}

/* draw server notebook section */
static gboolean setup_servers(GtkWidget *dialog)
{
  GtkWidget *vbox1, *vbox2;
  GtkWidget *frmServers;
  GtkWidget *hbox1;
  GtkWidget *vbuttonbox1;
  GtkWidget *cmdAdd;
  GtkWidget *cmdRemove;
  GtkWidget *cmdEdit;
  GtkWidget *frmUserInfo;
  GtkWidget *table1;
  GtkWidget *clistServers;
  GtkWidget *spin, *label, *scrollbox, *entry, *checkbox;
  GtkAdjustment *adj;
  GList *tmp;

  gchar *listlabels[3];

  setup_dialog = dialog;

  temp_setupservers = NULL;
  for (tmp = g_list_first(setupservers); tmp != NULL; tmp = tmp->next)
  {
    SETUP_SERVER_REC *rec = tmp->data;

    add_server_to_list(&temp_setupservers, rec->ircnet, rec->server,
                       rec->port, rec->password, rec->autoconnect, rec->cmd_queue_speed, NULL);
  }

  vbox1 = gtk_vbox_new (FALSE, 7);
  gtk_container_border_width (GTK_CONTAINER (vbox1), 7);
  
  /* 'IRC servers' frame */
  frmServers = gtk_frame_new (_("IRC Servers"));
  gtk_box_pack_start (GTK_BOX (vbox1), frmServers, TRUE, TRUE, 0);

  vbox2 = gtk_vbox_new (FALSE, 7);
  gtk_container_border_width (GTK_CONTAINER (vbox2), 7);
  gtk_container_add (GTK_CONTAINER (frmServers), vbox2);

  hbox1 = gtk_hbox_new (FALSE, 7);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox1, TRUE, TRUE, 0);

  scrollbox = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox1), scrollbox, TRUE, TRUE, 0);

  listlabels[0] = _("AC"); listlabels[1] = _("IRC Net"); listlabels[2] = _("Server");
  clistServers = gtk_clist_new_with_titles(3, listlabels);
  gtk_object_set_data (GTK_OBJECT (setup_dialog), "clistServers", clistServers);
  gtk_container_add(GTK_CONTAINER(scrollbox), clistServers);
  gtk_clist_set_column_width (GTK_CLIST (clistServers), 1, 60);
  gtk_clist_set_reorderable(GTK_CLIST(clistServers), TRUE);
  gtk_clist_column_titles_passive(GTK_CLIST(clistServers));
  gtk_clist_column_titles_show (GTK_CLIST (clistServers));

  gtk_signal_connect(GTK_OBJECT(clistServers), "row_move", GTK_SIGNAL_FUNC(sig_serverlist_move_row), NULL);
  gtk_signal_connect(GTK_OBJECT(clistServers), "select_row", GTK_SIGNAL_FUNC(sig_serverlist_select_row), NULL);
  gtk_signal_connect(GTK_OBJECT(clistServers), "unselect_row", GTK_SIGNAL_FUNC(sig_serverlist_unselect_row), NULL);

  fill_clist_with_servers(temp_setupservers);

  /* Command buttons for adding servers to 'selected servers' list */
  vbuttonbox1 = gtk_vbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (hbox1), vbuttonbox1, FALSE, TRUE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox1), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (vbuttonbox1), 7);
  cmdAdd = gtk_button_new_with_label (_("Add"));
  gtk_signal_connect_object(GTK_OBJECT(cmdAdd), "clicked",
			    GTK_SIGNAL_FUNC(sig_add_server), GTK_OBJECT(setup_dialog));
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), cmdAdd);
  cmdEdit = gtk_button_new_with_label (_("Edit"));
  gtk_signal_connect_object(GTK_OBJECT(cmdEdit), "clicked",
			    GTK_SIGNAL_FUNC(sig_edit_server), GTK_OBJECT(setup_dialog));
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), cmdEdit);
  cmdRemove = gtk_button_new_with_label (_("Remove"));
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), cmdRemove);
  gtk_signal_connect_object(GTK_OBJECT(cmdRemove), "clicked",
			    GTK_SIGNAL_FUNC(sig_delete_server), GTK_OBJECT(setup_dialog));
  frmUserInfo = gtk_frame_new (_("User information"));
  gtk_box_pack_start (GTK_BOX (vbox1), frmUserInfo, FALSE, TRUE, 0);

  /* skip motd */
  checkbox = gtk_check_button_new_with_label(_("Skip MOTD"));
  server_options[OPTION_SKIP_MOTD].widget = checkbox;
  gtk_box_pack_start(GTK_BOX(vbox2), checkbox, FALSE, FALSE, 0);

  /* reconnect to server time */
  hbox1 = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox1, FALSE, FALSE, 0);

  label = gtk_label_new(_("Seconds to wait before reconnecting (-1 = never)"));
  gtk_box_pack_start(GTK_BOX(hbox1), label, FALSE, FALSE, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (*((gint *) server_options[OPTION_SERVER_RECONNECT].data),
                                              -1, G_MAXINT, 1, 60, 0);
  spin = gtk_spin_button_new(adj, 0, 0);
  server_options[OPTION_SERVER_RECONNECT].widget = spin;
  gtk_widget_set_usize(spin, 80, -1);
  gtk_box_pack_start(GTK_BOX(hbox1), spin, FALSE, FALSE, 0);

  entry = gui_create_labelentry(vbox2, _("Source host (empty=default)"),
				*((gchar **) server_options[OPTION_SOURCE_HOST].data), FALSE);
  server_options[OPTION_SOURCE_HOST].widget = entry;

  /* User information widgets */
  table1 = gtk_table_new (3, 2, FALSE);
  gtk_container_add (GTK_CONTAINER (frmUserInfo), table1);
  gtk_container_border_width (GTK_CONTAINER (table1), 7);
  gtk_table_set_row_spacings (GTK_TABLE (table1), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table1), 7);

  entry = gui_create_tablelabelentry(table1, 0, 0, _("Nick"),
				*((gchar **) server_options[OPTION_DEFAULT_NICK].data));
  server_options[OPTION_DEFAULT_NICK].widget = entry;

  entry = gui_create_tablelabelentry(table1, 0, 1, _("Alternate nick"),
				*((gchar **) server_options[OPTION_ALTERNATE_NICK].data));
  server_options[OPTION_ALTERNATE_NICK].widget = entry;

  entry = gui_create_tablelabelentry(table1, 0, 2, _("User name"),
				*((gchar **) server_options[OPTION_USER_NAME].data));
  server_options[OPTION_USER_NAME].widget = entry;

  entry = gui_create_tablelabelentry(table1, 0, 3, _("Real name"),
				*((gchar **) server_options[OPTION_REAL_NAME].data));
  gtk_signal_connect(GTK_OBJECT(entry), "key_press_event",
		     GTK_SIGNAL_FUNC(setup_sig_keypress), NULL);
  server_options[OPTION_REAL_NAME].widget = entry;

  setup_finish_init(dialog, server_options);

  gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialog),
                                 vbox1, gtk_label_new(_("Servers")));
  return TRUE;
}

static gboolean sig_ircnet_changed(gchar *oldname, gchar *newname)
{
    GList *tmp;

    g_return_val_if_fail(oldname != NULL, FALSE);
    g_return_val_if_fail(newname != NULL, FALSE);

    if (*oldname == '\0') return TRUE;

    for (tmp = g_list_first(temp_setupservers); tmp != NULL; tmp = tmp->next)
    {
        SETUP_SERVER_REC *rec = tmp->data;

        if (strcmp(rec->ircnet, oldname) == 0)
        {
            g_free(rec->ircnet);
            rec->ircnet = g_strdup(newname);
        }
    }

    fill_clist_with_servers(temp_setupservers);
    return TRUE;
}

void init_servers(void)
{
    setup_dialog = NULL;

    signal_add("setup servers apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_add("setup servers destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup servers", (SIGNAL_FUNC) setup_servers);
    signal_add("setup ircnet changed", (SIGNAL_FUNC) sig_ircnet_changed);
}

void deinit_servers(void)
{
    if (setup_dialog != NULL)
        gtk_widget_destroy(setup_dialog);

    signal_remove("setup servers apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_remove("setup servers destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup servers", (SIGNAL_FUNC) setup_servers);
    signal_remove("setup ircnet changed", (SIGNAL_FUNC) sig_ircnet_changed);
}
