/*
 setup-themes.c : Themes notebook entry for preferences

    Copyright (C) 1999 Timo Sirainen
                       Kari Lavikka (tuner@bdb.fi)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../ui-common/txt.h"
#include "setup-int.h"

static GtkWidget *setup_dialog;

THEME_REC *temp_current_theme;
GList *temp_themes;

static void theme_plugin_copy(gchar *key, PLUGIN_THEME_REC *src, THEME_REC *dest)
{
    PLUGIN_THEME_REC *rec;
    gint num, size;
    gchar *str;

    rec = g_new0(PLUGIN_THEME_REC, 1);
    rec->plugin = g_strdup(src->plugin);

    /* copy the formatlist */
    size = 0;

    str = list2str(src->formatlist, " ");
    rec->formatlist = str2list(str, ' ');
    g_free(str);
    size = g_list_length(rec->formatlist);

    /* copy texts */
    rec->format = g_new0(gchar*, size);
    for (num = 0; num < size; num++)
    {
	if (src->format[num] != NULL)
	    rec->format[num] = g_strdup(src->format[num]);
    }

    g_hash_table_insert(dest->plugins, rec->plugin, rec);
}

void themes_copy(GList **src, GList **dest, THEME_REC **old, THEME_REC **new)
{
    THEME_REC *theme;
    GList *tmp;
    gint num;

    *dest = NULL; *new = NULL;
    for (tmp = g_list_first(*src); tmp != NULL; tmp = tmp->next)
    {
        THEME_REC *rec = tmp->data;

	theme = theme_create(rec->path, rec->name);
	*dest = g_list_append(*dest, theme);

	if (rec == *old)
	    *new = theme;

	theme->font = rec->font == NULL ? NULL : g_strdup(rec->font);
	theme->bg_pixmap = rec->bg_pixmap == NULL ? NULL : g_strdup(rec->bg_pixmap);
	theme->flags = rec->flags;
	theme->default_color = rec->default_color;

	/* copy texts */
        for (num = 0; num < TEXTS; num++)
        {
            if (rec->format[num] != NULL)
                theme->format[num] = g_strdup(rec->format[num]);
	}

	/* copy colors */
	theme->gui_data = g_new0(GUI_THEME_REC, 1);
        memcpy(&THEME_GUI(theme)->colors, &THEME_GUI(rec)->colors, sizeof(GdkColor)*COLORS);

	/* copy plugins .. */
        g_hash_table_foreach(rec->plugins, (GHFunc) theme_plugin_copy, theme);
    }
    if (*new == NULL)
	*new = (*dest)->data;
}

static gboolean sig_setup_apply(GtkWidget *dialog)
{
    proplist_t aprop;

    g_return_val_if_fail(dialog != NULL, FALSE);
    g_return_val_if_fail(temp_themes != NULL, TRUE);

    /* free memory used by themes */
    g_list_foreach(themes, (GFunc) theme_destroy, NULL);
    g_list_free(themes);

    /* temp -> real */
    themes_copy(&temp_themes, &themes, &temp_current_theme, &current_theme);

    /* write to configuration file */
    aprop = config_section(&cprop, "appearance");
    config_set_str(aprop, "current_theme", current_theme->name);

    signal_emit("setup themes apply again", 1, dialog);
    return TRUE;
}

static gboolean sig_setup_destroyed(void)
{
    setup_dialog = NULL;

    g_list_foreach(temp_themes, (GFunc) theme_destroy, NULL);
    g_list_free(temp_themes);
    temp_themes = NULL;

    return TRUE;
}

static void sig_change_theme(GtkEntry *entry)
{
    GList *tmp;
    gchar *theme;

    g_return_if_fail(entry != NULL);

    theme = gtk_entry_get_text(entry);

    for (tmp = g_list_first(temp_themes); tmp != NULL; tmp = tmp->next)
    {
        THEME_REC *rec = tmp->data;

        if (strcmp(rec->name, theme) == 0)
        {
	    signal_emit("setup themes redraw", 1, rec);
	    temp_current_theme = rec;
	    break;
        }
    }
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void set_popdown_strings(GtkCombo *combo)
{
    THEME_REC *cur;
    GList *tmp, *list;

    list = NULL; cur = NULL;
    for (tmp = g_list_first(temp_themes); tmp != NULL; tmp = tmp->next)
    {
        THEME_REC *rec = tmp->data;

        list = g_list_append(list, rec->name);
        if (rec == temp_current_theme) cur = rec;
    }
    if (cur == NULL) temp_current_theme = temp_themes->data;

    gtk_combo_set_popdown_strings(combo, list);
    gtk_entry_set_text(GTK_ENTRY(combo->entry), temp_current_theme->name);
}

static void sig_add_theme_ok(GtkWidget *dialog)
{
    THEME_REC *rec;
    GtkCombo *combo;
    gchar *path, *name;

    g_return_if_fail(dialog != NULL);

    combo = gtk_object_get_data(GTK_OBJECT(dialog), "data");
    name = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "entry")));

    if (*name != '\0')
    {
	path = g_strdup_printf("%s/.irssi/%s.theme", g_get_home_dir(), name);

	temp_current_theme = rec = theme_create(path, name);
        rec->default_color = ((THEME_REC *) (themes->data))->default_color;
	rec->gui_data = g_new0(GUI_THEME_REC, 1);
        memcpy(&THEME_GUI(rec)->colors, &THEME_GUI((THEME_REC *) (themes->data))->colors, sizeof(GdkColor)*COLORS);

        temp_themes = g_list_append(temp_themes, rec);
	g_free(path);

        set_popdown_strings(combo);
        gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
    }
    gtk_widget_destroy(dialog);
}

static void sig_add_theme(GtkCombo *combo)
{
    GtkWidget *dialog;

    g_return_if_fail(combo != NULL);

    dialog = gui_entry_dialog(_("Add new theme"), NULL, NULL,
                              GTK_SIGNAL_FUNC(sig_add_theme_ok), combo);
    gui_widget_depends(setup_dialog, dialog);
}

static void sig_delete_theme_ok(GtkWidget *dialog)
{
    GtkCombo *combo;

    combo = gtk_object_get_data(GTK_OBJECT(dialog), "combo");
    gtk_widget_destroy(dialog);

    theme_destroy(temp_current_theme);
    temp_themes = g_list_remove(temp_themes, temp_current_theme);

    set_popdown_strings(combo);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_delete_theme(GtkCombo *combo)
{
    GtkWidget *dialog, *label;
    gchar *str;

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
    gui_widget_depends(setup_dialog, dialog);
    gtk_object_set_data(GTK_OBJECT(dialog), "combo", combo);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_delete_theme_ok), GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    str = g_strdup_printf(_("Really delete theme %s"), temp_current_theme->name);
    label = gtk_label_new(str); g_free(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, TRUE, TRUE, 5);

    gtk_widget_show_all(dialog);
}

static gboolean setup_themes(GtkWidget *dialog)
{
    GtkWidget *vbox, *hbox, *frame, *notebook;
    GtkWidget *combo, *buttonbox, *button;

    setup_dialog = dialog;

    /* copy themes to temp list */
    themes_copy(&themes, &temp_themes, &current_theme, &temp_current_theme);

    /* theme selector */
    vbox = gtk_vbox_new(FALSE, 0);
    frame = gtk_frame_new(_("Theme selector"));
    gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 5);

    hbox = gtk_hbox_new(FALSE, 10);
    gtk_container_border_width(GTK_CONTAINER(hbox), 7);
    gtk_container_add(GTK_CONTAINER(frame), hbox);

    combo = gtk_combo_new();
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
    gtk_combo_set_value_in_list(GTK_COMBO(combo), TRUE, FALSE);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), temp_current_theme->name);
    gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);

    set_popdown_strings(GTK_COMBO(combo));

    buttonbox = gtk_vbutton_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), buttonbox, FALSE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 7);

    button = gtk_button_new_with_label(_("Add..."));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_add_theme), GTK_OBJECT(combo));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Delete"));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_delete_theme), GTK_OBJECT(combo));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);


    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

    signal_emit("setup themes show", 2, dialog, notebook);
    signal_emit("setup themes redraw", 2, temp_current_theme, GINT_TO_POINTER(TRUE));

    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed",
		       GTK_SIGNAL_FUNC(sig_change_theme), NULL);

    gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialog), vbox,
				   gtk_label_new("main"));

    gtk_widget_show_all(dialog); /* removes Gtk-CRITICAL message that hiding tabs creates. */
#ifdef HAVE_GNOME
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(dialog)->notebook), FALSE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(dialog)->notebook), FALSE);
#else
    {
	GnomePropertyBox *box;

	box = gtk_object_get_data(GTK_OBJECT(dialog), "propertybox");
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(box->notebook), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(box->notebook), FALSE);
    }
#endif

    return TRUE;
}

static gboolean sig_theme_destroyed(THEME_REC *theme)
{
    g_free(theme->gui_data);
    theme->gui_data = NULL;
    return TRUE;
}

void init_themes(void)
{
    setup_dialog = NULL;

    signal_add("setup themes apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_add("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup themes", (SIGNAL_FUNC) setup_themes);
    signal_add("theme destroyed", (SIGNAL_FUNC) sig_theme_destroyed);
}

void deinit_themes(void)
{
    g_list_foreach(themes, (GFunc) sig_theme_destroyed, NULL);

    if (setup_dialog != NULL)
        gtk_widget_destroy(setup_dialog);

    signal_remove("setup themes apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_remove("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup themes", (SIGNAL_FUNC) setup_themes);
    signal_remove("theme destroyed", (SIGNAL_FUNC) sig_theme_destroyed);
}
