/*
 server-redirect.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint redirect_group;

void server_eventtable_destroy(gchar *key, GList *value)
{
    GList *tmp;

    g_free(key);

    for (tmp = g_list_first(value); tmp != NULL; tmp = tmp->next)
    {
        REDIRECT_REC *rec = tmp->data;

	if (rec->arg != NULL) g_free(rec->arg);
        g_free(rec->name);
        g_free(rec);
    }
    g_list_free(value);
}

void server_eventgrouptable_destroy(gpointer key, GList *value)
{
    g_list_foreach(value, (GFunc) g_free, NULL);
    g_list_free(value);
}

void server_cmdtable_destroy(gchar *key, REDIRECT_CMD_REC *value)
{
    g_free(key);

    g_list_foreach(value->events, (GFunc) g_free, NULL);
    g_list_free(value->events);
    g_free(value);
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    if (server->eventtable != NULL)
    {
        g_hash_table_foreach(server->eventtable, (GHFunc) server_eventtable_destroy, NULL);
        g_hash_table_destroy(server->eventtable);
    }

    g_hash_table_foreach(server->eventgrouptable, (GHFunc) server_eventgrouptable_destroy, NULL);
    g_hash_table_destroy(server->eventgrouptable);

    if (server->cmdtable != NULL)
    {
        g_hash_table_foreach(server->cmdtable, (GHFunc) server_cmdtable_destroy, NULL);
        g_hash_table_destroy(server->cmdtable);
    }

    return TRUE;
}

void server_redirect_initv(SERVER_REC *server, gchar *command, gint last, GList *list)
{
    REDIRECT_CMD_REC *rec;

    g_return_if_fail(server != NULL);
    g_return_if_fail(command != NULL);
    g_return_if_fail(last > 0);

    if (g_hash_table_lookup(server->cmdtable, command) != NULL)
    {
        /* already in hash table. list of events SHOULD be the same... */
	g_list_foreach(list, (GFunc) g_free, NULL);
	g_list_free(list);
	return;
    }

    rec = g_new(REDIRECT_CMD_REC, 1);
    rec->last = last;
    rec->events = list;
    g_hash_table_insert(server->cmdtable, g_strdup(command), rec);
}

void server_redirect_init(SERVER_REC *server, gchar *command, gint last, ...)
{
    va_list args;
    GList *list;
    gchar *event;

    va_start(args, last);
    list = NULL;
    while ((event = va_arg(args, gchar *)) != NULL)
	list = g_list_append(list, g_strdup(event));
    va_end(args);

    server_redirect_initv(server, command, last, list);
}

gint server_redirect_single_event(SERVER_REC *server, gchar *arg, gboolean last, gint group, gchar *event, gchar *signal, gint argpos)
{
    REDIRECT_REC *rec;
    GList *list, *grouplist;
    gchar *origkey;

    g_return_val_if_fail(server != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(signal != NULL, 0);
    g_return_val_if_fail(arg != NULL || argpos == -1, 0);

    if (group == 0) group = ++redirect_group;

    rec = g_new0(REDIRECT_REC, 1);
    rec->arg = arg == NULL ? NULL : g_strdup(arg);
    rec->argpos = argpos;
    rec->name = g_strdup(signal);
    rec->group = group;
    rec->last = last;

    if (g_hash_table_lookup_extended(server->eventtable, event, (gpointer *) &origkey, (gpointer *) &list))
        g_hash_table_remove(server->eventtable, origkey);
    else
    {
        list = NULL;
        origkey = g_strdup(event);
    }

    grouplist = g_hash_table_lookup(server->eventgrouptable, GINT_TO_POINTER(group));
    if (grouplist != NULL) g_hash_table_remove(server->eventgrouptable, GINT_TO_POINTER(group));

    list = g_list_append(list, rec);
    grouplist = g_list_append(grouplist, g_strdup(event));

    g_hash_table_insert(server->eventtable, origkey, list);
    g_hash_table_insert(server->eventgrouptable, GINT_TO_POINTER(group), grouplist);

    return group;
}

void server_redirect_event(SERVER_REC *server, gchar *arg, gint last, ...)
{
    va_list args;
    gchar *event, *signal;
    gint argpos, group;

    g_return_if_fail(server != NULL);

    va_start(args, last);

    group = 0;
    while ((event = va_arg(args, gchar *)) != NULL)
    {
        signal = va_arg(args, gchar *);
	argpos = va_arg(args, gint);
	group = server_redirect_single_event(server, arg, last > 0, group, event, signal, argpos);
	last--;
    }

    va_end(args);
}

void server_redirect_default(SERVER_REC *server, gchar *command)
{
    REDIRECT_CMD_REC *cmdrec;
    REDIRECT_REC *rec;
    GList *events, *list, *grouplist;
    gchar *event, *origkey;
    gint last;

    g_return_if_fail(server != NULL);
    g_return_if_fail(command != NULL);

    if (server->cmdtable == NULL)
        return; /* not connected yet */

    cmdrec = g_hash_table_lookup(server->cmdtable, command);
    if (cmdrec == NULL) return;

    /* add all events used by command to eventtable and eventgrouptable */
    redirect_group++; grouplist = NULL; last = cmdrec->last;
    for (events = cmdrec->events; events != NULL; events = events->next, last--)
    {
        event = events->data;

        if (g_hash_table_lookup_extended(server->eventtable, event, (gpointer *) &origkey, (gpointer *) &list))
            g_hash_table_remove(server->eventtable, origkey);
        else
        {
            list = NULL;
            origkey = g_strdup(event);
        }

	rec = g_new0(REDIRECT_REC, 1);
	rec->argpos = -1;
        rec->name = g_strdup(event);
        rec->group = redirect_group;
        rec->last = last > 0;

        grouplist = g_list_append(grouplist, g_strdup(event));
        list = g_list_append(list, rec);
        g_hash_table_insert(server->eventtable, origkey, list);
    }

    g_hash_table_insert(server->eventgrouptable, GINT_TO_POINTER(redirect_group), grouplist);
}

void server_redirect_remove_next(SERVER_REC *server, gchar *event, GList *item)
{
    REDIRECT_REC *rec;
    GList *grouplist, *list, *events, *tmp;
    gchar *origkey;
    gint group;

    g_return_if_fail(server != NULL);
    g_return_if_fail(event != NULL);

    if (!g_hash_table_lookup_extended(server->eventtable, event, (gpointer *) &origkey, (gpointer *) &list))
        return;

    rec = item == NULL ? list->data : item->data;
    if (!rec->last)
    {
	/* this wasn't last expected event */
	return;
    }
    group = rec->group;

    /* get list of events from this group */
    grouplist = g_hash_table_lookup(server->eventgrouptable, GINT_TO_POINTER(group));

    /* remove all of them */
    for (list = g_list_first(grouplist); list != NULL; list = list->next)
    {
        event = list->data;

	if (!g_hash_table_lookup_extended(server->eventtable, event, (gpointer *) &origkey, (gpointer *) &events))
	{
	    g_warning("server_redirect_remove_next() : event in eventgrouptable but not in eventtable");
	    continue;
	}

	/* remove the right group */
	for (tmp = events; tmp != NULL; tmp = tmp->next)
	{
	    rec = tmp->data;

	    if (rec->group == group)
		break;
	}

	if (rec == NULL)
	{
	    g_warning("server_redirect_remove_next() : event in eventgrouptable but not in eventtable (group)");
	    continue;
	}

	g_free(event);

	events = g_list_remove(events, rec);
	if (rec->arg != NULL) g_free(rec->arg);
	g_free(rec->name);
	g_free(rec);

	/* update hash table */
	g_hash_table_remove(server->eventtable, origkey);
	if (events == NULL)
	    g_free(origkey);
	else
	    g_hash_table_insert(server->eventtable, origkey, events);
    }

    g_hash_table_remove(server->eventgrouptable, GINT_TO_POINTER(group));
    g_list_free(grouplist);
}

GList *server_redirect_getqueue(SERVER_REC *server, gchar *event, gchar *args)
{
    REDIRECT_REC *rec;
    GList *list, *arglist;
    gchar *arg;
    gboolean found;

    list = g_hash_table_lookup(server->eventtable, event);

    for (; list != NULL; list = list->next)
    {
	rec = list->data;
	if (rec->argpos == -1)
	    break;

	/* we need to check if the argument is right.. */
	arglist = str2list(args, ' ');
	arg = g_list_nth_data(arglist, rec->argpos);

	found = (rec->arg != NULL && arg != NULL && find_substr(rec->arg, arg));

	if (arglist != NULL)
	{
	    g_free(arglist->data);
	    g_list_free(arglist);
	}

	if (found) break;
    }

    return list;
}

void servers_redirect_init(void)
{
    redirect_group = 0;

    signal_add("server disconnected", (SIGNAL_FUNC) sig_disconnected);
}

void servers_redirect_deinit(void)
{
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);
}
