/*
 perl.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <config.h>
#undef PACKAGE

#ifdef HAVE_PERL
#include <EXTERN.h>
#ifndef _SEM_SEMUN_UNDEFINED
#define HAS_UNION_SEMUN
#endif
#include <perl.h>

#undef _
#undef PACKAGE
#include "irssi.h"

extern void xs_init(void);

typedef struct
{
    gchar *signal;
    gchar *args[7];
}
PERL_SIGNAL_ARGS_REC;

typedef struct
{
    gchar *signal;
    gchar *func;
}
PERL_SIGNAL_REC;

typedef struct
{
    gint tag;
    gchar *func;
    gchar *data;
}
PERL_TIMEOUT_REC;

#include "perl-signals.h"

static GSList *perl_signals, *perl_timeouts;
static PerlInterpreter *irssi_perl_interp;

static void perl_signal_destroy(PERL_SIGNAL_REC *rec)
{
    perl_signals = g_slist_remove(perl_signals, rec);

    if (strncmp(rec->signal, "command ", 8) == 0)
	command_unbind(rec->signal+8, NULL);

    g_free(rec->signal);
    g_free(rec->func);
    g_free(rec);
}

static void perl_timeout_destroy(PERL_TIMEOUT_REC *rec)
{
    perl_timeouts = g_slist_remove(perl_timeouts, rec);

    gui_timeout_remove(rec->tag);
    g_free(rec->func);
    g_free(rec->data);
    g_free(rec);
}

static void irssi_perl_start(void)
{
    /* stolen from xchat, thanks :) */
    gchar *args[] = {"", "-e", "0"};
    gchar load_file[] =
	"sub load_file()\n"
	"{\n"
	"  (my $file_name) = @_;\n"
	"  open FH, $file_name or return 2;\n"
	"  local($/) = undef;\n"
	"  $file = <FH>;\n"
	"  close FH;\n"
	"  eval $file;\n"
	"  eval $file if $@;\n"
	"  return 1 if $@;\n"
	"  return 0;\n"
	"}";

    perl_signals = NULL;
    perl_timeouts = NULL;

    irssi_perl_interp = perl_alloc();
    perl_construct(irssi_perl_interp);

    perl_parse(irssi_perl_interp, xs_init, 3, args, NULL);
    perl_eval_pv(load_file, TRUE);
}

static void irssi_perl_stop(void)
{
    while (perl_signals != NULL)
	perl_signal_destroy(perl_signals->data);

    while (perl_timeouts != NULL)
	perl_timeout_destroy(perl_timeouts->data);

    perl_destruct(irssi_perl_interp);
    perl_free(irssi_perl_interp);
    irssi_perl_interp = NULL;
}

static gboolean cmd_run(gchar *data)
{
    dSP;
    struct stat statbuf;
    gchar *fname;
    int retcount;

    /* add .pl suffix if it's missing */
    data = (strlen(data) <= 3 || strcmp(data+strlen(data)-3, ".pl") == 0) ?
	g_strdup(data) : g_strdup_printf("%s.pl", data);

    if (*data == G_DIR_SEPARATOR)
    {
	/* whole path specified */
	fname = g_strdup(data);
    }
    else
    {
        /* check from ~/.irssi/scripts/ */
	fname = g_strdup_printf("%s/.irssi/scripts/%s", g_get_home_dir(), data);
	if (stat(fname, &statbuf) != 0)
	{
	    /* check from SCRIPTDIR */
	    g_free(fname),
	    fname = g_strdup_printf(SCRIPTDIR"/%s", data);
	}
    }
    g_free(data);

    ENTER;
    SAVETMPS;

    PUSHMARK(SP);
    XPUSHs(sv_2mortal(newSVpv(fname, strlen(fname)))); g_free(fname);
    PUTBACK;

    retcount = perl_call_pv("load_file", G_EVAL|G_SCALAR);
    SPAGAIN;

    if (SvTRUE(ERRSV))
    {
	STRLEN n_a;

	signal_emit("perl error", 1, SvPV(ERRSV, n_a));
        (void)POPs;
    }
    else while (retcount--) (void)POPi;

    PUTBACK;
    FREETMPS;
    LEAVE;

    return TRUE;
}

static gboolean cmd_flush(gchar *data)
{
    irssi_perl_stop();
    irssi_perl_start();
    return TRUE;
}

void perl_signal_add(gchar *signal, gchar *func)
{
    PERL_SIGNAL_REC *rec;

    rec = g_new(PERL_SIGNAL_REC, 1);
    rec->signal = g_strdup(signal);
    rec->func = g_strdup(func);

    perl_signals = g_slist_append(perl_signals, rec);
}

void perl_signal_remove(gchar *signal, gchar *func)
{
    GSList *tmp;

    for (tmp = perl_signals; tmp != NULL; tmp = tmp->next)
    {
	PERL_SIGNAL_REC *rec = tmp->data;

	if (strcmp(signal, rec->signal) == 0 && strcmp(func, rec->func) == 0)
	{
	    perl_signal_destroy(rec);
	    break;
	}
    }
}


static gint perl_timeout(PERL_TIMEOUT_REC *rec)
{
    dSP;
    int retcount;

    ENTER;
    SAVETMPS;

    PUSHMARK(SP);
    XPUSHs(sv_2mortal(newSVpv(rec->data, strlen(rec->data))));
    PUTBACK;

    retcount = perl_call_pv(rec->func, G_EVAL|G_SCALAR);
    SPAGAIN;

    if (SvTRUE(ERRSV))
    {
	STRLEN n_a;

	signal_emit("perl error", 1, SvPV(ERRSV, n_a));
        (void)POPs;
    }
    else while (retcount--) (void)POPi;

    PUTBACK;
    FREETMPS;
    LEAVE;

    return 1;
}

gint perl_timeout_add(gint msecs, gchar *func, gchar *data)
{
    PERL_TIMEOUT_REC *rec;

    rec = g_new(PERL_TIMEOUT_REC, 1);
    rec->func = g_strdup(func);
    rec->data = g_strdup(data);
    rec->tag = gui_timeout_add(msecs, (GUITimeoutFunction) perl_timeout, rec);

    perl_timeouts = g_slist_append(perl_timeouts, rec);
    return rec->tag;
}

void perl_timeout_remove(gint tag)
{
    GSList *tmp;

    for (tmp = perl_timeouts; tmp != NULL; tmp = tmp->next)
    {
	PERL_TIMEOUT_REC *rec = tmp->data;

	if (rec->tag == tag)
	{
	    perl_timeout_destroy(rec);
	    break;
	}
    }
}

static gboolean call_perl(gchar *func, gchar *signal, va_list va)
{
    dSP;
    PERL_SIGNAL_ARGS_REC *rec;
    int retcount, n, ret;
    gpointer arg;
    HV *stash;

    /* find the signal argument types */
    rec = NULL;
    for (n = 0; perl_signal_args[n].signal != NULL; n++)
    {
	if (strncmp(signal, perl_signal_args[n].signal,
		    strlen(perl_signal_args[n].signal)) == 0)
	{
	    rec = &perl_signal_args[n];
	    break;
	}
    }

    ENTER;
    SAVETMPS;

    PUSHMARK(sp);

    if (rec != NULL)
    {
	/* put the arguments to perl stack */
	for (n = 0; n < 7; n++)
	{
	    arg = va_arg(va, gpointer);

            if (rec->args[n] == NULL)
                break;

	    if (strcmp(rec->args[n], "string") == 0)
		XPUSHs(sv_2mortal(newSVpv(arg == NULL ? "" : arg, arg == NULL ? 0 : strlen(arg))));
	    else if (strcmp(rec->args[n], "int") == 0)
		XPUSHs(sv_2mortal(newSViv(GPOINTER_TO_INT(arg))));
	    else if (strcmp(rec->args[n], "ulongptr") == 0)
		XPUSHs(sv_2mortal(newSViv(*(gulong *) arg)));
	    else if (strcmp(rec->args[n], "Irssi::List") == 0)
	    {
		LIST_REC *rec = arg;
		AV *av = newAV();

		av_push(av, newSVpv(rec->key, strlen(rec->key)));
		av_push(av, newSVpv(rec->value, strlen(rec->value)));
		XPUSHs(sv_2mortal(newRV_inc((SV*) av)));
	    }
	    else if (strncmp(rec->args[n], "glist_", 6) == 0)
	    {
		GList *tmp;

		stash = gv_stashpv(rec->args[n]+6, 0);
		for (tmp = arg; tmp != NULL; tmp = tmp->next)
		    XPUSHs(sv_2mortal(sv_bless(newRV_noinc(newSViv(GPOINTER_TO_INT(tmp->data))), stash)));
	    }
	    else
	    {
		stash = gv_stashpv(rec->args[n], 0);
		XPUSHs(sv_2mortal(sv_bless(newRV_noinc(newSViv(GPOINTER_TO_INT(arg))), stash)));
	    }
	}
    }

    PUTBACK;
    retcount = perl_call_pv(func, G_EVAL|G_SCALAR);
    SPAGAIN;

    ret = 0;
    if (SvTRUE(ERRSV))
    {
	STRLEN n_a;

	signal_emit("perl error", 1, SvPV(ERRSV, n_a));
        (void)POPs;
    }
    else
    {
	SV *sv;

	if (retcount > 0)
	{
	    sv = POPs;
            if (SvIOK(sv) && SvIV(sv) == 1) ret = 1;
	}
	for (n = 2; n <= retcount; n++)
	    (void)POPi;
    }

    PUTBACK;
    FREETMPS;
    LEAVE;

    return ret;
}

static gboolean sig_signal(gchar *signal, ...)
{
    GSList *tmp;
    va_list va;

    va_start(va, signal);

    for (tmp = perl_signals; tmp != NULL; tmp = tmp->next)
    {
	PERL_SIGNAL_REC *rec = tmp->data;

	if (strcmp(rec->signal, signal) != 0)
	    continue;

	if (call_perl(rec->func, signal, va))
	    return FALSE;
    }

    va_end(va);
    return TRUE;
}

static void irssi_perl_autorun(void)
{
    DIR *dirp;
    struct dirent *dp;
    gchar *path, *fname;

    path = g_strdup_printf("%s/.irssi/scripts/autorun", g_get_home_dir());
    dirp = opendir(path);
    if (dirp == NULL)
    {
	g_free(path);
	return;
    }

    for (;;)
    {
	dp = readdir(dirp);
	if (dp == NULL) break;

	fname = g_strdup_printf("%s/%s", path, dp->d_name);
        cmd_run(fname);
	g_free(fname);
    }
    closedir(dirp);
    g_free(path);
}

void irssi_perl_init(void)
{
    command_bind("run", NULL, (SIGNAL_FUNC) cmd_run);
    command_bind("perlflush", NULL, (SIGNAL_FUNC) cmd_flush);
    signal_add("signal", (SIGNAL_FUNC) sig_signal);

    irssi_perl_start();
    irssi_perl_autorun();
}

void irssi_perl_deinit(void)
{
    irssi_perl_stop();

    command_unbind("run", (SIGNAL_FUNC) cmd_run);
    command_unbind("perlflush", (SIGNAL_FUNC) cmd_flush);
    signal_remove("signal", (SIGNAL_FUNC) sig_signal);
}
#else
void irssi_perl_init(void)
{
}

void irssi_perl_deinit(void)
{
}
#endif
