/*
 setup-themes.c : Irssi settings

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* Add all *.theme files from directory to themes */
static void find_themes(gchar *path)
{
    DIR *dirp;
    struct dirent *dp;
    GList *tmp;
    gchar *fname, *name;
    gint len;

    dirp = opendir(path);
    if (dirp == NULL) return;

    for (;;)
    {
	dp = readdir(dirp);
	if (dp == NULL) break;

	len = strlen(dp->d_name);
	if (len > 6 && strcmp(dp->d_name+len-6, ".theme") == 0)
	{
	    /* first check if it's duplicate */
	    name = g_strdup(dp->d_name);
	    name[strlen(name)-6] = '\0';

	    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
	    {
		THEME_REC *rec = tmp->data;

		if (g_strcasecmp(rec->name, name) == 0)
		{
		    /* dupe! */
		    break;
		}
	    }

	    if (tmp == NULL)
	    {
                fname = g_strdup_printf("%s/%s", path, dp->d_name);
		themes = g_list_append(themes, theme_create(fname, name));
		g_free(fname);
	    }
	    g_free(name);
	}
    }
    closedir(dirp);
}

/* Read plugin texts into theme */
static void theme_read_plugin_texts(gchar *hashkey, PLUGIN_THEME_REC *rec, proplist_t tprop)
{
    proplist_t formats, formatkeys, pkey, pvalue;
    gchar *key, *value;
    gint num, max;
    GList *tmp;

    formats = config_get_prop(config_get_prop(tprop, "pluginformats"), rec->plugin);
    if (formats == NULL) return;

    formatkeys = PLGetAllDictionaryKeys(formats);
    max = formatkeys == NULL ? 0 : PLGetNumberOfElements(formatkeys);
    for (num = 0; num < max; num++)
    {
	/* get the key & value */
	pkey = PLGetArrayElement(formatkeys, num);
	pvalue = PLGetDictionaryEntry(formats, pkey);
	if (pkey == NULL || pvalue == NULL) continue; /* hm?? */

	key = PLGetString(pkey);
	value = PLGetString(pvalue);

	tmp = glist_find_icase_string(rec->formatlist, key);
	if (tmp != NULL)
	{
	    rec->format[g_list_position(rec->formatlist, tmp)] =
		g_strdup(value);
	}
    }
    if (formatkeys != NULL) PLRelease(formatkeys);
}

static gboolean theme_read(THEME_REC *theme, gchar *path)
{
    PLUGIN_THEME_REC *prec;
    proplist_t tprop, formats, formatkeys, pkey, pvalue;
    gchar *key, *value;
    gboolean errors;
    gint num, max, fnum;

    errors = FALSE;

    tprop = PLGetProplistWithPath(path);
    if (tprop == NULL)
    {
	/* Hm.. some error in theme, abort. */
	theme->default_color = 15;
        return FALSE;
    }

    /* default color */
    theme->default_color = config_get_int(tprop, "default_color", 15);
    /* get font */
    value = config_get_str(tprop, "font", NULL);
    theme->font = (value == NULL || *value == '\0') ? NULL : g_strdup(value);

    /* get background pixmap */
    value = config_get_str(tprop, "bg_pixmap", NULL);
    theme->bg_pixmap = (value == NULL || *value == '\0') ? NULL : g_strdup(value);
    /* get background pixmap properties */
    if (config_get_bool(tprop, "bg_scrollable", FALSE))
	theme->flags |= THEME_FLAG_BG_SCROLLABLE;
    if (config_get_bool(tprop, "bg_scaled", TRUE))
	theme->flags |= THEME_FLAG_BG_SCALED;
    if (config_get_bool(tprop, "bg_shaded", FALSE))
	theme->flags |= THEME_FLAG_BG_SHADED;

    /* Read the standard formats */
    formats = config_get_prop(tprop, "formats");
    formatkeys = formats == NULL ? NULL : PLGetAllDictionaryKeys(formats);

    max = formatkeys == NULL ? 0 : PLGetNumberOfElements(formatkeys);
    for (num = 0; num < max; num++)
    {
	/* get the key & value */
	pkey = PLGetArrayElement(formatkeys, num);
	pvalue = PLGetDictionaryEntry(formats, pkey);
	if (pkey == NULL || pvalue == NULL) continue; /* hm?? */

	key = PLGetString(pkey);
	value = PLGetString(pvalue);

	/* process.. */
	value = g_strdup(value);
	fnum = ui_formats_find(key);
	if (fnum == -1 || !ui_formats_check(fnum, value))
	{
	    /* tag not found or error in format.. */
	    PLRemoveDictionaryEntry(formats, pkey);
	    errors = TRUE;
	    g_free(value);
	}
	else
	{
	    theme->format[fnum] = value;
	}
    }
    if (formatkeys != NULL) PLRelease(formatkeys);

    /* Read plugins that are defined in this theme. */
    formats = config_get_prop(tprop, "plugins");
    formatkeys = formats == NULL ? NULL : PLGetAllDictionaryKeys(formats);

    max = formatkeys == NULL ? 0 : PLGetNumberOfElements(formatkeys);
    for (num = 0; num < max; num++)
    {
	/* get the key & value */
	pkey = PLGetArrayElement(formatkeys, num);
	pvalue = PLGetDictionaryEntry(formats, pkey);
	if (pkey == NULL || pvalue == NULL) continue; /* hm?? */

	key = PLGetString(pkey);
	value = PLGetString(pvalue);

	prec = g_new0(PLUGIN_THEME_REC, 1);
	prec->plugin = g_strdup(key);
	prec->formatlist = str2list(value, ' ');
	prec->format = g_new0(gchar*, g_list_length(prec->formatlist));
	g_hash_table_insert(theme->plugins, prec->plugin, prec);
    }

    /* Read the texts inside the plugin */
    g_hash_table_foreach(theme->plugins, (GHFunc) theme_read_plugin_texts, tprop);

    if (errors && !readonly)
    {
	/* errors fixed - save the theme */
	if (!PLSave(tprop, TRUE))
	{
	    /* we probably tried to save to global directory where we
               didn't have access.. try saving it to home dir instead. */
	    proplist_t fname;
	    gchar *str;

	    /* check that we really didn't try to save it to home dir.. */
	    str = g_strdup_printf("%s/.irssi/", g_get_home_dir());
	    if (strncmp(path, str, strlen(str)) != 0)
	    {
		g_free(str);
		str = g_strdup_printf("%s/.irssi/%s", g_get_home_dir(), g_basename(path));

		fname = PLMakeString(str);
		PLSetFilename(tprop, fname);
		PLRelease(fname);

		PLSave(tprop, TRUE);
	    }
	    g_free(str);
	}
    }
    PLRelease(tprop);

    return errors;
}

static gboolean sig_formats_error(void)
{
    signal_emit("gui dialog", 2, "warning",
		"Your theme(s) had some old format strings, "
		"these have been changed back to their default values.");
    signal_remove("irssi init finished", (SIGNAL_FUNC) sig_formats_error);
    return TRUE;
}

void setup_themes_init(void)
{
    THEME_REC *rec;
    GList *tmp;
    gchar *str, *value;
    gboolean errors;

    /* read list of themes */
    str = g_strdup_printf("%s/.irssi", g_get_home_dir());
    find_themes(str);
    g_free(str);
    find_themes(SYSCONFDIR"/irssi");

    /* read formats for all themes */
    errors = FALSE;
    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
        rec = tmp->data;

	if (theme_read(rec, rec->path))
	    errors = TRUE;
    }

    if (errors)
	signal_add("irssi init finished", (SIGNAL_FUNC) sig_formats_error);

    /* find the current theme to use */
    value = config_get_str(config_get_prop(cprop, "appearance"),
			   "current_theme", "default");

    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
	rec = tmp->data;

	if (g_strcasecmp(rec->name, value) == 0)
	{
	    current_theme = rec;
	    break;
	}
    }
}

void setup_themes_deinit(void)
{
}
