/*
 * user_list.cpp - implementation of user-list for side-bar
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <qfiledialog.h>
#include <qlistbox.h>
#include <qvbox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qcursor.h>
#include <qwhatsthis.h>

#include "user_list.h"
#include "client.h"
#include "system_environment.h"
#include "client_manager.h"
#include "embed.h"

#include "user_list.moc"


userList::userList( QWidget * _parent ) :
	sideBarWidget( tr( "Logged in users" ), embed::getIconPixmap( "users" ), _parent )
{
	m_list = new QListBox( contentParent() );
	m_list->setSelectionMode( QListBox::Single );
	QWhatsThis::add( m_list, tr( "Here you see the real names and the user-names of the users, logged in at currently visible "
					"clients." ) );
	connect( m_list, SIGNAL( contextMenuRequested( QListBoxItem *, const QPoint & ) ), this,
			SLOT( contextMenuHandler( QListBoxItem *, const QPoint & ) ) );

	m_exportToFileBtn = new QPushButton( embed::getIconPixmap( "filesave" ), tr( "Export to text-file" ), contentParent() );
	connect( m_exportToFileBtn, SIGNAL( clicked() ), this, SLOT( clickedExportToFile() ) );
	QWhatsThis::add( m_exportToFileBtn, tr( "Use this button for exporting this user-list into a text-file. You can use this file "
						"later for collecting files after an exam has finished. This is sometimes neccessary, "
						"because some users might have finished and logged out earlier and so you cannot collect "
						"their files at the end of the exam." ) );
}




userList::~userList()
{
}




void userList::reload( void )
{
	QStringList users = userList::getLoggedInUsers();

	QString selected_user;

#if QT_VERSION >= 0x030100
	if( m_list->selectedItem() != NULL )
	{
		selected_user = m_list->text( m_list->index(m_list->selectedItem() ) );
#else
	if( m_list->currentText() != QString::null )
	{
		selected_user = m_list->currentText();
#endif
	}

	m_list->clear();
	m_list->insertStringList( users );
	m_list->sort();

	if( selected_user != "" && m_list->findItem( selected_user ) != NULL )
	{
		m_list->setSelected( m_list->findItem( selected_user ), TRUE);
	}
}




client * userList::getClientFromUser( const QString _user )
{
	QValueVector<client *> clients = clientManager::inst()->visibleClients();

	// loop through all clients
	for( Q_UINT16 cl = 0; cl < clients.size(); ++cl )
	{
		QString cur_user = clients[cl]->user();
		if( cur_user != "none" && cur_user != "" && ( systemEnvironment::realUserName(cur_user)+" ("+cur_user+")" ) == _user )
		{
			return( clients[cl] );
		}
	}
	return( NULL );
}




QString userList::getUserFromClient( const QString _ip )
{
	QValueVector<client *> clients = clientManager::inst()->visibleClients();

	// loop through all clients
	for( Q_UINT16 cl = 0; cl < clients.size(); ++cl )
	{
		if( clients[cl]->ip() == _ip )
			return( clients[cl]->user() );

	}
	return( "" );
}




void userList::contextMenuHandler( QListBoxItem * _i, const QPoint & )
{
	QString cm_user = m_list->text( m_list->index( _i ) );

	client * c = userList::getClientFromUser( cm_user );

	if( c != NULL )
	{
		QPopupMenu context_menu( this );
		c->createActionMenu( &context_menu );
		context_menu.exec( QCursor::pos() );
	}
}




void userList::clickedExportToFile( void )
{
	QFileDialog ofd( QDir::home().path(), QString::null, this, "", TRUE );
	ofd.setMode( QFileDialog::AnyFile );
	ofd.setCaption( tr("Select output-file") );
	if( ofd.exec() == QDialog::Rejected )
	{
		return;
	}

	QStringList users = userList::getLoggedInUsers();
	QString output = QDateTime::currentDateTime().toString() + "\n";

	for( QStringList::Iterator it = users.begin(); it != users.end(); ++it )
	{
		output += *it + "\n";
	}

	QFile outfile( ofd.selectedFile() );
	outfile.open( IO_WriteOnly );
	outfile.writeBlock( output.data(), output.length() );
	outfile.close();
}




QStringList userList::getLoggedInUsers( void )
{
	QValueVector<client *> clients = clientManager::inst()->visibleClients();

	QStringList users;

	// loop through all clients
	for( Q_UINT16 cl = 0; cl < clients.size(); ++cl )
	{
		QString user = clients[cl]->user();
		if( user != "none" && user != "" )
		{
			users.push_back( systemEnvironment::realUserName( user )+" ("+user+")" );
		}
	}
	return( users );
}
