#****************************************************************************
#*                               sim_cmds.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements simulator-generic simulation commands
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

namespace eval SimCmds {
    variable d_simMsgBuf             ""
    variable d_sim                   ""
    variable d_defaultRunLength      100
    variable d_simTime               0
    variable d_simRunIncrement       1000
    variable d_breakRequest          0
    variable d_stdout                ""
    variable d_runstepStartCB        ""
    variable d_runstepUpdateCB       ""
    variable d_runstepEndCB          ""
    variable d_designFile            ""
    variable d_designLoaded          false
    variable d_simRunning            0

    variable d_resStrings            [list s ms us ns ps fs]
    variable d_simResolution         ""
    variable d_resMultList           [list 1 1000 1000000 1000000000 \
                                           1000000000000 1000000000000000]

    namespace export load_design
    namespace export run
    namespace export close_design
    namespace export restart_design
}

#********************************************************************
#* SimCmds::SimCmds()
#*
#* Constructor for the SimCmds class
#********************************************************************
proc SimCmds::SimCmds {} {

}

#********************************************************************
#* DesignLoaded
#********************************************************************
proc SimCmds::DesignLoaded {} {
    variable d_designLoaded

    return $d_designLoaded
}

#********************************************************************
#* SimMsgCB
#********************************************************************
proc SimCmds::SimMsgCB {ch} {
    variable d_simMsgBuf

    set tmp [read $ch]

    set d_simMsgBuf "${d_simMsgBuf}${tmp}"

    set idx 0

    while {$idx >= 0} {
        set idx [string first "\n" $d_simMsgBuf]

        if {$idx < 0} {
            break
        }

        set msg [string range $d_simMsgBuf 0 [expr $idx - 1]]
        set d_simMsgBuf [string range $d_simMsgBuf \
            [expr $idx + 1] end]
        ivi_puts "$msg"
    }
}

#********************************************************************
#* FindType
#********************************************************************
proc SimCmds::FindType {path} {

    set fs [file extension $path]
    set type ""

    foreach sim [sim_mgr sim_list] {
        foreach suffix [sim_mgr get_suffixes $sim] {
            if {$fs == $suffix} {
                set type $sim
                break
            }
        }

        if {$type != ""} {
            break
        }
    }

    return $type
}

#********************************************************************
#* load_design
#********************************************************************
proc SimCmds::load_design args {
    variable d_sim
    variable d_stdout
    variable d_designFile
    variable d_simResolution
    variable d_designLoaded
    global   CallbackTypes
    global   WidgetTypes

    if {$d_sim != ""} {
        error "Design already loaded"
    }

    set path [lindex $args 0]

    set      type ""

    set options [lrange $args 1 end]
    while {[llength $options] > 0} {
        set opt [lindex $options 0]
        set shift 1

        if {$opt == "-type"} {
            set type [lindex $options 1]
            set shift 2
        } else {
            error "unknown option \"$opt\""
        }

        set options [lrange $options $shift end]
    }

    if {$type == ""} {
        set type [FindType $path]

        if {$type == ""} {
            error "Cannot auto-detect type of design \"$path\""
        }
    }

    if {[catch {set d_sim [sim ::IviSim.default -type $type]} res]} {
        error "Cannot construct simulator: $res"
    }

    if {[catch {$d_sim load_design $path} res]} {
        ivi_puts "ERROR: Problem loading design"
        error    "problem loading design \"$res\""
    } else {

        set d_stdout [$d_sim get_stdout]
        fileevent $d_stdout readable [list SimCmds::SimMsgCB $d_stdout]

        set d_simResolution [$d_sim res_string]
        set d_runstepStartCB [callback add $CallbackTypes(SIM_RUNSTEP_START) \
            $d_sim SimCmds::UpdateTimeText]
        set d_runstepUpdateCB [callback add $CallbackTypes(SIM_RUNSTEP_UPDATE) \
            $d_sim SimCmds::UpdateTimeText]
        set d_runstepEndCB   [callback add $CallbackTypes(SIM_RUNSTEP_END) \
            $d_sim SimCmds::UpdateTimeText]

        set ddb [$d_sim cget -ddb]
        set top_lev [$ddb glob -modules .]

        set d_designFile $path

        SetStatBarText "Design: $top_lev"

        ivi_puts "NOTE: Loaded design \"$top_lev\" from file \"$path\""
    }

    callback invoke $CallbackTypes(SIM_DESIGN_LOAD) null
    set d_designLoaded true
}

#********************************************************************
#* restart_design
#********************************************************************
proc SimCmds::restart_design args {
    variable    d_sim
    variable    d_designFile
    variable    d_simRunning

    if {$d_sim == ""} {
        error "no design to restart"
    }

    if {$d_simRunning == 1} {
        ivi_puts "Cannot Restart - simulation running"
        return
    }

    ivi_puts "Note: Restarting Simulation"

    #**** First, must capture the state of windows
    tcl_channel_obj design_restart_channel
    if {[catch {wave_save_fmt_fp design_restart_channel -no_cursors}]} {
        error "problem saving setup"
    }

    wave_clear -win all

    set designFile $d_designFile

    if {[catch {close_design} res]} {
        error "problem closing design: $res"
    }

    #**** Some delete tasks are scheduled...
    update idletasks

    if {[catch {load_design $designFile} res]} {
        error "problem re-loading design: $res"
    }

    design_restart_channel execute
    design_restart_channel delete

    ivi_puts "Note: Simulation Restart Complete"
}

#********************************************************************
#* parse_run_amt
#********************************************************************
proc SimCmds::parse_run_amt {str} {
    variable d_resStrings
    variable d_simResolution
    variable d_resMultList

    set matchn [scan $str "%d%s" len units]

    if {$matchn == -1 || $matchn == 0} {
        error "time-length \"$str\" is invalid"
    } elseif {$matchn == 1} {
        return $len
    } elseif {$matchn == 2} {
        set simUnitIdx   [lsearch -exact $d_resStrings $d_simResolution]
        set specUnitIdx  [lsearch -exact $d_resStrings $units]

        if {$simUnitIdx < 0} {
            puts "Internal Error: simUnitIdx not found"
        }

        if {$specUnitIdx < 0} {
            error "Unknown unit \"$units\""
        }

        set run_length $len
        if {$simUnitIdx > $specUnitIdx} {
            set factor [lindex $d_resMultList [expr $simUnitIdx - $specUnitIdx]]

            set run_length [expr $len * $factor]
        } elseif {$simUnitIdx < $specUnitIdx} {
            #**** This isn't good: we will round off somewhat...
            set factor [lindex $d_resMultList [expr $specUnitIdx - $simUnitIdx]]

            set run_length [expr $len / $factor]

            if {$run_length <= 0} {
                ivi_puts [join [list "WARNING: run resolution \"$units\" is " \
                  "smaller than simulation resolution \"$d_simResolution\""]]
                set run_length 1
            } elseif {[expr $len % $factor] > 0} {
                ivi_puts [join [list "WARNING: run resolution \"$units\" is " \
                  "smaller than simulation resolution \"$d_simResolution\""]]
            }
        } else {
            #**** Units equal...
        }

        return $run_length
    } else {
        puts "Internal Error: SimCmds::parse_run_amt - matchn invalid ($matchn)"
    }
}

#********************************************************************
#* run
#********************************************************************
proc SimCmds::run args {
    variable d_defaultRunLength
    variable d_simTime
    variable d_breakRequest
    variable d_sim
    variable d_simRunIncrement
    variable d_simRunning
    global   CallbackTypes

    set d_breakRequest 0
    set run_all 0

    if {$d_sim == ""} {
        error "No design loaded"
    }

    set d_simRunning 1
    callback invoke $CallbackTypes(SIM_RUNSTEP_START) $d_sim [$d_sim time]

    if {[llength $args] == 0} {
        set run_len $d_defaultRunLength
    } elseif {[lindex $args 0] == "forever"} {
        set run_len 1000000
        set stp 0
        set run_all 1
    } else {
        set run_len [parse_run_amt [lindex $args 0]]
    }

    set target_time [expr [$d_sim time] + $run_len]

    while {1} {
        if {[$d_sim finished]} {
            update idletasks
            ivi_puts "NOTE: Simulation finished"
            break
        }

        set remain_time [expr $target_time - [$d_sim time]]
        if {$d_simRunIncrement < $remain_time || $run_all != 0} {
            set sim_run_time $d_simRunIncrement
        } else {
            set sim_run_time $remain_time
        }

        set exec_time [time {set time [$d_sim run $sim_run_time]} ]
        set exec_time [lindex $exec_time 0]
        set sec_time  [expr $exec_time / 1000]

        if {$sec_time  > 1000} {
            if {$d_simRunIncrement > 10000} {
#                puts "Reducing increment from $d_simRunIncrement to [expr $d_simRunIncrement - 10000]"

                set d_simRunIncrement [expr $d_simRunIncrement - 10000]
            }
        } elseif {$sec_time < 800} {
#                puts "Increasing increment from $d_simRunIncrement to [expr $d_simRunIncrement + 10000]"
            set d_simRunIncrement [expr $d_simRunIncrement + 10000]
        }

        if {$d_breakRequest == 1} {
            break
        }

        set d_simTime [$d_sim time]

        callback invoke $CallbackTypes(SIM_RUNSTEP_UPDATE) $d_sim $d_simTime
        update idletasks

        if {$run_all == 0} {
            if {$d_simTime >= $target_time} {
                break
            }
        }
    }

    set d_breakRequest 0

    callback invoke $CallbackTypes(SIM_RUNSTEP_END) $d_sim $d_simTime
    set d_simRunning 0

    return ""
}

#********************************************************************
#* stop
#********************************************************************
proc SimCmds::stop {} {
    variable d_breakRequest
    variable d_simRunning

    if {$d_simRunning} {
        ivi_puts "Break Requested"
        set d_breakRequest 1
    }
}

#********************************************************************
#* close_design
#********************************************************************
proc SimCmds::close_design {} {
    global   WidgetTypes
    variable d_sim
    variable d_designLoaded
    global   CallbackTypes

    if {$d_sim == ""} {
        error "ERROR: No design loaded"
    }

    $d_sim close
    set d_sim ""

    SetStatBarText "No Design Loaded"
    SetTimeBarText "0" ""

    callback invoke $CallbackTypes(SIM_DESIGN_CLOSE) null
    set d_designLoaded false
}

#********************************************************************
#* UpdateTimeText
#********************************************************************
proc SimCmds::UpdateTimeText args {
    variable d_sim

    SetTimeBarText [$d_sim format_time]
}

namespace import SimCmds::*

