/* 
 * $Id: ctkmain.c,v 1.89 2001/06/13 23:50:48 neuro Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <arr.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <locale.h>
#include <ctype.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <linux/vt.h>

#include "ctk.h"
#include "ctkstack.h"
#include "ctkcolor.h"
#include "config.h"

#ifdef HAVE_LIBGPM
#include <gpm.h>
extern int gpm_tried;

gint gpmTag     = -1;
gint gpmDataTag = -1;

gint gpmData_fd = -1;

#endif

gint ctk_mouse_x  = 0;
gint ctk_mouse_y  = 0;
gint ctk_mouse_dx = 0;
gint ctk_mouse_dy = 0;

gint stdinTag   = -1;

GNode *CtkHierarchy=NULL;

gboolean ctk_screen_dirty;
GSList *ctk_main_loops = NULL;
GSList *ctk_garbage_queue = NULL;

gboolean ctk_clean_redraw;

struct CtkClipboard_S CtkClipboard;

struct cur_loc_s {
	gint row;
	gint col;
	CtkWidget *widget;
};

gboolean ctk_mouse_draw = FALSE;

gint ctk_keyval   = 0;
gint ctk_scr_r	   = 0;
gint ctk_scr_c    = 0;

/* Close CTK and clean up arr */
gint ctk_close()
{
	/* Remove all Channels */
	ctk_close_mouse();

	ctk_main_quit();

	arr_scr_free();

	return 1;
}

gint ctk_shutdown_gpmdata();

static CtkWidget* ctk_mouse_over_widget  = NULL;
static CtkWidget* ctk_mouse_press_widget = NULL;

/* Release global vars */
gint ctk_main_mouse_remove_stale_global(CtkObject* object)
{
      if (object == CTK_OBJECT(ctk_mouse_over_widget))
	    ctk_mouse_over_widget = NULL;
      if (object == CTK_OBJECT(ctk_mouse_press_widget))
	    ctk_mouse_press_widget = NULL;
	
      return 0;
}

#ifdef HAVE_LIBGPM
void ctk_leave_mouse_over()
{
      if (ctk_mouse_over_widget)
	    ctk_event_emit_by_name(CTK_OBJECT(ctk_mouse_over_widget),
				   "leave_notify_event");
      ctk_mouse_over_widget = NULL;
}

void ctk_enter_mouse_over(CtkWidget* widget)
{
      ctk_mouse_over_widget = widget;
      ctk_event_emit_by_name(CTK_OBJECT(widget),
			     "enter_notify_event");
}

void ctk_release_mouse_press()
{
      if (ctk_mouse_press_widget)
	    ctk_event_emit_by_name(CTK_OBJECT(ctk_mouse_press_widget),
				   "button_release_event");
      ctk_mouse_press_widget = NULL;
}

void ctk_engage_mouse_press(CtkWidget* widget)
{
      ctk_mouse_press_widget = widget;
      ctk_event_emit_by_name(CTK_OBJECT(ctk_mouse_press_widget),
			     "button_press_event");
}

void ctk_drag_mouse()
{
      Gpm_Event event;

      if (ctk_mouse_press_widget)
      {
	    if (ctk_event_emit_by_name(CTK_OBJECT(ctk_mouse_press_widget),
				       "ctk_drag"))
		  return;
      }

      event.x = ctk_mouse_x+1;
      event.y = ctk_mouse_y+1;
      GPM_DRAWPOINTER(&event);
}
#endif

/* Pre-order step by step traversal */
GNode* ctk_g_node_next(GNode* node)
{
	if (node->children)
		return node->children;
	
	for (; node; node = node->parent)
		if (node->next)
			break;
	
	if (!node) return node;
	
	return node->next;
}

GNode* ctk_g_node_prev(GNode* node)
{
	if (!node->prev)
		return node->parent;
	
	node = node->prev;
	if (!node->children)
		return node;
	
	node = node->children;
	while (1)
	{
		if (node->next)
		{
			node = node->next;
			continue;
		}
		if (node->children)
		{
			node = node->children;
			continue;
		}
		break;
	}
	
	return node;
}

/* Move onto the next item */
void ctk_forward_tab(CtkWindow* window)
{
	GNode* node;
	
	ctk_assert(window && window->focus_widget && window->focus_widget->node,
		"CTK object linkages are invalid!");
	
	node = window->focus_widget->node;
	do
	{
		CtkWidget* widget;
		
		node = ctk_g_node_next(node);
		if (!node)
		{
			node = CTK_WIDGET(window)->node;
		}
		
		widget = CTK_WIDGET(node->data);
		ctk_assert(widget, "CTK packing tree has null widget!");
		
		/* If the widget is displayed (not hidden in a notebook)
		 * and sensitive, it is our new focus.
		 */
		if (widget->sensitive && widget->clip_width  > 0 &&
		                         widget->clip_height > 0)
		{
			break;
		}
	} while (node != window->focus_widget->node);
	
	ctk_window_set_focus(window, CTK_WIDGET(node->data));
}

/* Move onto the next item */
void ctk_backward_tab(CtkWindow* window)
{
	GNode* node;
	
	ctk_assert(window && window->focus_widget && window->focus_widget->node,
		"CTK object linkages are invalid!");
	
	node = window->focus_widget->node;
	do
	{
		CtkWidget* widget;
		
		node = ctk_g_node_prev(node);
		if (!node)
		{
			node = CTK_WIDGET(window)->node;
			while (1)
			{
				if (node->next)
				{
					node = node->next;
					continue;
				}
				if (node->children)
				{
					node = node->children;
					continue;
				}
				break;
			}
		}
		
		widget = CTK_WIDGET(node->data);
		ctk_assert(widget, "CTK packing tree has null widget!");
		
		/* If the widget is displayed (not hidden in a notebook)
		 * and sensitive, it is our new focus.
		 */
		if (widget->sensitive && widget->clip_width  > 0 && 
		                         widget->clip_height > 0)
		{
			break;
		}
	} while (node != window->focus_widget->node);
	
	ctk_window_set_focus(window, CTK_WIDGET(node->data));
}

/* Check to see if an object is behind the mouse cursor */
CtkWindow* ctk_find_window_clicked(gint row, gint col)
{
      gint       i;
      CtkWidget* widget;
      
      if (!focus_stack)
	    return NULL;
      
      for (i = 0; focus_stack[i]; i += 1)
      {
	    widget = CTK_WIDGET(focus_stack[i]->data);
	    
	    if (!widget)
		  return NULL;
	    
	    /* Check if clicking on the Window */
	    if (ctk_check_clipping_xy(widget, col, row))
	    {
		  if (i > 0 && CTK_WINDOW(focus_stack[0]->data)->modal)
			return NULL;
		  
		  return CTK_WINDOW(widget);
	    } 
      }	
      
      return NULL;
}
 
CtkWidget* ctk_find_widget_clicked(CtkWindow* window, gint row, gint col)
{
      GNode*     scan;
      CtkWidget* plum;
      CtkWidget* out;

      out = CTK_WIDGET(window);

      do
      {
	    for (scan = out->node->children; scan; scan = scan->next)
	    {
		  plum = CTK_WIDGET(scan->data);

		  if (ctk_check_clipping_xy(plum, col, row))
		  {
			out = plum;
			break; /* leaves scan != NULL */
		  }
	    }
      } while (scan);

      return out;
}

#ifdef HAVE_LIBGPM
/* Handle all Mouse Events */
static gboolean real_gpm_event = TRUE;
gint mouse_event_handler(Gpm_Event* event, gpointer data)
{
      CtkWindow* mouse_window;
      CtkWidget* mouse_widget;
      
      /* There are three cases of ctk run (and ctk can't move between them):
       *   1) text-mode
       *      Here, gpm sends events on gpmctl and sends nothing to gpmdata.
       *      However, if the user switches to X, leaving ctk running on a 
       *      console, then we don't want to be eating /dev/gpmdata b/c it
       *      means the X mouse is slow and chunky, and the user can click
       *      stuff in ctk while thinking they're clicking in X.
       *   2) kon graphics mode
       *      Here, we are in graphics mode, so gpm does not send to gpmctl,
       *      it repeats on gpmdata. So, we have an ms3 mouse driver that
       *      emulates gpm builtin to ctk. Unfortunately, we have to draw
       *      the mouse cursor (so it gets slow on high load).
       *   3) X - xterm
       *      Ctk receives mouse events from the xterm via stdin. libgpm does
       *      this for us already and pretends it comes from gpmctl. It would
       *      be very bad if we pulled data from gpmdata in this case.
       */
      /* The attempt to solve all cases above:
       *   Initially open gpmctl. If we find we are NOT on a VT, but 
       *   connected to a slave owned by a program on a VT, then open gpmdata
       *   and read off it whenever the person is on the parent VT.
       *   Then, if a real event comes from libgpm (real_gpm_event) 
       *   we're directly on a console or in an xterm. So shutdown gpmdata 
       *   immediately.
       */
      if (real_gpm_event)
      {
            ctk_shutdown_gpmdata();
      }
	
      /* One of the goals here is to try to minimize signal noise.
       * Thus only on a change of focus or change of position do we emit
       * signals. This isn't enough, but it will have to do.
       */

      /* Minus 1 b/c coords in ctk are from 0,0 */
      ctk_mouse_x = event->x-1;
      ctk_mouse_y = event->y-1;

      ctk_mouse_dx = event->dx;
      ctk_mouse_dy = event->dy;

      /* If the mouse is released, send the mouse release event */
      if ((event->type & GPM_UP) != 0)
      {
	    ctk_release_mouse_press();
      }

      if ((event->type & GPM_DRAG) != 0)
      {
	    ctk_drag_mouse();
      }

      /* Find the widget the mouse is now over */
      mouse_window = ctk_find_window_clicked(ctk_mouse_y, ctk_mouse_x);
      if (!mouse_window)
      {
	    /* No window -> no widget, so leave */
	    ctk_leave_mouse_over();
      }
      else
      {
	    mouse_widget = ctk_find_widget_clicked(mouse_window, ctk_mouse_y, ctk_mouse_x);
	    ctk_assert(mouse_widget != NULL, "Click inside window hits on no widgets?!");
	    
	    /* Only if there is a change in the widget: */
	    if (mouse_widget != ctk_mouse_over_widget)
	    {
		  /* Leave the current widget */
		  ctk_leave_mouse_over();
	
		  /* Enter the current widget */
		  ctk_enter_mouse_over(mouse_widget);
	    }
	    
	    /* If the mouse is pressed, send the mouse press event */
	    if ((event->type & GPM_DOWN) != 0)
	    {
		  GNode* scan;
		  
		  /* Foreground the window first so we can see the press behaviour */
		  ctk_window_raise(mouse_window);
		  
		  /* Scan up looking for a sensitive widget */
		  for (scan = mouse_widget->node; scan; scan = scan->parent)
		  {
			mouse_widget = CTK_WIDGET(scan->data);
			if (mouse_widget->sensitive) break;
		  }

		  /* Run the click handler */
		  if (mouse_widget)
		  {
			ctk_engage_mouse_press(mouse_widget);
		  }
		  
		  /* Now, the user may have been evil and destroyed or unpacked
		   * themselves. If so, we don't focus it.
		   */
		  
		  if (mouse_widget->destroyed || 
		  	ctk_window_container(mouse_widget) != mouse_window)
		  {
		  	mouse_widget = NULL;
		  }

		  if (!mouse_widget || mouse_widget == CTK_WIDGET(mouse_window))
		  {
			/* Nothing clicked - preserve focus */
			mouse_widget = mouse_window->focus_widget;
		  }
		  
		  /* If we changed focus, move the focus */
		  ctk_window_set_focus(mouse_window, mouse_widget);
	    }
      }
	
      return 256;
}

gint isgn(gint i)
{
	if (i > 0) return 1;
	if (i < 0) return -1;
	return 0;
}


static gint parent_vt   = -1;
static gint the_console = -1;

void ctk_ms3_handler(gpointer data, gint fd, gint condition)
{
	static int       mypos = 0;
	static Gpm_Event event = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	static gchar     ms3buf[4];
	static gint      jump_mod_x;
	static gint      jump_mod_y;
	
	signed char    ch;
	Gpm_Event      newEvent;
	int            olddraw;
	struct vt_stat my_current_vt;
	
	/* No parent VT?! We don't feed then */
	if (parent_vt == -1 || the_console == -1)
		return;
	
	ioctl(the_console, VT_GETSTATE, &my_current_vt);
	
	/* We're not on the right VT */
	if (my_current_vt.v_active != parent_vt)
		return;
		
	read(gpmData_fd, &ms3buf[mypos], 1);
	
	switch (mypos)
	{
	case 0:
		if ((ms3buf[0] & 0xC0) == 0x40)
			mypos++;
		break;
	case 1:
		if ((ms3buf[1] & 0xC0) == 0)
			mypos++;
		else
			mypos = 0;
		break;
	case 2:
		mypos++;
		break;
	case 3:
		/* We now have a complete packet */
		mypos = 0;
		
		newEvent.buttons = 0;
		if (ms3buf[0] & 0x20) newEvent.buttons |= GPM_B_LEFT;
		if (ms3buf[3] & 0x10) newEvent.buttons |= GPM_B_MIDDLE;
		if (ms3buf[0] & 0x10) newEvent.buttons |= GPM_B_RIGHT;
		
		newEvent.dx = ch =
			(((ms3buf[0] & 0x03) << 6) | (ms3buf[1] & 0x3F));
		newEvent.dy = ch =
			(((ms3buf[0] & 0x0C) << 4) | (ms3buf[2] & 0x3F));
			
		/* Slow down cowboy -- gpmdata is too fast */
		newEvent.dx = (jump_mod_x + newEvent.dx + isgn(newEvent.dx)*2);
		newEvent.dy = (jump_mod_y + newEvent.dy + isgn(newEvent.dy)*3);
		
		jump_mod_x = newEvent.dx % 5;
		newEvent.dx /= 5;
		
		jump_mod_y = newEvent.dy % 10;
		newEvent.dy /= 10;
				
		/* Pretend we're gpm - interpret the change */
		olddraw = ctk_mouse_draw;
		if (!event.buttons && newEvent.buttons)
		{
			ctk_mouse_draw = TRUE;
			newEvent.type = GPM_DOWN;
		}
		else if (event.buttons && !newEvent.buttons)
		{
			ctk_mouse_draw = TRUE;
			newEvent.type = GPM_UP;
		}
		else if (newEvent.buttons && (newEvent.dx || newEvent.dy))
		{
			ctk_mouse_draw = FALSE;
			newEvent.type = GPM_DRAG;
		}
		else
		{
			ctk_mouse_draw = TRUE;
			newEvent.type = GPM_MOVE;
		}
		
		/* Make the cursor position */
		newEvent.x = newEvent.dx + event.x;
		newEvent.y = newEvent.dy + event.y;
		
		if (newEvent.x < 1) newEvent.x = 1;
		if (newEvent.y < 1) newEvent.y = 1;
		if (newEvent.x > ctk_scr_c) newEvent.x = ctk_scr_c;
		if (newEvent.y > ctk_scr_r) newEvent.y = ctk_scr_r;
		
		/* If nothing important happened, just quit now */
		if (newEvent.x == event.x && newEvent.y == event.y &&
			newEvent.type == GPM_MOVE)
		{
			ctk_mouse_draw = olddraw;
			return;
		}
		
		/* Uninverse the char */
		if (olddraw)
		{
			ctk_inverse_char(event.x - 1, event.y - 1);
			arr_dump_screen();
		}
		
		if (ctk_mouse_draw)
		{
			ctk_inverse_char(newEvent.x - 1, newEvent.y - 1);
			arr_dump_screen();
		}
		      
		event = newEvent;
		
		real_gpm_event = FALSE;
		mouse_event_handler(&newEvent, NULL);
		real_gpm_event = TRUE;
		
		break;
	}
}
#endif

static guint ctk_arr_pushed = -1;

gint ctk_arr_getc_wrapper(FILE* f)
{
	if (ctk_arr_pushed != -1)
	{
		gint tmp = ctk_arr_pushed;
		ctk_arr_pushed = -1;
		return tmp;
	}
#ifdef HAVE_LIBGPM
	else 
		return Gpm_Getc(f);
#else
	return getc(f);
#endif
}

/* Main Loop.  Gets Keyboard, Mouse, and other events. */
void event_main_loop(gpointer data, gint fd, gint condition)
{
	arr_kqueue* queue_ptr;
	arr_kqueue* queue_head;
	guint ch;
	CtkWindow* focus_window;
	CtkWidget* focus_widget;

#ifdef HAVE_LIBGPM	
	ch = Gpm_Getc(stdin);
	ctk_assert(ch != EOF, "Gpm borked out from under us!");
#else
	ch = getc(stdin);
#endif
	
	/* 256 is returned by the mouse handler to say that there was
	 * a mouse event - not a keyboard one.
	 */
	if (ch == 256) 
		return;

	ctk_arr_pushed = ch;

	queue_head = arr_key(ctk_arr_getc_wrapper);
	ctk_assert(queue_head != NULL, "Stdin is borked!");

	focus_window = ctk_window_get_focus_window();
	if (!focus_window)
	      return;

	focus_widget = focus_window->focus_widget;
	if (!focus_widget)
	{
	      focus_window->focus_widget = focus_widget = CTK_WIDGET(focus_window);
	}

	for (queue_ptr = queue_head; queue_ptr; queue_ptr = queue_ptr->next)
	{
		ctk_keyval = queue_ptr->key;

		switch (queue_ptr->key)
		{
		case AK_CTRL('O'):	/* Ctrl-O quick exit */
			g_main_quit(ctk_main_loops->data);
			break;
		case AK_CTRL('R'):
			ctk_redraw_screen(CTK_REDRAW_ALL);
			break;
		case '\t':
			ctk_forward_tab(focus_window);
			break;
		case MODIFIER_SHIFT | '\t':
			ctk_backward_tab(focus_window);
			break;
		default:
			ctk_event_emit_by_name(CTK_OBJECT(focus_widget),
					"key_press_event");
			break;
		}
	}

	while (queue_head)
	{
		queue_ptr = queue_head->next;
		free(queue_head);
		queue_head = queue_ptr;
	}
}

#ifdef HAVE_LIBGPM
gint ctk_shutdown_gpmdata()
{
	if (gpmDataTag != -1)
	{
		ctk_input_remove(gpmDataTag);
		gpmDataTag = -1;
	}

	if (gpmData_fd != -1)
	{
		flock(gpmData_fd, LOCK_UN);
		close(gpmData_fd);
		gpmData_fd = -1;
	}
	
	ctk_mouse_draw = FALSE;
	
	return 1;
}
#endif

gint ctk_close_mouse()
{
#ifdef HAVE_LIBGPM
	if (gpmTag != -1)
	{
		ctk_input_remove(gpmTag);
		gpmTag = -1;
	}
	
	if (gpm_fd != -1)
	{
		Gpm_Close();
		gpm_fd = -1;
	}
	
	ctk_shutdown_gpmdata();
#endif
	return 1;
}

#ifdef HAVE_LIBGPM
static gint ctk_is_a_console(gint fd)
{
	struct vt_mode arg;
	return (ioctl(fd, VT_GETMODE, &arg) == 0);
}

static gint ctk_open_console(const gchar* file)
{
	gint console_fd;
	
	console_fd = open(file, O_RDONLY);
	if (console_fd < 0 && errno == EACCES)
	{
		console_fd = open(file, O_WRONLY);
	}
	
	if (console_fd < 0)
	{
		return -1;
	}
	
	if (!ctk_is_a_console(console_fd))
	{
		close(console_fd);
		return -1;
	}
	
	return console_fd;
		
}

static gint ctk_grab_console()
{
	gint console_fd;
	
	/* Try standard devices */
	if ((console_fd = ctk_open_console("/dev/console")) != -1) return console_fd;
	if ((console_fd = ctk_open_console("/dev/tty"))     != -1) return console_fd;
	if ((console_fd = ctk_open_console("/dev/tty0"))    != -1) return console_fd;
	
	/* Try our stdin/stdout/stderr */
	for (console_fd = 0; console_fd < 3; console_fd++)
	{
		if (ctk_is_a_console(console_fd))
		{
			return dup(console_fd);
		}
	}
	
	/* Unable to get a console */
	return -1;
}

static gint ctk_startup_gpmdata()
{
	gint   got_a_mouse;
	pid_t  scan_pid;
	pid_t  parent_pid;
	gint   device;
	gchar* name;
	FILE*  file;
	gchar  buf[80];
	gchar* line;
	gint   len;
	
	got_a_mouse = 0;
	
	/* Now, we have to be absolutely paranoid so that we never EVER
	 * use gpmdata unless we're under kon. We also need to find out
	 * what VT kon is running on, so that we will only ever read from
	 * it if we're on the kon VT.
	 */
	 
	for (scan_pid = getppid(); scan_pid != 1; scan_pid = parent_pid)
	{
		sprintf(buf, "/proc/%i/stat", scan_pid);
		
		file = fopen(buf, "r");
		if (!file)
			return got_a_mouse;
		
		line = NULL;
		len = 0;
		getline(&line, &len, file);
		fclose(file);
		
		if (!line)
			return got_a_mouse;
		
		name = NULL;
		len = sscanf(line, "%*d (%a[^)]) %*c %d %*d %*d %d",
			&name, &parent_pid, &device);
		
		/* We're done with this. */
		free(line);
		
		if (len != 3 || name == NULL)
		{
			if (name) free(name);
			return got_a_mouse;
		}
		
		if (!strcmp(name, "kon"))
		{
			free(name);
			/* Yay! We have kon! Happy */
			if ((device  & 0xFF00L) != 1024)
			{
				/* kon isn't on a vt! run away! run away! */
				return got_a_mouse;
			}
			
			parent_vt = device & 0xFFL;
			break;
		}
		free(name);
	}
	
	if (scan_pid == 1)
	{
		/* No kon found */
		return got_a_mouse;
	}
	
	if (the_console != -1)
	{
		close(the_console);
	}
	
	/* Try to get a console so we can find out where the user is hiding */
	the_console = ctk_grab_console();
	if (the_console == -1)
	{
		/* Failed to open the console - we need this to check VTs */
		parent_vt = -1;
		return got_a_mouse;
	}
	
	/* Cool! We now have enough data to know we're running under kon
	 * and figure out when the user is looking at the VT kon is on.
	 * This way we can read from gpmdata safely -- ie: only when it's
	 * got data for us!
	 */
	
	gpmData_fd = open("/dev/gpmdata", O_RDONLY | O_NONBLOCK);
	if (gpmData_fd != -1)
	{
		gpmDataTag = ctk_input_add(gpmData_fd, CTK_INPUT_READ, ctk_ms3_handler, NULL);
		ctk_assert(gpmDataTag != -1, "Failed to add /dev/gpmdata to ctk main loop after successful open");
		got_a_mouse = 1;
	}
	
	return got_a_mouse;
}
#endif


gint ctk_init_mouse()
{
#ifdef HAVE_LIBGPM
	Gpm_Connect conn;
#endif
	gint        got_a_mouse;
	
	got_a_mouse = 0;
	
	ctk_close_mouse();
	
#ifdef HAVE_LIBGPM
	conn.eventMask = ~0;
	conn.defaultMask = GPM_MOVE|GPM_HARD;
	conn.maxMod = ~0;
	conn.minMod = 0;
	
	gpm_tried = 0;
	gpm_fd = Gpm_Open(&conn, 0);
	
	if (gpm_fd != -1)
	{
		if (gpm_fd != -2)
		{
		      gpmTag = ctk_input_add(gpm_fd, CTK_INPUT_READ, event_main_loop, NULL);
		      ctk_assert(gpmTag != -1, "Failed to add non-xterm mouse to input loop");
		}
		
		got_a_mouse = 1;
		gpm_handler = mouse_event_handler;
	}
	
	if (ctk_startup_gpmdata())
	{
		got_a_mouse = 1;
	}
#endif
	
	return got_a_mouse;
}

/* Initialize CTK */
gint ctk_init(gint initmask)
{
	/* Create Ctk type hierarchy for fast lookups */
	ctk_type_init();

	if (arr_scr_init() < 0)
		exit(1);

	ctk_scr_r = arr_scr_y();
	ctk_scr_c = arr_scr_x();

	arr_buffer_init(' ', (FG_BLUE | BG_BLUE));
	arr_hide_cursor();

	/* Setup color Palette */
	ctk_color_init();
	
	stdinTag = ctk_input_add(0, CTK_INPUT_READ,
					event_main_loop, NULL);

	/* Setup GPM */
	if (initmask & CTK_USEMOUSE) 
	{
		if (!ctk_init_mouse()) 
		      return 0;
	}
	
	ctk_clean_redraw = ((initmask & CTK_CLEANDRAW) != 0);

	return 1;
}

/* Check Modal.  We want to focus the first window that we find that is
 * modal */

void ctk_main_check_modal(void)
{
	gint i;
	CtkWindow *window;
	CtkWidget *widget;

	if (!focus_stack)
	    return;
	
	for (i = 0; focus_stack[i]; i += 1)
	{
		widget = CTK_WIDGET(focus_stack[i]->data);
		window = (CtkWindow *)widget;
		
		if (window->modal)
		{
		      if (i != 0) 
			    ctk_window_raise(window);
		      break;
		}
	}
}

/* CTK Main Loop */
gint ctk_main()
{
	GMainLoop *loop;
	
	loop = g_main_new(TRUE);
	
	ctk_main_loops = g_slist_prepend(ctk_main_loops, loop);
	
	ctk_main_check_modal();
	
	ctk_redraw_screen(CTK_REDRAW_CHANGED);
	g_main_run(ctk_main_loops->data);
	
	ctk_main_loops = g_slist_remove(ctk_main_loops, loop);

	g_main_destroy(loop);

	return 0;
}

/* Ctk Main Quit */
void ctk_main_quit(void)
{
	if (ctk_main_loops)
	    g_main_quit(ctk_main_loops->data);
}

/* Ctk Exit */
void ctk_exit(gint errorcode)
{	
	ctk_close();
	
	exit(errorcode);
}

/* Set Locale, but not worry about */
gchar* ctk_set_locale(void)
{
	return setlocale(LC_ALL, "");
}

/* Add a node to the garbage queue */
void ctk_main_garbage_add(CtkWidget* widget)
{
	ctk_garbage_queue = g_slist_prepend(ctk_garbage_queue, (gpointer)widget);
}

void ctk_main_garbage_cleanup()
{
	GSList* list;
	GSList* head;

	/* This zeroing the head before traverse allows destroys in the
	 * callbacks.
	 */
	head = ctk_garbage_queue;
	ctk_garbage_queue = NULL;

	for (list = head; list; list = list->next)
	{
		ctk_widget_destroy_real((CtkWidget *)list->data);
	}
	
	g_slist_free(head);
}
