/***************************************************************************
                           csearchindex.cpp  -  description
                             -------------------
    begin                : Mon May 14 2003
    copyright            : (C) 2003-2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#endif

#include <string.h>

#include <dclib/dcos.h>
#include <dclib/core/cbytearray.h>
#include <dclib/cconfig.h>
#include <dclib/core/cstring.h>
#include <dclib/core/cstringlist.h>
#include <dclib/core/cdir.h>
#include <dclib/core/cfile.h>
#include <dclib/cdownloadmanager.h>
#include <dclib/dclib.h>
#include <dclib/core/cmanager.h>
#include <dclib/core/cbz.h>
#include <dclib/core/che3.h>
#include <dclib/cfilemanager.h>
#include <dclib/hash/tigertree.h>
#include <dclib/core/cbase32.h>

#include "csearchindex.h"

/** */
CSearchIndex::CSearchIndex()
{
	m_pBaseArray       = new CByteArray();
	m_pFileBaseArray   = new CByteArray();
	m_pPathBaseArray   = new CByteArray();
	m_pSearchIndex     = new CByteArray();
	m_pSearchFileIndex = new CByteArray();
	m_pSearchBase      = new CByteArray();

	m_pHashBaseArray = new CByteArray();
	m_pHashFileBaseArray = new CByteArray();
	m_pHashPathBaseArray = new CByteArray();
	m_pHashIndex = new CByteArray();
	
	memset(m_pSearchArray,0,sizeof(m_pSearchArray));

	m_nSearchFileIndex = 0;

	LoadIndex();
}

/** */
CSearchIndex::~CSearchIndex()
{
	delete m_pBaseArray;
	delete m_pFileBaseArray;
	delete m_pPathBaseArray;
	delete m_pSearchIndex;
	delete m_pSearchFileIndex;
	delete m_pSearchBase;
	
	delete m_pHashBaseArray;
	delete m_pHashFileBaseArray;
	delete m_pHashPathBaseArray;
	delete m_pHashIndex;
}

void CSearchIndex::Reset()
{
	m_pBaseArray->SetSize(0);
	m_pFileBaseArray->SetSize(0);
	m_pPathBaseArray->SetSize(0);

	ResetIndex();
}

void CSearchIndex::ResetIndex()
{
	m_pSearchIndex->SetSize(0);
	m_pSearchFileIndex->SetSize(0);
	m_pSearchBase->SetSize(0);

	memset(m_pSearchArray,0,sizeof(m_pSearchArray));

	m_nSearchFileIndex = 0;
}

void CSearchIndex::ResetHashIndex()
{
	m_pHashBaseArray->SetSize(0);
	m_pHashFileBaseArray->SetSize(0);
	m_pHashPathBaseArray->SetSize(0);
	m_pHashIndex->SetSize(0);
}

/** */
CStringList * CSearchIndex::Search( CString s )
{
	CString *ps;
	CStringList *sl=0;
	struct fileindexobject *pfio;
	struct searchindexobject * psio;

	if ( (psio = SearchIndex(s) ) != 0 )
	{
		sl = new CStringList();

		pfio = (struct fileindexobject*) (m_pSearchIndex->Data()+psio->m_nFileIndex);

		while(pfio)
		{
			ps = new CString( CString().setNum(pfio->m_nFileBaseIndex));
			sl->Add( *ps, ps );

			if ( pfio->m_nNext == 0 )
			{
				pfio = 0;
			}
			else
			{
				pfio = (struct fileindexobject*) (m_pSearchIndex->Data()+pfio->m_nNext);
			}
		}
	}

	return sl;
}

/** */
CStringList * CSearchIndex::SearchHash( unsigned char * hash )
{
	ulonglong hbi,hi,bi;
	CStringList *sl=0;
	CString *ps;

	hi = 0;
	
	while ( FindHash( hash, &hi ) == TRUE )
	{
		if ( HashBaseIndexFromHashIndex( hi, &hbi ) )
		{
			if ( BaseIndexFromHashBaseIndex( hbi, &bi ) )
			{
				if ( !sl )
					sl = new CStringList();
				ps = new CString( CString().setNum(bi));
				sl->Add( *ps, ps );
			}
		}
		hi++;
	}

	return sl;
}

/** */
bool CSearchIndex::GetFileBaseObject( ulonglong index, struct filebaseobject * fbo )
{
	bool res = FALSE;

	if ( (index*sizeof(struct filebaseobject)) < m_pBaseArray->Size() )
	{
		memcpy( fbo, m_pBaseArray->Data()+(index*sizeof(struct filebaseobject)), sizeof(struct filebaseobject) );

		res = TRUE;
	}

	return res;
}

/** */
bool CSearchIndex::GetFileBaseObject( CString id, struct filebaseobject * fbo, CString & filename )
{
	bool res = FALSE;
	ulonglong index;

	index = id.asULL();

	if ( GetFileBaseObject(index,fbo) == TRUE )
	{
		if ( m_pPathBaseArray->Size() > fbo->m_nPathIndex )
		{
			filename = (char*)m_pPathBaseArray->Data()+fbo->m_nPathIndex;

			if ( filename != "" )
			{
				filename += DIRSEPARATOR;
			}
		}

		if ( m_pFileBaseArray->Size() > fbo->m_nFileIndex )
		{
			filename += (char*)m_pFileBaseArray->Data()+fbo->m_nFileIndex;
		}

		res = TRUE;
	}

	return res;
}

/** */
bool CSearchIndex::LoadIndex()
{
	bool err = FALSE;
	CDir d;
	ulonglong filesize;

	// first try to load hash index
	if ( m_pHashBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"hashbase.bin") == FALSE )
	{
		err = TRUE;
	}
	if ( err == FALSE )
	{
		if ( m_pHashFileBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"hashfilebase.bin") == FALSE )
		{
			err = TRUE;
		}
	}
	if ( err == FALSE )
	{
		if ( m_pHashPathBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"hashpathbase.bin") == FALSE )
		{
			err = TRUE;
		}
	}
	if ( err == FALSE )
	{
		if ( m_pHashIndex->LoadFromFile(CConfig::Instance()->GetConfigPath()+"hashindex.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err == TRUE )
	{
		ResetHashIndex();
		
		err = FALSE;
	}
	
	// sanity check
	filesize = d.getFileSize(CConfig::Instance()->GetConfigPath()+"database.bin",FALSE);

	if ( (filesize%sizeof(struct filebaseobject)) != 0 )
	{
		err = TRUE;
	}

	// load the filelist
	if ( err == FALSE )
	{
		if ( m_pBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"database.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err == FALSE )
	{
		if ( m_pFileBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"filebase.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err == FALSE )
	{
		if ( m_pPathBaseArray->LoadFromFile(CConfig::Instance()->GetConfigPath()+"pathbase.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err )
	{
		m_pBaseArray->SetSize(0);
		m_pFileBaseArray->SetSize(0);
		m_pPathBaseArray->SetSize(0);
	}

	if ( err == FALSE )
	{
		if ( m_pSearchBase->LoadFromFile(CConfig::Instance()->GetConfigPath()+"searchbase.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err == FALSE )
	{
		if ( m_pSearchIndex->LoadFromFile(CConfig::Instance()->GetConfigPath()+"searchindex.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err == FALSE )
	{
		if ( m_pSearchFileIndex->LoadFromFile(CConfig::Instance()->GetConfigPath()+"searchfileindex.bin") == FALSE )
		{
			err = TRUE;
		}
	}

	if ( err )
	{
		m_pSearchBase->SetSize(0);
		m_pSearchIndex->SetSize(0);
		m_pSearchFileIndex->SetSize(0);
		m_nSearchFileIndex = 0;

		// TODO: recreate index
	}
	else
	{
		m_nSearchFileIndex = (struct long_s*)m_pSearchFileIndex->Data();
	}

	return !err;
}

/** */
void CSearchIndex::SaveIndex()
{
	m_pSearchBase->SaveToFile(CConfig::Instance()->GetConfigPath()+"searchbase.bin");
	m_pSearchIndex->SaveToFile(CConfig::Instance()->GetConfigPath()+"searchindex.bin");
	m_pSearchFileIndex->SaveToFile(CConfig::Instance()->GetConfigPath()+"searchfileindex.bin");
	m_pBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"database.bin");
	m_pFileBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"filebase.bin");
	m_pPathBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"pathbase.bin");
	
	m_pHashBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"hashbase.bin");
	m_pHashFileBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"hashfilebase.bin");
	m_pHashPathBaseArray->SaveToFile(CConfig::Instance()->GetConfigPath()+"hashpathbase.bin");
	m_pHashIndex->SaveToFile(CConfig::Instance()->GetConfigPath()+"hashindex.bin");
}

/** */
unsigned long CSearchIndex::IndexCount()
{
	unsigned long i = 0;

	if ( m_pBaseArray )
	{
		i = m_pBaseArray->Size()/sizeof(struct filebaseobject);
	}

	return i;
}

/** */
unsigned long CSearchIndex::AddIndex( CFileInfo *fileinfo, CString path, eFileTypes filetype )
{
	struct filebaseobject fbo;
	unsigned long i;

	fbo.m_eFileType  = filetype;
	fbo.m_nFileIndex = m_pFileBaseArray->Size();
	fbo.m_nPathIndex = m_pPathBaseArray->Size();
	fbo.m_nHashIndex = (unsigned long)-1;
	fbo.m_nSize      = fileinfo->size;
	fbo.m_tModTime   = fileinfo->st_m_time;

	i = m_pBaseArray->Size()/sizeof(struct filebaseobject);

	m_pBaseArray->Append( (const char*)&fbo, sizeof(struct filebaseobject) );
	m_pFileBaseArray->Append( fileinfo->name.Data(), fileinfo->name.Length()+1 );
	m_pPathBaseArray->Append( path.Data(), path.Length()+1 );
	
	return i;
}

/** */
void CSearchIndex::UpdateIndex( ulonglong index, struct filebaseobject * fbo )
{
	if ( (index*sizeof(struct filebaseobject)) < m_pBaseArray->Size() )
	{
		memcpy( m_pBaseArray->Data()+(index*sizeof(struct filebaseobject)), fbo, sizeof(struct filebaseobject) );
	}
}

/** */
bool CSearchIndex::FindHash( unsigned char * hash, ulonglong * hi )
{
	ulonglong i;
	
	for(i=*hi;i<m_pHashIndex->Size();i+=TIGERSIZE)
		if ( memcmp(hash,m_pHashIndex->Data()+i,TIGERSIZE) == 0 )
		{
			*hi = i;
			return TRUE;
		}
		
	return FALSE;
}

/** */
CString CSearchIndex::GetHash( ulonglong hbi )
{
	CString s;
	CBase32 base32;
	CByteArray dst,src;
	
	if ( hbi < m_pHashIndex->Size() )
	{
		src.Append(m_pHashIndex->Data()+hbi,TIGERSIZE);
		base32.Encode( &dst, &src );
		if ( dst.Size() > 0 )
		{
			s.Set((const char*)dst.Data(),dst.Size());
			s = "TTH:"+s;
		}
	}
	
	return s;
}

/** */
bool CSearchIndex::HashBaseIndexFromHashIndex( ulonglong hi, ulonglong * hbi )
{
	ulonglong i;
	struct hashbaseobject * hbo;

	for(i=0;i<m_pHashBaseArray->Size();i+=sizeof(struct hashbaseobject))
	{
		hbo = (struct hashbaseobject *)(m_pHashBaseArray->Data()+i);
		if ( hbo->m_nHashIndex == hi )
		{
			*hbi = i;
			return TRUE;
		}
	}
			
	return FALSE;
}

/** */
bool CSearchIndex::BaseIndexFromHashBaseIndex( ulonglong hbi, ulonglong * bi )
{
	ulonglong i;
	struct filebaseobject * fbo;
	
	for(i=0;i<m_pBaseArray->Size();i+=sizeof(struct filebaseobject))
	{
		fbo = (struct filebaseobject*)(m_pBaseArray->Data()+i);
		
		if ( fbo->m_nHashIndex == hbi )
		{
			*bi = i/sizeof(struct filebaseobject);
			return TRUE;
		}
	}
	
	return FALSE;
}

/** */
bool CSearchIndex::FindHashBaseIndex( struct filebaseobject * fbo, ulonglong * hbi )
{
	ulonglong i;
	struct hashbaseobject * hbo;

	for(i=0;i<m_pHashBaseArray->Size();i+=sizeof(struct hashbaseobject))
	{
		hbo = (struct hashbaseobject *)(m_pHashBaseArray->Data()+i);
		
		if ( Compare(fbo,hbo) )
		{
			*hbi = i;
			return TRUE;
		}
	}
	
	return FALSE;
}

/** */
bool CSearchIndex::Compare( struct filebaseobject * fbo, struct hashbaseobject * hbo )
{
	if ( fbo->m_nSize != hbo->m_nSize )
		return FALSE;
	if ( fbo->m_tModTime != hbo->m_tModTime )
		return FALSE;

	CString s1,s2;
	
	s1 = (char*)m_pFileBaseArray->Data()+fbo->m_nFileIndex;
	s2 = (char*)m_pHashFileBaseArray->Data()+hbo->m_nFileIndex;
	
	if ( s1 != s2 )
		return FALSE;
	
	s1 = (char*)m_pPathBaseArray->Data()+fbo->m_nPathIndex;
	s2 = (char*)m_pHashPathBaseArray->Data()+hbo->m_nPathIndex;
	
	if ( s1 != s2 )
		return FALSE;

	return TRUE;
}

/** */
void CSearchIndex::AddHashIndex( ulonglong filebaseindex, unsigned char * hash )
{
	struct filebaseobject fbo;
	struct hashbaseobject hbo;
	char * c;
	ulonglong hi,hbi;
	
	if ( !GetFileBaseObject(filebaseindex,&fbo) )
	{
		return;
	}
	
	hi = 0;
	if ( FindHash(hash,&hi) )
	{
		if ( HashBaseIndexFromHashIndex(hi,&hbi) )
		{
			if ( Compare(&fbo,(struct hashbaseobject *)(m_pHashBaseArray->Data()+hbi)) )
			{
				printf("hash found\n");
				fbo.m_nHashIndex = hbi;
				UpdateIndex(filebaseindex,&fbo);
				return;
			}
		}
	}
	
	hbo.m_nSize = fbo.m_nSize;
	hbo.m_tModTime = fbo.m_tModTime;
	hbo.m_nFileIndex = m_pHashFileBaseArray->Size();
	hbo.m_nPathIndex = m_pHashPathBaseArray->Size();
	hbo.m_nHashIndex = m_pHashIndex->Size();

	fbo.m_nHashIndex = m_pHashBaseArray->Size();

	m_pHashBaseArray->Append((const char*)&hbo, sizeof(struct hashbaseobject));
	c = (char*)m_pFileBaseArray->Data()+fbo.m_nFileIndex;
	m_pHashFileBaseArray->Append(c,strlen(c)+1);
	c = (char*)m_pPathBaseArray->Data()+fbo.m_nPathIndex;
	m_pHashPathBaseArray->Append(c,strlen(c)+1);
	m_pHashIndex->Append( hash, TIGERSIZE );

	UpdateIndex(filebaseindex,&fbo);
}

/** */
void CSearchIndex::InitIndex()
{
	int i;

	// init array
	m_pSearchFileIndex->SetSize(sizeof(struct long_s)*0x100);

	for(i=0;i<=0xFF;i++)
	{
		m_nSearchFileIndex = (struct long_s*)m_pSearchFileIndex->Data();

		m_nSearchFileIndex[i].n = 0;

		if ( m_pSearchArray[i] != 0 )
		{
			m_nSearchFileIndex[i].n = m_pSearchFileIndex->Size();
			m_pSearchFileIndex->Append(m_pSearchArray[i]->Data(),m_pSearchArray[i]->Size());
			delete m_pSearchArray[i];
		}
	}
}

/** */
CString CSearchIndex::GetFileName( ulonglong i )
{
	CString s = "";
	struct filebaseobject * fbo;

	if ( (i*sizeof(struct filebaseobject)) < m_pBaseArray->Size() )
	{
		fbo = (struct filebaseobject *) (m_pBaseArray->Data()+(i*sizeof(struct filebaseobject)));

		s = (char*) (m_pFileBaseArray->Data()+fbo->m_nFileIndex);
	}

	return s;
}

struct searchindexobject * CSearchIndex::SearchIndex( CString & s )
{
	unsigned long i,is,ie;
	unsigned int in = s.Data()[0]&0xFF;
	struct searchindexobject * psio = 0;

	if ( (!m_nSearchFileIndex) || (m_nSearchFileIndex[in].n == 0) )
	{
		return 0;
	}

	// set start
	is = m_nSearchFileIndex[in].n;

	// find end
	for(ie=0,i=in+1;i<=0xFF;i++)
	{
		if ( m_nSearchFileIndex[i].n != 0 )
		{
			ie = m_nSearchFileIndex[i].n;
			break;
		}
	}

	if ( ie == 0 )
	{
		ie = m_pSearchFileIndex->Size();
	}

	i = 0;

	while( (is+(i*sizeof(struct searchindexobject))) < ie )
	{
		psio = (struct searchindexobject *) (m_pSearchFileIndex->Data()+(is+(i*sizeof(struct searchindexobject))));

		if ( s == (char*)(m_pSearchBase->Data()+psio->m_nIndex) )
		{
			break;
		}

		i++;
		psio = 0;
	}

	return psio;
}

struct searchindexobject * CSearchIndex::FindIndex( CString & s )
{
	long i=0;
	unsigned int in = s.Data()[0]&0xFF;
	struct searchindexobject * psio = 0;

	if ( m_pSearchArray[in] == 0 )
	{
		return 0;
	}

	while( (i*sizeof(struct searchindexobject)) < m_pSearchArray[in]->Size() )
	{
		psio = (struct searchindexobject *) (m_pSearchArray[in]->Data()+(i*sizeof(struct searchindexobject)));

		if ( s == (char*)(m_pSearchBase->Data()+psio->m_nIndex) )
		{
			break;
		}

		i++;
		psio = 0;
	}

	return psio;
}

void CSearchIndex::AddIndex( CString & s, ulonglong filebaseindex )
{
	unsigned int in;
	struct searchindexobject sio, *psio;
	struct fileindexobject fio, *pfio;

	in = s.Data()[0]&0xFF;
			
	if ( m_pSearchArray[in] == 0 )
	{
		m_pSearchArray[in] = new CByteArray();
	}
			
	if ( (psio = FindIndex(s) ) != 0 )
	{
		pfio = (struct fileindexobject*) (m_pSearchIndex->Data()+psio->m_nFileIndex);

		while(pfio->m_nNext!=0)
		{
			pfio = (struct fileindexobject*) (m_pSearchIndex->Data()+pfio->m_nNext);
		}

		pfio->m_nNext = m_pSearchIndex->Size();
	}
	else
	{
		sio.m_nIndex     = m_pSearchBase->Size();
		sio.m_nFileIndex = m_pSearchIndex->Size();

		m_pSearchBase->Append(s.Data(),s.Length()+1);
		m_pSearchArray[in]->Append( (const char*)&sio, sizeof(struct searchindexobject) );
	}

	fio.m_nFileBaseIndex = filebaseindex;
	fio.m_nNext          = 0;

	m_pSearchIndex->Append( (const char*)&fio, sizeof(struct fileindexobject) );
}
