/**
 * @file manager.cpp
 * @author Michael Vogt <mvo@debian.org>
 * @author Peter Rockai <me@mornfall.net>
 */

#include <apt-pkg/error.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/acquire.h>
#include <apt-pkg/acquire-item.h>
#include <libintl.h>

#include <ept/config.h>
#include <ept/error.h>
#include <ept/manager.h>
#include <ept/cache/cache.h>
#include <ept/cache/apt/state.h>
#include <ept/cache/apt/packages.tcc>

namespace ept {
namespace t {

using namespace wibble;

template< typename C >
Manager< C >::Manager( typename C::Aggregator &p )
    : m_packages( p ), m_progress( 0 ), m_updateInterval( 200000 ), m_pm( 0 ),
      m_fetcher( 0 )
{
}

template< typename C >
Manager< C >::~Manager() {
    delete m_pm;
    delete m_fetcher;
}

template< typename C >
void Manager< C >::setProgressCallback( ProgressCallback *progress )
{
   m_progress = progress;
}

template< typename C >
void Manager< C >::download()
{
    // for some reason, if this throws, a subsequent update
    // will fail to grab lock on the lists directory
    // WEIRD ... (and eventually obsolete)
    if ( _system->Lock() == false )
        throw wibble::exception::System( "Could not lock APT database" );

    if ( m_packages.state().brokenCount() != 0 )
        throw exception::Consistency( "Can't download/commit with broken packages." );

    createFetcher();
    m_pm = _system->CreatePM(&m_packages.state().aptState());
    getArchives();
}

template< typename C >
void Manager< C >::getArchives()
{
    if ( !m_pm )
        throw exception::Consistency( "Tried to get archives without PM" );
    if ( !m_fetcher )
        throw exception::Consistency( "Tried to get archives without fetcher" );

    m_fetcher->Shutdown(); // reload

    pkgSourceList SList;

    // FIXME: we really need an apt-front sources list class that'd be
    // safe (that is, throw exceptions itself)
    if(!SList.ReadMainList())
        throw exception::Generic( "The list of sources could not be read.");
    pkgRecords *r = new pkgRecords( m_packages.index().aptCache() );
    m_pm->GetArchives( m_fetcher, &SList, r );
    delete r;

    fetch();
}

template< typename C >
void Manager< C >::fetch() {

    if ( !m_fetcher )
        throw exception::Consistency( "Tried to fetch without fetcher" );

#ifndef RPM
    pkgAcquire::RunResult res = m_fetcher->Run( m_updateInterval );
#else
    pkgAcquire::RunResult res = m_fetcher->Run();
#endif

    if( res == pkgAcquire::Failed )
        throw exception::Generic( "Download failed" );

    if( res == pkgAcquire::Cancelled )
        throw OperationCancelled();
}

template< typename C >
void Manager< C >::createFetcher() {
    if ( m_progress == 0 ) {
        throw exception::Consistency(
            "Manager: createFetcher called without progress object");
    }
    m_fetcher = new pkgAcquire( m_progress );
}

template< typename C >
void Manager< C >::commit()
{
    if (!m_pm)
        throw exception::Consistency( "Tried to commit without PM" );

    // mornfall: is it me or this looks like a race?
    while (1) {
        _system->UnLock(); // unlock for dpkg to take over
#ifndef RPM
        pkgPackageManager::OrderResult Res = m_pm->DoInstall(-1);
#else
        pkgPackageManager::OrderResult Res = m_pm->DoInstall();
#endif
        if (Res == pkgPackageManager::Completed) {
            // _system->Lock(); // regain lock
            delete m_pm;
            m_pm = 0;
            // m_packages.invalidate();
            m_packages.reload();
            return;
        } else if (Res == pkgPackageManager::Failed) {
            m_packages.reload();
            // m_packages.invalidate();
            throw exception::Generic( "Error installing packages" );
        }
        checkGlobalError( "Error installing packages" );
        getArchives();
    }
}

// mornfall: this needs to be unfucked (exceptions, make it safe, etc)
template< typename C >
void Manager< C >::update()
{
    // m_packages.invalidate();
    try {
        pkgSourceList SList;
        if(!SList.ReadMainList())
            throw exception::Generic( "The list of sources could not be read.");

        FileFd Lock;
        Lock.Fd(GetLock(_config->FindDir("Dir::State::Lists") + "lock"));
        checkGlobalError( "Unable to lock the list directory" );

        createFetcher();
        SList.GetIndexes( m_fetcher );

        fetch();

        bool Failed = false;
        string FailedURI;
        for (pkgAcquire::ItemIterator I = m_fetcher->ItemsBegin();
             I != m_fetcher->ItemsEnd(); I++)
        {
            if ((*I)->Status == pkgAcquire::Item::StatDone)
                continue;

            (*I)->Finished();

            if((*I)->ErrorText.empty())
                FailedURI += (*I)->DescURI() + "\n";
            else
                FailedURI += (*I)->DescURI() + ": " + (*I)->ErrorText + "\n";
            Failed = true;
        }

        // Clean out any old list files
        if (_config->FindB("APT::Get::List-Cleanup",true) == true)
        {
            if ( m_fetcher->Clean(_config->FindDir("Dir::State::lists")) == false ||
                 m_fetcher->Clean(_config->FindDir("Dir::State::lists")
                                  + "partial/") == false )
                ; //return false;
        }

        m_packages.reload();
    } catch (...) {
        m_packages.reload();
        throw;
    }

}

}
}
