/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#ifndef TT_H
#define TT_H 1


#ifdef LIBFLUX_BUILD
# include "types.h"
#else
# include <flux/types.h>
#endif

#include <stdio.h>
#include <regex.h>


#define TRUE 1
#define FALSE 0


#define TT_FOR_EACH(tt, c) \
  for((c) = tt_get_first_child(tt); (c); (c) = tt_get_next(c))

#define TT_FOR_ALL(tt, c) \
  for((c) = tt_get_first_child(tt); (c); c = tt_get_next_infix(c, tt))


/* 32 bytes */

typedef struct _tt
{
  struct _tt *parent,
             *child_first, *child_last,
             *next, *prev;

  u16 data_is_internal : 1;  /* Data is easily accessible; resides in RAM */
  u16 data_is_local : 1;     /* Data is allocated locally; kill on delete */
  u16 data_is_ready : 1;     /* Data is fully loaded (for constructors) */
  u16 is_fake_root : 1;

  short handle;  /* Handle is for file operations */
  u32 size;
  void *data;  /* Pointer to internal mem block or (string) name of ext. file */
} TT;

#define TT(x) ((TT *) x)

typedef enum
{
  TT_PRINT_COMPACT = 0, TT_PRINT_KNR, TT_PRINT_ALLMAN
} TT_PRINT_MODE;


#define TT_GENERIC_BLOCK_SIZE 1024


/* Allocation */

TT *tt_new();
TT *tt_new_with_data(void *src, int len);
TT *tt_new_with_parent_and_data(TT *parent, void *src, int len);
void tt_del(TT *tt);
TT *tt_dup(TT *tt);      /* Duplicate and detach node */
TT *tt_dup_all(TT *tt);  /* Duplicate and detach branch */

TT *tt_split(TT *tt, u32 pos);  /* FIXME: External data. */

/* Status set */

#define tt_set_ready(tt, ready) ((tt)->data_is_ready = ready)
#define tt_set_internal(tt, internal) ((tt)->data_is_internal = (internal))
#define tt_set_fake_root(tt, fake_root) ((tt)->is_fake_root = (fake_root))

/* Status get */

#define tt_has_data(tt) ((tt)->data ? 1 : 0)
#define tt_is_ready(tt) ((tt)->data_is_ready ? 1 : 0)
#define tt_data_is_internal(tt) ((tt)->data_is_internal)
#define tt_data_is_local(tt) ((tt)->data_is_local)
#define tt_is_fake_root(tt) ((tt)->is_fake_root)

/* Statistics */

u32 tt_depth(TT *tt);  /* Root is at level 0 */
void tt_stat_children_all(TT *root, u32 *count, u32 *size);

#define tt_size(tt) ((tt)->size)
u32 tt_size_children(TT *root);
u32 tt_size_children_all(TT *root);

u32 tt_count_children(TT *tt);
u32 tt_count_children_all(TT *tt);
u32 tt_count_siblings(TT *tt);

/* Navigation */

#define tt_is_root(tt) ((tt)->parent ? 0 : 1)
#define tt_is_leaf(tt) ((tt)->child_first ? 0 : 1)
#define tt_is_first(tt) ((tt)->prev ? 0 : 1)
#define tt_is_last(tt) ((tt)->next ? 0 : 1)
#define tt_is_sibling(tt0, tt1) ((tt0)->parent == (tt1)->parent ? 1 : 0)

int tt_has_child(TT *parent, TT *child);

TT *tt_get_root(TT *tt);
TT *tt_get_first_sibling(TT *tt);
TT *tt_get_last_sibling(TT *tt);
#define tt_get_prev(tt) ((tt)->prev)
#define tt_get_next(tt) ((tt)->next)
#define tt_get_parent(tt) ((tt)->parent)
#define tt_get_first_child(tt) ((tt)->child_first)
#define tt_get_last_child(tt) ((tt)->child_last)
TT *tt_get_next_infix(TT *tt, TT *top);
TT *tt_get_common_parent(TT *tt0, TT *tt1);
TT *tt_get_next_in_breadth_with_level(TT *tt, int depth, int level);
#define tt_get_next_in_breadth(tt, depth) tt_get_next_in_breadth_with_level(tt, depth, tt_depth(tt))
TT *tt_get_next_in_same_depth(TT *tt);

/* Linking */

void tt_add_as_first_child(TT *parent_tt, TT *tt);
void tt_add_as_last_child(TT *parent_tt, TT *tt);
void tt_add_as_first_sibling(TT *sibling_tt, TT *tt);
void tt_add_as_last_sibling(TT *sibling_tt, TT *tt);
void tt_add_before(TT *next_tt, TT *tt);
void tt_add_after(TT *prev_tt, TT *tt);
#define tt_add(parent_tt, tt) tt_add_as_last_child(parent_tt, tt)

void tt_swap(TT *tt0, TT *tt1);
void tt_detach(TT *tt);

int tt_is_in_path(TT *tt0, TT *tt1);  /* tt0 is in path of tt1? */

/* Data */

int tt_get_external_handle(TT *tt);

void tt_data_swap(TT *tt0, TT *tt1);
void tt_data_del(TT *tt);

void tt_data_set_internal(TT *tt, void *src, u32 len, unsigned int copy);
int tt_data_set_file(TT *tt, const char *path, int local);  /* Returns success indicator */

void tt_data_set_bytes(TT *tt, void *src, u32 start, u32 len);
void tt_data_append_bytes(TT *tt, void *src, u32 len);
void tt_data_prepend_bytes(TT *tt, void *src, u32 len);
void tt_data_set_int(TT *tt, int val);
void tt_data_set_ptr(TT *tt, void *ptr);
void tt_data_set_str(TT *tt, const char *str);

void *tt_data_get(TT *tt);  /* Returns file path (string) or pointer to data */
u32 tt_data_get_bytes(TT *tt, void *dest, u32 start, u32 len);
int tt_data_get_int(TT *tt);
void *tt_data_get_ptr(TT *tt);
char *tt_data_get_str(TT *tt);

/* Matching */

int tt_cmp(TT *tt0, TT *tt1);
int tt_casecmp(TT *tt0, TT *tt1);
int tt_memcmp(TT *tt, void *p, u32 len);
int tt_memcasecmp(TT *tt, void *p, u32 len);
#define tt_strcmp(tt, s) (tt_memcmp((tt), (s), strlen(s)))
#define tt_strcasecmp(tt, s) (tt_memcasecmp((tt), (s), strlen(s)))

size_t tt_spn(TT *tt, const char *accept);
size_t tt_cspn(TT *tt, const char *reject);
int tt_chr(TT *tt, int c);
int tt_rchr(TT *tt, int c);

int tt_regcmp_precomp(TT *tt, regex_t *preg);
int tt_regcmp(TT *tt, const char *regex);
int tt_regcasecmp(TT *tt, const char *regex);


/* Searching */

TT *tt_find_first_sibling(TT *tt, void *data, u32 len);
TT *tt_find_last_sibling(TT *tt, void *data, u32 len);
TT *tt_find_next(TT *tt, void *data, u32 len);
TT *tt_find_prev(TT *tt, void *data, u32 len);

TT *tt_find_first_child(TT *tt, void *data, u32 len);
TT *tt_find_last_child(TT *tt, void *data, u32 len);

TT *tt_match_first_sibling(TT *tt, const char *regexp);
TT *tt_match_last_sibling(TT *tt, const char *regexp);
TT *tt_match_next(TT *tt, const char *regexp);
TT *tt_match_prev(TT *tt, const char *regexp);

TT *tt_match_first_child(TT *tt, const char *regexp);
TT *tt_match_last_child(TT *tt, const char *regexp);

/* Hashing */

u32 tt_hash(TT *tt);
u32 tt_hash_all(TT *tt);

/* Scanning/printing */

TT *tt_scan_from_file(FILE *in);
void tt_print_to_file(TT *tt, FILE *out, TT_PRINT_MODE mode, int honour_meta);

#endif /* TT_H */
