/******************************************************************************************************************************************
 cobject.c
******************************************************************************************************************************************/

#include "cobject.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CObjectListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CObjectListener::CObjectListener ()
		:m_Owners	 ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CObjectListener::~CObjectListener ()
{
	for (size_t i=m_Owners.GetLength(), j=0; i>0; i--, j++) (*m_Owners[j]) -> m_Listener = NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CObject);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CObject::CObject  	     (const CObjectListener *inListener)
	:CMetaModule	     (),
	 m_Listener  	     (const_cast <CObjectListener *> (inListener)),
	 m_SerializeListener (true)
{
	if (m_Listener != NULL)
	{
		m_Listener -> m_Owners += this;
		m_Listener -> OnConstruct (this);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CObject::~CObject ()
{
	if (m_Listener != NULL)
	{
		m_Listener -> OnDestruct (this);
		m_Listener -> m_Owners -= this;
		if (m_Listener -> m_Owners.GetLength() == 0) delete m_Listener;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener must be
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CObject::ListenerMustBe () const
{
	return __metaclass(CObjectListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CObject::AssignListener (const CObjectListener *inListener)
{
	if (m_Listener != NULL)
	{
		m_Listener -> m_Owners -= this;
		if (m_Listener -> m_Owners.GetLength() == 0) delete m_Listener;
		m_Listener = NULL;
	}
	if (inListener != NULL)
	{
		if (inListener -> ClassIs (ListenerMustBe())) 
		{
			m_Listener = const_cast <CObjectListener *> (inListener);
			m_Listener -> m_Owners += this;
		}
		else
			return false;
	}
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener removal
//-----------------------------------------------------------------------------------------------------------------------------------------
CObjectListener * CObject::RemoveListener ()
{
	CObjectListener *outListener = m_Listener;
	if (m_Listener != NULL) m_Listener -> m_Owners -= this;
	m_Listener = NULL;
	return outListener;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener reader
//-----------------------------------------------------------------------------------------------------------------------------------------
CObjectListener * CObject::GetListener () const
{
	return const_cast <CObjectListener *> (m_Listener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CObject::Serialize (CArchive &ioArchive) THROWABLE
{
	// generic call
	CMetaModule::Serialize (ioArchive);

	// do the object have to handle the listener serialization ?
	if (!m_SerializeListener) return;

	// process analyse
	switch (ioArchive.GetProcess())
	{
		// storing object
		case ARCHIVEPROCESS_STORING :
		{
			
			// save potential listener specifications
			if (m_Listener != NULL && metaclass_cast(m_Listener) -> MetaClassType == METACLASS_DYNAMIC)
				ioArchive << *m_Listener;
			else
				ioArchive << static_cast <unsigned long> (0L);
		}
		break;

		// loading object
		case ARCHIVEPROCESS_LOADING :
		{
			// delete listener if any
			if (m_Listener != NULL) delete m_Listener; m_Listener = NULL;

			// is next data in archive some type of unsigned long (metaclass signature expected or 0L) ?
			if (ioArchive.NextDataIs() != UINT32)
				throw new CException 
					(CString("Error while looking for listener specifications in archive, wrong data type."),
					 __exception(DATATYPE));

			// retreive metaclass, instanciate the associated class and load it if defined, jump over unsigned long if any
			if ((m_Listener = static_cast <CObjectListener *> (CSerialized::Instanciate (ioArchive))) == NULL)
				ioArchive++;
		}
		break;
	}	
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CObject::Serialize	(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call
	CMetaModule::Serialize (ioXMLElementNode, inMode);

	// do the object have to handle the listener serialization ?
	if (!m_SerializeListener) return;

	// serialization request analyse
	switch (inMode)
	{
		// load in process
		case XML_READ :
		{
			// delete the listener if any
			if (m_Listener != NULL) delete m_Listener; m_Listener = NULL;

			// foreach child of the current xml node...
			for (size_t i=0; i<xml_node_get_children_number(ioXMLElementNode); i++)
			{
				// current child node read in
				CXMLElementNode *inXMLNode (::xml_node_get_child (ioXMLElementNode, i));

				// check we got a metaclass associated with it
				const CMetaClass *inMetaClass = CSerialized::GetMetaClass (inXMLNode);	

				// check it is a listener one and that it can be instanciated to avoid exception to be thrown away...
				if (inMetaClass != NULL && CMetaClass::MetaClassIs (__metaclass(CObjectListener), inMetaClass) && 
				    inMetaClass -> MetaClassType == METACLASS_DYNAMIC && inMetaClass -> ClassInstanciate != NULL)
				{
					// there is no mistake, instanciate it !
					m_Listener = static_cast <CObjectListener *> (inMetaClass -> ClassInstanciate ());

					// so now, load our listener from the current node
					m_Listener -> Serialize (inXMLNode, XML_READ);

					// ok
					break;
				}
			}
		}
		break;

		// storing process
		case XML_WRITE :
		{
			// keep a local copy of the xml node that will be modified by the serialization process
			CXMLElementNode *outXMLElementNode = ioXMLElementNode;

			// is there any listener associated to this object and is it a dynamic one ?
			if (m_Listener != NULL && metaclass_cast(m_Listener) -> MetaClassType == METACLASS_DYNAMIC &&
			    metaclass_cast(m_Listener) -> ClassInstanciate != NULL)
				m_Listener -> Serialize (outXMLElementNode, XML_WRITE);
		}
		break;
	}
}
