/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-----------------------------------------------------------------------
FILENAME:               post.h
LIBRARY:                libgpiv
EXTERNAL FUNCTIONS:
SOURCE:                 post_par.c
                        gpiv_valid_parameters_logic
                        gpiv_valid_default_parameters
                        gpiv_post_read_parameters
                        gpiv_post_check_parameters_read
			gpiv_post_test_parameter
			gpiv_post_print_parameters
			gpiv_post_fprint_parameters
                        gpiv_post_fread_hdf5_parameters
                        gpiv_post_fwrite_hdf5_parameters

SOURCE:                 post.c
			gpiv_post_manipiv 
			gpiv_post_savg
			gpiv_post_uvhisto
                        gpiv_post_subtract_dxdy
                        gpiv_post_scale
                        gpiv_post_inverse_scale
                        gpiv_post_vorstra


LAST MODIFICATION DATE: $Id: post.h,v 1.16 2006/01/31 13:27:22 gerber Exp $
 ------------------------------------------------------------------- */
#ifndef __LIBGPIV_POST_H__
#define __LIBGPIV_POST_H__

/*
 * Default values of GpivPostPar
 */
#define GPIV_GPIVPOSTPAR_DEFAULT__OPERATOR_MANIPIV 0
#define GPIV_GPIVPOSTPAR_DEFAULT__SET 1
#define GPIV_GPIVPOSTPAR_DEFAULT__SET_DX 0.0
#define GPIV_GPIVPOSTPAR_DEFAULT__SET_DY 0.0
#define GPIV_GPIVPOSTPAR_DEFAULT__SUBTRACT 0
#define GPIV_GPIVPOSTPAR_DEFAULT__Z_OFF_DX 0.0
#define GPIV_GPIVPOSTPAR_DEFAULT__Z_OFF_DY 0.0
#define GPIV_GPIVPOSTPAR_DEFAULT__DIFF_TYPE 2
#define GPIV_GPIVPOSTPAR_DEFAULT__OPERATOR_VORSTRA 0
#define GPIV_GPIVPOSTPAR_DEFAULT__SCALE_TYPE 0

/*
 * Variables for manipulating PIV data
 */


enum GpivOperationManipiv {	/* what to do with the input data */
    GPIV_FAST_Y = 0,			/* returns fast running y-positions of data */
    GPIV_FLIP_X = 1,			/* flip in x-direction */
    GPIV_FLIP_Y = 2,			/* flip in y-direction */
    GPIV_REVERT = 3,			/* reverts array indexes  */
    GPIV_ROT90 = 4,			/* rotates over 90 degrees */
    GPIV_ROT180 = 5,			/* rotates over 180 degrees */
    GPIV_FILTER_BLOCK = 6,		/* filters out data */
    GPIV_PASS_BLOCK = 7,
    GPIV_ADD_XY = 8                  /* adds specified displacement/velocity in x and y direction */
} OPERATOR_MANIPIV;		/* passes through data */

typedef struct __GpivRoi GpivRoi;
struct __GpivRoi {		/* Region of interest */
    float x_1;			/* lowest x value */
    float y_1;			/* lowest y value */
    float x_2;			/* highest x value */
    float y_2;			/* highest x value */
} block;



/*
 * Variables for vorticity and strain
 */
enum GpivDifferentiation {
    GPIV_CENTRAL = 0,
    GPIV_LEAST_SQUARES = 1,
    GPIV_RICHARDSON = 2,
    GPIV_CIRCULATION = 3
};

enum GpivOperation {
    GPIV_VORTICITY = 0,
    GPIV_S_STRAIN = 1,
    GPIV_N_STRAIN = 2
};

/*
 * Variables for scaling PIV data
 */
enum GpivScaleType {            /* enumerate for normal and inverse scaling */
    GPIV_SCALE = 0,
    GPIV_SCALE_INV = 1
} GPIV_SCALE_TYPE;


/*
 * Velocity component U (horizontal), V (vertical)
 */
enum GpivVelComponent {
    GPIV_U,
    GPIV_V
};

typedef struct __GpivPostPar GpivPostPar;
struct __GpivPostPar {

/*
 * Parameters for manipulating PIV data
 */
    int operator_manipiv;
    int operator_manipiv_logic;	/* variable to check if operator_manipiv has 
                                   been read */

    gboolean set;
    gboolean set_logic;
    float set_dx;
    gboolean set_dx_logic;
    float set_dy;
    gboolean set_dy_logic;

/*
 * Parameters for s-avg
 */
    int subtract;
    gboolean subtract_logic;
    float z_off_dx;
    gboolean z_off_dx_logic;
    float z_off_dy;
    gboolean z_off_dy_logic;

/*
 * Parameters for vorticity and strain
 */
    int diff_type;
    gboolean operator_vorstra;

    int diff_type_logic;
    gboolean operator_vorstra_logic;

/*
 * Parameters for scaling PIV data. Other parameters for scaling are in 
 * __GpivImagePar.
 */
    enum GpivScaleType scale_type;   /* normal (0) or inverse (1) scaling */
    gboolean scale_type_logic;
};



void
gpiv_post_parameters_logic(GpivPostPar * piv_post_par,
                           gboolean flag
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Set flag for piv_post_par _logic
 *
 * INPUTS:
 *      piv_post_par:   post-processing parameters
 *      flag:           boolean to set
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_post_default_parameters(GpivPostPar *piv_post_par_default,
                             gboolean flag
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets post-processing parameters to default values
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *      piv_post_par_default:   default post-processing parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_post_read_parameters(FILE * fp_par, 
                          GpivPostPar * piv_post_par,
                          gboolean print_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read all parameters for PIV data validation
 *
 * INPUTS:
 *      fp_par:         pointer to input file
 *
 * OUTPUTS:
 *      piv_post_par:   post-processing parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_post_check_parameters_read(GpivPostPar * piv_post_par,
                                GpivPostPar piv_post_par_default
                                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Check out if all parameters have been read. Else, set to default value
 *
 * INPUTS:
 *      piv_post_par:   post-processing parameters
 *      piv_post_par_default:   default post-processing parameters
 *
 * OUTPUTS:
 *      piv_post_par:   post-processing parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_post_test_parameter(GpivPostPar piv_post_par
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Testing parameters on valid values and initializing derived variables
 *
 * INPUTS:
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_post_print_parameters(GpivPostPar piv_post_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints parameters to stdout
 *
 * INPUTS:
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/




void 
gpiv_post_fprint_parameters(FILE * fp_par_out,
                            GpivPostPar piv_post_par
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints parameters to fp_par_out
 *
 * INPUTS:
 *      piv_post_par:   post-processing parameters
 *      fp_par_out:     pointer to output file
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_post_fread_hdf5_parameters(char *fname, 
                                GpivPostPar * piv_post_par 
                                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads post parameters from hdf5 data file
 *
 * INPUTS:
 *      fname:          file name of data file
 *
 * OUTPUTS:
 *      piv_post_par:   post-processing parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_post_fwrite_hdf5_parameters(char *fname, 
                                 GpivPostPar * piv_post_par 
                                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes post parameters to an existing hdf5 data file
 *
 * INPUTS:
 *      fname:          file name of data file
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_post_manipiv(char *fname_in_piv,
                  gboolean fname_logic,
                  GpivPivData in_data,
                  GpivPivData * out_data,
                  GpivPostPar piv_post_par,
                  char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS],
                  int nd_lines,
                  char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int *nc_lines
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to manipulate data; flipping rotating, etc
 *
 * INPUTS:
 *      fname_in_piv:   file name of input piv data
 *      fname_logic:    flag whether files are to be used
 *      in_data:        input piv data
 *      piv_post_par:   post-processing parameters
 *      d_line: 	data lines
 *      nd_lines:       number of data lines
 *      c_line:         commented lines
 *      nc_lines:       number of commented lines
 *      
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_post_savg(GpivPivData in_data,
               GpivPivData * out_data,
               GpivPostPar piv_post_par,
               int *ndata
               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to calculate spatial mean, variances
 *
 * INPUTS:
 *      in_data:        input piv data
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_post_uvhisto(GpivPivData in_data, 
                 GpivBinData * klass,
                 enum GpivVelComponent velcomp
                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculating histogram of U (horizontal) or V (vertical) particle 
 *     displacements
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *      in_data:        input piv data
 *      velcomp:        velocity component from which histogram is calculated
 *
 * OUTPUTS:
 *      klass:          histogram
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_post_subtract_dxdy(GpivPivData in_data,
                        GpivPivData *out_data, 
                        float z_off_dx, 
                        float z_off_dy
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Subtracts displacements or velocity from piv data
 *
 * INPUTS:
 *      in_data:        input piv data
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int 
gpiv_post_scale(GpivPivData in_data,
                GpivPivData * out_data,
                GpivImagePar image_par
                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to adjust for time and spatial scales
 *
 * INPUTS:
 *      in_data:        input piv data
 *      image_par:      image parameter structure
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int 
gpiv_post_inverse_scale(GpivPivData in_data,
                        GpivPivData * out_data,
                        GpivImagePar image_par
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to adjust for time and spatial scales
 *
 * INPUTS:
 *      in_data:        input piv data
 *      image_par:      image parameter structure
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_post_vorstra(GpivPivData in_data,
                  GpivScalarData * out_data,
                  GpivPostPar piv_post_par
                  );

/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to calculate vorticity and strain
 *
 * INPUTS:
 *      in_data:        input piv data
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



#endif /* __LIBGPIV_POST_H__ */
