/*******************************************************************************************************************************************
 serialize.c

 simple compilation directives :
 c++ -Wno-multichar -O2 -o serialize `pkg-config --cflags libgtkol-1.2` serialize.c `pkg-config --libs libgtkol-1.2`

 This sample demonstrates gtkol serialization capabilities.
*******************************************************************************************************************************************/

#include "capplication.h"

const static CString DEMO_XML_DUMP ("/tmp/serialize.xml");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CDemoFormSerializeListener definition
//-----------------------------------------------------------------------------------------------------------------------------------------
class CDemoFormSerializeListener : public CFormListener
{
	// called on the inSender form when it is closed
	virtual void				OnClose		(CObject *inSender);

	// because the form is to be serialized and we want its associated listener to be instanciated again from scratch when reloaded
	// we have to associate it to a dynamic metaclass so that the libgenerics serialization engine will know how to handle it
	SECTION_DYNAMIC_METACLASS;
};

// declare the specific metaclass and resolve its code
DECLARE_DYNAMIC_METACLASS ('_osz', CDemoFormSerializeListener, CFormListener);
RESOLVE_DYNAMIC_METACLASS (CDemoFormSerializeListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CFileExitMenuItemListener definition
//-----------------------------------------------------------------------------------------------------------------------------------------
class CFileExitMenuItemListener : public CMenuItemListener
{
	// called on the inSender menuitem when it is clicked
	virtual void				OnClick		(CObject *inSender);

	// because the menuitem is to be serialized and we want its associated listener to be instanciated again from scratch when reloaded
	// we have to associate it to a dynamic metaclass so that the libgenerics serialization engine will know how to handle it
	SECTION_DYNAMIC_METACLASS;
};

// declare the specific metaclass and resolve its code
DECLARE_DYNAMIC_METACLASS ('_ext', CFileExitMenuItemListener, CMenuItemListener);
RESOLVE_DYNAMIC_METACLASS (CFileExitMenuItemListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CDemoFormSerializeListener::OnClose definition
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDemoFormSerializeListener::OnClose (CObject *)
{
	// just open an xml document giving it gtkol-demo as a root element name 
	CXMLDocument XMLDoc (DEMO_XML_DUMP, XML_WRITE, CString("gtkol-demo"));

	// dump the whole application into the xml file i.e. retreive the absolute component node (the capplication one with id always 1)
	// and redirect it into the document
	XMLDoc << *CComponent::GetComponent (1L);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// CFileExitMenuItemListener::OnClick definition
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileExitMenuItemListener::OnClick (CObject *inSender)
{
	// retreive the menuitem instance and get its owner form to request a close action on it
	static_cast <CForm *> (static_cast <CMenuItem *> (inSender) -> GetOwner (__metaclass(CForm))) -> Close ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// execution entry point
//-----------------------------------------------------------------------------------------------------------------------------------------
int main (int argc, char **argv)
{
	// if the xml dump file does not exist, we have to explicitly instanciate the demo
	if (!CFile::Exists (DEMO_XML_DUMP))
	{
		// declare an application
		CApplication Application (argc, argv);

		// instanciate a form with the given listener and set some properties
		CForm *Form = new CForm (&Application, new CDemoFormSerializeListener());
		Form -> SetCaption (CString("Serialized Form"));
		Form -> SetBounds (TBounds (TBounds ((CApplication::GetScreenSize().w-420)/2, 
						     (CApplication::GetScreenSize().h-350)/2, 420, 350)));

		// instanciate a vertical layout and set some properties
		CVBoxLayout *VBoxLayout = new CVBoxLayout (Form);
		VBoxLayout -> SetHomogeneous (false);
		VBoxLayout -> SetSpacing (1);
		VBoxLayout -> SetBoxPack (BOXPACK_START, false, false, 0);

		// instanciate a menu bar and its menu items
		CMenuBar  *MenuBar = new CMenuBar  (VBoxLayout);
		CMenuItem *File    = new CMenuItem (MenuBar, CString("_File"));
		CMenuItem *Exit    = new CMenuItem (File, CString("_Quit"), _HKNone_, new CFileExitMenuItemListener());

		// ok, instanciate a treeview with setting the model to handle some strings and pixbufs and set it dropsiteable
		VBoxLayout -> SetBoxPack  (BOXPACK_START, true, true, 0);
		CTreeView *TreeView  = new CTreeView (VBoxLayout, _IFVString_+_IFVPixbuf_);
		TreeView   -> SetDropSite (true);

		// instanciate a tree view item, setting its field values and to be a draggable and dropsiteable control
		CTreeViewItem &Item0 = *new CTreeViewItem (TreeView);
		Item0.SetDraggable (true);
		Item0.SetDropSite  (true);
		Item0[0] = CString("Copy");
		Item0[1] = new CPixbuf(GTK_STOCK_COPY);

		// another one...
		CTreeViewItem &Item1 = *new CTreeViewItem (TreeView);
		Item1.SetDraggable (true);
		Item1.SetDropSite  (true);
		Item1[0] = CString("Cut"); 
		Item1[1] = new CPixbuf(GTK_STOCK_CUT);
		
		// and a last one
		CTreeViewItem &Item2 = *new CTreeViewItem (TreeView);
		Item2.SetDraggable (true);
		Item2.SetDropSite  (true);
		Item2[0] = CString("Paste");
		Item2[1] = new CPixbuf(GTK_STOCK_PASTE);

		// show the job
		Form -> Show ();

		// run the application
		Application.Run ();
	}
	// the xml dump file exists, relaod the whole application from it
	else
	{
		// be proper as the serialization process may generate some exceptions...
		try
		{
			// open the xml dump file, check we got the specified root element node and get it
			CXMLDocument XMLDoc (DEMO_XML_DUMP, XML_READ, CString("gtkol-demo"));
			CXMLElementNode *inXMLRoot = XMLDoc.GetRootElement ();

			// reload the whole application from scratch...
			CSerialized *inSerialized = CSerialized::Instanciate (inXMLRoot);

			// check we got an application instance as excepted and launch it !
			if (inSerialized -> ClassIs (__metaclass(CApplication))) static_cast <CApplication *> (inSerialized) -> Run ();

			// delete the instances
			delete inXMLRoot;
			delete inSerialized;
		}
		// an exception occured
		catch (CException *inException)
		{
			// just print it
			::fprintf (stderr, "Exception : %s\n", inException->GetMessage().Get());

			// stop
			return -1;
		}
	}

	// ok
	return 0;
}

