/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  drawable.cc - GdkDrawable C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "drawable.h"
#include "private/drawable_p.h"
#include "color.h"
#include "display.h"
#include "image.h"
#include "region.h"
#include "screen.h"
#include "visual.h"
#include "../gdk-pixbuf/pixbuf.h"
#include "../pango/font.h"
#include "../pango/glyph.h"
#include "../pango/layout.h"
#include <gdk/gdkpixbuf.h>

using namespace Inti;

/*  Gdk::Drawable
 */

Gdk::Drawable::Drawable(GdkDrawable *drawable, bool reference)
: G::Object((GObject*)drawable, reference)
{
}

Gdk::Drawable::Drawable()
: G::Object((GObject*)DrawableClass::create())
{
}

Gdk::Drawable::~Drawable()
{
}

GdkDrawableClass* 
Gdk::Drawable::gdk_drawable_class() const 
{ 
	return get_class<GdkDrawableClass>();
}
	
Gdk::Drawable::operator GdkDrawable* () const 
{ 
	return this ? gdk_drawable() : 0; 
}
	
void
Gdk::Drawable::get_size(int *width, int *height) const
{
	gdk_drawable_get_size(gdk_drawable(), width, height);
}

Gdk::Colormap*
Gdk::Drawable::get_colormap() const
{
	GdkColormap *colormap = gdk_drawable_get_colormap(gdk_drawable());
	return colormap ? G::Object::wrap<Colormap>(colormap) : 0;
}

const Gdk::Visual*
Gdk::Drawable::get_visual() const
{
	return G::Object::wrap<Visual>(gdk_drawable_get_visual(gdk_drawable()));
}

int
Gdk::Drawable::get_depth() const
{
	return gdk_drawable_get_depth(gdk_drawable());
}

Gdk::Screen*
Gdk::Drawable::get_screen() const
{
	return G::Object::wrap<Screen>(gdk_drawable_get_screen(gdk_drawable()));
}

Gdk::Display*
Gdk::Drawable::get_display() const
{
	return G::Object::wrap<Display>(gdk_drawable_get_display(gdk_drawable()));
}

Pointer<Gdk::Image>
Gdk::Drawable::get_image(int x, int y, int width, int height) const
{
	return G::Object::wrap_new<Image>(gdk_drawable_get_image(gdk_drawable(), x, y, width, height), true);
}
	
Pointer<Gdk::Region>
Gdk::Drawable::get_clip_region() const
{
	return new Region(gdk_drawable_get_clip_region(gdk_drawable()));
}

Pointer<Gdk::Region>
Gdk::Drawable::get_visible_region() const
{
	return new Region(gdk_drawable_get_visible_region(gdk_drawable()));
}

Pointer<Gdk::Pixbuf>
Gdk::Drawable::get_pixbuf(int x, int y, int width, int height, Colormap *colormap) const
{
	GdkPixbuf *pixbuf = gdk_pixbuf_get_from_drawable(0, gdk_drawable(), *colormap, x, y, 0, 0, width, height);
	return pixbuf ? G::Object::wrap_new<Pixbuf>(pixbuf, true) : 0;
}
	
void 
Gdk::Drawable::set_colormap(Colormap *colormap)
{
	gdk_drawable_set_colormap(gdk_drawable(), colormap->gdk_colormap());
}

void 
Gdk::Drawable::draw_polygon(const GC& gc, const std::vector<Point>& points, bool filled)
{
	g_return_if_fail(!points.empty());

	int count = points.size();
	GdkPoint *tmp_points = new GdkPoint[count];
	
	int i = 0;
	while (i < count)
	{
		tmp_points[i] = *points[i].gdk_point();
		++i;
	}
	
	gdk_draw_polygon(gdk_drawable(), gc.gdk_gc(), filled, tmp_points, count);
	delete [] tmp_points;
}		 

namespace {

void draw_point_array(GdkDrawable *drawable, GdkGC *gc, const std::vector<Gdk::Point>& points, bool draw_as_points)
{
	g_return_if_fail(!points.empty());

	int count = points.size();
	GdkPoint *tmp_points = new GdkPoint[count];
	
	int i = 0;
	while (i < count)
	{
		tmp_points[i] = *points[i].gdk_point();
		++i;
	}
	
	(draw_as_points ? gdk_draw_points : gdk_draw_lines)(drawable, gc, tmp_points, count);
	delete [] tmp_points;
}

} // namespace
	
void 
Gdk::Drawable::draw_points(const GC& gc, const std::vector<Gdk::Point>& points)
{
	draw_point_array(gdk_drawable(), gc.gdk_gc(), points, true);
}

void 
Gdk::Drawable::draw_segments(const GC& gc, const std::vector<Gdk::Segment>& segs)
{	
	g_return_if_fail(!segs.empty());

	int count = segs.size();
	GdkSegment *tmp_segs = new GdkSegment[count];
	
	int i = 0;
	while (i < count)
	{
		tmp_segs[i] = *segs[i].gdk_segment();
		++i;
	}

	gdk_draw_segments(gdk_drawable(), gc.gdk_gc(), tmp_segs, count);
	delete [] tmp_segs;
}

void 
Gdk::Drawable::draw_lines(const GC& gc, const std::vector<Gdk::Point>& points)
{	
	draw_point_array(gdk_drawable(), gc.gdk_gc(), points, false);
}

void 
Gdk::Drawable::draw_pixbuf(const GC *gc, const Pixbuf& pixbuf, int src_x, int src_y, int dest_x, int dest_y,
                           int width, int height, RgbDither dither, int x_dither, int y_dither)
{
	gdk_draw_pixbuf(gdk_drawable(), *gc, pixbuf.gdk_pixbuf(), src_x, src_y,
	                dest_x, dest_y, width, height, (GdkRgbDither)dither, x_dither, y_dither);
}

void
Gdk::Drawable::draw_point(const GC& gc, int x, int y)
{
	gdk_draw_point(gdk_drawable(), gc.gdk_gc(), x, y);
}

void 
Gdk::Drawable::draw_point(const GC& gc, const Point& point)
{
	gdk_draw_point(gdk_drawable(), gc.gdk_gc(), point.x(), point.y());
}

void 
Gdk::Drawable::draw_line(const GC& gc, int x1, int y1, int x2, int y2)
{
	gdk_draw_line(gdk_drawable(), gc.gdk_gc(), x1, y1, x2, y2);
}

void 
Gdk::Drawable::draw_line(const GC& gc, const Segment& seg)
{
	gdk_draw_line(gdk_drawable(), gc.gdk_gc(), seg.x1(), seg.y1(), seg.x2(), seg.y2());
}

void 
Gdk::Drawable::draw_rectangle(const GC& gc, int x, int y, int width, int height, bool filled)
{
	gdk_draw_rectangle(gdk_drawable(), gc.gdk_gc(), filled, x, y, width, height);
}

void 
Gdk::Drawable::draw_rectangle(const GC& gc, const Rectangle& rectangle, bool filled)
{
	gdk_draw_rectangle(gdk_drawable(), gc.gdk_gc(), filled, rectangle.x(), rectangle.y(), 
	                   rectangle.width(), rectangle.height());
}

void 
Gdk::Drawable::draw_arc(const GC& gc, int x, int y, int width, int height, int angle1, int angle2, bool filled)
{
	gdk_draw_arc(gdk_drawable(), gc.gdk_gc(), filled, x, y, width, height, angle1, angle2);
}

void 
Gdk::Drawable::draw_arc(const GC& gc, const Rectangle& rectangle, int angle1, int angle2, bool filled)
{
	gdk_draw_arc(gdk_drawable(), gc.gdk_gc(), filled, rectangle.x(), rectangle.y(), 
	             rectangle.width(), rectangle.height(), angle1, angle2);
}

void
Gdk::Drawable::draw_drawable(const GC& gc, const Drawable& src, int xsrc, int ysrc, int xdest, int ydest, int width, int height)
{
	gdk_draw_drawable(gdk_drawable(), gc.gdk_gc(), src.gdk_drawable(), xsrc, ysrc, xdest, ydest, width, height);
}

void
Gdk::Drawable::draw_drawable(const GC& gc, const Drawable& src, int xsrc, int ysrc, const Rectangle& dest)
{
	gdk_draw_drawable(gdk_drawable(), gc.gdk_gc(), src.gdk_drawable(), xsrc, ysrc, 
	                  dest.x(), dest.y(), dest.width(), dest.height());
}

void
Gdk::Drawable::draw_image(const GC& gc, const Image& image, int xsrc, int ysrc, int xdest, int ydest, int width, int height)
{
	gdk_draw_image(gdk_drawable(), gc.gdk_gc(), image.gdk_image(), xsrc, ysrc, xdest, ydest, width, height);
}

void
Gdk::Drawable::draw_glyphs(const GC& gc, const Pango::Font *font, int x, int y, const Pango::GlyphString *glyphs)
{
	gdk_draw_glyphs(gdk_drawable(), gc.gdk_gc(), *font, x, y, *glyphs);
}

void
Gdk::Drawable::draw_layout_line(const GC& gc, int x, int y, const Pango::LayoutLine& line,
                                            const Color *foreground, const Color *background)
{
	gdk_draw_layout_line_with_colors(gdk_drawable(), gc.gdk_gc(), x, y, line.pango_layout_line(),
	                                 *foreground, *background);
}

void 
Gdk::Drawable::draw_layout(const GC& gc, int x, int y, const Pango::Layout& layout,
                                       const Color *foreground, const Color *background)
{
	gdk_draw_layout_with_colors(gdk_drawable(), gc.gdk_gc(), x, y, layout.pango_layout(),
	                            *foreground, *background);
}

void 
Gdk::Drawable::draw_rgb_image(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                              unsigned char *rgb_buf, int rowstride)
{
 	gdk_draw_rgb_image(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, rgb_buf, rowstride);
}

void 
Gdk::Drawable::draw_rgb_image_dithalign(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                                        unsigned char *rgb_buf, int rowstride, int xdith, int ydith)
{
	gdk_draw_rgb_image_dithalign(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, rgb_buf,
	                             rowstride, xdith, ydith);
}

void 
Gdk::Drawable::draw_rgb_32_image(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                                 unsigned char *buf, int rowstride)
{
	gdk_draw_rgb_32_image(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, buf, rowstride);
}

void 
Gdk::Drawable::draw_rgb_32_image_dithalign(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                                           unsigned char *buf, int rowstride, int xdith, int ydith)
{
	gdk_draw_rgb_32_image_dithalign(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, buf,
	                                rowstride, xdith, ydith);
}

void 
Gdk::Drawable::draw_gray_image(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                               unsigned char *buf, int rowstride)
{
	gdk_draw_gray_image(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, buf, rowstride);
}

void 
Gdk::Drawable::draw_indexed_image(const GC& gc, int x, int y, int width, int height, RgbDither dith,
                                  unsigned char *buf, int rowstride, GdkRgbCmap *cmap)
{
	gdk_draw_indexed_image(gdk_drawable(), gc.gdk_gc(), x, y, width, height, (GdkRgbDither)dith, buf, rowstride, cmap);
}

/*  Gdk::DrawableClass
 */

void
Gdk::DrawableClass::init(GdkDrawableClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->create_gc = &create_gc_proxy;
	g_class->draw_rectangle = &draw_rectangle_proxy;
	g_class->draw_arc = &draw_arc_proxy;
	g_class->draw_polygon = &draw_polygon_proxy;
	g_class->draw_drawable = &draw_drawable_proxy;
	g_class->draw_points = &draw_points_proxy;
	g_class->draw_segments = &draw_segments_proxy;
	g_class->draw_lines = &draw_lines_proxy;
	g_class->draw_glyphs = &draw_glyphs_proxy;
	g_class->draw_image = &draw_image_proxy;
	g_class->get_depth = &get_depth_proxy;
	g_class->get_size = &get_size_proxy;
	g_class->set_colormap = &set_colormap_proxy;
	g_class->get_colormap = &get_colormap_proxy;
	g_class->get_visual = &get_visual_proxy;
	g_class->get_image = &get_image_proxy;
	g_class->get_clip_region = &get_clip_region_proxy;
	g_class->get_visible_region = &get_visible_region_proxy;
}

GType
Gdk::DrawableClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GDK_TYPE_DRAWABLE, (GClassInitFunc)init);
	}
	return type;
}

void*
Gdk::DrawableClass::create()
{
	return g_object_new(get_type(), 0);
}

GdkGC*
Gdk::DrawableClass::create_gc_proxy(GdkDrawable *drawable, GdkGCValues *values, GdkGCValuesMask mask)
{
	GdkGC *gc = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		gc = tmp_drawable->do_create_gc(values, mask);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->create_gc)
			gc = g_class->create_gc(drawable, values, mask);
	}
	return gc;
}

void
Gdk::DrawableClass::draw_rectangle_proxy(GdkDrawable *drawable, GdkGC *gc, gint filled, gint x, gint y, gint width, gint height)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_rectangle(gc, filled, x, y, width, height);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_rectangle)
			g_class->draw_rectangle(drawable, gc, filled, x, y, width, height);
	}
}

void
Gdk::DrawableClass::draw_arc_proxy(GdkDrawable *drawable, GdkGC *gc, gint filled, gint x, gint y, gint width, gint height, gint angle1, gint angle2)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_arc(gc, filled, x, y, width, height, angle1, angle2);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_arc)
			g_class->draw_arc(drawable, gc, filled, x, y, width, height, angle1, angle2);
	}
}

void
Gdk::DrawableClass::draw_polygon_proxy(GdkDrawable *drawable, GdkGC *gc, gint filled, GdkPoint *points, gint npoints)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_polygon(gc, filled, points, npoints);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_polygon)
			g_class->draw_polygon(drawable, gc, filled, points, npoints);
	}
}

void
Gdk::DrawableClass::draw_drawable_proxy(GdkDrawable *drawable, GdkGC *gc, GdkDrawable *src, gint xsrc, gint ysrc, gint xdest, gint ydest, gint width, gint height)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_drawable(gc, src, xsrc, ysrc, xdest, ydest, width, height);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_drawable)
			g_class->draw_drawable(drawable, gc, src, xsrc, ysrc, xdest, ydest, width, height);
	}
}

void
Gdk::DrawableClass::draw_points_proxy(GdkDrawable *drawable, GdkGC *gc, GdkPoint *points, gint npoints)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_points(gc, points, npoints);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_points)
			g_class->draw_points(drawable, gc, points, npoints);
	}
}

void
Gdk::DrawableClass::draw_segments_proxy(GdkDrawable *drawable, GdkGC *gc, GdkSegment *segs, gint nsegs)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_segments(gc, segs, nsegs);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_segments)
			g_class->draw_segments(drawable, gc, segs, nsegs);
	}
}

void
Gdk::DrawableClass::draw_lines_proxy(GdkDrawable *drawable, GdkGC *gc, GdkPoint *points, gint npoints)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_lines(gc, points, npoints);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_lines)
			g_class->draw_lines(drawable, gc, points, npoints);
	}
}

void
Gdk::DrawableClass::draw_glyphs_proxy(GdkDrawable *drawable, GdkGC *gc, PangoFont *font, gint  x, gint y, PangoGlyphString *glyphs)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_glyphs(gc, font, x, y, glyphs);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_glyphs)
			g_class->draw_glyphs(drawable, gc, font, x, y, glyphs);
	}
}

void
Gdk::DrawableClass::draw_image_proxy(GdkDrawable *drawable, GdkGC *gc, GdkImage *image, gint xsrc, gint ysrc, gint xdest, gint ydest, gint width, gint height)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_draw_image(gc, image, xsrc, ysrc, xdest, ydest, width, height);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->draw_image)
			g_class->draw_image(drawable, gc, image, xsrc, ysrc, xdest, ydest, width, height);
	}
}

gint
Gdk::DrawableClass::get_depth_proxy(GdkDrawable *drawable)
{
	int depth = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		depth = tmp_drawable->do_get_depth();
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_depth)
			depth = g_class->get_depth(drawable);
	}
	return depth;
}

void
Gdk::DrawableClass::get_size_proxy(GdkDrawable *drawable, gint *width, gint *height)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_get_size(width, height);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_size)
			g_class->get_size(drawable, width, height);
	}
}

void
Gdk::DrawableClass::set_colormap_proxy(GdkDrawable *drawable, GdkColormap *cmap)
{
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		tmp_drawable->do_set_colormap(cmap);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->set_colormap)
			g_class->set_colormap(drawable, cmap);
	}
}

GdkColormap*
Gdk::DrawableClass::get_colormap_proxy(GdkDrawable *drawable)
{
	GdkColormap *cmap = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		cmap = tmp_drawable->do_get_colormap();
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_colormap)
			cmap = g_class->get_colormap(drawable);
	}
	return cmap;
}

GdkVisual*
Gdk::DrawableClass::get_visual_proxy(GdkDrawable *drawable)
{
	GdkVisual *visual = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		visual = tmp_drawable->do_get_visual();
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_visual)
			visual = g_class->get_visual(drawable);
	}
	return visual;
}

GdkImage*
Gdk::DrawableClass::get_image_proxy(GdkDrawable *drawable, gint x, gint  y, gint width, gint height)
{
	GdkImage *image = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		image = tmp_drawable->do_get_image(x, y, width, height);
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_image)
			image = g_class->get_image(drawable, x, y, width, height);
	}
	return image;
}

GdkRegion*
Gdk::DrawableClass::get_clip_region_proxy(GdkDrawable *drawable)
{
	GdkRegion *region = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		region = tmp_drawable->do_get_clip_region();
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_clip_region)
			region = g_class->get_clip_region(drawable);
	}
	return region;
}

GdkRegion*
Gdk::DrawableClass::get_visible_region_proxy(GdkDrawable *drawable)
{
	GdkRegion *region = 0;
	Drawable *tmp_drawable = G::Object::pointer<Drawable>(drawable);
	if (tmp_drawable)
		region = tmp_drawable->do_get_visible_region();
	else
	{
		GdkDrawableClass *g_class = G::TypeInstance::class_peek_parent<GdkDrawableClass>(GDK_DRAWABLE_GET_CLASS(drawable));
		if (g_class->get_visible_region)
			region = g_class->get_visible_region(drawable);
	}
	return region;
}

/*  Overridable GdkDrawable methods
 */

GdkGC*
Gdk::Drawable::do_create_gc(GdkGCValues *values, GdkGCValuesMask mask)
{
	GdkGC *gc = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->create_gc)
		gc = g_class->create_gc(gdk_drawable(), values, mask);
	return gc;
}

void
Gdk::Drawable::do_draw_rectangle(GdkGC *gc, int filled, int x, int y, int width, int height)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_rectangle)
		g_class->draw_rectangle(gdk_drawable(), gc, filled, x, y, width, height);
}

void
Gdk::Drawable::do_draw_arc(GdkGC *gc, int filled, int x, int y, int width, int height, int angle1, int angle2)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_arc)
		g_class->draw_arc(gdk_drawable(), gc, filled, x, y, width, height, angle1, angle2);
}

void
Gdk::Drawable::do_draw_polygon(GdkGC *gc, int filled, GdkPoint *points, int npoints)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_polygon)
		g_class->draw_polygon(gdk_drawable(), gc, filled, points, npoints);
}

void
Gdk::Drawable::do_draw_drawable(GdkGC *gc, GdkDrawable *src, int xsrc, int ysrc, int xdest, int ydest, int width, int height)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_drawable)
		g_class->draw_drawable(gdk_drawable(), gc, src, xsrc, ysrc, xdest, ydest, width, height);
}

void
Gdk::Drawable::do_draw_points(GdkGC *gc, GdkPoint *points, int npoints)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_points)
		g_class->draw_points(gdk_drawable(), gc, points, npoints);
}

void
Gdk::Drawable::do_draw_segments(GdkGC *gc, GdkSegment *segments, int nsegments)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_segments)
		g_class->draw_segments(gdk_drawable(), gc, segments, nsegments);
}

void
Gdk::Drawable::do_draw_lines(GdkGC *gc, GdkPoint *points, int npoints)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_lines)
		g_class->draw_lines(gdk_drawable(), gc, points, npoints);
}

void
Gdk::Drawable::do_draw_glyphs(GdkGC *gc, PangoFont *font, int  x, int y, PangoGlyphString *glyphs)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_glyphs)
		g_class->draw_glyphs(gdk_drawable(), gc, font, x, y, glyphs);
}

void
Gdk::Drawable::do_draw_image(GdkGC *gc, GdkImage *image, int xsrc, int ysrc, int xdest, int ydest, int width, int height)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->draw_image)
		g_class->draw_image(gdk_drawable(), gc, image, xsrc, ysrc, xdest, ydest, width, height);
}

int
Gdk::Drawable::do_get_depth()
{
	int depth = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_depth)
		depth = g_class->get_depth(gdk_drawable());
	return depth;
}

void
Gdk::Drawable::do_get_size(int *width, int *height)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_size)
		g_class->get_size(gdk_drawable(), width, height);
}

void
Gdk::Drawable::do_set_colormap(GdkColormap *cmap)
{
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->set_colormap)
		g_class->set_colormap(gdk_drawable(), cmap);
}

GdkColormap*
Gdk::Drawable::do_get_colormap()
{
	GdkColormap *cmap = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_colormap)
		cmap = g_class->get_colormap(gdk_drawable());
	return cmap;
}

GdkVisual*
Gdk::Drawable::do_get_visual()
{
	GdkVisual *visual = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_visual)
		visual = g_class->get_visual(gdk_drawable());
	return visual;
}

GdkImage*
Gdk::Drawable::do_get_image(int x, int  y, int width, int height)
{
	GdkImage *image = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_image)
		image = g_class->get_image(gdk_drawable(), x, y, width, height);
	return image;
}

GdkRegion*
Gdk::Drawable::do_get_clip_region()
{
	GdkRegion *region = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_clip_region)
		region = g_class->get_clip_region(gdk_drawable());
	return region;
}

GdkRegion*
Gdk::Drawable::do_get_visible_region()
{
	GdkRegion *region = 0;
	GdkDrawableClass *g_class = class_peek_parent<GdkDrawableClass>(gdk_drawable_class());
	if (g_class->get_visible_region)
		region = g_class->get_visible_region(gdk_drawable());
	return region;
}

