/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/settings.h
//! @brief A GtkSettings C++ wrapper interface.
//!
//! Provides Settings, an object that holds the current settings for a screen.

#ifndef INTI_GTK_SETTINGS_H
#define INTI_GTK_SETTINGS_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef __GTK_SETTINGS_H__
#include <gtk/gtksettings.h>
#endif

namespace Inti {

namespace Gdk {
class Screen;
}

namespace Gtk {

//! @class Settings settings.h inti/gtk/settings.h
//! @brief A GtkSettings C++ wrapper class.
//!
//! The Settings object holds the current settings for a GDK screen.

class Settings : public G::Object
{
	friend class G::Object;

	Settings(const Settings&);
	Settings& operator=(const Settings&);
	
protected:
//! @name Constructors
//! @{

	explicit Settings(GtkSettings *settings, bool reference = true);
	//!< Construct a new Settings from an existing GtkSettings.
	//!< @param settings A pointer to a GtkSettings.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>settings</EM> can be a newly created GtkSettings or an existing
	//!< GtkSettings (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<int> GtkDoubleClickTimePropertyType;
	typedef G::PropertyProxy<G::Object, GtkDoubleClickTimePropertyType> GtkDoubleClickTimePropertyProxy;
	static const GtkDoubleClickTimePropertyType gtk_double_click_time_property;

	typedef G::Property<bool> GtkCursorBlinkPropertyType;
	typedef G::PropertyProxy<G::Object, GtkCursorBlinkPropertyType> GtkCursorBlinkPropertyProxy;
	static const GtkCursorBlinkPropertyType gtk_cursor_blink_property;

	typedef G::Property<int> GtkCursorBlinkTimePropertyType;
	typedef G::PropertyProxy<G::Object, GtkCursorBlinkTimePropertyType> GtkCursorBlinkTimePropertyProxy;
	static const GtkCursorBlinkTimePropertyType gtk_cursor_blink_time_property;

	typedef G::Property<bool> GtkSplitCursorPropertyType;
	typedef G::PropertyProxy<G::Object, GtkSplitCursorPropertyType> GtkSplitCursorPropertyProxy;
	static const GtkSplitCursorPropertyType gtk_split_cursor_property;

	typedef G::Property<String> GtkThemeNamePropertyType;
	typedef G::PropertyProxy<G::Object, GtkThemeNamePropertyType> GtkThemeNamePropertyProxy;
	static const GtkThemeNamePropertyType gtk_theme_name_property;

	typedef G::Property<String> GtkKeyThemeNamePropertyType;
	typedef G::PropertyProxy<G::Object, GtkKeyThemeNamePropertyType> GtkKeyThemeNamePropertyProxy;
	static const GtkKeyThemeNamePropertyType gtk_key_theme_name_property;

	typedef G::Property<String> GtkMenuBarAccelPropertyType;
	typedef G::PropertyProxy<G::Object, GtkMenuBarAccelPropertyType> GtkMenuBarAccelPropertyProxy;
	static const GtkMenuBarAccelPropertyType gtk_menu_bar_accel_property;

	typedef G::Property<int> GtkDndDragThresholdPropertyType;
	typedef G::PropertyProxy<G::Object, GtkDndDragThresholdPropertyType> GtkDndDragThresholdPropertyProxy;
	static const GtkDndDragThresholdPropertyType gtk_dnd_drag_threshold_property;

	typedef G::Property<String> GtkFontNamePropertyType;
	typedef G::PropertyProxy<G::Object, GtkFontNamePropertyType> GtkFontNamePropertyProxy;
	static const GtkFontNamePropertyType gtk_font_name_property;

	typedef G::Property<String> GtkIconSizesPropertyType;
	typedef G::PropertyProxy<G::Object, GtkIconSizesPropertyType> GtkIconSizesPropertyProxy;
	static const GtkIconSizesPropertyType gtk_icon_sizes_property;

public:
//! @name Constructors
//! @{

	virtual ~Settings();
	//!< Destructor.
	
//! @}

	static Settings* get(const Gdk::Screen *screen = 0);
	//!< Gets the Settings object for the specified screen, creating it if necessary.
	//!< @param screen A Gdk::Screen, or null for the default screen.
	//!< @return A Settings object.
	//!<
	//!< <BR><B>Note:</B> The default GDK screen is the only screen in the absence of MultiHead support.

//! @name Accessors
//! @{

	GtkSettings* gtk_settings() const { return (GtkSettings*)instance; }
	//!< Get a pointer to the GtkSettings structure.

	GtkSettingsClass* gtk_settings_class() const;
	//!< Get a pointer to the GtkSettingsClass structure.

	operator GtkSettings* () const;
	//!< Conversion operator; safely converts a Settings to a GtkSettings pointer.

//! @}
//! @name Property Proxies
//! @{

	const GtkDoubleClickTimePropertyProxy prop_gtk_double_click_time()
	{
		return GtkDoubleClickTimePropertyProxy(this, &gtk_double_click_time_property);
	}
	//!< Maximum time allowed between two clicks for them to be considered a double click
	//!< (in milliseconds) (int : Read / Write).

	const GtkCursorBlinkPropertyProxy prop_gtk_cursor_blink()
	{
		return GtkCursorBlinkPropertyProxy(this, &gtk_cursor_blink_property);
	}
	//!< Whether the cursor should blink (bool : Read / Write).

	const GtkCursorBlinkTimePropertyProxy prop_gtk_cursor_blink_time()
	{
		return GtkCursorBlinkTimePropertyProxy(this, &gtk_cursor_blink_time_property);
	}
	//!< Length of the cursor blink cycle, in milleseconds (int : Read / Write).

	const GtkSplitCursorPropertyProxy prop_gtk_split_cursor()
	{
		return GtkSplitCursorPropertyProxy(this, &gtk_split_cursor_property);
	}
	//!< Whether two cursors should be displayed for mixed left-to-right and right-to-left text
	//!< (bool : Read / Write).

	const GtkThemeNamePropertyProxy prop_gtk_theme_name()
	{
		return GtkThemeNamePropertyProxy(this, &gtk_theme_name_property);
	}
	//!< Name of theme RC file to load (String : Read / Write).

	const GtkKeyThemeNamePropertyProxy prop_gtk_key_theme_name()
	{
		return GtkKeyThemeNamePropertyProxy(this, &gtk_key_theme_name_property);
	}
	//!< Name of key theme RC file to load (String : Read / Write).

	const GtkMenuBarAccelPropertyProxy prop_gtk_menu_bar_accel()
	{
		return GtkMenuBarAccelPropertyProxy(this, &gtk_menu_bar_accel_property);
	}
	//!< Keybinding to activate the menu bar (String : Read / Write).

	const GtkDndDragThresholdPropertyProxy prop_gtk_dnd_drag_threshold()
	{
		return GtkDndDragThresholdPropertyProxy(this, &gtk_dnd_drag_threshold_property);
	}
	//!< Number of pixels the cursor can move before dragging (int : Read / Write).

	const GtkFontNamePropertyProxy prop_gtk_font_name()
	{
		return GtkFontNamePropertyProxy(this, &gtk_font_name_property);
	}
	//!< Name of default font to use (String : Read / Write).

	const GtkIconSizesPropertyProxy prop_gtk_icon_sizes()
	{
		return GtkIconSizesPropertyProxy(this, &gtk_icon_sizes_property);
	}
	//!< List of icon sizes (gtk-menu=16,16;gtk-button=20,20...) (String : Read / Write).

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_SETTINGS_H

