/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.functor;

import java.util.Set;

/**
 * The arguments for executing an {@link IFunctor} implementation.
 * 
 */
public interface IArgs {
	/**
	 * Generic option for the transport of IArgs.
	 */
	public static final String ARG_ARGS = "args"; //$NON-NLS-1$

	public void add(Object object);

	public void clear();

	/**
	 * The argument at position <code>index</code>.
	 * 
	 * @param index
	 *            The index of the argument to return.
	 * 
	 * @return The argument at position <code>index</code>.
	 */
	public Object get(int index);

	/**
	 * The argument at position <code>index</code>
	 * 
	 * @param index
	 *            The index of the argument to return.
	 * @param defaultValue
	 *            The default value to be returned if argument is not available.
	 * 
	 * @return The argument at position <code>index</code>
	 */
	public Object get(int index, Object defaultValue);

	/**
	 * The argument named <code>name</code>.
	 * 
	 * @param name
	 *            The name of the argument to return.
	 * 
	 * @return The argument named <code>name</code> .
	 */
	public Object get(String name);

	/**
	 * The argument named <code>name</code> or the defaultValue if not
	 * available.
	 * 
	 * @param name
	 *            The name of the argument to return.
	 * @param defaultValue
	 *            The default value to be returned if argument is not available.
	 * 
	 * @return The argument named <code>name</code>
	 */
	public Object get(String name, Object defaultValue);

	/**
	 * <code>true</code> if an argument at <code>index</code> is defined.
	 * 
	 * @param index
	 * @return <code>true</code> if an argument at <code>index</code> is
	 *         defined.
	 */
	public boolean isDefined(int index);

	/**
	 * <code>true</code> if an argument named <code>name</code> is defined.
	 * 
	 * @param name
	 * @return <code>true</code> if an argument named <code>name</code> is
	 *         defined.
	 */
	public boolean isDefined(String name);

	/**
	 * <code>true</code> if this argument list is indexed. This means its
	 * elements are available via integer indexes.
	 * <p>
	 * This does NOT mean the arguments are not available via names - there are
	 * implementations that can support both.
	 * 
	 * @return <code>true</code> if this argument list is indexed.
	 */
	public boolean isIndexed();

	/**
	 * <code>true</code> if this argument list is named. This means its
	 * elements are available via names.
	 * <p>
	 * This does NOT mean the arguments are not available via indexes - there
	 * are implementations that can support both.
	 * 
	 * @return <code>true</code> if this argument list is named.
	 */
	public boolean isNamed();

	/**
	 * The set of all argument names in the argument list if this argument list
	 * is not indexed or null.
	 * 
	 * @return The set of all argument names in the argument list if this
	 *         argument list is not indexed or null.
	 */
	public Set names();

	public void put(int index, Object value);

	public void put(String name, Object value);

	/**
	 * The total number of arguments.
	 * 
	 * @return The total number of arguments.
	 */
	public int size();
}
