/************************************************************************
 * $Id: prayer.c,v 1.24 2005/03/14 22:09:07 thamer Exp $
 *
 * ------------
 * Description:
 * ------------
 *  Copyright (c) 2003-2005, Arabeyes, Thamer Mahmoud
 *
 *  A full featured Muslim Prayer Times calculator
 *
 *
 * -----------------
 * Revision Details:    (Updated by Revision Control System)
 * -----------------
 *  $Date: 2005/03/14 22:09:07 $
 *  $Author: thamer $
 *  $Revision: 1.24 $
 *  $Source: /home/arabeyes/cvs/projects/itl/libs/prayertime/src/prayer.c,v $
 *
 * (www.arabeyes.org - under LGPL license - see COPYING file)
 ************************************************************************/


#include "prayer.h"
#include "astro.h"

/* Defaults */
#define KAABA_LAT 21.423333
#define KAABA_LONG 39.823333
#define DEF_NEAREST_LATITUDE 48.5
#define DEF_IMSAAK_ANGLE 1.5
#define DEF_IMSAAK_INTERVAL 10
#define DEFAULT_ROUND_SEC 30 
#define AGGRESSIVE_ROUND_SEC 1



enum exmethods  { NONE_EX,
                  LAT_ALL,
                  LAT_ALWAYS,
                  LAT_INVALID,
                  GOOD_ALL,
                  GOOD_INVALID,
                  SEVEN_NIGHT_ALWAYS,
                  SEVEN_NIGHT_INVALID,
                  SEVEN_DAY_ALWAYS,
                  SEVEN_DAY_INVALID,
                  HALF_ALWAYS,
                  HALF_INVALID,
                  MIN_ALWAYS,
                  MIN_INVALID,
                  GOOD_DIF };

enum methods    { NONE,
                  EGYPT_SURVEY,
                  KARACHI_SHAF,
                  KARACHI_HANAF,
                  NORTH_AMERICA,
                  MUSLIM_LEAGUE,
                  UMM_ALQURRA,
                  FIXED_ISHAA };

enum salatType  { FAJR,
                  SHUROOQ,
                  THUHR,
                  ASSR,
                  MAGHRIB,
                  ISHAA,
                  IMSAAK,
                  NEXTFAJR };


Astro astroCache; /* This is Used for storing some formulae results between
                   * multiple getPrayerTimes calls*/

static double getThuhr (double lon, const Astro* astro);
static double getShoMag ( const Location* loc, const Astro* astro, int type);
static double getFajIsh (double Lat, double dec, double Ang);
static double getAssr (double Lat, double dec, int mathhab);
static void base6hm(double bs, const Location* loc, const Method* conf,
                    Prayer* pt, int type);
static void getDayInfo( const Date* date, double gmt, int *lastDay, double *julianDay);
static void getPrayerTimesByDay ( const Location* loc, const Method* conf, int lastDay,
                                  double julianDay, Prayer* pt, int type);


void getPrayerTimes ( const Location* loc, const Method* conf, const Date* date, 
                      Prayer* pt)
{  
    int lastDay;
    double julianDay;

    getDayInfo ( date, loc->gmtDiff, &lastDay, &julianDay);
    getPrayerTimesByDay( loc, conf, lastDay, julianDay, pt, 0);
}

 
static void getPrayerTimesByDay ( const Location* loc, const Method* conf, 
                                  int lastDay, double julianDay, Prayer* pt, 
                                  int type)
{
    int i, invalid;
    double th, sh, mg, fj, is, ar;
    double lat, lon, dec;
    double tempPrayer[6];
    Astro tAstro;

    lat = loc->degreeLat; 
    lon = loc->degreeLong;
    invalid = 0;

    /* Start by filling the tAstro structure with the appropriate astronomical
     * values for this day. We also pass the cache structure to update and check
     * if the actual values are already available. */
    getAstroValuesByDay(julianDay, loc, &astroCache, &tAstro);
    dec = DEG_TO_RAD(tAstro.dec[1]);

    /* Get Prayer Times formulae results for this day of year and this
     * location. The results are NOT the actual prayer times */
    fj   = getFajIsh (lat, dec, conf->fajrAng);
    sh   = getShoMag (loc, &tAstro, SHUROOQ);
    th   = getThuhr (lon, &tAstro);
    ar   = getAssr (lat, dec, conf->mathhab);
    mg   = getShoMag (loc, &tAstro, MAGHRIB);
    is   = getFajIsh (lat, dec, conf->ishaaAng);
    
    /* Calculate all prayer times as Base-10 numbers in Normal circumstances */ 
    /* Fajr */   
    if (fj == 99) {
        tempPrayer[0] = 99;
        invalid = 1;
    } 
    else tempPrayer[0] = th - fj;
     
    if (sh == 99)
        invalid = 1;

    tempPrayer[1] = sh;
    tempPrayer[2] = th;
    tempPrayer[3] = th + ar;
    tempPrayer[4] = mg;

    if (mg == 99)
        invalid = 1;

    /* Ishaa */
    if (is == 99) {
        tempPrayer[5] = 99;
        invalid = 1;
    } 
    else tempPrayer[5] = th + is;
    
    
    /* Calculate all prayer times as Base-10 numbers in Extreme Latitudes (if
     * needed) */
  
    /* Reset status of extreme switches */
    for (i=0; i<6; i++)
        pt[i].isExtreme = 0; 
     
     
    if ((conf->extreme != NONE_EX) && !((conf->extreme == GOOD_INVALID || 
                                         conf->extreme == LAT_INVALID ||
                                         conf->extreme == SEVEN_NIGHT_INVALID ||
                                         conf->extreme == SEVEN_DAY_INVALID ||
                                         conf->extreme == HALF_INVALID) &&
                                        (invalid == 0)))
    {
        double exdecPrev, exdecNext;
        double exTh=99, exFj=99, exIs=99, exAr=99, exIm=99, exSh=99, exMg=99;
        double portion = 0;
        double nGoodDay = 0;
        int exinterval = 0;
        Location exLoc = *loc;
        Astro exAstroPrev;
        Astro exAstroNext;

        switch(conf->extreme)
        {
        /* Nearest Latitude (Method.nearestLat) */
        case LAT_ALL:
        case LAT_ALWAYS:
        case LAT_INVALID:

            /* xxxthamer: we cannot compute this when interval is set because
             * angle==0 . Only the if-invalid methods would work */
            exLoc.degreeLat = conf->nearestLat;
            exFj = getFajIsh(conf->nearestLat, dec, conf->fajrAng);
            exIm = getFajIsh(conf->nearestLat, dec, conf->imsaakAng);
            exIs = getFajIsh(conf->nearestLat, dec, conf->ishaaAng);
            exAr = getAssr(conf->nearestLat, dec, conf->mathhab);
            exSh = getShoMag (&exLoc, &tAstro, SHUROOQ);
            exMg = getShoMag (&exLoc, &tAstro, MAGHRIB);


            switch(conf->extreme)
            {
            case LAT_ALL:
                tempPrayer[0] = th - exFj;
                tempPrayer[1] = exSh;
                tempPrayer[3] = th + exAr;
                tempPrayer[4] = exMg;
                tempPrayer[5] = th + exIs;
                pt[0].isExtreme = 1;
                pt[1].isExtreme = 1;
                pt[2].isExtreme = 1;
                pt[3].isExtreme = 1;
                pt[4].isExtreme = 1;
                pt[5].isExtreme = 1;
                break;
        
            case LAT_ALWAYS:
                tempPrayer[0] = th - exFj;
                tempPrayer[5] = th + exIs;
                pt[0].isExtreme = 1;
                pt[5].isExtreme = 1;
                break;
        
            case LAT_INVALID:
                if (tempPrayer[0] == 99) {
                    tempPrayer[0] = th - exFj;
                    pt[0].isExtreme = 1;
                }
                if (tempPrayer[5] == 99) {
                    tempPrayer[5] = th + exIs;
                    pt[5].isExtreme = 1;
                }
                break;
            }
            break;

           
        /* Nearest Good Day */
        case GOOD_ALL:
        case GOOD_INVALID:
        case GOOD_DIF:
            
            exAstroPrev = astroCache;
            exAstroNext = astroCache;

            /* Start by getting last or next nearest Good Day */
            for(i=0; i <= lastDay; i++)
            {

                /* last closest day */
                nGoodDay = julianDay - i;
                getAstroValuesByDay(nGoodDay, loc, &exAstroPrev, &tAstro);
                exdecPrev = DEG_TO_RAD(tAstro.dec[1]);
                exFj = getFajIsh(lat, exdecPrev, conf->fajrAng);

                
                if (exFj != 99)
                {
                    exIs = getFajIsh(lat, exdecPrev, conf->ishaaAng);
                    if (exIs != 99)
                    {
                        exTh = getThuhr (lon, &tAstro);
                        exSh = getShoMag (loc, &tAstro, SHUROOQ);
                        exMg = getShoMag (loc, &tAstro, MAGHRIB);
                        exAr = getAssr (lat, exdecPrev, conf->mathhab);
                        break;
                    }
                }

                /* Next closest day */
                nGoodDay = julianDay + i;
                getAstroValuesByDay(nGoodDay, loc, &exAstroNext, &tAstro);
                exdecNext = DEG_TO_RAD(tAstro.dec[1]);
                exFj = getFajIsh(lat, exdecNext, conf->fajrAng);
                if (exFj != 99)
                {
                    exIs = getFajIsh(lat, exdecNext, conf->ishaaAng);
                    if (exIs != 99)
                    {
                        exTh = getThuhr (lon, &tAstro);
                        exSh = getShoMag (loc, &tAstro, SHUROOQ);
                        exMg = getShoMag (loc, &tAstro, MAGHRIB);
                        exAr = getAssr (lat, exdecNext, conf->mathhab);
                        break;
                    }
                }
            }

            switch(conf->extreme)
            {
            case GOOD_ALL:
                tempPrayer[0] = exTh - exFj;
                tempPrayer[1] = exSh;
                tempPrayer[2] = exTh;
                tempPrayer[3] = exTh + exAr;
                tempPrayer[4] = exMg;
                tempPrayer[5] = exTh + exIs;
                for (i=0; i<6; i++)
                    pt[i].isExtreme = 1;
                break;
            case GOOD_INVALID:
                if (tempPrayer[0] == 99) {
                    tempPrayer[0] = exTh - exFj;
                    pt[0].isExtreme = 1;
                }
                if (tempPrayer[5] == 99) {
                    tempPrayer[5] = exTh + exIs;
                    pt[5].isExtreme = 1;
                }
                break;

            case GOOD_DIF:
                /* Nearest Good Day: Different good days for Fajr and Ishaa (Not
                 * implemented) */
                break;
            }
            break;
      
        case SEVEN_NIGHT_ALWAYS:
        case SEVEN_NIGHT_INVALID:
        case SEVEN_DAY_ALWAYS:
        case SEVEN_DAY_INVALID:
        case HALF_ALWAYS:
        case HALF_INVALID:

            /* xxxthamer: For clarity, we may need to move the HALF_* methods
             * into their own separate case statement. */    
            switch(conf->extreme)
            {
            case SEVEN_NIGHT_ALWAYS:
            case SEVEN_NIGHT_INVALID:
                portion = (24 - (tempPrayer[4] - tempPrayer[1])) * (1/7.0);
                break;
            case SEVEN_DAY_ALWAYS:
            case SEVEN_DAY_INVALID:
                portion = (tempPrayer[4] - tempPrayer[1]) * (1/7.0);
                break;
            case HALF_ALWAYS:
            case HALF_INVALID:
                portion = (24 - tempPrayer[4] - tempPrayer[1]) * (1/2.0);
                break;
            }


            if (conf->extreme == SEVEN_NIGHT_INVALID ||
                conf->extreme == SEVEN_DAY_INVALID ||
                conf->extreme == HALF_INVALID)
            {
                if (tempPrayer[0] == 99) {
                    if  (conf->extreme == HALF_INVALID)
                        tempPrayer[0] =  portion - (conf->fajrInv / 60.0);
                    else tempPrayer[0] = tempPrayer[1] - portion;
                    pt[0].isExtreme = 1;
                }
                if (tempPrayer[5] == 99) {
                    if  (conf->extreme == HALF_INVALID)
                        tempPrayer[5] = portion + (conf->ishaaInv / 60.0) ;
                    else tempPrayer[5] = tempPrayer[4] + portion;
                    pt[5].isExtreme = 1;
                }
            } else { /* for the always methods */
                
                if  (conf->extreme == HALF_ALWAYS) {
                    tempPrayer[0] = portion - (conf->fajrInv / 60.0);
                    tempPrayer[5] = portion + (conf->ishaaInv / 60.0) ;
                }

                else {
                    tempPrayer[0] = tempPrayer[1] - portion;
                    tempPrayer[5] = tempPrayer[4] + portion;
                }
                pt[0].isExtreme = 1;
                pt[5].isExtreme = 1;
            }
            break;

        case MIN_ALWAYS:
            /* Do nothing here because this is implemented through fajrInv and
             * ishaaInv structure members */
            tempPrayer[0] = tempPrayer[1];
            tempPrayer[5] = tempPrayer[4];
            pt[0].isExtreme = 1;
            pt[5].isExtreme = 1;
            break;
      
        case MIN_INVALID:
            if (tempPrayer[0] == 99) {
                exinterval = conf->fajrInv / 60.0;
                tempPrayer[0] = tempPrayer[1] - exinterval;
                pt[0].isExtreme = 1;
            }
            if (tempPrayer[5] == 99) {
                exinterval = conf->ishaaInv / 60.0;
                tempPrayer[5] = tempPrayer[4] + exinterval;
                pt[5].isExtreme = 1;
            }
            break;
        } /* end switch */
    } /* end extreme */

    
    /* Apply intervals if set */
    if (conf->extreme != MIN_INVALID && 
        conf->extreme != HALF_INVALID &&
        conf->extreme != HALF_ALWAYS) {
        if (conf->fajrInv != 0)
            tempPrayer[0] = tempPrayer[1] - (conf->fajrInv / 60.0);    
        if (conf->ishaaInv != 0)
            tempPrayer[5] = tempPrayer[4] + (conf->ishaaInv / 60.0); 
    }

 
    /* Final Step: Fill the Prayer array by doing decimal degree to
     * Prayer structure conversion*/
    if (type == IMSAAK || type == NEXTFAJR)
        base6hm(tempPrayer[0], loc, conf, &pt[0], type);
    else {
        for (i=0; i<6; i++)
            base6hm(tempPrayer[i], loc, conf, &pt[i], i);
    }

}



static void base6hm(double bs, const Location* loc, const Method* conf, 
                    Prayer* pt, int type)
{
    double min, sec;

    if (bs == 99)
    {
        pt->hour    = 99;
        pt->minute = 99;
        pt->second = 0;
        return;
    }
        
    /* Add offsets */
    if (conf->offset == 1) {
        if (type == IMSAAK || type == NEXTFAJR)
            bs += (conf->offList[0] / 60.0);
        else  bs += (conf->offList[type] / 60.0);
    }


    /* Fix after minus offsets before midnight */
    if (bs < 0) {
        while (bs < 0)
            bs = 24 + bs;
    }

    min = (bs - floor(bs)) * 60;
    sec = (min - floor(min)) * 60;


    /* Add rounding minutes */
    if (conf->round == 1)
    {
        if (sec >= DEFAULT_ROUND_SEC)
            bs += 1/60.0;
        /* compute again */
        min = (bs - floor(bs)) * 60;
        sec = 0;

    } else if (conf->round == 2 || conf->round == 3)
    {
        switch(type)
        {
        case FAJR:
        case THUHR:
        case ASSR:
        case MAGHRIB:
        case ISHAA:
        case NEXTFAJR:
            
            if (conf->round == 2) {
                if (sec >= DEFAULT_ROUND_SEC) {
                    bs += 1/60.0;
                    min = (bs - floor(bs)) * 60;
                }
            } else if (conf->round == 3)
            {
                if (sec >= AGGRESSIVE_ROUND_SEC) {
                    bs += 1/60.0;
                    min = (bs - floor(bs)) * 60;
                }
            }
            sec = 0;
            break;

        case SHUROOQ:
        case IMSAAK:
            sec = 0;
            break;
        }
    }

    /* Add daylight saving time and fix after midnight times */
    bs += loc->dst;
    if (bs >= 24)
        bs = fmod(bs, 24);
          
    pt->hour   = (int)bs;
    pt->minute = (int)min;
    pt->second = (int)sec;

}

void getImsaak (const Location* loc, const Method* conf, const Date* date, 
                Prayer* pt)
{

    Method tmpConf;
    int lastDay;
    double julianDay;
    Prayer temp[6];

    tmpConf = *conf;

    if (conf->fajrInv != 0) { 
        if (conf->imsaakInv == 0)
            tmpConf.fajrInv += DEF_IMSAAK_INTERVAL;
        else tmpConf.fajrInv += conf->imsaakInv;
        
    } else if (conf->imsaakInv != 0) {
        /* use an inv even if al-Fajr is computed (Indonesia?) */       
        tmpConf.offList[0] += (conf->imsaakInv * -1);
        tmpConf.offset = 1;
    } else { 
        tmpConf.fajrAng += conf->imsaakAng;
    }

    getDayInfo ( date, loc->gmtDiff, &lastDay, &julianDay);
    getPrayerTimesByDay( loc, &tmpConf, lastDay, julianDay, temp, IMSAAK);

    /* xxxthamer: We probably need to check whether it's possible to compute
     * Imsaak normally for some extreme methods first */
    /* In case of an extreme Fajr time calculation use intervals for Imsaak and
     * compute again */
    if (temp[0].isExtreme != 0)
    {
        tmpConf = *conf;
        if ( conf->imsaakInv == 0)
        {
            tmpConf.offList[0] -= DEF_IMSAAK_INTERVAL;
            tmpConf.offset = 1;
        } else
        {
            tmpConf.offList[0] -= conf->imsaakInv;
            tmpConf.offset = 1;
        }
        getPrayerTimesByDay( loc, &tmpConf, lastDay, julianDay, temp, IMSAAK);
    }
    
    *pt = temp[0];

}

void getNextDayImsaak (const Location* loc, const Method* conf, const Date* date, 
                       Prayer* pt)
{
    /* Copy the date structure and increment for next day.*/
    Prayer temppt;
    Date tempd = *date; 
    tempd.day++;
     
    getImsaak (loc, conf, &tempd, &temppt);

    *pt = temppt; 

}

void getNextDayFajr (const Location* loc, const Method* conf, const Date* date, 
                     Prayer* pt)
{

    Prayer temp[6];
    int lastDay;
    double julianDay;

    getDayInfo ( date, loc->gmtDiff, &lastDay, &julianDay);
    getPrayerTimesByDay( loc, conf, lastDay, julianDay+1, temp, NEXTFAJR);
 
    *pt = temp[0]; 
}


static double getFajIsh(double Lat, double dec, double Ang)
{

    double part1 = cos(DEG_TO_RAD(Lat)) * cos(dec);
    double part2 = -sin(DEG_TO_RAD(Ang)) - sin(DEG_TO_RAD(Lat))
        * sin(dec);
    
    double part3 = part2 / part1;
    if ( part3 <= INVALID_TRIGGER)
        return 99;

    return DEG_TO_10_BASE * RAD_TO_DEG (acos(part3) );

}


static double getShoMag (const Location* loc, const Astro* astro, int type)
{
    double lhour, M, sidG, ra0=astro->ra[0], ra2=astro->ra[2];
    double A, B, H, sunAlt, R, tH;

    double part1 = sin (DEG_TO_RAD (loc->degreeLat)) * sin (DEG_TO_RAD(astro->dec[1]));
    double part2a =  CENTER_OF_SUN_ANGLE;
    double part2 = sin (DEG_TO_RAD (part2a)) - part1;
    double part3 = cos (DEG_TO_RAD (loc->degreeLat)) * cos (DEG_TO_RAD(astro->dec[1]));

    double part4 = part2 / part3;
    
    if (part4 <= -1 || part4 >= 1)
        return 99;

    lhour =  limitAngle180 (( RAD_TO_DEG (acos (part4))));
    M = ((astro->ra[1] - loc->degreeLong - astro->sid[1]) / 360.0);

    if (type ==  SHUROOQ)
        M = M - (lhour/360.0);
    if (type == MAGHRIB)
        M = M + (lhour/360.0);
        
    M = limitAngle111(M);

    sidG = limitAngle(astro->sid[1] + 360.985647 * M);

    ra0 = astro->ra[0];
    ra2 = astro->ra[2];

    if (astro->ra[1] > 350 && astro->ra[2] < 10)
        ra2 += 360;
    if (astro->ra[0] > 350 && astro->ra[1] < 10)
        ra0 = 0;

    A = astro->ra[1] + (M * (( astro->ra[1] - ra0) +
                             (ra2 - astro->ra[1] ) +
                             (( ra2 - astro->ra[1] ) -
                              ( astro->ra[1]  -  ra0)) * M) / 2.0 );

    B = astro->dec[1] + (M * ((astro->dec[1] - astro->dec[0]) + 
                              (astro->dec[2] - astro->dec[1]) + 
                              ((astro->dec[2] - astro->dec[1]) -  
                               (astro->dec[1] - astro->dec[0])) * M) / 2.0 );

    H =  limitAngle180between(sidG +  loc->degreeLong -  A);

    tH =  H - RAD_TO_DEG(astro->dra[1]);

    sunAlt = RAD_TO_DEG(asin (  sin(DEG_TO_RAD( loc->degreeLat)) * sin(DEG_TO_RAD (B)) 
                                + cos(DEG_TO_RAD( loc->degreeLat)) * cos(DEG_TO_RAD (B)) 
                                * cos(DEG_TO_RAD(tH)) ));

    sunAlt += getRefraction(loc, sunAlt);
    
    R = (M + (( sunAlt - CENTER_OF_SUN_ANGLE+ (ALTITUDE_REFRACTION * 
                                               pow (loc->seaLevel,0.5)))
              /(360.0 * cos(DEG_TO_RAD (B)) *  cos(DEG_TO_RAD( loc->degreeLat)) * 
                                               sin(DEG_TO_RAD (tH)))));

    return  (R * 24.0);

}

static double getThuhr(double lon, const Astro* astro)
{

    double M, sidG;
    double ra0=astro->ra[0], ra2=astro->ra[2];
    double A, H;

    M = ((astro->ra[1] - lon - astro->sid[1]) / 360.0);
    M = limitAngle111(M);
    sidG =  astro->sid[1] + 360.985647 * M;

    if (astro->ra[1] > 350 && astro->ra[2] < 10)
        ra2 += 360;
    if (astro->ra[0] > 350 && astro->ra[1] < 10)
        ra0 = 0;

    A = astro->ra[1] + (M * ((astro->ra[1] - ra0) 
                             + ( ra2 - astro->ra[1]) + 
                             (( ra2 - astro->ra[1]) -  
                              (astro->ra[1] - ra0)) * M) / 2.0 );

    H =  limitAngle180between(sidG + lon - A);

    return  24.0 * (M - H/360.0);
}

static double getAssr(double Lat, double dec, int mathhab)
{
    double part1, part2, part3, part4;
    
    part1 = mathhab + tan(DEG_TO_RAD(Lat) - dec);
    if (part1 < 1 || Lat < 0)
        part1 = mathhab - tan(DEG_TO_RAD(Lat) - dec);

    part2 = (PI/2.0) - atan(part1);
    part3 = sin(part2) - sin(DEG_TO_RAD(Lat)) * sin(dec);
    part4 = (part3 / (cos(DEG_TO_RAD(Lat)) * cos(dec)));

/*  if (part4 > 1) */
/*      return 99; */

    return DEG_TO_10_BASE * RAD_TO_DEG (acos(part4));
}

 
int getDayofYear(int year, int month, int day)
{
    int i;
    int isLeap = ((year & 3) == 0) && ((year % 100) != 0 
                                       || (year % 400) == 0);
  
    static char dayList[2][13] = {
        {0,31,28,31,30,31,30,31,31,30,31,30,31},
        {0,31,29,31,30,31,30,31,31,30,31,30,31}
    };

    for (i=1; i<month; i++)
        day += dayList[isLeap][i];
  
    
    return day;
}

double dms2Decimal(int deg, int min, double sec, char dir)
{
    double sum = deg + ((min/60.0)+(sec/3600.0));
    if (dir == 'S' || dir == 'W' || dir == 's' || dir == 'w')
        return sum * (-1.0);
    return sum;
}

void decimal2Dms(double decimal, int *deg, int *min, double *sec)
{
    double tempmin, tempsec, n1, n2;

    tempmin = modf(decimal, &n1) * 60.0;
    tempsec = modf(tempmin, &n2) * 60.0;

    *deg = (int)n1;
    *min = (int)n2;
    *sec = tempsec;

}

static void getDayInfo ( const Date* date, double gmt, int *lastDay, 
                         double *julianDay)
{
    int ld;
    double jd;
    ld = getDayofYear(date->year, 12, 31);
    jd = getJulianDay(date, gmt);
    *lastDay = ld;
    *julianDay = jd;
}

void getMethod(int n, Method* conf)
{
    int i;
    conf->fajrInv = 0; 
    conf->ishaaInv = 0; 
    conf->imsaakInv = 0;
    conf->mathhab = 1;
    conf->round = 2;
    conf->nearestLat = DEF_NEAREST_LATITUDE;
    conf->imsaakAng = DEF_IMSAAK_ANGLE;
    conf->extreme = 5;
    conf->offset = 0;
    for (i = 0; i < 6; i++) {
        conf->offList[i] = 0; 
    }
    
    switch(n)
    {
    case NONE:
        conf->fajrAng = 0.0;
        conf->ishaaAng = 0.0;
        break;

    case EGYPT_SURVEY:
        conf->fajrAng = 20;
        conf->ishaaAng = 18;
        break;

    case KARACHI_SHAF:
        conf->fajrAng = 18;
        conf->ishaaAng = 18;
        break;

    case KARACHI_HANAF: 
        conf->fajrAng = 18;
        conf->ishaaAng = 18;
        conf->mathhab = 2;
        break;

    case NORTH_AMERICA:
        conf->fajrAng = 15;
        conf->ishaaAng = 15;
        break;

    case MUSLIM_LEAGUE: 
        conf->fajrAng = 18;
        conf->ishaaAng = 17;
        break;

    case UMM_ALQURRA: 
        conf->fajrAng = 19;
        conf->ishaaAng = 0.0;
        conf->ishaaInv = 90;
        break;

    case FIXED_ISHAA:
        conf->fajrAng = 19.5;
        conf->ishaaAng = 0.0;
        conf->ishaaInv = 90;
        break;
    }
}

/* Obtaining the direction of the shortest distance towards Qibla by using the
 * great circle formula */ 
double getNorthQibla(const Location* loc)
{
    /* xxxthamer: reduce DEG_TO_RAD usage */
    double num, denom;
    num = sin (DEG_TO_RAD (loc->degreeLong) - DEG_TO_RAD (KAABA_LONG));
    denom = (cos (DEG_TO_RAD (loc->degreeLat)) * tan (DEG_TO_RAD (KAABA_LAT))) -
        (sin (DEG_TO_RAD (loc->degreeLat)) * ((cos ((DEG_TO_RAD (loc->degreeLong) -
                                                     DEG_TO_RAD(KAABA_LONG))))));
    return RAD_TO_DEG (atan2 (num, denom));

}
