/**
 * ================================================
 * LibLoader : a free Java resource loading library
 * ================================================
 *
 * Project Info:  http://reporting.pentaho.org/libloader/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * ResourceBundleLoader.java
 * ------------
 */

package org.pentaho.reporting.libraries.resourceloader;

import java.util.Map;

/**
 * A resource bundle is a compound archive that holds several resources.
 *
 * @author Thomas Morgner
 */
public interface ResourceBundleLoader
{
  /**
   * Tries to load the bundle. If the key does not point to a usable resource-bundle, this method returns
   * null. The Exception is only thrown if the bundle is not readable because of IO-Errors.
   * <p/>
   * A resource-bundle loader should only load the bundle for the key itself, never for any of the derived subkeys.
   * It is the ResourceManager's responsibility to search the key's hierachy for the correct key.
   * 
   * @param key the resource key pointing to the bundle.
   * @return the loaded bundle or null, if the resource was not understood.
   * @throws ResourceLoadingException if something goes wrong.
   */
  public ResourceBundleData loadBundle
      (final ResourceManager resourceManager, final ResourceKey key) throws ResourceLoadingException;

  /**
   * Checks, whether this resource loader implementation was responsible for
   * creating this key.
   *
   * @param key the key that should be tested.
   * @return true, if the key is supported.
   */
  public boolean isSupportedKey (ResourceKey key);

  /**
   * Derives a new resource key from the given key. If neither a path nor new
   * factory-keys are given, the parent key is returned.
   *
   * @param parent the parent
   * @param path the derived path (can be null).
   * @param factoryKeys the optional factory keys (can be null).
   * @return the derived key.
   * @throws ResourceKeyCreationException if the key cannot be derived for any
   * reason.
   */
  public ResourceKey deriveKey (ResourceKey parent,
                                String path,
                                Map factoryKeys)
      throws ResourceKeyCreationException;
}
