/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.heartbeat;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import mx4j.log.Log;
import mx4j.log.Logger;
import mx4j.tools.connector.RemoteMBeanServer;


/**
 * HeartbeatListener  supports multiple observers listening to heartbeat
 * from multiple sources.
 *
 * @version $Revision: 1.9 $
 */
public class HeartBeatListener implements HeartBeatListenerMBean, HeartBeatListenerControl
{
   private ObjectName m_name = null;
   private HashMap m_sources = new HashMap();
   private Thread m_bareListenerThread = null;
   private ListenerThread m_listenerThread = new ListenerThread();
   private boolean m_started[] = new boolean[1];

   public HeartBeatListener(String name)
   {
      try
      {
         m_name = new ObjectName(name);
      }
      catch (Exception e)
      {
      }

      m_started[0] = false;
      m_bareListenerThread = new Thread(m_listenerThread, name);
      m_bareListenerThread.start();

      // synchronize with the thread
      synchronized (m_started)
      {
         while (!m_started[0])
         {
            try
            {
               m_started.wait();
            }
            catch (Exception e)
            {
            }
         }
      }
   }

   private Logger getLogger()
   {
      return Log.getLogger(getClass().getName());
   }

   public void processHeartBeat(String heartBeatSource)
   {
      ObserverSession os = (ObserverSession)m_sources.get(heartBeatSource);

      if (os != null)
      {
         os.reset();
      }
   }

   public ObjectName getObjectName()
   {
      return m_name;
   }

   /**
    * Add this listener to a remote heartbeat source and register application to
    * receive heart beat failure notification from HeartBeatListener.
    */
   public void registerObserver(RemoteMBeanServer connector, Object connectorType, Object address, String heartBeatCanonName, NotificationListener observer)
           throws MalformedObjectNameException, InstanceNotFoundException, MBeanException, ReflectionException, IOException
   {
      /**
       * FIXME: provide for capability to have several observers getting
       * heartbeats from the same source
       */
      ObjectName obName = new ObjectName(heartBeatCanonName);
      Object[] params = new Object[3];
      String[] signature = new String[3];
      params[0] = m_name.getCanonicalName();
      params[1] = connectorType;
      params[2] = address;
      signature[0] = "java.lang.String";
      signature[1] = "java.lang.Object";
      signature[2] = "java.lang.Object";
      connector.invoke(obName, "addHeartBeatListener", params, signature);

      ObserverSession os = new ObserverSession(heartBeatCanonName, observer);
      synchronized (m_sources)
      {
         m_sources.put(heartBeatCanonName, os);
      }
   }

   /**
    * Removes the specified heartbeat observer.
    */
   public void unregisterObserver(String sourceCanonName, NotificationListener observer)
   {
      synchronized (m_sources)
      {
         m_sources.remove(sourceCanonName);
      }
   }

   public synchronized void stop()
   {
      m_listenerThread.stop();
      m_bareListenerThread.interrupt();

      // give listener thread a chance to react
      Thread.currentThread().yield();
   }

   private class ObserverSession
   {
      private String m_heartBeatName;
      private NotificationListener m_observer;
      private long m_heartBeatTime;
      private int FAIL_COUNTER_LIMIT = HeartBeatMBean.DEFAULT_RETRIES;
      private int m_failCount = 0;

      ObserverSession(String name, NotificationListener observer)
      {
         m_heartBeatName = name;
         m_observer = observer;
         m_heartBeatTime = System.currentTimeMillis();

         Properties env = System.getProperties();
         String p = env.getProperty(HeartBeatMBean.TRIES_PROP);

         if (p != null)
         {
            FAIL_COUNTER_LIMIT = (new Integer(p)).intValue();
         }
      }

      public void reset()
      {
         m_failCount = 0;
      }

      public void incrementCounter()
      {
         m_failCount++;
         Logger logger = getLogger();
         if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("ListenerThread.incrementCounter: " + m_heartBeatName + " count " + m_failCount);
      }

      public NotificationListener getObserver()
      {
         return m_observer;
      }

      public String getHeartBeatName()
      {
         return m_heartBeatName;
      }

      public boolean shouldContinue()
      {
         if (m_failCount > FAIL_COUNTER_LIMIT)
         {
            return false;
         }
         return true;
      }
   }

   private class ListenerThread implements Runnable
   {
      private int m_period = HeartBeatMBean.DEFAULT_PERIOD;
      private boolean m_stop = false;

      public ListenerThread()
      {
         Properties env = System.getProperties();
         String p = env.getProperty(HeartBeatMBean.PERIOD_PROP);

         if (p != null)
         {
            m_period = (new Integer(p)).intValue();
         }
      }

      public void run()
      {
         // notify creator that this thread is running
         synchronized (m_started)
         {
            m_started[0] = true;
            m_started.notifyAll();
         }

         while (true)
         {
            try
            {
               HashMap sources;

               synchronized (m_sources)
               {
                  sources = (HashMap)m_sources.clone();
               }

               // loop thru heartbeat sources
               Set keys = sources.keySet();
               for (Iterator it = keys.iterator(); it.hasNext();)
               {
                  String sourceName = (String)it.next();
                  ObserverSession sess = (ObserverSession)sources.get(sourceName);
                  sess.incrementCounter();

                  if (!sess.shouldContinue())
                  {
                     NotificationListener observer = sess.getObserver();
                     Notification not = new Notification(HeartBeatListenerControl.LOST_HEARTBEAT, sourceName, 0);

                     if (observer != null)
                     {
                        observer.handleNotification(not, null);
                     }

                     Logger logger = getLogger();

                     // remove this heartbeat source
                     synchronized (m_sources)
                     {
                        m_sources.remove(sourceName);
                        if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("ListenerThread.run: removed source=" + sourceName);
                     }
                  }
               } // for
            }
            catch (Exception ex)
            {
            }

            try
            {
               Thread.sleep(1000 * m_period);
            }
            catch (Exception ex)
            {
            }

            if (m_stop)
            {
               return;
            }
         } // while(true)
      } // run

      public synchronized void stop()
      {
         m_stop = true;
      }
   }
}
