/*
    Copyright (C) 2000 Steve Brown
    Copyright (C) 2000,2001 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: group.c,v 1.77 2002/01/15 11:27:14 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/group.c,v $
    $Date: 2002/01/15 11:27:14 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <nss.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <grp.h>
#include <pthread.h>
#include <mysql/mysql.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#ifdef INCLUDE_MY_SYS
	#include <mysql/my_pthread.h>
#endif

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "group.h"
#include "ent.h"

#define ENT_TYPE 1

#if USE_GROUP

/* This structure is used to share information between
 * all ***grent functions. This is a cache of all GIDs */

enum nss_status _nss_mysql_initgroups (const char *user, gid_t group, long int *start,
              long int *size, gid_t * groups, long int limit,int *errnop);
enum nss_status _nss_mysql_setgrent (void);
enum nss_status _nss_mysql_endgrent (void);
enum nss_status _nss_mysql_getgrent_r (struct group *gr,
                char * buffer, size_t buflen,int * errnop);
enum nss_status _nss_mysql_getgrnam_r (const char * name, struct group *gr,
                char * buffer, size_t buflen,int *errnop);
enum nss_status _nss_mysql_getgrgid_r (const gid_t gid, struct group *gr,
                char * buffer, size_t buflen,int *errnop);
static int check_connection(MYSQL ** mysql_auth,struct passwdoptions * options,pthread_mutex_t * mutex);

/*EOP*/

/* light */
#define light_clean_quit() \
if (sql) free(sql); \
if (result) mysql_free_result(result); \
*errnop = ENOENT; /* most common situation */

/* generic quit macro */
#define clean_quit() _nss_mysql_free_groups(&goptions); \
_nss_mysql_free_users(&options); \
light_clean_quit();

/* check if the argument is NULL, if so, it sets errno and quit */
#define check_mem(x) if ( (x) == NULL ) { \
	_nss_mysql_log(LOG_ERR,"group.c: not enough memory"); \
	clean_quit(); \
	*errnop = EAGAIN; \
	return NSS_STATUS_TRYAGAIN; \
}


#define light_check_mem(x) if ( (x) == NULL ) { \
	_nss_mysql_log(LOG_ERR,"group.c: not enough memory"); \
	light_clean_quit(); \
	*errnop = EAGAIN; \
	return NSS_STATUS_TRYAGAIN; \
}

/* _nss_mysql_group_fill_struct
 * Fills an struct group
 * if uname and name are not NULL, looks for a group by _name_
 * else looks for a group by gid
 * Arguments:
 * uname: name sent to the library (could be NULL)
 * name: name ? space to escape all name bytes : NULL
 * gid: name ? NULL : gid of the group
 * gr: ptr to an _alloced_ group struct
 * errnop: pointer to application errno
 */


#define lockm() \
if (m) pthread_mutex_lock(m);

#define unlockm() \
if (m) pthread_mutex_unlock(m);

enum nss_status _nss_mysql_group_fill_struct(const char * uname,char * name,const gid_t gid,
		struct group * gr, int * errnop, MYSQL * mysql_auth,struct passwdoptions  * options,
		struct groupoptions * goptions,pthread_mutex_t * m) {
	enum nss_status status = NSS_STATUS_SUCCESS;
	char * sql = NULL;
	MYSQL_RES *result = NULL;
	unsigned long int num_rows;
	
	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_group_fill_struct called");
	
	if (uname) {
#ifndef HAVE_MYSQL_REAL_ESCAPE_STRING
		mysql_escape_string(name,uname,strlen(uname));
#else
		mysql_real_escape_string(mysql_auth,name,uname,strlen(uname));
#endif
	}

	if (name) {
		/* we look by name */
		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s from %s "
				"LEFT JOIN %s on %s=%s LEFT JOIN %s on %s=%s "
				"where %s = '%s' and %s and %s",
			goptions->groupnamecolumn,
			goptions->gidcolumn,
			goptions->passwordcolumn,
			options->usercolumn,
			/* from */
			goptions->groupinfotable,
			/* left join */
			goptions->memberstable,
			/* on */
			goptions->groupidcolumn,
			goptions->mgroupidcolumn,
			/* left join */
			options->table,
			/* on */
			goptions->museridcolumn,
			options->useridcolumn,
			/* where */
			goptions->groupnamecolumn,
			name,
			goptions->where[0] ? goptions->where : "1=1",
			options->where[0] ? options->where : "1=1");
	} else {
		/* we look by GID */
		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s from %s "
				"LEFT JOIN %s on %s=%s LEFT JOIN %s on %s=%s "
				"where %s = %d and %s and %s",
			goptions->groupnamecolumn,
			goptions->gidcolumn,
			goptions->passwordcolumn,
			options->usercolumn,
			/* from */
			goptions->groupinfotable,
			/* left join */
			goptions->memberstable,
			/* on */
			goptions->groupidcolumn,
			goptions->mgroupidcolumn,
			/* left join */
			options->table,
			/* on */
			goptions->museridcolumn,
			options->useridcolumn,
			/* where */
			goptions->gidcolumn,
			gid,
			goptions->where[0] ? goptions->where : "1=1",
			options->where[0] ? options->where : "1=1");
	}
	light_check_mem(sql);
	if (DEBUG) _nss_mysql_log(LOG_ERR,"setgrent: SQL statement: %s",sql);

	lockm();
	/* Sending query */
	if (mysql_query(mysql_auth, sql)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_fill_struct: mysql query failed: %s",mysql_error(mysql_auth));
		light_clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}
		
	/* Getting result set */
	result = mysql_store_result(mysql_auth);
	if (! result) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_fill_struct: mysql_store_result failed: %s",mysql_error(mysql_auth));
		light_clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}
	unlockm();
	/* 1 row is success, 0 is notfound */
	if ((num_rows = mysql_num_rows(result)) == 0) {
		/* notfound */
		if (DEBUG) _nss_mysql_log(LOG_ERR,"group not found");
		light_clean_quit();
		return NSS_STATUS_NOTFOUND;
		
	}

	if ((status = _nss_mysql_group_result_to_struct(gr, errnop, mysql_auth, result, num_rows)) != NSS_STATUS_SUCCESS) {
		light_clean_quit();
		return status;
	}

	light_clean_quit();
	*errnop = 0;
	return status;
}

#define X_check_mem(x) if ( (x) == NULL ) { \
	_nss_mysql_log(LOG_ERR,"group.c(result_to_struct): not enough memory"); \
	*errnop = EAGAIN; \
	return NSS_STATUS_TRYAGAIN; \
}

/* _nss_mysql_group_result_to_struct
 * Translate a set of rows (available in the result arg)
 * into a proper group structure (in the gr arg) using the
 * current MySQL connection (in the mysql_auth arg). It 
 * will consume the exact number of rows specified (in the num_rows
 * arg). The errnop argument will be set if an error occurs.  
 *
 * It is used as a backend function for _nss_mysql_group_fill_struct.
 */

enum nss_status _nss_mysql_group_result_to_struct(struct group * gr, int * errnop, MYSQL * mysql_auth, MYSQL_RES *result, unsigned long int num_rows) {
	MYSQL_ROW sql_row;
	unsigned long i = 0;
	int error;

	sql_row = mysql_fetch_row(result);
	if (sql_row == NULL) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: mysql_fetch_row failed: %s."
				        " This should NOT happen. Contact current maintainer.",
						mysql_error(mysql_auth));
		return NSS_STATUS_UNAVAIL;
	}

	gr->gr_gid = _nss_mysql_strtol(sql_row[1],-1,&error);
	if (error) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: gid field(%s) cannot be "
			               "converted to an integer. Fix your database.",
						   sql_row[1]);
		return NSS_STATUS_UNAVAIL;
	}
		
	if (_nss_mysql_isempty(sql_row[0])) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: name is empty for GID (%d)."
				               " Fix your database.",gr->gr_gid);
		return NSS_STATUS_UNAVAIL;
	}
	gr->gr_name = strdup(sql_row[0]);
	X_check_mem(gr->gr_name);

	if (_nss_mysql_isempty(sql_row[2])) {
		_nss_mysql_log(LOG_ERR,"g_fill_strict: %s has an empty or null password. "
				               "Fix your database.",gr->gr_name);
		gr->gr_passwd = strdup("x");
	} else 
		gr->gr_passwd = strdup(sql_row[2]);
	X_check_mem(gr->gr_passwd);

	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: %s's gid is %d",gr->gr_name,gr->gr_gid);
	
	if (num_rows == 1 && sql_row[3] == NULL) {
		/* no member in group */
		if (DEBUG) _nss_mysql_log(LOG_ERR,"no members in group %s (GID %d)",gr->gr_name,gr->gr_gid);
		gr->gr_mem = malloc(sizeof(char *));
		X_check_mem(gr->gr_mem);
		gr->gr_mem[0] = NULL;
	} else {
		/* we get one row if there is no members of one */
		gr->gr_mem = malloc((num_rows + 1) * sizeof(char *));
		X_check_mem(gr->gr_mem);

		/* we fill the members array */
		for (i = 0 ; i < num_rows && (!i || (sql_row = mysql_fetch_row(result))) ; ++i) {
			
			if (!sql_row) {
				_nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: mysql_fetch_row failed: %s",
					               mysql_error(mysql_auth));
				return NSS_STATUS_UNAVAIL;
			}
	
			/* user name MUST not be empty */
			if (_nss_mysql_isempty(sql_row[3])) {
				_nss_mysql_log(LOG_ERR,"Empty or NULL member for group %s(%d). Fix your database",
					               gr->gr_name,gr->gr_gid);
			}

			gr->gr_mem[i] = sql_row[3] ? strdup(sql_row[3]) : strdup("unknown");
			X_check_mem(gr->gr_mem[i]);

			if (DEBUG) _nss_mysql_log(LOG_ERR,"%s is a member of %s(%d)",sql_row[3],
				                   gr->gr_name,gr->gr_gid);
		}
		/* we indicate the end */
		gr->gr_mem[i] = NULL;
		
		if (strncmp("",mysql_error(mysql_auth),4)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: not all group members have been returned."
			        " mysql_fetch_row has failed: %s",mysql_error(mysql_auth));
		}
	}

	return NSS_STATUS_SUCCESS;
}

/* Initgroups
 * Returns all groups which an user is in
 * Arguments :
 * user : user name
 * group: this gid must be excluded from groups
 * start: first element in groups
 * size: size of groups
 * groups: array of gid the user is in
 * limit: size limit of groups
 * errnop: pointer to the application errno
 */

#undef lockm
#undef unlockm
#define lockm() \
pthread_mutex_lock(&initgroups_mutex);

#define unlockm() \
pthread_mutex_unlock(&initgroups_mutex);

static pthread_mutex_t initgroups_mutex = PTHREAD_MUTEX_INITIALIZER;
static MYSQL * initgroups_auth = NULL;

enum nss_status _nss_mysql_initgroups (const char *user, gid_t group, long int *start,
              long int *size, gid_t * groups, long int limit,int *errnop) {
	char * sql = NULL;
	MYSQL_RES * result = NULL;
	MYSQL_ROW sql_row;
	unsigned int i;
	struct passwdoptions options;
	struct groupoptions goptions;
	char * secure_user;

	memset(&options,0,sizeof(struct passwdoptions));
	memset(&goptions,0,sizeof(struct groupoptions));

	if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups called with user=%s,group=%d,*start=%d,*size=%d,limit=%d",user,*start,*size,limit);
	
	if (!user) {
		*errnop = ENOENT;
		return NSS_STATUS_NOTFOUND;
	}

	/* we parse conf files */
	if (! _nss_mysql_read_conf_file("groups",&options,&goptions,NULL)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_initgroups: conf file parsing failed");
		_nss_mysql_free_users(&options);
		_nss_mysql_free_groups(&goptions);
		*errnop = ENOENT;
		return NSS_STATUS_UNAVAIL;
	}

	if (! check_connection(&initgroups_auth,&options,&initgroups_mutex)) {
		_nss_mysql_free_users(&options);
		_nss_mysql_free_groups(&goptions);
		*errnop = ENOENT;
		return NSS_STATUS_UNAVAIL;
	}

	/* we escape the user string */
	secure_user = malloc(strlen(user) * 2 + 1);
	if (secure_user == NULL) {
		_nss_mysql_log(LOG_ERR,"initgroups: not enough memory to escape the user string");
		*errnop = EAGAIN;
		my_thread_end();
		return NSS_STATUS_TRYAGAIN;
	}

#ifndef HAVE_MYSQL_REAL_ESCAPE_STRING
	mysql_escape_string(secure_user,user,strlen(user));
#else
	mysql_real_escape_string(initgroups_auth,secure_user,user,strlen(user));
#endif

	/* we look for the group which the user is a member of
	 * no aliasing for groupinfotable and optionstable because of the where
	 * clauses
	 */
	sql = _nss_mysql_sqlprintf("select %s from %s, %s, %s where %s=%s"
			" and %s=%s and %s='%s' and %s <> %d",
			goptions.gidcolumn,
			/* from */
			options.table,
			goptions.memberstable,
			goptions.groupinfotable,
			/* where */
			options.useridcolumn,
			goptions.museridcolumn,
			/* and */
			goptions.mgroupidcolumn,
			goptions.groupidcolumn,
			/* and */
			options.usercolumn,
			secure_user,
			/* and */
			goptions.gidcolumn,
			group);
	check_mem(sql);

	/* conf files where clauses */
	if (options.where[0]) {
		char * tmp;
		/* we try to alloc the space we need */
		tmp = realloc(sql,strlen(sql) + sizeof(" and ") + strlen(options.where));
		if (tmp == NULL) {
			/* we could not, so we free the old one and leave */
			_nss_mysql_log(LOG_ERR,"initgroups: can't allocate memory for where clauses");
			free(sql);
		}
		sql = tmp;
		check_mem(sql);
		strcat(sql," and ");
		strcat(sql,options.where);
	}

	if (goptions.where[0]) {
		/* same as above */
		char * tmp;
		tmp = realloc(sql,strlen(sql) + sizeof(" and ") + strlen(goptions.where));
		if (tmp == NULL) {
			_nss_mysql_log(LOG_ERR,"initgroups: can't allocate memory for where clauses");
			free(sql);
		}
		sql = tmp;
		check_mem(sql);
		strcat(sql," and ");
		strcat(sql,goptions.where);
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: SQL statement: %s",sql);

	lockm();
	if (mysql_query(initgroups_auth,sql)) {
		_nss_mysql_log(LOG_ERR,"initgroups: mysql_query has failed: %s",mysql_error(initgroups_auth));
		clean_quit();
		unlockm();
		my_thread_end();
		return NSS_STATUS_UNAVAIL;
	}
	
	result = mysql_store_result(initgroups_auth);
	if (! result) {
		_nss_mysql_log(LOG_ERR,"initgroups: mysql_store_result has failed: %s",mysql_error(initgroups_auth));
		clean_quit();
		unlockm();
		my_thread_end();
		return NSS_STATUS_UNAVAIL;
	}
	unlockm();
	
	i = mysql_num_rows(result);
	if (i == 0)	{
		if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: no results found. Exiting");
		clean_quit();
		my_thread_end();
		*errnop = 0;
		return NSS_STATUS_SUCCESS;
	}

	while ((sql_row = mysql_fetch_row(result)))	{
		if (!sql_row) {
			_nss_mysql_log(LOG_ERR,"initgroups: mysql_fetch_row failed: %s.",
					       mysql_error(initgroups_auth));
			clean_quit();
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}
		
		/* do we have enought memory space ? */
		if (*start == *size && limit <= 0) {
			/* we have to write below the array limit, we reallocate twice more
			 * space 
			 */
			gid_t * tmp;
			tmp = realloc (groups, 2 * *size * sizeof (*groups));
			if (tmp == NULL) {
				_nss_mysql_log(LOG_ERR,"initgroups: We could not have allocated the needed space. Stopping");
				break;
			}
			groups = tmp;
			*size *= 2;
		}
		
		groups[*start] = _nss_mysql_strtol(sql_row[0],-1,&i);
		if (i) {
			_nss_mysql_log(LOG_ERR,"initgroups: cannot convert group_id (%s) for %s. "
				        "Fix your database.",sql_row[0],user);
		}
		if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: %s is a member of %d",user,groups[*start]);
		
		++*start; /* next element */
		if (*start == limit) {
			/* we cannot go on */
			if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: reached limit at %d",*start);
			break;
		}
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups exited");
	clean_quit();
	*errnop = 0;
	my_thread_end();
	return NSS_STATUS_SUCCESS;
}

/* setgrent
 * Initializes data for ...grent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_setgrent (void) {
	return _nss_mysql_setent(ENT_TYPE);
}

/* endgrent
 * Kills all data for ...grent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_endgrent (void) {
	return _nss_mysql_endent(ENT_TYPE);
}

/* getgrent
 * Gets info for every group
 * Arguments:
 * gr: array to fill
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

enum nss_status _nss_mysql_getgrent_r (struct group *gr,
                char * buffer, size_t buflen,int * errnop) {
	return _nss_mysql_getent_r(ENT_TYPE,gr,buffer,buflen,errnop);
}

/* check_connection 
 * checks if a connection has been opened and if it is still alive
 * tries to open a connection if not
 */

int check_connection(MYSQL ** mysql_auth,struct passwdoptions * options,pthread_mutex_t * mutex) {
	/* Is the server still alive ? */
	pthread_mutex_lock(mutex);
	if (*mysql_auth != NULL) {
		my_thread_init();
		if (mysql_ping(*mysql_auth)) {
			if (DEBUG) _nss_mysql_log(LOG_ERR,"check_connection: can't sustain connection : %s",
					mysql_error(*mysql_auth));
			*mysql_auth = NULL;
		}
	}
	
	/* DB connection */
	if (*mysql_auth == NULL) {
		if  (! _nss_mysql_db_connect(mysql_auth,options->host,options->dbuser,options->dbpasswd,options->database)) {
			pthread_mutex_unlock(mutex);
			my_thread_end();
			return 0;
		}
	} 
	pthread_mutex_unlock(mutex);
	return 1;
}


/* getgrnam
 * Looks for a group by its name
 * Arguments:
 * name: name of the group
 * gr: pointer to the gr struct we'll filll
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

#define getgr_quit() \
_nss_mysql_free_users(&options); \
_nss_mysql_free_groups(&goptions); 

static pthread_mutex_t nam_mutex = PTHREAD_MUTEX_INITIALIZER;
static MYSQL * nam_auth = NULL;

enum nss_status _nss_mysql_getgrnam_r (const char * name, struct group *gr,
                char * buffer, size_t buflen,int *errnop) {
	char * buff;
	enum nss_status toreturn;
	struct passwdoptions options;
	struct groupoptions goptions;

	memset(&options,0,sizeof(struct passwdoptions));
	memset(&goptions,0,sizeof(struct groupoptions));
	*errnop = ENOENT;
	
	/* gr and name must NOT be NULL */
	if ( ! gr || ! name) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_getgrnam_r: called with bad arguments");
		return NSS_STATUS_UNAVAIL;
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_getgrnam_r called for %s",name);
	
	/* conf files parsing */
	if (! _nss_mysql_read_conf_file("groups",&options,&goptions,NULL)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_getgrnam_r: conf file parsing failed");
		getgr_quit();
		return NSS_STATUS_UNAVAIL;
	}

	/* DB connection */
	if (! check_connection(&nam_auth,&options,&nam_mutex)) {
		getgr_quit();
		return NSS_STATUS_UNAVAIL;
	}
	
	buff = malloc(strlen(name)*2+1);
	if (buff == NULL) {
		/* XXX if (DEBUG) */ _nss_mysql_log(LOG_ERR,"_nss_mysql_getgrnam_r: not enough memory to "
				                             "escape buffer");
		*errnop = EAGAIN;
		getgr_quit();
		my_thread_end();
		return NSS_STATUS_TRYAGAIN;
	}
	
	/* actual lookup */
	toreturn = _nss_mysql_group_fill_struct(name,buff,0,gr,errnop,nam_auth,&options,&goptions,&nam_mutex);
	free(buff); /* I am so nice */
	getgr_quit();
	my_thread_end();
	return toreturn;
}	


/* getgrgid
 * Looks for a group by its GID
 * Arguments:
 * gid: gid of the group
 * gr: pointer to the gr struct we'll filll
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */


static pthread_mutex_t gid_mutex = PTHREAD_MUTEX_INITIALIZER;
static MYSQL * gid_auth = NULL;

enum nss_status _nss_mysql_getgrgid_r (const gid_t gid, struct group *gr,
                char * buffer, size_t buflen,int *errnop) {
	enum nss_status status;
	struct passwdoptions options;
	struct groupoptions goptions;
	
	memset(&options,0,sizeof(struct passwdoptions));
	memset(&goptions,0,sizeof(struct groupoptions));
	
	*errnop = ENOENT;
	/* sanity checks, trust no one */
	if (! gr) {
		_nss_mysql_log(LOG_ERR,"getgrgid: called with bad arguments");
		return NSS_STATUS_UNAVAIL;
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getgrgid called for group %d",gid);
	
	/* conf files parsing */
	if (! _nss_mysql_read_conf_file("groups",&options,&goptions,NULL)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_group_fill_struct: conf file parsing failed");
		getgr_quit();
		return NSS_STATUS_UNAVAIL;
	}

	/* DB connection */
	if (! check_connection(&gid_auth,&options,&gid_mutex)) {
		getgr_quit();
		return NSS_STATUS_UNAVAIL;
	}

	/* actual lookup */
	status = _nss_mysql_group_fill_struct(NULL,NULL,gid,gr,errnop,gid_auth,&options,&goptions,&gid_mutex);
	getgr_quit();
	my_thread_end();
	return status;
}

#ifdef TEST

int main(void) {
	struct group gr;
	enum nss_status s;
	_nss_mysql_setgrent();
	while ((s=_nss_mysql_getgrent_r(&gr,NULL,0,&errno))!=NSS_STATUS_NOTFOUND);
	_nss_mysql_endgrent();
	return 0;
}

#endif

#endif /* if USE_GROUP */
