/*
    Copyright (C) 2000 Steve Brown
    Copyright (C) 2000,2001 Guillaume Morin, Alcve
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: parser.c,v 1.51 2002/01/16 17:32:31 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/parser.c,v $
    $Date: 2002/01/16 17:32:31 $
    $Author: gmorin $
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include "parser.h"
#include "nss-shadow.h"
#include "passwd.h"
#include "group.h"
#include "lib.h"

#define CONFFILE "/etc/nss-mysql.conf"
#define ROOT_CONFFILE "/etc/nss-mysql-root.conf"

#define FILE_BUF 1024

#define s_free(x) if (x) {free(x); x = NULL;}

/* 
here is the conf file format :
# begins a comment till the end of the line
a regular assignment will be
section.parameter=value;
eg. groups.dbuser=nss;
see the sample configuration files for the list of all parameters.
*/


/* for parse file return value */
struct result {
	char param[32];
	char * value;
};

struct check {
	char option[32]; /* configuration option */
	int mandatory; /* can be empty or null */
	char ** value_ptr; /* temporary storage place */
};

static void process(char *,char *,struct check *);
static int parse_line(char *,struct result *);
static int check(char *,struct check *);

/* _nss_mysql__nss_mysql_read_conf_file(): reads parameters from a section of the config file
 *	section can be either "users", "groups", or "shadow". */
int _nss_mysql_read_conf_file(char * section,struct passwdoptions * options,
		struct groupoptions * goptions, struct shadowoptions * soptions) {
	FILE * fd;
	char buffer[FILE_BUF];
	char * ptr;
	char * ptr2;
	char * conf_version = 0;
	char * conf_file = "(none)";
	struct check check_conf[] = {
		{"version",1,&conf_version},
		{"",1,NULL}
	};
	struct check check_users[] = {
		{"host",1,&options->host},
		{"database",1,&options->database},
		{"db_user",1,&options->dbuser},
		{"db_password",0,&options->dbpasswd},
		{"table",1,&options->table},
		{"where_clause",0,&options->where},
		{"user_column",1,&options->usercolumn},
#if ! USE_SHADOW
		{"password_column",1,&options->passwdcolumn},
# else
		{"password_column",2,NULL}, /* no value will be read, but no warning will 
									   appear in the logs */
#endif
		{"userid_column",1,&options->useridcolumn},
		{"gid_column",1,&options->gidcolumn},
		{"uid_column",1,&options->uidcolumn},
		{"realname_column",1,&options->realnamecolumn},
		{"shell_column",1,&options->shellcolumn},
		{"homedir_column",1,&options->homedircolumn},
		{"",1,NULL}
	};
#if USE_GROUP
	struct check check_groups[] = {
		{"group_info_table",1,&goptions->groupinfotable},
		{"where_clause",0,&goptions->where},
		{"group_name_column",1,&goptions->groupnamecolumn},
		{"groupid_column",1,&goptions->groupidcolumn},
		{"gid_column",1,&goptions->gidcolumn},
		{"password_column",1,&goptions->passwordcolumn},
		{"members_table",1,&goptions->memberstable},
		{"member_userid_column",1,&goptions->museridcolumn},
		{"member_groupid_column",1,&goptions->mgroupidcolumn},
		{"",1,NULL}
	};
#endif
#if USE_SHADOW
	struct check check_shadow[] = {
		{"host",1,&soptions->host},
		{"database",1,&soptions->database},
		{"db_user",1,&soptions->dbuser},
		{"db_password",0,&soptions->dbpasswd},
		{"table",1,&soptions->table},
		{"where_clause",0,&soptions->where},
		{"user_column",1,&soptions->usercolumn},
		{"password_column",1,&soptions->passwdcolumn},
		{"userid_column",1,&soptions->useridcolumn},
		{"lastchange_column",1,&soptions->lastchange},
		{"min_column",1,&soptions->min},
		{"max_column",1,&soptions->max},
		{"warn_column",1,&soptions->warn},
		{"inact_column",1,&soptions->inact},
		{"expire_column",1,&soptions->expire},
		{"",1,NULL}
	};
#endif
	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file: called for section %s",section);
	
	if (strcmp(section,"users") != 0 && strcmp(section,"groups") != 0 &&
			strcmp(section,"shadow")) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file: called with unknow section %s",section);
		return 0;
	}

	/* memory allocation */
	if (strcmp(section,"users") == 0 || strcmp(section,"groups") == 0)	{
		/* the groups function needs user section information */
		if (options == NULL) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file: called for section %s with ",
					               "options == NULL", section);
			return 0;
		} else
			/* XXX should be a free */
			memset(options,0,sizeof(struct passwdoptions));
	}

#if USE_GROUP
	if (strcmp(section,"groups") == 0) {
		if (goptions == NULL)	{
			_nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file: called for section groups with "
					               "goptions == NULL");
			return 0;
		} else
			/* XXX should be a free */
			memset(goptions,0,sizeof(struct groupoptions));
	}
#endif

#if USE_SHADOW
	if (strcmp(section,"shadow") == 0) {
		if (soptions == NULL)	{
 			_nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file: called for section shadow with "
					               "soptions == NULL");
			return 0;
		} else
			/* XXX should be a free */
			memset(soptions,0,sizeof(struct shadowoptions));
	}  
#endif
	
	/* we open the conf file */
	if (strcmp("shadow",section) == 0) {
		/* the shadow information are stored in a ROOT_CONFFILE */
		struct stat buf;
		fd = NULL;
		if ( stat(ROOT_CONFFILE,&buf) == 0 )
		{
			/* we want to be sure that that ROOT_CONFFILE is owned by root
			 * and his mode is 600 or 400
			 */
			if ( buf.st_uid == 0 && (buf.st_mode == (S_IRUSR | S_IWUSR | S_IFREG) || buf.st_mode == (S_IRUSR | S_IFREG)) ) {
				fd = fopen(ROOT_CONFFILE,"r");
				conf_file = ROOT_CONFFILE;
			} else {
				_nss_mysql_log(LOG_ERR,ROOT_CONFFILE " MUST be only root readable");
			}
		}
	} else {
		fd = fopen(CONFFILE,"r");
		conf_file = CONFFILE;
	}
	
	if (fd == NULL) {
		_nss_mysql_log(LOG_ERR, "Cannot open %s configuration file: %s. (euid=%d, uid=%d)", conf_file,strerror(errno),geteuid,getuid);
		return 0;
	}

	while (fgets(buffer,FILE_BUF,fd))	{
		/* we skip the empty lines */
		if (strlen(buffer) == 0)
			continue;

		ptr = buffer;
		/* we chop the buffer, Perl style :-) */
		buffer[strlen(buffer) - 1] = '\0';

		/* we skip the spaces */
		while (*ptr == ' ' || *ptr == '\t')
			++ptr;
		
		if (strlen(ptr) == 0 || *ptr == '#' || *ptr == '.')
			continue;

		/* ptr represents the beginning of the section string */
		ptr2=ptr;
		
		/* we are looking for the '.' */
		while (*ptr2 != '.' && *ptr2 != '\0')
			++ptr2;
		
		if (*ptr2 == '\0' || *(ptr2 + 1) == '\0')
			/* we did not  find a '.' */
			continue;
		
		if (strncmp(ptr,"users",ptr2-ptr) == 0 && (strcmp(section,"users") == 0 || strcmp(section,"groups") == 0))
			process(ptr2+1,"users",check_users);
#if USE_GROUP
		else if (strncmp(ptr,"groups",ptr2-ptr) == 0 && strcmp(section,"groups") == 0)
			process(ptr2+1,"groups",check_groups);
#endif
#if USE_SHADOW
		else if (strncmp(ptr,"shadow",ptr2-ptr) == 0 && strcmp(section,"shadow") == 0)
			process(ptr2+1,"shadow",check_shadow);
#endif
		else if (strncmp(ptr,"conf",ptr2-ptr) == 0)
			process(ptr2+1,"conf",check_conf);

	}
	fclose(fd);

	/* configuration file version sanity check. */
	{
		int version = conf_version ? atoi(conf_version) : 1;
		if (version == 1 && CONF_VERSION == 2) {
			_nss_mysql_log(LOG_ERR,"%s is a file without a conf.version line"
" and is therefore a configuration file version 1."
" You must upgrade it to version 2 by doing the following:"
" A) Replace ALL field names by fully qualified field names."
" For instance if you have users.uid_column = uid; replace it with"
" users.uid_column = user.uid; provided \"user\" is the table name."
" B) Add a conf.version = 2; line somewhere in this file so that the parser"
" is aware that you took care of the conversion.",
										 conf_file
										 );
			return 0;
		} else if (version != CONF_VERSION) {
			_nss_mysql_log(LOG_ERR,"%s: unexpected conf.version = %d, only conf.version = %d is valid\n", conf_file, version, CONF_VERSION);
			return 0;
		}
	}

	/* preparing return value */
	if (strcmp("users",section) == 0) {
		if (!check("users",check_users)) {
			_nss_mysql_free_users(options);
			return 0;
		}
#if USE_GROUP
	} else if (strcmp("groups",section) == 0 ) {
		if (!check("users",check_users) || !check("groups",check_groups)) {
			_nss_mysql_free_users(options);
			_nss_mysql_free_groups(goptions);
			return 0;
		}
#endif
#if USE_SHADOW
	} else {
		if (!check("shadow",check_shadow)) {
			_nss_mysql_free_shadow(soptions);
			return 0;
		}
#endif
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_read_conf_file ended for section %s",section);
	
	return 1;
}

/* _nss_mysql_free_users
 * frees a struct options
 */
void _nss_mysql_free_users(struct passwdoptions *pw) {
	s_free(pw->host);
	s_free(pw->where);
	s_free(pw->database);
	s_free(pw->dbuser);
	s_free(pw->dbpasswd);
	s_free(pw->table);
	s_free(pw->usercolumn);
#if ! USE_SHADOW
	s_free(pw->passwdcolumn);
#endif
	s_free(pw->useridcolumn);
	s_free(pw->uidcolumn);
	s_free(pw->gidcolumn);
	s_free(pw->realnamecolumn);
	s_free(pw->homedircolumn);
	s_free(pw->shellcolumn);
}

void process(char * begin_ptr,char * section, struct check * c) {
	struct result res;
	
	if (parse_line(begin_ptr,&res) == -1) {
		_nss_mysql_log(LOG_ERR,"process: Malformed line : %s\n",begin_ptr);
		return;
	}
	while (c->option[0]) {
		if (strcmp(res.param,c->option) == 0) {
			if (c->value_ptr) 
				*(c->value_ptr) = res.value;
			else 
				free(res.value);
			break;
		}
		++c;
	}
	if (!c->option[0]) {
		_nss_mysql_log(LOG_ERR,"Unknown %s parameter : %s\n",section,res.param);
		free(res.value);
	}
}

/* check
 * Check if all mandatory fields have been set
 */
int check(char * section,struct check *c) {
	while(c->option[0]) {
	
		/* we skip variables with mandory == 2 */
		if (c->mandatory == 2) {
			++c;
			continue;
		}

		/* mandatory variables MUST be set, the others will become empty if ==
		 * NULL
		 */
		if(_nss_mysql_isempty(*(c->value_ptr))) {
			if (c->mandatory) {
				_nss_mysql_log(LOG_ERR,"check: %s.%s must be set. Fix your "
						               "configuration files",section,c->option);
				return 0;
			}
			if (*(c->value_ptr) == NULL) {
				if ( !(*(c->value_ptr) = strdup(""))) {
					_nss_mysql_log(LOG_ERR,"check: not enough memory");
					return 0;
				}
			}
		}
		++c;
	}
	return 1;
}

/* parse_line
 * Find the paramater and the value
 * begin_ptr: beginning of the parameter
 * res: struct we'll fill
 */
int parse_line(char * begin_ptr,struct result *res) {
	char * ptr;
	char * end_of_param;
	char * beginning_of_value;
	
	ptr = begin_ptr;

	/* we are looking for the '=' */
	while (*ptr != '\0' && *ptr != '=') 
		++ptr;
	
	if (*ptr == '\0')
		return -1;

	end_of_param = begin_ptr;

	/* we are looking for the end of the param string */
	while (*end_of_param != ' ' && *end_of_param != '\t' && *end_of_param != '=')
		++end_of_param;
	
	if (end_of_param == begin_ptr)
		/* there is no param */
		return -1;

	/* we skip the spaces between section.param and '=' */
	while (*ptr == ' ' || *ptr == '\t') 
		++ptr;
	
	/* we should reach the '=', if not we leave */
	if (*ptr != '=' )
		return -1;

	++ptr;	
	/* we skip the spaces between '=' and the value */
	while (*ptr == ' ' || *ptr == '\t')
		++ptr;

	beginning_of_value = ptr;
	
	/* we are looking for the end of the value */
	while (*ptr != '\0' && *ptr != ';' && *ptr != '#')
		++ptr;

	if (*ptr != ';')
		return -1;
	
	/* I do this because strncpy does not add a \0 at the end and strncat does*/
	res->param[0] = '\0';	
	strncat(res->param,begin_ptr,end_of_param-begin_ptr);
	
	res->value = malloc((ptr-beginning_of_value+1) * sizeof(char) );
	if (res->value == NULL)	{
		_nss_mysql_log(LOG_ERR,"parse_line: not enough memory");
		free(res->param);
		return -1;
	}
	res->value[0] = '\0';
	strncat(res->value,beginning_of_value,ptr-beginning_of_value);
	return 0;
}	


#if USE_SHADOW
void _nss_mysql_free_shadow(struct shadowoptions *sp) {
	s_free(sp->host);
	s_free(sp->where);
	s_free(sp->database);
	s_free(sp->dbuser);
	s_free(sp->dbpasswd);
	s_free(sp->table);
	s_free(sp->usercolumn);
	s_free(sp->passwdcolumn);
	s_free(sp->useridcolumn);
}
#endif /* ifdef USE_SHADOW */

#if USE_GROUP
void _nss_mysql_free_groups(struct groupoptions * gr) {
	s_free(gr->where);
	s_free(gr->groupinfotable);
	s_free(gr->groupnamecolumn);
	s_free(gr->groupidcolumn);
	s_free(gr->memberstable);
	s_free(gr->museridcolumn);
	s_free(gr->mgroupidcolumn);
	s_free(gr->passwordcolumn);
}
#endif /* ifdef USE_GROUP */

/* EOF */
