/* outboxstojobs.cpp

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "assert.h"

#include "outboxstojobs.h"
#include <accountjobs.h>
#include <accountimpl.h>
#include <bankimpl.h>
#include <hbcistring.h>

namespace HBCI {

// This const_cast is used frequently in this file since the
// Pointer to the customer won't be changed (thus doesn't violate
// his const declaration) but it is impossible to express this with
// the Pointer, unfortunately.
    static Pointer<Customer> 
    custPointer_const_cast(const Customer *c)
    {
	Pointer<Customer> cp = const_cast<Customer*>(c);
	cp.setAutoDelete(false);
	return cp;
    }
    static Pointer<Account> accPointer(Account *a)
    {
	Pointer<Account> ap = a;
	ap.setAutoDelete(false);
	return ap;
    }



OutboxJobGetStandingOrders::OutboxJobGetStandingOrders(
    Pointer<Customer> c,
    Pointer<Account> a)
:OutboxAccountJob(c,a)
{
}


OutboxJobGetStandingOrders::~OutboxJobGetStandingOrders(){
}


bool OutboxJobGetStandingOrders::createHBCIJobs(Pointer<MessageQueue> mbox,
                                        int n){

    if (0 == n)
        _job=new JOBGetStandingOrders(_cust,_acc);
    else
        _job = new JOBGetStandingOrders(_cust, _job);


    mbox.ref().addJob(_job);
    addSignersToQueue(mbox);

    return true;
}

bool OutboxJobGetStandingOrders::stillMessagesToSend(int nextMsg) const
{
  // zero? of course we have a message to send!
  if (0 == nextMsg)
	return true;

  // any more?
  return dynamic_cast<JOBGetStandingOrders&>(_job.ref()).attachMore();
}

bool OutboxJobGetStandingOrders::evaluate(){
    if (_job.ref().hasErrors())
        _result=HBCI_JOB_RESULT_FAILED;
    else
        _result=HBCI_JOB_RESULT_SUCCESS;

    return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobGetStandingOrders::commit(int msgNumber){
    list<Pointer<StandingOrder> >::const_iterator sit;

    // don't do anything when commit is called for the whole
    // job. otherwise, we would add the result of the last subjob twice
    if (HBCI_COMMIT_WHOLE_JOB == msgNumber)
        return true;

    // in retrieval-only mode? 
    if (_bank.ref().hbci()->isRetrievalOnly())
	return true;

    try {
        // only commit changes for the whole job
        for (sit = dynamic_cast<JOBGetStandingOrders&>(_job.ref()).
		 getOrders().begin();
             sit != dynamic_cast<JOBGetStandingOrders&>(_job.ref()).
		 getOrders().end();
             sit++) {
            // add all standing orders

	    // therefore set the account-information for the new orders
	    // (the segment-parser does not know about this account)
	    (*sit).ref().setOurAccountId(_acc.ref().accountId());
	    (*sit).ref().setOurBankCode(_acc.ref().bank().ref().bankCode());
	    (*sit).ref().setOurCountryCode(_acc.ref().bank().ref().countryCode());
            _acc.ref().addStandingOrder((*sit).ref());
        } // for

    } catch (Error) {
        return false;
    }

    return true;
}


string OutboxJobGetStandingOrders::description() const 
{
    return _makeDescription("Get standing orders");
}


const list<Pointer<StandingOrder> > &
OutboxJobGetStandingOrders::getOrders() const 
{
  return _orders;
}
list<int> OutboxJobGetStandingOrders::resultCodes() const
{
    list<int> res;
    if (_job.isValid()) 
	res = resultCodesFromJob(_job.ref());
    return res;
}

bool OutboxJobGetStandingOrders::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
  
  return (NULL != accountImpl.updForJob("HKDAB"));
}



OutboxJobNewStandingOrder::OutboxJobNewStandingOrder(
    Pointer<Customer> c,
    Pointer<Account> a, const StandingOrder &newOrder)
:OutboxAccountJob(c,a), _newOrder(newOrder)
{
}


OutboxJobNewStandingOrder::~OutboxJobNewStandingOrder(){
}


bool OutboxJobNewStandingOrder::createHBCIJobs(Pointer<MessageQueue> mbox,
                                        int n){

    _job=new JOBNewStandingOrder(_cust, _acc, _newOrder);
    mbox.ref().addJob(_job);
    addSignersToQueue(mbox);
    return true;
}

bool OutboxJobNewStandingOrder::stillMessagesToSend(int nextMsg) const 
{
  return (0 == nextMsg);
}

bool OutboxJobNewStandingOrder::evaluate(){
    if (_job.ref().hasErrors())
        _result=HBCI_JOB_RESULT_FAILED;
    else
        _result=HBCI_JOB_RESULT_SUCCESS;

    return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobNewStandingOrder::commit(int msgNumber){
    return true;
}


string OutboxJobNewStandingOrder::description() const 
{
    return _makeDescription("Create a new standing orders");
}
list<int> OutboxJobNewStandingOrder::resultCodes() const
{
    list<int> res;
    if (_job.isValid()) 
	res = resultCodesFromJob(_job.ref());
    return res;
}


// all the bpd-information-methods
int OutboxJobNewStandingOrder::maxDescriptionLines(const Bank &forBank)
{
    const BankImpl &bank =
	    dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // the first entry is for purpose lines
    return atoi(String::nextDEG(param, 0).c_str());
}


int OutboxJobNewStandingOrder::minDaysToExec(const Bank &forBank) {
    const BankImpl &bank =
	    dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // second entry
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    return atoi(String::nextDEG(param, pos).c_str());
}


int OutboxJobNewStandingOrder::maxDaysToExec(const Bank &forBank) {
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // third entry
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    return atoi(String::nextDEG(param, pos).c_str());
}


list<int> 
OutboxJobNewStandingOrder::monthlyCycle(const Bank &forBank) {
    list<int> months;
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return months;

    string param = jobData->parameter();

    // 4th entry
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    string tmp = String::nextDEG(param, pos);

    for (unsigned int i=0; i<tmp.length() / 2; i++) {
        int tmpMonth = atoi(tmp.substr(i*2, 2).c_str());
        months.push_back(tmpMonth);
    }

    return months;
}


list<int> 
OutboxJobNewStandingOrder::execDaysMonth(const Bank &forBank) {
  list<int> days;
  const BankImpl &bank =
      dynamic_cast<const BankImpl&>(forBank);
  const bpdJob *jobData = bank.findJob("HIDAES");

  if (NULL == jobData)
      return days;

  string param = jobData->parameter();

  // 5th entry
  int pos = 0;
  pos += String::nextDEG(param, pos).length() + 1;
  pos += String::nextDEG(param, pos).length() + 1;
  pos += String::nextDEG(param, pos).length() + 1;
  pos += String::nextDEG(param, pos).length() + 1;
  string tmp = String::nextDEG(param, pos);

  for (unsigned int i=0; i<tmp.length() / 2; i++) {
      int tmpDay = atoi(tmp.substr(i*2, 2).c_str());
      days.push_back(tmpDay);
  }

  return days;
}


list<int> 
OutboxJobNewStandingOrder::weeklyCycle(const Bank &forBank) {
    list<int> weeks;
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return weeks;

    string param = jobData->parameter();

    // 6th entry - not mandatory
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    string tmp = String::nextDEG(param, pos);

    for (unsigned int i=0; i<tmp.length() / 2; i++) {
        int tmpWeek = atoi(tmp.substr(i*2, 2).c_str());
        weeks.push_back(tmpWeek);
    }

    return weeks;
}


list<int> 
OutboxJobNewStandingOrder::execDaysWeek(const Bank &forBank) {
    list<int> days;
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return days;

    string param = jobData->parameter();

    // 7th entry - not mandatory
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    string tmp = String::nextDEG(param, pos);

    for (unsigned int i=0; i<tmp.length() / 2; i++) {
        int tmpDay = atoi(tmp.substr(i*2, 2).c_str());
        days.push_back(tmpDay);
    }

    return days;
}


list<int> 
OutboxJobNewStandingOrder::transactionCodes(const Bank &forBank) {
    list<int> keys;
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDAES");

    if (NULL == jobData)
        return keys;

    string param = jobData->parameter();

    // 8th entry - not mandatory
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
    string tmp = String::nextDEG(param, pos);

    for (unsigned int i=0; i<tmp.length() / 2; i++) {
        string tmpKey = tmp.substr(i*2, 2).c_str();
        keys.push_back(atoi(tmpKey.c_str()));
    }

    return keys;
}

    const Limit OutboxJobNewStandingOrder::limit(Pointer<Account> forAccount) {
	    AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
		const updJob *upd = accountImpl.updForJob("HKDAE");

		if (NULL == upd)
		    return Limit();
		else
		    return upd->limit();
	}

bool OutboxJobNewStandingOrder::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
  
  return (NULL != accountImpl.updForJob("HKDAE"));
}

/**************** OutboxJobDeleteStandingOrder **********************/

OutboxJobDeleteStandingOrder::OutboxJobDeleteStandingOrder(
    Pointer<Customer> c,
    Pointer<Account> a, 
    const StandingOrder &order2Delete)
    :OutboxAccountJob(c,a), 
     _order2Delete(order2Delete)
{
}


OutboxJobDeleteStandingOrder::~OutboxJobDeleteStandingOrder(){
}


bool OutboxJobDeleteStandingOrder::createHBCIJobs(Pointer<MessageQueue> mbox,
                                        int n){

    _job=new JOBDeleteStandingOrder(_cust, _acc, _order2Delete);
    mbox.ref().addJob(_job);
    addSignersToQueue(mbox);
    return true;
}

bool 
OutboxJobDeleteStandingOrder::stillMessagesToSend(int nextMsg) const 
{
  return (0 == nextMsg);
}

bool OutboxJobDeleteStandingOrder::evaluate(){
    if (_job.ref().hasErrors())
        _result=HBCI_JOB_RESULT_FAILED;
    else
        _result=HBCI_JOB_RESULT_SUCCESS;

    return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobDeleteStandingOrder::commit(int msgNumber){
  list<StandingOrder> orders;
  list<StandingOrder> newOrders;
  list<StandingOrder>::const_iterator oit;
  
    // in retrieval-only mode? 
    if (_bank.ref().hbci()->isRetrievalOnly())
	return true;

    try {
	  orders = _acc.ref().standingOrders();
	  for (oit=orders.begin(); oit!=orders.end(); oit++) {
	      if (! ((*oit) == _order2Delete))
		  newOrders.push_back((*oit));
	  }

	  // all orders except the delete one are stored back
	  orders.clear();
	  orders.splice(orders.begin(), newOrders);
    } catch (Error) {
        return false;
    }

    return true;
}


string OutboxJobDeleteStandingOrder::description() const 
{
    return _makeDescription("Delete a standing order");
}
list<int> OutboxJobDeleteStandingOrder::resultCodes() const
{
    list<int> res;
    if (_job.isValid()) 
	res = resultCodesFromJob(_job.ref());
    return res;
}

int OutboxJobDeleteStandingOrder::minDaysBeforeDelete(const Bank &forBank) {
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDALS");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // 1st entry
    int pos = 0;

    return atoi(String::nextDEG(param, pos).c_str());
}

int OutboxJobDeleteStandingOrder::maxDaysBeforeDelete(const Bank &forBank) {
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDALS");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // 2nd entry
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;

    return atoi(String::nextDEG(param, pos).c_str());
}

bool OutboxJobDeleteStandingOrder::deletionTerminable(const Bank &forBank) {
    const BankImpl &bank =
        dynamic_cast<const BankImpl&>(forBank);
    const bpdJob *jobData = bank.findJob("HIDALS");

    if (NULL == jobData)
        return 0;

    string param = jobData->parameter();

    // 3rd entry
    int pos = 0;
    pos += String::nextDEG(param, pos).length() + 1;
    pos += String::nextDEG(param, pos).length() + 1;
	string boolFlag = String::nextDEG(param, pos).c_str();

    return ("J" == boolFlag || "j" == boolFlag);
}

bool OutboxJobDeleteStandingOrder::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
  
  return (NULL != accountImpl.updForJob("HKDAL"));
}

} // namespace HBCI



HBCI_OutboxJobGetStOs *
HBCI_OutboxJobGetStOs_new(const HBCI_Customer *c, 
			  HBCI_Account *a)
{
    assert(c);
    assert(a);
    return new HBCI_OutboxJobGetStOs(custPointer_const_cast(c), 
				     accPointer(a));
}
HBCI_OutboxJob *
HBCI_OutboxJobGetStOs_OutboxJob(HBCI_OutboxJobGetStOs *j)
{
    return j;
}
const list_HBCI_StO *
HBCI_OutboxJobGetStOs_getOrders(const HBCI_OutboxJobGetStOs *j)
{
    assert (j);
    return &(j->getOrders());
}


HBCI_OutboxJobNewStO *
HBCI_OutboxJobNewStO_new(const HBCI_Customer *c, 
			 HBCI_Account *a,
			 const HBCI_StandingOrder *sto)
{
    assert(c);
    assert(a);
    assert(sto);
    return new HBCI_OutboxJobNewStO(custPointer_const_cast(c), 
				    accPointer(a), *sto);
}
HBCI_OutboxJob *
HBCI_OutboxJobNewStO_OutboxJob(HBCI_OutboxJobNewStO *j)
{
    return j;
}

HBCI_OutboxJobDeleteStO *
HBCI_OutboxJobDeleteStO_new(const HBCI_Customer *c, 
			    HBCI_Account *a,
			    const HBCI_StandingOrder *sto)
{
    assert(c);
    assert(a);
    assert(sto);
    return new HBCI::OutboxJobDeleteStandingOrder(custPointer_const_cast(c), 
						  accPointer(a), *sto);
}
HBCI_OutboxJob *
HBCI_OutboxJobDeleteStO_OutboxJob(HBCI_OutboxJobDeleteStO *j)
{
    return j;
}
