#include "osl/move_generator/escape_.h"
#include "osl/move_generator/allMoves.h"
#include "osl/move_action/store.h"
#include "osl/move_classifier/safeMove.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

typedef osl::NumEffectState test_state_t;

#include <algorithm>
#include <iostream>
#include <fstream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class EscapeMovesTest : public CppUnit::TestFixture 
#if OSL_WORDSIZE == 32
		      , public osl::misc::Align16New
#endif
{
  CPPUNIT_TEST_SUITE( EscapeMovesTest );
  CPPUNIT_TEST(testBug);
  CPPUNIT_TEST(testValid);
  CPPUNIT_TEST(testCapture);
  CPPUNIT_TEST(testEscape);
  CPPUNIT_TEST(testBlocking);
  CPPUNIT_TEST(testMoveMember);
  CPPUNIT_TEST_SUITE_END();
private:
  osl::NumEffectState state;
public:
  EscapeMovesTest() : state((osl::SimpleState(osl::HIRATE)))
  {
  }
  void setUp();
  void testBug();
  void testValid();
  void testCapture();
  void testEscape();
  void testBlocking();
  void testMoveMember();
};

CPPUNIT_TEST_SUITE_REGISTRATION(EscapeMovesTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;
extern bool isShortTest;

void assertSafe(const NumEffectState& state, const MoveVector& moves)
{
  if (! isShortTest)
    std::cerr << moves;
  for (size_t i=0; i<moves.size(); ++i)
  {
    CPPUNIT_ASSERT(move_classifier::SafeMove<BLACK>::isMember
		   (state, moves[i].ptype(), moves[i].from(),
		     moves[i].to()));
  }
}

void EscapeMovesTest::testBug()
{
  NumEffectState state(CsaString(
			 "P1-KY *  *  *  *  *  *  * +UM\n"
			 "P2 * +KI *  * +HI *  *  * -KY\n"
			 "P3 *  * -KE *  *  *  *  *  * \n"
			 "P4-FU-OU *  * -KI *  * -FU-FU\n"
			 "P5 * +KE-GI-FU *  *  * +KE * \n"
			 "P6+FU+OU *  * +FU+FU *  * +FU\n"
			 "P7 *  *  *  * +GI *  *  *  * \n"
			 "P8 *  *  *  *  *  *  * +HI * \n"
			 "P9+KY *  * -UM *  *  *  * +KY\n"
			 "P+00FU00FU00FU00FU00FU00FU00FU00FU00KI\n"
			 "P-00FU00FU00KE00GI00GI00KI\n"
			 "+\n").getInitialState());
  test_state_t eState(state);
  const Position kingPosition = Position(8,6);
  {
    effect::Liberty8<BLACK> liberty(eState, kingPosition);
    CPPUNIT_ASSERT_EQUAL(2, liberty.count());
  }
  {
    effect::Liberty8<BLACK> liberty(state, kingPosition);
    CPPUNIT_ASSERT_EQUAL(2, liberty.count());
  }
  
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store>::generateCaptureKing<BLACK>(eState,state.getPieceAt(kingPosition),
			  Position(7,5), store);
  }
  assertSafe(eState, moves);
  moves.clear();
  {
    Store store(moves);
    Escape<Store>::
      generateEscape<BLACK,KING>(eState,state.getPieceAt(kingPosition),store);
  }
  assertSafe(eState, moves);
  CPPUNIT_ASSERT_EQUAL((size_t)2, moves.size());

  moves.clear();
  {
    Store store(moves);
    Escape<Store>::
      generateKingEscape<BLACK,false>(eState, store);
  }
  assertSafe(eState, moves);
  CPPUNIT_ASSERT_EQUAL((size_t)2, moves.size());
}


void EscapeMovesTest::setUp(){
  state=CsaString(
"P1+NY *  *  *  *  * -OU * -KY\n"
"P2 * +TO *  *  * -GI-KI *  *\n"
"P3 * -RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5+KE * -KE * +FU *  * +FU *\n"
"P6 *  *  * +FU+GI-FU *  * +FU\n"
"P7+KE * -UM *  *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n"
).getInitialState();
}


static void testValid(NumEffectState& state){
  {
    NumEffectState state1=CsaString(
"P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
"P2 *  *  *  *  *  * -KI *  * \n"
"P3-FU * -FU-FU-FU-FU *  * -FU\n"
"P4 *  * +HI *  *  *  *  *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 * +GI+KI *  *  *  *  *  * \n"
"P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00KA\n"
"P-00KA\n"
"P+00HI\n"
"-\n"
).getInitialState();
    test_state_t eState(state1);
    MoveVector moves;
    {
      Store store(moves);
      Escape<Store >::
	generateMoves<WHITE,false>(eState,eState.getPieceAt(Position(7,3)),eState.getPieceAt(Position(7,4)),store);
    }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
    
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(7,4),PAWN,ROOK,false,WHITE)));
  }
  {
    NumEffectState state1=CsaString(
"P1-KY+RY *  * -FU-OU * -KE-KY\n"
"P2 *  *  *  * +GI-GI-KI *  * \n"
"P3-FU * -FU-FU *  *  *  *  * \n"
"P4 * -FU *  *  * -FU+FU * -FU\n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU+FU *  *  *  *  * \n"
"P7+FU+FU+GI *  * +FU+GI * +FU\n"
"P8 *  * +KI * +KI *  *  *  * \n"
"P9+KY+KE * +OU+KE * -HI+KE+KY\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00KI\n"
"P-00KA\n"
"P-00KA\n"
"+\n"
).getInitialState();
    test_state_t eState(state1);
    MoveVector moves;
    {
      Store store(moves);
      Escape<Store >::
	generateMoves<WHITE,false>(eState,eState.getPieceAt(Position(4,1)),eState.getPieceAt(Position(5,2)),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,1),Position(5,2),KING,SILVER,false,WHITE)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,1),Position(3,1),KING,PTYPE_EMPTY,false,WHITE)));
    // 開き王手になっている
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,1),Position(5,2),PAWN,SILVER,false,WHITE)));
  }

  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.getPieceAt(Position(8,9)),eState.getPieceAt(Position(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  //  std::cerr << moves << std::endl;
  for (size_t i=0;i<moves.size();i++)
    CPPUNIT_ASSERT(eState.isValidMove(moves[i]) ||
		   (std::cerr << eState << moves[i] << std::endl,0));
  // 本当に escape になっているかはチェックしない
}

void EscapeMovesTest::testValid(){
  ::testValid(state);
}

static void testCapture(NumEffectState& state){

  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.getPieceAt(Position(8,9)),
		    eState.getPieceAt(Position(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  
  CPPUNIT_ASSERT(moves.isMember(Move(Position(9,5),Position(8,3),PKNIGHT,PROOK,true,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(9,5),Position(8,3),KNIGHT,PROOK,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,2),Position(8,3),PPAWN,PROOK,false,BLACK)));
}

void EscapeMovesTest::testCapture(){
  ::testCapture(state);
}

/**
 * 当たりをつけられている駒を逃げる手を生成する
 */
static void testEscape(NumEffectState& state){
  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.getPieceAt(Position(8,9)),
		    eState.getPieceAt(Position(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  // std::cerr << moves << std::endl;
  // valid escape
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,9),Position(9,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,9),Position(7,9),KING,PTYPE_EMPTY,false,BLACK)));
  // 
  CPPUNIT_ASSERT(!moves.isMember(Move(Position(8,9),Position(9,9),KING,PTYPE_EMPTY,false,BLACK)));
  //
  CPPUNIT_ASSERT(eState.hasEffectBy(WHITE,Position(8,8)));
  CPPUNIT_ASSERT(!moves.isMember(Move(Position(8,9),Position(8,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(!moves.isMember(Move(Position(8,9),Position(7,8),KING,PTYPE_EMPTY,false,BLACK)));
}

void EscapeMovesTest::testEscape(){
  ::testEscape(state);
}


static void testBlocking(NumEffectState& state){
  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.getPieceAt(Position(8,9)),
		    eState.getPieceAt(Position(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  
  CPPUNIT_ASSERT(moves.isMember(Move(Position(9,7),Position(8,5),KNIGHT,PTYPE_EMPTY,false,BLACK)));
  //
  CPPUNIT_ASSERT(!moves.isMember(Move(Position(9,8),Position(8,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,4),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,5),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,6),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,7),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,8),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,4),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,5),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,6),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,7),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,8),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,4),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,5),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,6),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,7),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,8),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,4),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,5),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,6),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,7),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Position(8,8),GOLD,BLACK)));
  {
  NumEffectState state1=CsaString(
"P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
"P2 *  *  *  *  *  * -KI-KA * \n"
"P3-FU * -FU-FU-FU-FU *  * -FU\n"
"P4 *  *  *  *  *  * +HI *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 * -HI+FU *  *  *  *  *  * \n"
"P7+FU *  * +FU+FU+FU+FU * +FU\n"
"P8 * +KA+KI *  *  *  *  *  * \n"
"P9+KY+KE+GI * +OU+KI+GI+KE+KY\n"
"P+00FU\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"-\n"
).getInitialState();
  moves.clear();
  test_state_t eState1(state1);
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<WHITE,false>(eState1,eState1.getPieceAt(Position(3,2)),
		    eState1.getPieceAt(Position(3,4)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  CPPUNIT_ASSERT(moves.isMember(Move(Position(3,3),PAWN,WHITE)) ||
		 (std::cerr << eState1 << moves << std::endl,0));
  }
}
// 
void EscapeMovesTest::testBlocking(){
  ::testBlocking(state);
}

static void testMoveFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  test_state_t neState(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for (unsigned int i=0;i<moves.size();i++){
    MoveVector allMoves;
    {
    Store store(allMoves);
    //    std::cerr << "Before AllGene: " << neState.pawnMask[0] << "," << neState.pawnMask[1] << std::endl;
    AllMoves<Store>::
      generate(neState.getTurn(),neState,store);
    }
    //    std::cerr << "After AllGene: " << neState.pawnMask[0] << "," << neState.pawnMask[1] << std::endl;
    {
      size_t origSize=allMoves.size();
      allMoves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,allMoves.size());
    }
    for (int num=0;num<40;num++){
      Piece p=neState.getPieceOf(num);
      const Player pl=neState.getTurn();
      if (p.isOnBoard() && p.owner()==pl){
	Position pos=p.position();
	if (neState.hasEffectBy(alt(pl),pos)){
	  if(p.ptype()!=KING &&
	     neState.hasEffectBy(alt(pl),neState.getKingPosition(pl)))
	    continue;
	  //
	  Piece p1=neState.findCheapThreat(alt(pl),pos);
	  //
	  MoveVector escapeMoves, escape_cheap;
	  {
	    Store storeEscape(escapeMoves), store_cheap(escape_cheap);
	    if(pl==BLACK) {
	      Escape<Store >::
		generateMoves<BLACK,false>(neState,p,p1,storeEscape);
	      Escape<Store>::
		generateMoves<BLACK,true>(neState,p,p1,store_cheap);
	    }
	    else
	    {
	      Escape<Store >::
		generateMoves<WHITE,false>(neState,p,p1,storeEscape);
	      Escape<Store>::
		generateMoves<WHITE,true>(neState,p,p1,store_cheap);
	    }
	  }
	  // 逃げる手は全部生成する
	  for (size_t i=0;i<allMoves.size();i++)
	  {
	    if(!isSafeMove(neState,allMoves[i])) continue;
	    assert(neState.isConsistent(true));
	    NumEffectState next_state = neState;
	    ApplyMoveOfTurn::doMove(next_state, allMoves[i]);
	    assert(next_state.isConsistent(true));
	    if (!next_state.hasEffectBy(next_state.getTurn(),
					next_state.getPieceOf(num).position()))
	    {
	      CPPUNIT_ASSERT(escapeMoves.isMember(allMoves[i])
			     || (std::cerr << "state=\n" << next_state << "\nlast move=" << allMoves[i] << "\n" << escapeMoves << std::endl,0)
		);
	    }
	  }
	  // 王手の場合は生成した手は全部逃げる手
	  if(p.ptype()==KING){
	    for (size_t i=0;i<escapeMoves.size();i++){
	      assert(neState.isConsistent(true));
	      CPPUNIT_ASSERT(!escapeMoves[i].ignoreUnpromote(neState.getTurn()));
	      
	      NumEffectState next_state = neState;
	      ApplyMoveOfTurn::doMove(next_state, escapeMoves[i]);
	      CPPUNIT_ASSERT(!next_state.
			     hasEffectBy(next_state.getTurn(),
					 next_state.getPieceOf(num).position())
			     || (std::cerr << "next_state=\n" << next_state << "\nlast move=" << escapeMoves[i] << std::endl,0)
		);
	    }
	  }
	}
      }
    }
    assert(neState.isConsistent(true));
    Move move=moves[i];
    ApplyMoveOfTurn::doMove(neState, move);
    assert(neState.isConsistent(true) ||
	   (std::cerr << move << std::endl << neState << std::endl,0)
	   );
  }
}

void EscapeMovesTest::testMoveMember(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=1000;
  if (isShortTest) 
      count=10;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if (fileName == "") 
	break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testMoveFile(OslConfig::testCsaFile(fileName));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
