#include "osl/hash/boardKey.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>

using namespace osl;
using namespace osl::hash;

class BoardKeyTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(BoardKeyTest);
  // ここに(テストのための)メソッドを列挙する．
  CPPUNIT_TEST(testAccess);
  CPPUNIT_TEST(testSize);
  CPPUNIT_TEST(testEqual);
  CPPUNIT_TEST(testRandom);
  CPPUNIT_TEST(testPlayer);
  CPPUNIT_TEST(testAddSub);
  CPPUNIT_TEST_SUITE_END();
public:
  void testAccess();
  void testSize();
  void testEqual();
  void testRandom();
  void testPlayer();
  void testAddSub();
};

CPPUNIT_TEST_SUITE_REGISTRATION(BoardKeyTest);


template<typename T>
static void testAccessBoard()
{
  typedef typename T::int_t int_t;
  T key;
  for(size_t i=0;i<key.size();i++)
    key[i]=static_cast<int_t>(i);
  for(size_t i=0;i<key.size();i++)
    CPPUNIT_ASSERT_EQUAL(static_cast<int_t>(i),key[i]);
  for(size_t i=0;i<key.size();i++)
    key[i]=static_cast<int_t>(i+1);
  for(size_t i=0;i<key.size();i++)
    CPPUNIT_ASSERT_EQUAL(static_cast<int_t>(i+1),key[i]);
}

template<typename T>
static void testAccessHash()
{
  T key;
  key.setPieceStand(PieceStand(3,2,1,0,1,1,1,1));
  CPPUNIT_ASSERT_EQUAL(PieceStand(3,2,1,0,1,1,1,1),key.pieceStand());
  key.setPieceStand(PieceStand(1,1,1,1,3,2,1,0));
  CPPUNIT_ASSERT_EQUAL(PieceStand(1,1,1,1,3,2,1,0),key.pieceStand());
}

void BoardKeyTest::testAccess()
{
  testAccessBoard<BoardKey32>();
  testAccessBoard<BoardKey64>();
  testAccessHash<HashKey32>();
  testAccessHash<HashKey64>();
}

void BoardKeyTest::testSize()
{
  {
    BoardKey32 key;
    CPPUNIT_ASSERT_EQUAL(static_cast<size_t>(4),key.size());
  }
  {
    BoardKey64 key;
    CPPUNIT_ASSERT_EQUAL(static_cast<size_t>(2),key.size());
  }
  {
    HashKey32 key;
    CPPUNIT_ASSERT_EQUAL(static_cast<size_t>(4),key.size());
  }
  {
    HashKey64 key;
    CPPUNIT_ASSERT_EQUAL(static_cast<size_t>(2),key.size());
  }
}

template<class T>
static void testEqualBoard(){
    T key1;
    for(size_t i=0;i<key1.size();i++)
      key1[i]=i;
    T key2(key1);
    CPPUNIT_ASSERT(key1==key2);
    for(size_t i=0;i<key1.size();i++){
      T key3(key1);
      key3[i]=i+10;
      CPPUNIT_ASSERT(key1!=key3);
    }
}

template<class T>
static void testEqualSignature()
{
  testEqualBoard<T>();
  T key1;
  key1.setRandom();
  T key2(key1);
  CPPUNIT_ASSERT(key1==key2);
  CPPUNIT_ASSERT(key1.signature() != key2.signature()+10);
}

template<class T>
static void testEqualHash()
{
  testEqualSignature<T>();
  T key1;
  key1.setRandom();
  key1.setPieceStand(PieceStand(1,1,1,1,1,1,1,1));
  T key2(key1);
  CPPUNIT_ASSERT(key1==key2);
  key2.setPieceStand(PieceStand(1,1,1,1,1,0,1,1));
  CPPUNIT_ASSERT(key1!=key2);
}


void BoardKeyTest::testEqual()
{
  testEqualBoard<BoardKey32>();
  testEqualBoard<BoardKey64>();
  testEqualHash<HashKey32>();
  testEqualHash<HashKey64>();
}

template<class T>
static void testRandomBoard(){
  typedef typename T::int_t int_t;
  T key1;
  key1.setRandom();
  /**
   * 確率的に10000回程度で衝突が起きてしまうのは困る． 
   */
  for(int i=0;i<10000;i++){
    T key2(key1);
    key2.setRandom();
    CPPUNIT_ASSERT(key1!=key2);
    /**
     * 0ワード目の最下位ビットは0とする
     */
    CPPUNIT_ASSERT_EQUAL(static_cast<int_t>(0),(key2[0]&static_cast<int_t>(1)));
  }
  /** すべてのbitが100回変わらないことはない */
  for(size_t j=0;j<key1.size();j++){
    T key2(key1);
    int_t v=static_cast<int_t>(0);
    for(int i=0;i<100;i++){
      key2.setRandom();
      v|=key2[j]^key1[j];
    }
    if(j==0)
      CPPUNIT_ASSERT_EQUAL(static_cast<int_t>(1),~v);
    else
      CPPUNIT_ASSERT_EQUAL(static_cast<int_t>(0),~v);
  }
}

template<class T>
static void testRandomSignature(){
  testRandomBoard<T>();
  T key1;
  key1.setRandom();
  /**
   * 確率的に10000回程度で衝突が起きてしまうのは困る． 
   */
  for(int i=0;i<10000;i++){
    T key2(key1);
    key2.setRandom();
    CPPUNIT_ASSERT(key1.signature()!=key2.signature());
  }
  /** すべてのbitが100回変わらないことはない */
  T key2(key1);
  unsigned int v=0u;
  for(int i=0;i<100;i++){
    key2.setRandom();
    v|=key2.signature()^key1.signature();
  }
  CPPUNIT_ASSERT_EQUAL(0u,~v);
}

template<class T>
static void testRandomHash(){
  testRandomSignature<T>();
}

void BoardKeyTest::testRandom()
{
  testRandomBoard<BoardKey32>();
  testRandomBoard<BoardKey64>();
  testRandomHash<HashKey32>();
  testRandomHash<HashKey64>();
}

template<class T>
static void testPlayerBoard(){
  T key1;
  typedef typename T::int_t int_t;
  for(int i=0;i<100;i++){
    key1.setRandom();
    {
      T key2(key1);
      key2.setPlayer(BLACK);
      CPPUNIT_ASSERT_EQUAL(key1[0], (key2[0] & (~(int_t)0x1)));
      CPPUNIT_ASSERT(key2.isPlayerOfTurn(BLACK));
      CPPUNIT_ASSERT(!key2.isPlayerOfTurn(WHITE));
      key2.changeTurn();
      CPPUNIT_ASSERT_EQUAL(key1[0], key2[0] & (~(int_t)0x1));
      CPPUNIT_ASSERT(!key2.isPlayerOfTurn(BLACK));
      CPPUNIT_ASSERT(key2.isPlayerOfTurn(WHITE));
    }
    {
      T key2(key1);
      key2.setPlayer(WHITE);
      CPPUNIT_ASSERT_EQUAL(key1[0], key2[0] & (~(int_t)0x1));
      CPPUNIT_ASSERT(!key2.isPlayerOfTurn(BLACK));
      CPPUNIT_ASSERT(key2.isPlayerOfTurn(WHITE));
      key2.changeTurn();
      CPPUNIT_ASSERT_EQUAL(key1[0], key2[0] & (~(int_t)0x1));
      CPPUNIT_ASSERT(key2.isPlayerOfTurn(BLACK));
      CPPUNIT_ASSERT(!key2.isPlayerOfTurn(WHITE));

      key2.setPlayer(WHITE);
      CPPUNIT_ASSERT_EQUAL(key1[0], key2[0] & (~(int_t)0x1));
      CPPUNIT_ASSERT(!key2.isPlayerOfTurn(BLACK));
      CPPUNIT_ASSERT(key2.isPlayerOfTurn(WHITE));
    }
  }
}

void BoardKeyTest::testPlayer()
{
  testPlayerBoard<BoardKey32>();
  testPlayerBoard<BoardKey64>();
  testPlayerBoard<HashKey32>();
  testPlayerBoard<HashKey64>();
}

template<class T>
static void testAddSub(){
  T key1;
  for(int i=0;i<100;i++){
    {
      key1.setRandom();
      key1.setPlayer(BLACK);
      T key3(key1);
      T key2;
      key2.setRandom();
      key3+=key2;
      /**
       * たまたまsetRandomの結果が0でなければ成立
       */
      CPPUNIT_ASSERT(key1!=key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(BLACK));
      key3-=key2;
      CPPUNIT_ASSERT(key1==key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(BLACK));
    }
    {
      key1.setRandom();
      key1.setPlayer(WHITE);
      T key3(key1);
      T key2;
      key2.setRandom();
      key3+=key2;
      /**
       * たまたまsetRandomの結果が0でなければ成立
       */
      CPPUNIT_ASSERT(key1!=key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(WHITE));
      key3-=key2;
      CPPUNIT_ASSERT(key1==key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(WHITE));
    }
    {
      key1.setRandom();
      key1.setPlayer(BLACK);
      T key3(key1);
      T key2;
      key2.setRandom();
      key2.setPlayer(BLACK);
      key3+=key2;
      /**
       * たまたまsetRandomの結果が0でなければ成立
       */
      CPPUNIT_ASSERT(key1!=key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(BLACK));
      key3-=key2;
      CPPUNIT_ASSERT(key1==key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(BLACK));
    }
    {
      key1.setRandom();
      key1.setPlayer(WHITE);
      T key3(key1);
      T key2;
      key2.setRandom();
      key2.setPlayer(BLACK);
      key3+=key2;
      /**
       * たまたまsetRandomの結果が0でなければ成立
       */
      CPPUNIT_ASSERT(key1!=key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(WHITE));
      key3-=key2;
      CPPUNIT_ASSERT(key1==key3);
      CPPUNIT_ASSERT(key3.isPlayerOfTurn(WHITE));
    }
  }
}

void BoardKeyTest::testAddSub()
{
  ::testAddSub<BoardKey32>();
  ::testAddSub<BoardKey64>();
  ::testAddSub<HashKey32>();
  ::testAddSub<HashKey64>();
}


// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
