/* 
 * pam_rsa PAM-module for local authentication with RSA keypairs
 * copyright (c) 2006 Vesa-Matti Kari <hyperllama@laamanaama.helsinki.fi>
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/pem.h>
#include <openssl/evp.h>
#include <pthread.h>
#include <syslog.h>
#include <unistd.h>
#include "util.h"
#include "conf.h"
#include "sha1.h"
#include "pam_rsa.h"
#include "rsa.h"


int opensslinit = 0;
pthread_mutex_t opensslinit_mutex = PTHREAD_MUTEX_INITIALIZER;

/* IMPORTANT WARNING: Contrary to common intuitions, OpenSSL's API requires 
 * that certain pointer contents must be initialized before passing them as arguments.
 * This is a quote from pem(3) manual page: 
 *
 * --------------------------------begin quote----------------------------------
 * A frequent cause of problems is attempting to use the PEM routines like this:
 *
 *      X509 *x;
 *      PEM_read_bio_X509(bp, &x, 0, NULL);
 *
 *     this is a bug because an attempt will be made to reuse the data at x 
 *     which is an uninitialised pointer.
 * ----------------------------------end quote----------------------------------
 */


static int pass_cb(char *buf, int size, int rwflag, void *u);
static int seed_prng(int bytes);
static X509 *cert_load(const char *file);
static RSA *rsa_get(const char *file);
static RSA *rsa_priv_get(const char *file, char *passphrase, int debug);
static int rsa_encrypt(int flen, RSA *rsa, unsigned char *from, unsigned char *to);
static int pubkey_encrypt(int flen, const char *file, unsigned char *from, unsigned char *to);
static int rsa_decrypt(int flen, RSA *rsa, unsigned char *from, unsigned char *to);
static int privkey_decrypt(int flen, const char *file,
	unsigned char *from, unsigned char *to, char *passphrase, int debug);
static char *pubkey_location(struct pamrsaconf *cf, const char *user);
static char *privkey_location(struct pamrsaconf *cf, const char *user);


static int seed_prng(int bytes)
{
	/* Directories, symlinks and character devices allowed, 
     * group readable directories and files allowed,
     * world readable directories and files allowed.
     * (We allow symlinks because at least the unofficial
     * /dev/urandom OS patch for Solaris 8 creates /dev/urandom
     * as a symbolic link.
     */
	if (!is_safepath("/dev/urandom", "cdl", "RrFf")) {
		return 0;	
	}

	if (!RAND_load_file("/dev/urandom", bytes)) {
		return 0;
	}
	return 1;
}


static X509 *cert_load(const char *file)
{
	X509 *cert = NULL;
	FILE *fp;

	/* Directories and regular files allowed,
     * group readable directories and files allowed,
     * world readable directories and files allowed 
     */
	if (!is_safepath(file, "dr", "RrFf")) {
		return NULL;
	}

	if ((fp = fopen(file, "r")) == NULL) {
		pamrsa_log(LOG_NOTICE, "unable to open certificate file '%s'", file);
		return NULL;
	}

	if (PEM_read_X509(fp, &cert, NULL, NULL) == NULL) {
		pamrsa_log(LOG_ALERT, "unable to load X509 certificate from '%s'", file);
		(void)fclose(fp);
		return NULL;
	}

	if (fclose(fp) != 0) {
		X509_free(cert);
		pamrsa_log(LOG_CRIT, "unable to close certificate file '%s'", file);
		return NULL;
	}
	
	return cert;
}


static RSA *rsa_get(const char *file)
{
	X509 *cert = NULL;
	EVP_PKEY *pkey = NULL;
	RSA *rsa = NULL;
	
	if ((cert = cert_load(file)) == NULL) {
		return NULL;
	}

	if ((pkey = X509_get_pubkey(cert)) == NULL) {
		X509_free(cert);
		return NULL;
	}

	/* Public Key for encryption must be RSA */
	if ((rsa = EVP_PKEY_get1_RSA(pkey)) == NULL) {
		X509_free(cert);
		EVP_PKEY_free(pkey);
		return NULL;
	}

	EVP_PKEY_free(pkey);
	X509_free(cert);
	
	return rsa;
}


static int pass_cb(char *buf, int size, int rwflag, void *u)
{
	int len;

	if (u == NULL) {
		return 0;
	}

	if ((len = strlen((char *)u)) <= 0) {
		return 0;
	}

	if (len > size) {
		len = size;
	}
	
	memcpy(buf, (char *)u, len);
	
	return len;
}


static RSA *rsa_priv_get(const char *file, char *passphrase, int debug)
{
	FILE *fp;
	RSA *rsa = NULL;

	/* Private key: No group readable files,
     * no world readable files. Allow group and
     * world readable directories. 
     */
	if (!is_safepath(file, "dr", "RF")) {
		return NULL;
	}

	if ((fp = fopen(file, "r")) == NULL) {
		pamrsa_log(LOG_NOTICE, "unable to open private keyfile '%s'", file);
		return NULL;
	}

	if (PEM_read_RSAPrivateKey(fp, &rsa, pass_cb, passphrase) == NULL) {

		pamrsa_log(LOG_ALERT, "unable to load RSA private key from '%s'", file);

		if (debug) {
			char ebuf[512];
			int e;
			e = ERR_get_error();
			memset(ebuf, '\0', sizeof(ebuf));
			ERR_error_string_n(e, ebuf, sizeof(ebuf) - 1);
			pamrsa_log(LOG_DEBUG, "%s", ebuf);
		} 
		if (fclose(fp) != 0) {
			pamrsa_log(LOG_CRIT, "unable to close private keyfile '%s'", file);
		}
		return NULL;
	}

	if (fclose(fp) != 0) {
		RSA_free(rsa);
		pamrsa_log(LOG_CRIT, "unable to close private keyfile '%s'", file);
		return NULL;
	}
	
	return rsa;
}


static int rsa_encrypt(int flen, RSA *rsa, unsigned char *from, unsigned char *to)
{
	int r;

	/* maximum flen depends on padding type, see RSA_public_encrypt(3) */
	if ((flen >= (RSA_size(rsa) - 11))) {
		return -1;
	}

	if ((r = RSA_public_encrypt(flen, from, to, rsa, RSA_PKCS1_PADDING)) < 0) {
		return -2;
	}

	if (r != RSA_size(rsa)) {
		return -3;
	}

	return r;
}


static int pubkey_encrypt(int flen, const char *file, unsigned char *from, unsigned char *to)
{
	RSA *rsa = NULL;
	int r;

	if ((rsa = rsa_get(file)) == NULL) {
		return -1;
	}

	r = rsa_encrypt(flen, rsa, from, to);
	
	RSA_free(rsa);

	return r;
}


static int rsa_decrypt(int flen, RSA *rsa, unsigned char *from, unsigned char *to)
{
	int r;
#ifdef DEBUG
	unsigned long e;
	char ebuf[512];
#endif

	if ((r = RSA_private_decrypt(flen, from, to, rsa, RSA_PKCS1_PADDING)) < 0) {
#ifdef DEBUG
		e = ERR_get_error();
		memset(ebuf, '\0', sizeof(ebuf));
		ERR_error_string_n(e, ebuf, sizeof(ebuf) - 1);
		pamrsa_log(LOG_DEBUG, "RSA_private_decrypt(): %s", ebuf);
#endif
		return -1;
	}
	return r;
}


static int privkey_decrypt(int flen, const char *file,
	unsigned char *from, unsigned char *to, char *passphrase, int debug)
{
	RSA *rsa = NULL;
	int r;

	if ((rsa = rsa_priv_get(file, passphrase, debug)) == NULL) {
		return -1;
	}
	
	r = rsa_decrypt(flen, rsa, from, to);

	RSA_free(rsa);

	return r;
}


static char *pubkey_location(struct pamrsaconf *cf, const char *user)
{
	const char *dir;
	const char *last;
	char buf[MAXIMUM_PATH];
	char *p;

	memset(buf, '\0', sizeof(buf));

	dir = cf->pubkey_dir;

	last = dir + strlen(dir) - 1; 	
	if (*last == '/') {
		snprintf(buf, sizeof(buf) - 1, "%s%s.pem", dir, user);
	} else {
		snprintf(buf, sizeof(buf) - 1, "%s/%s.pem", dir, user);
	}

	if ((p = strdup(buf)) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
	}

	return p;
}


static char *privkey_location(struct pamrsaconf *cf, const char *user)
{
	const char *dir;
	const char *fn;
	const char *last;
	char *p;
	char buf[MAXIMUM_PATH];
	char sha1priv[8+1];
	char sha1host[8+1];	/* hostname is always hashed */
	char host[256];

	memset(buf, '\0', sizeof(buf));
	sha1priv[sizeof(sha1priv)-1] = '\0';
	sha1host[sizeof(sha1host)-1] = '\0';
	host[sizeof(host)-1] = '\0';

	dir = cf->privkey_dir;

	if (cf->privkey_name_hash == HASH_SHA1) {
		if ((sha1head(user, sha1priv, sizeof(sha1priv) - 1)) != 0) {
			pamrsa_log(LOG_ERR, "could not hash private key filename");
			return NULL;	
		}
		fn = sha1priv;
	} else {	/* HASH_NONE */
		fn = user;
	}

	if (gethostname(host, sizeof(host) - 1) != 0) { 
		pamrsa_log(LOG_CRIT, "could not determine my own hostname");
		return NULL;
	}

	if ((sha1head(host, sha1host, sizeof(sha1host) - 1)) != 0) {
		pamrsa_log(LOG_ERR, "could not hash hostname");
		return NULL;	
	}

	last = dir + strlen(dir) - 1; 	
	if (*last == '/') {
		snprintf(buf, sizeof(buf) - 1, "%s%s/%s.pem", dir, sha1host, fn);
	} else {
		snprintf(buf, sizeof(buf) - 1, "%s/%s/%s.pem", dir, sha1host, fn);
	}

	if ((p = strdup(buf)) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
	}

	return p;
}


int rsa_auth_okay(struct pamrsaconf *cf, struct pamrsaarg *parg,
	const char *user, int *authokay, char *passphrase)
{
	unsigned char *msg = (unsigned char *)"KOLLOS IS A PURE MIND WITHOUT RESTRICTIONS OF THE PHYSICAL UNIVERSE";
	char *pubkey = NULL;
	char *privkey = NULL;
	unsigned char *crypt = NULL;
	unsigned char *plain = NULL;
	int msglen, clen, plen;
	int ret;

	*authokay = 0; /* Never change this pessimistic assumption */


	if (pthread_mutex_lock(&opensslinit_mutex) != 0) {
		return -1;
	}

	if (opensslinit == 0) {
		ERR_load_crypto_strings();
		OpenSSL_add_all_algorithms();
		if (seed_prng(MAX_KEY_BITS / 8) == 0) {
			pamrsa_log(LOG_CRIT, "could not seed prng");
			if (pthread_mutex_unlock(&opensslinit_mutex) != 0) {
				pamrsa_log(LOG_CRIT, "could not unlock opensslinit_mutex");
			}
			return -1;
		}
		opensslinit = 1;
	}

	if (pthread_mutex_unlock(&opensslinit_mutex) != 0) {
		return -2;
	}


	if ((crypt = malloc(MAX_KEY_BITS / 8)) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
		return -3;		
	}

	if ((plain = malloc(MAX_KEY_BITS / 8)) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
		ret = -3;		
		goto endfree;
	}

	memset(crypt, 0, MAX_KEY_BITS / 8);
	memset(plain, 0, MAX_KEY_BITS / 8);

	if ((pubkey = pubkey_location(cf, user)) == NULL) {
		ret = -3;
		goto endfree;
	}

	if ((privkey = privkey_location(cf, user)) == NULL) {
		ret = -3;
		goto endfree;
	}

	if (parg->debug) {
		pamrsa_log(LOG_DEBUG, "user='%s' pubkey='%s' privkey='%s' ask_passphrase=%s",
			 user, pubkey, privkey,
			(passphrase)?"yes":"no");
	}

	msglen = strlen((char *)msg) + 1;


	if ((clen = pubkey_encrypt(msglen, pubkey, msg, crypt)) < 0) {
		if (parg->debug) {
			pamrsa_log(LOG_DEBUG, "encrypting with '%s' failed: %d",
				pubkey, clen);
		}
		ret = -4;
		goto endfree;
	}

	if (parg->debug) {
		pamrsa_log(LOG_DEBUG, "encrypting with '%s' succeeded", pubkey);
	}

	if ((plen = privkey_decrypt(clen, privkey, crypt, plain, passphrase, parg->debug)) < 0) {
		if (parg->debug) {
			pamrsa_log(LOG_DEBUG, "decrypting with '%s' failed: %d: %s",
				privkey, plen);
		}
		ret = -5;
		goto endfree;
	}

	if (parg->debug) {
		pamrsa_log(LOG_DEBUG, "decrypting with '%s' succeeded", privkey);
	}

	if (!strcmp((char *)msg, (char *)plain)) {
		*authokay = 1;
		ret = 0;
	} 

/* some of these may be NULL if goto brought us here, 
   but that is perfectly okay: free(NULL) is safe [see ANSI/ISO C standard(s)] */
endfree:
	free(pubkey);
	free(privkey);
	free(crypt);
	free(plain);

	return ret;
}
