#ifndef S11N_NS_ABSTRACT_CREATOR_H_INCLUDED
#define S11N_NS_ABSTRACT_CREATOR_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// abstract_creatore.h:
// Defines a difficult-to-describe class. See the docs.
//
//    Author: stephan beal <stephan@s11n.net>
//    License: Public Domain
////////////////////////////////////////////////////////////////////////

#include <cassert>
#ifdef NDEBUG
#  undef NDEBUG // force assert()
#endif

#include <string>
#include "s11n_core.h" // classload
#include <S11N_NS/class_name.h>

namespace S11N_NS
{

        /**
           abstract_creator is a helper to avoid some code having to
           know if a type is created on a stack or the heap. This
           makes some template code easier to write, as it avoids
           syntax errors when trying something like:

<pre>
if( object type is a pointer type ) delete(object);
else { ... }
</pre>

           This implementation creates items the stack via the default
           ctor. If instantiated with (T *) a pointer/heap-based
           specialization is activated instead.

           Designed for use with, e.g., ListType::value_type, which
           will may be, e.g., either T or (T *).

           These objects contain no state information.
        */
        template <typename T>
        struct abstract_creator
        {
                /** Same as (T). */
                typedef T value_type;

                /** Same as (T). */
                typedef T base_value_type;

                /**
                   This implementation does nothing and returns true.

                   The string argument is bogus for this implementation,
                   and is used by the pointer specialization to implement
                   polymorphic classloading of value_type.
                */
                static bool create( value_type & v,
                                    const std::string & = std::string()  )
                {
                        return true;
                }

                /**
                   Assigs lhs to rhs and true.
                */
                static void assign( value_type &lhs, const value_type & rhs )
                {
                        lhs = rhs;
                        return;
                }

                /**
                   Returns a copy of.
                */
                static value_type copy( const value_type & rhs )
                {
                        return rhs;
                }

                /**
                   Does nothing in this specialization. Specializations
                   which allocate resources are expected to release them here.

                   Note that release() exists because using a static boolean
                   to flag a clean-up mode won't work:

<pre>
typedef double T;
T val;
if( val is a pointer type ) { delete( val ); }
else ...
</pre>

                   That will only work when T is actually a pointer type.
                   Thus the following workaround:

<pre>
typedef [some type] T;
T val;
typedef abstract_creator&lt;T&gt; AC;
assert( AC::create(val) ); // fine for (T) or (T *)
... use val ...
AC::release(val); // deletes pointers. No-op for value types
</pre>
		That works with pointers or non-pointers, and simply does nothing
                for non-pointers.

                See object_reference_wrapper for a type which can wrap
                function calls to objects using the dot notation,
                regardless of their pointerness.
                 */
                static void release( value_type & )
                {
                        return;
                }

        }; // abstract_creator<T>

        /**
           A specialization of abstract_creator to create objects on
           the heap, using the s11n classloader.
        */
        template <typename T>
        struct abstract_creator<T *>
        {
                /** Same as (T *). */
                typedef T * value_type;
 
               /** Same as (T). */
                typedef T base_value_type;

                /**
                   Tries to create a value_type object,
                   using classload&lt;base_value_type&gt;( key )
                   to create it. v is assigned to it's value, which
                   may be 0. Returns true if an object is created,
                   else false.

                   The caller owns the returned object.

                   Maintenance note: new classloader registrations may
                   need to be installed as new types show up,
                   especially for streamables/PODs (because those
                   aren't normally registered as classes), or this
                   function won't handle them. In s11n this
                   registration is handled by most of the various
                   proxy installation macros.
                */
                static bool create(  value_type & v,
                                     const std::string & implclass = std::string() /* guess! */  )
                {
                        const std::string key = 
                                (! implclass.empty())
                                ? implclass
                                : ::classname<value_type>();

                        v = classload<base_value_type>( key );
                        //CERR << "create("<<key<<")=" <<std::hex<<v <<"\n";
                        return 0 != &v;
                }

                /**
                   Creates a MONOMORPHIC copy of rhs and returns it.

                   This copy is only useful for non-polymorphic
                   types.
                */
                static value_type copy( const value_type & rhs )
                {
                        return new base_value_type(rhs);
                }

                /**
                   Returns rhs.
                */
                static void assign( value_type &lhs, const value_type & rhs )
                {
                        lhs = rhs;
                        return;
                }


                /**
                   Deletes v and assigns it to 0.
                 */
                static void release( value_type & v )
                {
                        //CERR << "release(" << std::hex<<v<<")\n";
                        delete( v );
                        v = 0;
                }
        }; // abstract_creator<T *>


} // namespace

#endif // S11N_NS_ABSTRACT_CREATOR_H_INCLUDED

