/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

import java.sql.*;
import java.math.BigDecimal;
import java.io.IOException;

import com.sap.dbtech.util.*;
import com.sap.dbtech.jdbc.exceptions.*;
/**
 *
 */
class StringTranslator
    extends CharDataTranslator
{
    /**
     * @param mode
     * @param ioType
     * @param dataType
     * @param len
     * @param ioLen
     * @param bufpos
     * @param readOnly
     * @param autoIncr
     */
    StringTranslator(int mode, int ioType, int dataType, int len, int ioLen,
            int bufpos, boolean readOnly, boolean autoIncr) {
        super(mode, ioType, dataType, len, ioLen, bufpos, readOnly, autoIncr);
    }
    /**
     *
     * @return java.lang.String
     */
    public Object getObject (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.getString (controller, mem);
    }
    /**
     *
     * @return BigDecimal
     */
    public BigDecimal getBigDecimal (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        String value = this.getString (controller, mem);
        BigDecimal result = null;

        if (value == null) {
            // throw this.newGetException ("(null)");
            result = new BigDecimal (0);
        }
        else {
            try {
                result = new BigDecimal (value);
            }
            catch (NumberFormatException exc) {
              try {
                result = new BigDecimal (value.trim());
              }
              catch (NumberFormatException ex) {
                throw this.newParseException (value, "BigDecimal");
              }
            }
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public boolean getBoolean (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        String value = this.getString (controller, mem);
        if (value == null) {
            // throw this.newGetException ("(null)");
            return false;
        }
        Boolean parsedBool = Boolean.valueOf (value);
        if (parsedBool.booleanValue ()) {
            return true;
        }
        else {
            /* try parsing as number */
            try {
                double doubleVal = Double.parseDouble (value);
                return (doubleVal != 0.0);
            }
            catch (NumberFormatException exc) {
                return false;
            }
        }
    }

    /**
     *
     * @return byte
     */
    public byte[] getBytes (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        String result = this.getString(controller,mem);
        if (result != null)
          return result.getBytes();
        else
          return null;
    }
    /**
     *
     * @return byte
     */
    public byte getByte (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Byte (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return java.io.Reader
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.Reader getCharacterStream (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData) throws SQLException
    {
        String asString = this.getString (controller, mem);
        java.io.Reader result;

        if (asString == null) {
            return null;
        }
        result = new java.io.StringReader (asString);
        return result;
    }
    /**
     *
     * @return double
     */
    public double getDouble (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Double (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return float
     */
    public float getFloat (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Float (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return int
     */
    public int getInt (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Int (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return long
     */
    public long getLong (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Long (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return short
     */
    public short getShort (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.bigDecimal2Short (this.getBigDecimal (controller, mem));
    }
    /**
     *
     * @return java.lang.String
     */
    public String getString (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        String result = null;

        if (!this.isNull (controller, mem)) {
            result = mem.getStrippedString (this.bufpos, this.logicalLength);
        }
        return result;
    }
    /**
     *
     * @return byte[]
     */
    public Object transBytesForInput (
            byte [] val) throws SQLException
    {
        if (val == null) {
            return null;
        }
        else {
            return this.transStringForInput (new String (val));
        }
    }

    public Object transCharacterStreamForInput(java.io.Reader stream, int length)
        throws SQLException
    {
        if(length <=0) {
            return null;
        }
        try {
            char[] ba=new char[length];
            int r=stream.read(ba);
            if(r!=length) {
                if(r==-1) r=0;
                char[] ba2=ba;
                ba=new char[r];
                System.arraycopy(ba2, 0, ba, 0, r);
            }
            return transStringForInput(new String(ba));
        } catch(IOException ioex) {
            throw new SQLExceptionSapDB(ioex.getMessage());
        }
    }


    public Object transBinaryStreamForInput(java.io.InputStream stream, int length)
        throws SQLException
    {
        if(length <=0) {
            return null;
        }
        try {
            byte[] ba=new byte[length];
            int r=stream.read(ba);
            if(r!=length) {
                if(r==-1) r=0;
                byte[] ba2=ba;
                ba=new byte[r];
                System.arraycopy(ba2, 0, ba, 0, r);
            }
            return transBytesForInput(ba);
        } catch(IOException ioex) {
            throw new SQLExceptionSapDB(ioex.getMessage());
        }
    }

    /**
     *
     * @param obj java.lang.Object
     * @exception java.sql.SQLException The exception description.
     */
    protected Object transSpecificForInput (
        Object obj)
    throws SQLException
    {
        // conversion to string handled by super.putObject ()
        return null;
    }

    /**
     * Performs specific string checks for string insert (length check).
     * (The string is not inserted here, but will be used unmodified on
     *  packet creation later).
     * @param arg the String to insert
     * @return <code>arg</code> unmodified.
     */
    public Object transStringForInput (String arg)
    throws SQLException
    {
        if (arg == null) {
            return null;
        }
        byte [] bytes = StringUtil.bytes_iso8859_1(arg);
        this.checkFieldLimits (bytes.length);
        return arg;
    }
    
    public Clob getClob(SQLParamController controller, StructuredMem mem, StructuredMem longData) throws SQLException {
        Clob result = null;
        String stringResult = this.getString(controller,mem); 
        if (stringResult != null) {
            result = new GetvalChar(stringResult);
        }
        return result;
    }
}
