#include "TePDIContrast.hpp"

#include "TePDIAgnostic.hpp"
#include "TePDITypes.hpp"
#include "TePDIUtils.hpp"

#include <sstream>
#include <cmath>


TePDIContrast::TePDIContrast( const TePDIParameters& params )
: TePDIAlgorithm( params )
{
}

TePDIContrast::TePDIContrast()
{
}


TePDIContrast::~TePDIContrast()
{
}


bool TePDIContrast::RunImplementation()
{
  /* Getting general parameters */

  int contrast_type;
  params_.GetParameter( "contrast_type", contrast_type );

  TePDITypes::TePDIRasterPtrType inRaster;
  params_.GetParameter( "input_image", inRaster );

  std::vector< int > channels;
  params_.GetParameter( "channels", channels );

  int histo_levels;
  params_.GetParameter( "histo_levels", histo_levels );

  TePDITypes::TePDIRasterPtrType outRaster;
  params_.GetParameter( "output_image", outRaster );

  /* Generating histograms */

  BuildHistograms( inRaster, (unsigned int)histo_levels, channels );

  /* Processing each algorithm */

  TePDITypes::TePDILutType lut;

  switch( contrast_type ) {
    case TePDIContrastMinMax :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetMinMaxLut( histo_cache_[ channels[ channels_index ] ] );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );
      }

      break;
    }
    case TePDIContrastLinear :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      double min_level = 0;
      double max_level = 0;

      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetLinearLut( histo_cache_[ channels[ channels_index ] ],
          min_level, max_level );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );
      }

      break;
    }
    case TePDIContrastSquareRoot :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      double min_level = 0;
      double max_level = 0;

      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetSquareRootLut( histo_cache_[ channels[ channels_index ] ],
          min_level, max_level );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );
      }

      break;
    }
    case TePDIContrastSquare :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      double min_level = 0;
      double max_level = 0;

      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetSquareLut( histo_cache_[ channels[ channels_index ] ],
          min_level, max_level );
        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );
      }

      break;
    }
    case TePDIContrastLog :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      double min_level = 0;
      double max_level = 0;

      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetLogLut( histo_cache_[ channels[ channels_index ] ],
          min_level, max_level );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );
      }

      break;
    }
    case TePDIContrastNegative :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      double min_level = 0;
      double max_level = 0;

      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetNegativeLut( histo_cache_[ channels[ channels_index ] ],
          min_level, max_level );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );

      }

      break;
    }
    case TePDIContrastHistEqualizer :
    {
      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, channels.size(),
        inRaster->params().nlines_, inRaster->params().ncols_ ),
        "Raster reset error" );

      for( unsigned int channels_index = 0 ;
           channels_index < channels.size() ;
           ++channels_index ) {

        lut = GetHistEqualizerLut( histo_cache_[ channels[ channels_index ] ] );

        RemapLevels( inRaster, lut, channels[ channels_index ],
          channels_index, outRaster );

      }

      break;
    }
    case TePDIContrastSimpleSlicer :
    {
      /* Getting user parameters */

      TePDIRgbPalette::pointer palette;
      params_.GetParameter( "rgb_palette", palette );

      double min_level = 0;
      double max_level = 0;
      params_.GetParameter( "min_level", min_level );
      params_.GetParameter( "max_level", max_level );

      PDIAGN_TRUE_OR_RETURN(
        TePDIUtils::TeResetRaster( outRaster, 1,
        inRaster->params().nlines_, inRaster->params().ncols_,
        palette.NakedPointer() ), "Raster reset error" );

      /* Slicing process */

      GetSimpleSlicerLut( histo_cache_[ channels[ 0 ] ],
        palette,  min_level, max_level, lut );

      RemapLevels( inRaster, lut, channels[ 0 ], 0,
        outRaster );

      break;
    }
    default :
    {
      PDIAGN_NOT_IMPLEMENTED;
      break;
    }
  }


  return true;
}


bool TePDIContrast::CheckParameters( const TePDIParameters& parameters )
{
  /* Checking for general required parameters */

  TePDITypes::TePDIRasterPtrType inRaster;
  if( ! parameters.GetParameter( "input_image", inRaster ) ) {

    PDIAGN_LOGERR( "Missing parameter: input_image" );
    return false;
  }
  if( ! inRaster.isActive() ) {

    PDIAGN_LOGERR( "Invalid parameter: input_image inactive" );
    return false;
  }
  if( inRaster->status() == TeNOTREADY ) {

    PDIAGN_LOGERR( "Invalid parameter: input_image not ready" );
    return false;
  }

  TePDITypes::TePDIRasterPtrType outRaster;
  if( ! parameters.GetParameter( "output_image", outRaster ) ) {

    PDIAGN_LOGERR( "Missing parameter: output_image" );
    return false;
  }
  if( ! outRaster.isActive() ) {

    PDIAGN_LOGERR( "Invalid parameter: output_image inactive" );
    return false;
  }
  if( inRaster->status() == TeREADYTOWRITE ) {

    PDIAGN_LOGERR( "Invalid parameter: output_image not ready" );
    return false;
  }

  int histo_levels = 0;
  if( ! parameters.GetParameter( "histo_levels", histo_levels ) ||
      ( histo_levels <= 0 ) ) {

    PDIAGN_LOGERR( "Invalid parameter: histo_levels" );
    return false;
  }

  /* Checking for the correct allowed contrast types */

  int contrast_type;
  if( ! parameters.GetParameter( "contrast_type", contrast_type ) ) {
    PDIAGN_LOGERR( "Missing parameter: contrast_type" );
    return false;
  }
  PDIAGN_CHECK_NOTEQUAL( contrast_type, 0 ,
    "Invalid Contrast Type" );
  if( ( contrast_type != TePDIContrastMinMax ) &&
      ( contrast_type != TePDIContrastLinear ) &&
      ( contrast_type != TePDIContrastSquareRoot ) &&
      ( contrast_type != TePDIContrastSquare ) &&
      ( contrast_type != TePDIContrastLog ) &&
      ( contrast_type != TePDIContrastNegative ) &&
      ( contrast_type != TePDIContrastHistEqualizer ) &&
      ( contrast_type != TePDIContrastSimpleSlicer ) ) {

    PDIAGN_LOGERR( "Invalid parameter: contrast_type" );
    return false;
  }

  /* channels parameter checking */

  std::vector< int > channels;
  if( ! parameters.GetParameter( "channels", channels ) ) {

    PDIAGN_LOGERR( "Missing parameter: channels" );
    return false;
  }
  for( unsigned int index = 0 ; index < channels.size() ; ++index ) {
    if( channels[ index ] >= inRaster->nBands() ) {
      PDIAGN_LOGERR( "Invalid parameter: channels" );
      return false;
    }
  }
  if( ( contrast_type == TePDIContrastSimpleSlicer ) ) {
    if( channels.size() != 1 ) {
      PDIAGN_LOGERR( "Invalid channels number" );
      return false;
    }
  } else {
    /* For the other algorithms */
    if( channels.size() == 0 ) {
      PDIAGN_LOGERR( "Invalid channels number" );
      return false;
    }
  }

  /* Checking for min and max required parameters */

  if( ( contrast_type == TePDIContrastLinear ) ||
      ( contrast_type == TePDIContrastSquareRoot ) ||
      ( contrast_type == TePDIContrastSquare ) ||
      ( contrast_type == TePDIContrastLog ) ||
      ( contrast_type == TePDIContrastNegative ) ||
      ( contrast_type == TePDIContrastSimpleSlicer ) ) {

    if( ! parameters.CheckParameter( "min_level",
          TePDIParameters::TePDIDoubleParam ) ) {

      PDIAGN_LOGERR( "Missing parameter: min_level" );
      return false;
    }
    if( ! parameters.CheckParameter( "max_level",
          TePDIParameters::TePDIDoubleParam ) ) {

      PDIAGN_LOGERR( "Missing parameter: max_level" );
      return false;
    }
  }

  /* Checking for RGB Palette required parameters */

  if( ( contrast_type == TePDIContrastSimpleSlicer ) ) {
    TePDIRgbPalette::pointer rgb_palette;

    if( ( ! parameters.GetParameter( "rgb_palette", rgb_palette ) ) ||
        ( ! rgb_palette.isActive() ) ) {

      PDIAGN_LOGERR( "Missing parameter: rgb_palette" );
      return false;
    }
  }

  return true;
}


void TePDIContrast::ResetState()
{
  histo_cache_.clear();
}


void TePDIContrast::BuildHistograms(
  TePDITypes::TePDIRasterPtrType& inRaster,
  unsigned int histo_levels,
  std::vector< int >& channels,
  bool force )
{
  PDIAGN_TRUE_OR_THROW( inRaster.isActive(),
    "inRaster inactive" );
  PDIAGN_TRUE_OR_THROW( inRaster->status() != TeNOTREADY,
    "inRaster not ready" );

  for( unsigned int channels_index = 0 ; channels_index < channels.size() ;
       ++channels_index ) {

    PDIAGN_TRUE_OR_THROW( channels[ channels_index ] < inRaster->nBands(),
      "Trying to creat histogram from an invalid band" );

    bool hist_cached =
      ( histo_cache_.find( channels[ channels_index ] ) !=
        histo_cache_.end() ) ? true : false;

    if( ( ! hist_cached ) || force ) {
      if( hist_cached ) {
        histo_cache_[ channels[ channels_index ] ].clear();
      }

      TePDITypes::TePDIHistogramType temp_hist;

      PDIAGN_TRUE_OR_THROW( TePDIUtils::TeGenerateHistogram( inRaster,
        histo_levels, channels[ channels_index ], temp_hist ),
        "Histogram Generation Error" );

      PDIAGN_CHECK_EQUAL( temp_hist.size(), histo_levels,
        "Generated histogram size error" );

      histo_cache_[ channels[ channels_index ] ] = temp_hist;
    }
  }
}


void TePDIContrast::RemapLevels(
  TePDITypes::TePDIRasterPtrType& inRaster,
  TePDITypes::TePDILutType& lut,
  int in_channel,
  int out_channel,
  TePDITypes::TePDIRasterPtrType& outRaster )
{
  PDIAGN_TRUE_OR_THROW( inRaster.isActive(),
    "inRaster inactive" );
  PDIAGN_TRUE_OR_THROW( outRaster.isActive(),
    "outRaster inactive" );
  PDIAGN_TRUE_OR_THROW( inRaster->status() != TeNOTREADY,
    "inRaster not ready" );
  PDIAGN_TRUE_OR_THROW( outRaster->status() == TeREADYTOWRITE,
    "outRaster not ready" );
  PDIAGN_CHECK_EQUAL( inRaster->params().nlines_,
    outRaster->params().nlines_,
    "Lines number mismatch between input and output image" );
  PDIAGN_CHECK_EQUAL( inRaster->params().ncols_,
    outRaster->params().ncols_,
    "Columns number mismatch between input and output image" );

  int raster_lines = inRaster->params().nlines_;
  int raster_columns = inRaster->params().ncols_;

  double current_level;
  double best_lut_reference;

  TePDITypes::TePDILutType::iterator lut_it;
  TePDITypes::TePDILutType::iterator lut_next_it;
  TePDITypes::TePDILutType::iterator lut_it_end = lut.end();

  for( int line = 0 ; line < raster_lines ; ++line ) {
    for( int column = 0 ; column < raster_columns ; ++column ) {
      if( inRaster->getElement( column, line, current_level,
          in_channel ) ) {

        /* Finding mapped level by using the best LUT reference Level */

        lut_it = lut.begin();
        lut_next_it = lut_it;
        ++lut_next_it;

        while( lut_next_it != lut_it_end ) {
          if( lut_next_it->first >= current_level ) {
            break;
          }

          ++lut_it;
          ++lut_next_it;
        }

        if( lut_next_it == lut_it_end ) {
          best_lut_reference = lut_it->first;
        } else {
          if( std::abs( lut_it->first - current_level ) <
              std::abs( lut_next_it->first - current_level ) ) {

            best_lut_reference = lut_it->first;
          } else {
            best_lut_reference = lut_next_it->first;
          }
        }

        /* Pixel Output level mapping */

        PDIAGN_TRUE_OR_THROW( outRaster->setElement( column, line,
          lut[ best_lut_reference ], out_channel ),
          "Level remmaping error at " + TePDIUtils::to_string( line ) +
          "," + TePDIUtils::to_string( column ) );
      }
    }
  }

}

double TePDIContrast::GetHistMax( TePDITypes::TePDIHistogramType& hist )
{
  PDIAGN_TRUE_OR_THROW( hist.size() != 0, "Invalid histogram" );

  TePDITypes::TePDIHistogramType::reverse_iterator hist_it = hist.rbegin();
  TePDITypes::TePDIHistogramType::reverse_iterator hist_it_end = hist.rend();

  while( hist_it != hist_it_end ) {
    if( hist_it->second != 0 ) {
      return hist_it->first;
    }

    ++hist_it;
  }

  PDIAGN_LOG_AND_THROW( "Empty invalid Histogram" );

  return 0;
}

double TePDIContrast::GetHistMin( TePDITypes::TePDIHistogramType& hist )
{
  PDIAGN_TRUE_OR_THROW( hist.size() != 0, "Invalid histogram" );

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->second != 0 ) {
      return hist_it->first;
    }

    ++hist_it;
  }

  PDIAGN_LOG_AND_THROW( "Empty invalid Histogram" );

  return 0;
}


TePDITypes::TePDILutType TePDIContrast::GetMinMaxLut(
  TePDITypes::TePDIHistogramType& hist )
{
  PDIAGN_TRUE_OR_THROW( hist.size() != 0, "Invalid histogram" );

  return GetLinearLut( hist, GetHistMin( hist ), GetHistMax( hist ) );
}


TePDITypes::TePDILutType TePDIContrast::GetLinearLut(
  TePDITypes::TePDIHistogramType& hist,
  double min, double max )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( max > min, "Invalid max and min values" );

  /* Calculating parameters */

  unsigned int levels = hist.size();

  double a = 0;
  double b = 0;

  if( max == min ) {
    a = (double)levels;
    b = -1. * ((double)levels) * min;
  } else {
    a = ((double)levels) / ( max - min );
    b = ( -1. * ((double)levels) * min ) / ( max - min );
  }

  /* Generating LUT map using the existing histogram levels */

  TePDITypes::TePDILutType out_lut;

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->first <= min ) {
      out_lut[ hist_it->first ] = min;
    } else if( hist_it->first >= max ) {
      out_lut[ hist_it->first ] = max;
    } else {
      out_lut[ hist_it->first ] = ( a * hist_it->first ) + b;
    }

    ++hist_it;
  }

  return out_lut;
}


TePDITypes::TePDILutType TePDIContrast::GetSquareRootLut(
  TePDITypes::TePDIHistogramType& hist,
  double min, double max )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( min < max, "Invalid min and max values" );

  unsigned int levels = hist.size();

  double factor = ((double)levels) / std::sqrt( max - min );

  /* Generating LUT map using the existing histogram levels */

  TePDITypes::TePDILutType out_lut;

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->first <= min ) {
      out_lut[ hist_it->first ] = min;
    } else if( hist_it->first >= max ) {
      out_lut[ hist_it->first ] = max;
    } else {
      out_lut[ hist_it->first ] = factor * std::sqrt( hist_it->first - min );
    }

    ++hist_it;
  }

  return out_lut;
}


TePDITypes::TePDILutType TePDIContrast::GetSquareLut(
  TePDITypes::TePDIHistogramType& hist,
  double min, double max )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( min < max, "Invalid min and max values" );

  unsigned int levels = hist.size();

  double factor = ((double)levels) / std::pow( (max - min), 2 );

  /* Generating LUT map using the existing histogram levels */

  TePDITypes::TePDILutType out_lut;

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->first <= min ) {
      out_lut[ hist_it->first ] = min;
    } else if( hist_it->first >= max ) {
      out_lut[ hist_it->first ] = max;
    } else {
      out_lut[ hist_it->first ] =
        factor * std::pow( hist_it->first - min, 2 );
    }

    ++hist_it;
  }

  return out_lut;
}


TePDITypes::TePDILutType TePDIContrast::GetLogLut(
  TePDITypes::TePDIHistogramType& hist,
  double min, double max )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( max > ( min+1 ), "Invalid min and max values" );

  unsigned int levels = hist.size();

  double factor = ((double)levels) / std::log10( max - min + 1 );

  /* Generating LUT map using the existing histogram levels */

  TePDITypes::TePDILutType out_lut;

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->first <= min ) {
      out_lut[ hist_it->first ] = min;
    } else if( hist_it->first >= max ) {
      out_lut[ hist_it->first ] = max;
    } else {
      out_lut[ hist_it->first ] =
        factor * std::log10( hist_it->first - min + 1 );
    }

    ++hist_it;
  }

  return out_lut;
}


TePDITypes::TePDILutType TePDIContrast::GetNegativeLut(
  TePDITypes::TePDIHistogramType& hist,
  double min, double max )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( max > min, "Invalid max and min values" );

  /* Calculating parameters */

  unsigned int levels = hist.size();

  double a = -1. * ((double)levels) / ( max - min );
  double b = ( ((double)levels) * max ) / ( max - min );

  /* Generating LUT map using the existing histogram levels */

  TePDITypes::TePDILutType out_lut;

  TePDITypes::TePDIHistogramType::iterator hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator hist_it_end = hist.end();

  while( hist_it != hist_it_end ) {
    if( hist_it->first <= min ) {
      out_lut[ hist_it->first ] = max;
    } else if( hist_it->first >= max ) {
      out_lut[ hist_it->first ] = min;
    } else {
      out_lut[ hist_it->first ] = ( a * hist_it->first ) + b;
    }

    ++hist_it;
  }

  return out_lut;
}

TePDITypes::TePDILutType TePDIContrast::GetHistEqualizerLut(
  TePDITypes::TePDIHistogramType& hist )
{
  PDIAGN_TRUE_OR_THROW( hist.size() > 1, "Invalid histogram size" );

  TePDITypes::TePDIHistogramType::iterator in_hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator in_hist_it_past = hist.begin();
  TePDITypes::TePDIHistogramType::iterator in_hist_it_end = hist.end();

  int hist_population = 0;

  /* Generating the accumulative histogram */

  TePDITypes::TePDIHistogramType accumulative_hist;

  accumulative_hist[ in_hist_it->first ] = in_hist_it->second;
  hist_population += in_hist_it->second;
  ++in_hist_it;

  TePDITypes::TePDIHistogramType::iterator accumulative_hist_it =
    accumulative_hist.begin();
  TePDITypes::TePDIHistogramType::iterator accumulative_hist_it_past =
    accumulative_hist.begin();
  ++accumulative_hist_it;

  while( in_hist_it != in_hist_it_end ) {
    hist_population += in_hist_it->second;

    /* Building accumulative histogram */

    accumulative_hist[ in_hist_it->first ] =
      accumulative_hist_it_past->second + in_hist_it->second;

    ++accumulative_hist_it_past;
    ++accumulative_hist_it;

    ++in_hist_it_past;
    ++in_hist_it;
  }

  /* Creating the look-up table */

  double levels = (double)hist.size();

  TePDITypes::TePDILutType out_lut;

  in_hist_it = hist.begin();

  while( in_hist_it != in_hist_it_end ) {
    out_lut[ in_hist_it->first ] = accumulative_hist[ in_hist_it->first ] *
      levels / ((double)hist_population);

    ++in_hist_it;
  }

  return out_lut;
}


void TePDIContrast::GetSimpleSlicerLut(
  TePDITypes::TePDIHistogramType& hist,
  TePDIRgbPalette::pointer in_palette,
  double min,
  double max,
  TePDITypes::TePDILutType& out_lut )
{
  PDIAGN_CHECK_NOTEQUAL( hist.size(), 0, "Invalid histogram size" );
  PDIAGN_TRUE_OR_THROW( max > min, "Invalid max and min values" );
  PDIAGN_TRUE_OR_THROW( in_palette->size() > 0,
    "Invalid input palette size" );

  out_lut.clear();

  /* Extracting palette levels */

  std::vector< double > palette_levels;
  TePDIRgbPalette::iterator pal_it = in_palette->begin();
  TePDIRgbPalette::iterator pal_it_end = in_palette->end();

  while( pal_it != pal_it_end ) {
    palette_levels.push_back( pal_it.level() );

    ++pal_it;
  }

  /* min and max adjusting to the supplied histogram range */

  double in_hist_max = GetHistMax( hist );
  double in_hist_min = GetHistMin( hist );

  min = ( min < in_hist_min ) ? in_hist_min : min;
  max = ( max > in_hist_max ) ? in_hist_max : max;

  /* Output LUT generation */

  double slice_size = ( max - min ) / ((double)in_palette->size());

  double first_pal_level = palette_levels[ 0 ];
  double last_pal_level = palette_levels[ palette_levels.size() - 1 ];

  TePDITypes::TePDIHistogramType::iterator in_hist_it = hist.begin();
  TePDITypes::TePDIHistogramType::iterator in_hist_it_end = hist.end();

  double current_level;
  unsigned int current_slice;

  while( in_hist_it != in_hist_it_end ) {
    current_level = in_hist_it->first;

    if( current_level < min ) {
      out_lut[ current_level ] = first_pal_level;
    } else if ( current_level > max ) {
      out_lut[ current_level ] = last_pal_level;
    } else {
      current_slice =
        (unsigned int) std::floor( ( current_level - min ) / slice_size );

      out_lut[ current_level ] = palette_levels[ current_slice ];
    }

    ++in_hist_it;
  }


}
