/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

/**
 *   This file contains the class for RGB Palettes.
 */

#ifndef TEPDIRGBPALETTE_HPP
  #define TEPDIRGBPALETTE_HPP

  #include "TePDISharedPtr.cpp"

  #include <map>
  #include <cctype>

  class TePDIRgbPalette{
    public :
      typedef TePDISharedPtr< TePDIRgbPalette > pointer;
      typedef const TePDISharedPtr< TePDIRgbPalette > const_pointer;

      /**
       * Default Constructor
       *
       */
      TePDIRgbPalette();

      /**
       * Default Destructor
       */
      virtual ~TePDIRgbPalette();

      /**
       * Stores RGB values associated with a paletted level.
       *
       * @param level Palette Level.
       * @param red Red Level.
       * @param green Green Level.
       * @param blue Blue Level.
       */
      void set( double level, unsigned int red, unsigned int green,
        unsigned int blue );

      /**
       * Retrives Stored RGB values associated with a paletted level.
       *
       * @param level Palette Level.
       * @param red Red Level.
       * @param green Green Level.
       * @param blue Blue Level.
       *
       * @return true if the paletted level was found, false on error.
       */
      bool get( double level, unsigned int& red, unsigned int& green,
        unsigned int& blue ) const;

      /**
       * Clears the entire palette.
       */
      void clear();

      /**
       * The current palette size.
       *
       * @return The current palette size.
       */
      unsigned long int size() const;

      /**
       * Operator = overload.
       *
       * @param external_reference A external palette reference.
       */
      void operator=( TePDIRgbPalette& external_reference );

      /**
       * Automatic Level Based Palette creation.
       *
       * @param levels The number of levels.
       * @return The generated palette.
       */
      static TePDISharedPtr< TePDIRgbPalette > createLB( unsigned int levels );

      /**
       * Palette iterator.
       */
      class iterator{
        public :
          /**
           * Default Constructor.
           *
           * @param red_it Red channel iterator.
           * @param green_it Green channel iterator.
           * @param blue_it Blue channel iterator.
           */
          explicit iterator(
            const std::map< double, unsigned int >::iterator& red_it,
            const std::map< double, unsigned int >::iterator& green_it,
            const std::map< double, unsigned int >::iterator& blue_it );

          /**
           * Default Destructor.
           */
          ~iterator();

          /**
           * Operator ++ overload.
           */
          void operator++();

          /**
           * Operator -- overload.
           */
          void operator--();

          /**
           * Operator = overload.
           *
           * @param ext_ref External iterator reference.
           */
          void operator=( const iterator& ext_ref );

          /**
           * Operator != overload.
           *
           * @param ext_ref External iterator reference.
           * @return true if both iterators are pointing to the same
           * position.
           */
          bool operator!=( const iterator& ext_ref );

          /**
           * The current iterator pointed level.
           *
           * @return The current iterator pointed level.
           */
          double level() const;

          /**
           * The current iterator red channel value.
           *
           * @return The current iterator red channel value.
           */
          unsigned int red() const;

          /**
           * The current iterator green channel value.
           *
           * @return The current iterator green channel value.
           */
          unsigned int green() const;

          /**
           * The current iterator blue channel value.
           *
           * @return The current iterator blue channel value.
           */
          unsigned int blue() const;

        protected :
          /**
           * Internal red map iterator;
           */
          mutable std::map< double, unsigned int >::iterator red_it_;

          /**
           * Internal green map iterator;
           */
          mutable std::map< double, unsigned int >::iterator green_it_;

          /**
           * Internal blue map iterator;
           */
          mutable std::map< double, unsigned int >::iterator blue_it_;
      };

      /**
       * Iterator to the first element.
       *
       * @return An Iterator to the first element.
       */
      iterator begin() const;

      /**
       * Iterator pointing to the position after the
       * last element.
       *
       * @return The iterator.
       */
      iterator end() const;

    protected :
      /**
       * Internal Red channel map.
       *
       * @param double Paletted Level.
       * @param unsigned int Relative Red Level.
       */
      mutable std::map< double, unsigned int > red_map_;

      /**
       * Internal Green channel map.
       *
       * @param double Paletted Level.
       * @param unsigned int Relative Green Level.
       */
      mutable std::map< double, unsigned int > green_map_;

      /**
       * Internal Blue channel map.
       *
       * @param double Paletted Level.
       * @param unsigned int Relative Blue Level.
       */
      mutable std::map< double, unsigned int > blue_map_;
  };

#endif //TEPDIRGBPALETTE_HPP
