/************************************************************************************
TerraView - visualization and exploration of geographical databases
using TerraLib.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.
This file is part of TerraView. TerraView is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

You should have received a copy of the GNU General Public License
along with TerraView.
The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The software provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use of
this program and its documentation.
*************************************************************************************/

#include <TeQtBasicCanvas.h>
#include <TeQtCanvas.h>
#include <TeDecoderMemory.h>
#include <TeQtProgress.h>
#include <TeVectorRemap.h>
#include <TeGeometryAlgorithms.h>
#include <TeDefines.h>
#include <TeRasterRemap.h>
#include <TeDecoderQtImage.h>

#include <qpaintdevice.h> 
#include <qpaintdevicemetrics.h> 
#include <qapplication.h>

#include <algorithm>	
#include <time.h>


//TeQtBasicCanvas::TeQtBasicCanvas( )
TeQtBasicCanvas::TeQtBasicCanvas(QWidget *parent, const char *name ) : QScrollView(parent,name,WNorthWestGravity)
{
	pixmap0_ = 0;
	pixmap1_ = 0;
	pixmap2_ = 0;
	pixmap3_ = 0;
	backRaster_ = 0;

	canvasProjection_ = 0;
	dataProjection_ = 0;
	pointStyle_ = 0;
	nodeStyle_ = 0;
	pointSize_ = 3;
	nodeSize_ = 4;

#ifdef WIN32
	int winVersion = qApp->winVersion();
	if (winVersion == Qt::WV_2000 || winVersion == Qt::WV_NT)
		setTextTTStyle (QString("c:/winnt/fonts/tahoma.ttf"),10);
	else
		setTextTTStyle (QString("c:/windows/fonts/tahoma.ttf"),10);
#else
	char *s = "/usr/share/fonts/ttf/western/Bluehigc.ttf";
	QString qs = s;
	setTextTTStyle (qs,10);
#endif
	textTT_ = true;
	db_ = 0;

	//Mounting the brushStyleMap
	brushStyleMap_[TePolyTypeTransparent] = Qt::NoBrush;
	brushStyleMap_[TePolyTypeFill] = Qt::SolidPattern;
	brushStyleMap_[TePolyTypeHorizontal] = Qt::HorPattern;
	brushStyleMap_[TePolyTypeVertical] = Qt::VerPattern;
	brushStyleMap_[TePolyTypeFDiagonal] = Qt::FDiagPattern;
	brushStyleMap_[TePolyTypeBDiagonal] = Qt::BDiagPattern;
	brushStyleMap_[TePolyTypeCross] = Qt::CrossPattern;
	brushStyleMap_[TePolyTypeDiagonalCross] = Qt::DiagCrossPattern;

	//Mounting the penStyleMap
	penStyleMap_[TeLnTypeContinuous] = Qt::SolidLine;
	penStyleMap_[TeLnTypeDashed] = Qt::DashLine;
	penStyleMap_[TeLnTypeDotted] = Qt::DotLine;
	penStyleMap_[TeLnTypeDashDot] = Qt::DashDotLine;
	penStyleMap_[TeLnTypeDashDotDot] = Qt::DashDotDotLine;
	penStyleMap_[TeLnTypeNone] = Qt::NoPen;
}

TeQtBasicCanvas::~TeQtBasicCanvas()
{
	if (dataProjection_)
		delete dataProjection_;
	if (canvasProjection_)
		delete canvasProjection_;
}

void
TeQtBasicCanvas::setDataProjection ( TeProjection* proj )
{	
	if(dataProjection_)
		delete dataProjection_;
	dataProjection_ = TeProjectionFactory::make(proj->params());
	if (canvasProjection_ && dataProjection_ && !(*canvasProjection_== *dataProjection_))
	{
		dataProjection_->setDestinationProjection(canvasProjection_);
		canvasProjection_->setDestinationProjection(dataProjection_);
	}
}

void
TeQtBasicCanvas::setProjection ( TeProjection* proj )
{	
	if (canvasProjection_)
		delete canvasProjection_;
	canvasProjection_ = TeProjectionFactory::make(proj->params()); 
	params_.projection(canvasProjection_);
}

void TeQtBasicCanvas::plotOnWindow ()
{
	QPaintDevice* dev = painter_.device();
	if(dev == viewport())
		return;
	if(dev)
		painter_.end();
	painter_.begin(viewport());
}

void TeQtBasicCanvas::plotOnPixmap0 ()
{
	QPaintDevice* dev = painter_.device();
	if(dev == pixmap0_)
		return;
	if(dev)
		painter_.end();
	painter_.begin(pixmap0_);
}

void TeQtBasicCanvas::plotOnPixmap1 ()
{
	QPaintDevice* dev = painter_.device();
	if(dev == pixmap1_)
		return;
	if(dev)
		painter_.end();
	painter_.begin(pixmap1_);
}

void TeQtBasicCanvas::setWorld(TeBox b, int w, int h, QPaintDevice *pd)
{
	xmin_= b.x1();
	xmax_= b.x2();
	ymin_= b.y1();
	ymax_= b.y2();
	setView (w,h,pd);
	bool extend = pd ? false : true;
	TeQtBasicCanvas::setTransformation (b.x1(),b.x2(),b.y1(),b.y2(),extend);
	params_.nBands(3);
	double resx = pixelSize();
	double resy = pixelSize();
	TeBox box(getWorld());
	params_.lowerLeftResolutionSize(box.x1_+0.5*resx,box.y1_+0.5*resy,resx,resy,width_,height_);
	params_.projection();
	params_.mode_ = 'w';
	params_.decoderIdentifier_ = "";
}
void TeQtBasicCanvas::setTransformation(double xmin, double xmax, double ymin, double ymax, bool extend)
{
	if (painter_.device() == 0)
		return;
	
	xmin_= xmin;
	xmax_= xmax;
	ymin_= ymin;
	ymax_= ymax;
	double	dxw = xmax_ - xmin_,
			dyw = ymax_ - ymin_,
			dxv = width_,
			dyv = height_;
					
	double	fx = dxv/dxw,
			fy = dyv/dyw;

	if (fx > fy)
	{
		f_ = fy;
		if (extend)
		{
			dxw = width_/f_;
			xmax_ = xmin_ + dxw;
		}
		else
			width_ = (int)(f_*dxw + .5);
	}
	else
	{
		f_ = fx;
		if (extend)
		{
			dyw = height_/f_;
			ymax_ = ymin_ + dyw;
		}
		else
			height_ = (int)(f_*dyw + .5);
	}

	wc_ = TeBox (xmin_,ymin_,xmax_,ymax_);
	params_.boundingBoxResolution(xmin_,ymin_,xmax_,ymax_,params_.resx_,params_.resy_);

	QPaintDeviceMetrics devMetric(viewport());
	double wpixels = devMetric.width();
	double wMM = devMetric.widthMM();
	double wT = wMM;
	if(canvasProjection_)
	{
		string unit = canvasProjection_->units();
		if(unit == "Meters")
			wT = wMM / 1000.;
		else if(unit == "Kilometers")
			wT = wMM / 1000000.;
		else if(unit == "Centimeters")
			wT = wMM / 100.;
		else if(unit == "Millimeters")
			wT = wMM;
		else if(unit == "Feet")
			wT = wMM / (12. * 25.4);
		else if(unit == "Inches")
			wT = wMM / 25.4;
		else if(unit == "DecimalDegrees")
			wT = wMM / 110000000.;
	}
	double wp = wT / wpixels;
	scaleApx_ = (1. / f_) / wp;
}

void TeQtBasicCanvas::scaleApx(double scale)
{
	QPaintDeviceMetrics devMetric(viewport());
	double wMM = devMetric.widthMM();
	double wT = wMM;
	if(canvasProjection_)
	{
		string unit = canvasProjection_->units();
		if(unit == "Meters")
			wT = wMM / 1000.;
		else if(unit == "Kilometers")
			wT = wMM / 1000000.;
		else if(unit == "Centimeters")
			wT = wMM / 100.;
		else if(unit == "Millimeters")
			wT = wMM;
		else if(unit == "Feet")
			wT = wMM / (12. * 25.4);
		else if(unit == "Inches")
			wT = wMM / 25.4;
		else if(unit == "DecimalDegrees")
			wT = wMM / 110000000.;
	}

	double ff = scale / scaleApx_;
	double xmin, ymin, xmax, ymax;
	if(ff < 1)
	{
		double dx = (wc_.width() - (wc_.width() * ff)) / 2.;
		double dy = (wc_.height() - (wc_.height() * ff)) / 2.;
		xmin = xmin_ + dx;
		ymin = ymin_ + dy;
		xmax = xmax_ - dx;
		ymax = ymax_ - dy;
	}
	else
	{
		double dx = ((wc_.width() * ff) - wc_.width()) / 2.;
		double dy = ((wc_.height() * ff) - wc_.height()) / 2.;
		xmin = xmin_ - dx;
		ymin = ymin_ - dy;
		xmax = xmax_ + dx;
		ymax = ymax_ + dy;
	}

	TeBox box(xmin, ymin, xmax, ymax);
	setWorld(box);
}

double TeQtBasicCanvas::mapVtoDW (int pixels)
{
	TeBox wbox = getDataWorld();
	TeCoord2D	wpc(wbox.x1_ + wbox.width()/2., wbox.y1_ + wbox.height()/2.);
	QPoint pc = mapWtoV(wpc);
	QPoint qp(pc.x()+pixels, pc.y());
	TeCoord2D	wp = mapVtoDW(qp);
	wpc = mapVtoDW(pc);
	double d = fabs(wp.x() - wpc.x());
	return d;
}

TeCoord2D TeQtBasicCanvas::mapVtoDW (QPoint v)
{
	TeCoord2D w((v.x()-x0_)/f_ + xmin_,
		(height_ - y0_ - v.y())/f_ + ymin_);

	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		w = canvasProjection_->PC2LL (w);
		w = dataProjection_->LL2PC (w);
	}
	return w;
}

double TeQtBasicCanvas::mapVtoW (int pixels)
{
	TeBox wbox = getDataWorld();
	TeCoord2D	wpc(wbox.x1_ + wbox.width()/2., wbox.y1_ + wbox.height()/2.);
	QPoint pc = mapWtoV(wpc);
	QPoint qp(pc.x()+pixels, pc.y());
	TeCoord2D	wp = mapVtoW(qp);
	wpc = mapVtoW(pc);
	double d = fabs(wp.x() - wpc.x());
	return d;
}

TeCoord2D TeQtBasicCanvas::mapVtoW (QPoint v)
{
	TeCoord2D w((v.x()-x0_)/f_ + xmin_,
		(height_ - y0_ - v.y())/f_ + ymin_);
	return w;
}

TeBox TeQtBasicCanvas::getDataWorld()
{
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		TeBox b = TeRemapBox ( wc_, canvasProjection_, dataProjection_);
		return b;
	}
	return wc_ ;
}

QPoint TeQtBasicCanvas::mapWtoV (TeCoord2D w)
{
	TeCoord2D tw = w;
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		tw = dataProjection_->PC2LL (tw);
		tw = canvasProjection_->LL2PC (tw);
	}

	QPoint v((int)((tw.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((tw.y() - ymin_)*f_ + 0.5));
	correctScrolling (v);
	return v;
}

void TeQtBasicCanvas::mapWtoV (TeBox& w)
{
	TeCoord2D tw = w.lowerLeft();
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		tw = dataProjection_->PC2LL (tw);
		tw = canvasProjection_->LL2PC (tw);
	}

	QPoint p1((int)((tw.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((tw.y() - ymin_)*f_ + 0.5));
	correctScrolling (p1);

	tw = w.upperRight();
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		tw = dataProjection_->PC2LL (tw);
		tw = canvasProjection_->LL2PC (tw);
	}

	QPoint p2((int)((tw.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((tw.y() - ymin_)*f_ + 0.5));
	correctScrolling (p2);
	w.x1_ = p1.x();
	w.y1_ = p2.y();
	w.x2_ = p2.x();
	w.y2_ = p1.y();
}

void TeQtBasicCanvas::mapCanvasWorldToDataWorld (TeBox& w)
{
	TeCoord2D wll = w.lowerLeft();
	TeCoord2D wur = w.upperRight();
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		canvasProjection_->setDestinationProjection(dataProjection_);
		wll = canvasProjection_->PC2LL (wll);
		wll = dataProjection_->LL2PC (wll);
		wur = canvasProjection_->PC2LL (wur);
		wur = dataProjection_->LL2PC (wur);
	}

	w = TeBox(wll, wur);
}

void TeQtBasicCanvas::mapDataWorldToCanvasWorld (TeBox& w)
{
	TeCoord2D wll = w.lowerLeft();
	TeCoord2D wur = w.upperRight();
	if ( canvasProjection_ && dataProjection_ )
	if ( canvasProjection_ != dataProjection_ && !(*canvasProjection_ == *dataProjection_) )
	{
		dataProjection_->setDestinationProjection(canvasProjection_);
		wll = dataProjection_->PC2LL (wll);
		wll = canvasProjection_->LL2PC (wll);
		wur = dataProjection_->PC2LL (wur);
		wur = canvasProjection_->LL2PC (wur);
	}

	w = TeBox(wll, wur);
}

void TeQtBasicCanvas::mapCanvasWorldToViewport (TeBox& w)
{
	QPoint p1((int)((w.x1() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((w.y1() - ymin_)*f_ + 0.5));
	correctScrolling (p1);

	QPoint p2((int)((w.x2() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((w.y2() - ymin_)*f_ + 0.5));
	correctScrolling (p2);

// swap y value
	w.x1_ = p1.x();
	w.y1_ = p2.y();
	w.x2_ = p2.x();
	w.y2_ = p1.y();
}

void TeQtBasicCanvas::mapViewportToCanvasWorld (TeBox& v)
{
// swap y value

	TeCoord2D w1((v.x1()-x0_)/f_ + xmin_,
		(height_ - y0_ - v.y2())/f_ + ymin_);

	TeCoord2D w2((v.x2()-x0_)/f_ + xmin_,
		(height_ - y0_ - v.y1())/f_ + ymin_);

	v = TeBox(w1, w2);
}

QPoint TeQtBasicCanvas::mapCanvasWorldToViewport (TeCoord2D& c)
{
	QPoint p((int)((c.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((c.y() - ymin_)*f_ + 0.5));
	correctScrolling (p);

	return p;
}

void TeQtBasicCanvas::setPolygonColor (int r, int g, int b)
{
//	polygonColor_.setRgb(r,g,b);
//	polygonBrush_.setColor (polygonColor_);

	QRgb cor = qRgba(r, g, b, 50);
	polygonColor_.setRgb(cor);
	polygonBrush_.setColor (polygonColor_);
}

void TeQtBasicCanvas::setPolygonStyle (int s, int t)
{
	polygonBrush_.setStyle(brushStyleMap_[(TePolyBasicType)s]);
	polygonTransparency_ = t;
}

void TeQtBasicCanvas::setPolygonLineColor (int r, int g, int b)
{
	QColor cor(r, g, b);
	polygonPen_.setColor (cor);
}

void TeQtBasicCanvas::setPolygonLineStyle (int s, int w)
{
	polygonPen_.setStyle(penStyleMap_[(TeLnBasicType)s]);
	polygonPen_.setWidth(w);
}

void TeQtBasicCanvas::setLineColor (int r, int g, int b)
{
	lineColor_.setRgb(r,g,b);
	linePen_.setColor (lineColor_);
}

void TeQtBasicCanvas::setLineStyle (int s, int w)
{
	linePen_.setStyle(penStyleMap_[(TeLnBasicType)s]);
	linePen_.setWidth(w);
}

void TeQtBasicCanvas::setTextColor (int r, int g, int b)
{
	textColor_.setRgb(r,g,b);
	textPen_.setColor (textColor_);
}

void TeQtBasicCanvas::setTextStyle (string& family, int size, bool bold, bool italic )
{
	textFont_.setFamily (family.c_str());
	textFont_.setBold (bold);
	textFont_.setItalic (italic);
	if(size <= 0)
		textSize_ = 1;
	else
		textSize_ = size;
	textFont_.setPointSize (textSize_);
}

void TeQtBasicCanvas::setTextSize (int size)
{
	if(size <= 0)
		textSize_ = 1;
	else
		textSize_ = size;
	textFont_.setPointSize (textSize_);
}

void TeQtBasicCanvas::setTextTTStyle (QString& font, int size)
{
	textTTFont_ = font.latin1();
	textSize_ = size;
}

void TeQtBasicCanvas::plotText (TeCoord2D &pt, string &str, double angle, double /*alignh*/, double /*alignv*/)
{
	painter_.setPen(textPen_);
	painter_.setFont(textFont_);
	QPoint p = mapWtoV (pt);
	if (angle != 0.)
	{
		painter_.save ();
		painter_.translate (p.x(),p.y());
		painter_.rotate (-angle);
		painter_.drawText (0,-4,QString(str.c_str()));
		painter_.restore ();
	}
	else
		painter_.drawText (p,QString(str.c_str()));
}

void TeQtBasicCanvas::plotText (TeText& tx, TeVisual& visual)
{
	if(tx.textValue().empty())
		return;

	int	x, y;

	TeCoord2D pt = tx.location();
	QPoint p = mapWtoV (pt);
	double angle = tx.angle();

	QRect brect = textRect(tx, visual);
	painter_.setFont(textFont_);
	painter_.setPen(textPen_);
	painter_.setRasterOp (Qt::CopyROP);
	string st = tx.textValue();

	if (angle != 0.)
	{
		x = - brect.width()/2;
		y = - brect.height()/2;
		painter_.save ();
		painter_.translate (p.x(), p.y());
		painter_.rotate (-angle);
		painter_.drawText(x, y, brect.width(), brect.height(), Qt::AlignLeft|Qt::DontClip, st.c_str());
		painter_.restore ();
	}
	else
	{
		x = p.x() - brect.width()/2;
		y = p.y() - brect.height()/2;
		painter_.drawText(x, y, brect.width(), brect.height(), Qt::AlignLeft|Qt::DontClip, st.c_str());
	}
}

QRect TeQtBasicCanvas::textRect (TeText& tx, TeVisual visual)
{
	QRect rect;
	if(tx.textValue().empty())
		return rect;

	int size;
	int fixedSize = visual.fixedSize(); // font size is fixed
	if(fixedSize == false && tx.height() > 0.)
	{
		TeBox wbox = getDataWorld();
		TeCoord2D p1(wbox.x1_, wbox.y1_);
		TeCoord2D p2(double(p1.x() + tx.height()), double(p1.y() + tx.height()));
		TeBox box(p1, p2);
		mapWtoV(box);
		size = int(box.height());
	}
	else
		size = visual.size(); // font size
	if(size == 0)
		size = 1;

	setTextSize(size);
	textFont_.setFamily (visual.family().c_str());

	TeCoord2D pt = tx.location();
	QPoint p = mapWtoV (pt);

	string st = tx.textValue();
	QFontMetrics fm(textFont_);
//	int hh = fm.height();							//hh is not used
//	int ww = fm.width(st.c_str());			//ww is not used
	rect = fm.boundingRect(st.c_str());
	QPoint cc = rect.center();
	QPoint tr = p;
	tr = tr - cc;
	rect.setRight(rect.right()+tr.x());
	rect.setLeft(rect.left()+tr.x());
	rect.setTop(rect.top()+tr.y());
	rect.setBottom(rect.bottom()+tr.y());
	QPoint o = offset();
	rect.moveCenter(rect.center() + o); // do offset translation on window
	return rect;
}

void TeQtBasicCanvas::plotXorPolyline (QPointArray& PA, bool cdev)
{
//	if(painter_ == 0)
//		return;

	QPoint o = offset();
	QPen pen(QColor("green"));

	plotOnPixmap0();
	painter_.save ();
	if(cdev == false)
		painter_.translate(o.x(), o.y());
	painter_.setRasterOp (Qt::XorROP);
	painter_.setPen(pen);
	painter_.drawPolyline(PA);
	painter_.restore ();

	plotOnWindow(); // do offset translation on window
	painter_.save ();
	if(cdev)
		painter_.translate(-o.x(), -o.y());
	painter_.setRasterOp (Qt::XorROP);
	painter_.setPen(pen);
	painter_.drawPolyline(PA);
	painter_.restore ();

}

void TeQtBasicCanvas::plotTextRects (TeText& tx, TeVisual visual)
{
	if(tx.textValue().empty())
		return;

	QPoint o = offset();
	QRect rect, l, r, t, b, c;

	TeCoord2D pt = tx.location();
	plotOnWindow();
	QPoint p = mapWtoV (pt) + o;
	double angle = tx.angle();

	rect = textRect(tx, visual);
	if(angle != 0.)
	{
		rect.setTop(rect.top()-p.y());
		rect.setBottom(rect.bottom()-p.y());
		rect.setLeft(rect.left()-p.x());
		rect.setRight(rect.right()-p.x());
	}

	l = rect;
	l.setTop(rect.top()+rect.height()/2-3);
	l.setBottom(rect.top()+rect.height()/2+3);
	l.setRight(rect.left() + 6);

	r = rect;
	r.setTop(l.top());
	r.setBottom(l.bottom());
	r.setLeft(rect.right() - 6);

	t = rect;
	t.setLeft(rect.left()+rect.width()/2-3);
	t.setRight(rect.left()+rect.width()/2+3);
	t.setBottom(rect.top() + 6);

	b = rect;
	b.setLeft(t.left());
	b.setRight(t.right());
	b.setTop(rect.bottom() - 6);

	c.setLeft(rect.left()+rect.width()/2-3);
	c.setRight(c.left() + 6);
	c.setTop(rect.top()+rect.height()/2-3);
	c.setBottom(c.top() + 6);

	QPointArray parL(4);
	parL.setPoint(0, l.bottomLeft());
	parL.setPoint(1, l.bottomRight());
	parL.setPoint(2, l.topRight());
	parL.setPoint(3, l.topLeft());

	QPointArray parR(4);
	parR.setPoint(0, r.bottomRight());
	parR.setPoint(1, r.bottomLeft());
	parR.setPoint(2, r.topLeft());
	parR.setPoint(3, r.topRight());

	QPointArray parT(4);
	parT.setPoint(0, t.topLeft());
	parT.setPoint(1, t.bottomLeft());
	parT.setPoint(2, t.bottomRight());
	parT.setPoint(3, t.topRight());

	QPointArray parB(4);
	parB.setPoint(0, b.bottomLeft());
	parB.setPoint(1, b.topLeft());
	parB.setPoint(2, b.topRight());
	parB.setPoint(3, b.bottomRight());

	QBrush	brush(QColor("green"));

	if (angle != 0.)
	{
		plotOnPixmap0();
		painter_.save ();
		painter_.setRasterOp (Qt::XorROP);
		painter_.setBrush(Qt::NoBrush);
		QPen pen(QColor("green"));
		painter_.setPen(pen);
		painter_.translate (p.x(), p.y());
		painter_.rotate (-angle);
		painter_.drawRect(rect);
		painter_.drawPolyline(parL);
		painter_.drawPolyline(parR);
		painter_.drawPolyline(parT);
		painter_.drawPolyline(parB);
		painter_.fillRect(c, brush);
		painter_.restore ();

		plotOnWindow(); // do offset translation on window
		painter_.save ();
		painter_.setRasterOp (Qt::XorROP);
		painter_.setBrush(Qt::NoBrush);
		pen = QPen(QColor("green"));
		painter_.setPen(pen);
		painter_.translate (p.x()-o.x(), p.y()-o.y());
		painter_.rotate (-angle);
		painter_.drawRect(rect);
		painter_.drawPolyline(parL);
		painter_.drawPolyline(parR);
		painter_.drawPolyline(parT);
		painter_.drawPolyline(parB);
		painter_.fillRect(c, brush);
		painter_.restore ();
	}
	else
	{
		plotOnPixmap0();
		painter_.save ();
		painter_.setRasterOp (Qt::XorROP);
		painter_.setBrush(Qt::NoBrush);
		QPen pen(QColor("green"));
		painter_.setPen(pen);
		painter_.drawRect(rect);
		painter_.drawPolyline(parL);
		painter_.drawPolyline(parR);
		painter_.drawPolyline(parT);
		painter_.drawPolyline(parB);
		painter_.fillRect(c, brush);
		painter_.restore ();

		plotOnWindow(); // do offset translation on window
		painter_.save ();
		painter_.setRasterOp (Qt::XorROP);
		painter_.setBrush(Qt::NoBrush);
		pen = QPen(QColor("green"));
		painter_.setPen(pen);
		painter_.translate (-o.x(), -o.y());
		painter_.drawRect(rect);
		painter_.drawPolyline(parL);
		painter_.drawPolyline(parR);
		painter_.drawPolyline(parT);
		painter_.drawPolyline(parB);
		painter_.fillRect(c, brush);
		painter_.restore ();
	}
}

void TeQtBasicCanvas::setArcColor (int r, int g, int b)
{
	arcColor_.setRgb(r,g,b);
}

void TeQtBasicCanvas::setArcStyle (int s, int w )
{
	arcPen_.setStyle ((Qt::PenStyle)s);
	arcPen_.setWidth(w);
}

void TeQtBasicCanvas::setPointColor (int r, int g, int b)
{
	pointColor_.setRgb(r,g,b);
}

void TeQtBasicCanvas::setPointStyle (int s, int w)
{
	pointStyle_ = (TePtBasicType) s;
	pointSize_ = w;
}

void TeQtBasicCanvas::plotPoint (TeCoord2D &pt)
{	
	pointPen_.setColor (pointColor_);
	painter_.setPen(pointPen_);

	QPoint p = mapWtoV (pt);

	plotMark(p,pointStyle_, pointSize_);
}

void TeQtBasicCanvas::setNodeColor (int r, int g, int b)
{
	nodeColor_.setRgb(r,g,b);
}

void TeQtBasicCanvas::setNodeStyle (int s, int w)
{
	nodeStyle_ = s;
	nodeSize_ = w;
}

void TeQtBasicCanvas::plotNode (TeNode &pt)
{	
	nodePen_.setColor (nodeColor_);
	painter_.setPen(nodePen_);

	QPoint p = mapWtoV (pt.location());
	plotMark(p,nodeStyle_, nodeSize_);
}


void TeQtBasicCanvas::plotMark(QPoint &p, int s, int w)
{
	painter_.setPen(pointColor_);
	if (s == TePtTypePlus)
	{
		painter_.drawLine (p.x()-w/2,p.y(),p.x()+w/2,p.y());
		painter_.drawLine (p.x(),p.y()-w/2,p.x(),p.y()+w/2);
	}
	else if (s == TePtTypeStar)
	{
		painter_.save ();
		painter_.translate (p.x(),p.y());
		painter_.drawLine (0,-w/2,0,w/2);
		painter_.rotate (45);
		painter_.drawLine (0,-w/2,0,w/2);
		painter_.rotate (-90);
		painter_.drawLine (0,-w/2,0,w/2);
		painter_.restore ();
	}
	else if (s == TePtTypeCircle)
	{
		painter_.setBrush(pointColor_);
		painter_.drawChord (p.x()-w/2,p.y()-w/2,w,w,0,360*16);
	}
	else if (s == TePtTypeX)
	{
		painter_.drawLine (p.x()-w/2,p.y()-w/2,p.x()+w/2,p.y()+w/2);
		painter_.drawLine (p.x()-w/2,p.y()+w/2,p.x()+w/2,p.y()-w/2);
	}
	else if (s == TePtTypeBox)
	{
		painter_.fillRect (p.x()-w/2,p.y()-w/2,w,w,pointColor_);
	}
	else if (s == TePtTypeDiamond)
	{
		QPointArray pa(5);
		pa.setPoint(0, p.x()-w/2, p.y());
		pa.setPoint(1, p.x(), p.y()-w/2);
		pa.setPoint(2, p.x()+w/2, p.y());
		pa.setPoint(3, p.x(), p.y()+w/2);
 		pa.setPoint(4, p.x()-w/2, p.y());
		painter_.setBrush(pointColor_);
		painter_.drawPolygon(pa);
	}
	else if (s == TePtTypeHollowCircle)
	{
		painter_.drawArc (p.x()-w/2,p.y()-w/2,w,w,0,360*16);
	}
	else if (s == TePtTypeHollowBox)
	{
		painter_.setBrush(Qt::NoBrush);
		painter_.drawRect (p.x()-w/2,p.y()-w/2,w,w);
	}
	else if (s == TePtTypeHollowDiamond)
	{
		painter_.drawLine (p.x()-w/2,p.y(),p.x(),p.y()-w/2);
		painter_.drawLine (p.x(),p.y()-w/2,p.x()+w/2,p.y());
		painter_.drawLine (p.x()+w/2,p.y(),p.x(),p.y()+w/2);
		painter_.drawLine (p.x(),p.y()+w/2,p.x()-w/2,p.y());
	}
}


void TeQtBasicCanvas::plotCell (TeCell &cell)
{
	TeBox b = getDataWorld ();

	if (!TeIntersects (b, cell.box ()))
		return;

	QPoint pfrom, pto;

	pfrom = mapWtoV (cell.box().lowerLeft());
	pto   = mapWtoV (cell.box().upperRight());

	painter_.setBrush(polygonBrush_);
	painter_.setPen (polygonPen_);

	if((polygonTransparency_ == 0 && polygonBrush_.style() != Qt::NoBrush) || // 100% opaque or
		(polygonTransparency_ == 100 || polygonBrush_.style() == Qt::NoBrush)) // 100% transparent
	{
		if(polygonTransparency_ == 100 && polygonBrush_.style() != Qt::NoBrush)
			painter_.setBrush(Qt::NoBrush);
		painter_.drawRect( pfrom.x()+1, pto.y()+1,pto.x()-pfrom.x(), pfrom.y()-pto.y() );
	}
	else
	{
		QRect viewRect = painter_.viewport();

		TeBox	box = cell.box();
		mapWtoV (box); // data coordinate to viewport coordinate
		QRect polyRect((int)box.x1_, (int)box.y1_, (int)box.width(), (int)box.height());
		QRect interRect = viewRect & polyRect;
		QPoint pOffset = interRect.topLeft();

		int width = interRect.width();
		int height = interRect.height();

		int r = width%8;
		if(r)
			width += (8-r);
		r = height%8;
		if(r)
			height += (8-r);

		QBitmap	bm;
		bm.resize(width, height);
		//Fill bitmap with 0-bits: clipping region
		bm.fill(Qt::color0);
		QPainter maskPainter(&bm);

		// Draw cell with 1-bits: drawing region
		QBrush bs(Qt::color1, polygonBrush_.style());
		maskPainter.setBrush(bs);
		QPen pen(Qt::color1, polygonPen_.width());
		maskPainter.setPen(pen);
		maskPainter.translate(-pOffset.x(), -pOffset.y());
 		maskPainter.drawRect( interRect );
		maskPainter.end();

		QRegion clipRegion(bm);
		clipRegion.translate(pOffset.x(), pOffset.y());

		int transp = 255 - (polygonTransparency_ * 255 / 100);
		painter_.setClipRegion(clipRegion);
		// restore background from pixmap2_
//		bitBlt(painter_.device(), pOffset.x(), pOffset.y(), pixmap2_, pOffset.x(), pOffset.y(), interRect.width(), interRect.height());
		painter_.drawPixmap(pOffset.x(), pOffset.y(), *pixmap2_, pOffset.x(),
			pOffset.y(), interRect.width(), interRect.height());

		// set alpha buffer and color
		QImage img(interRect.width(), interRect.height(), 32);
		unsigned int val = (transp << 24) | (polygonColor_.red() << 16) | (polygonColor_.green() << 8) | polygonColor_.blue();
		img.fill(val);
		img.setAlphaBuffer(true);

		// plot transparency
//		bitBlt(painter_.device(), pOffset.x(), pOffset.y(), &img);
		painter_.drawPixmap(pOffset.x(), pOffset.y(), img);


		// plot contours
		painter_.setClipping(false);
		painter_.setBrush(Qt::NoBrush);
		painter_.drawRect( interRect );
	}
}

void TeQtBasicCanvas::plotPolygon (TePolygon &poly)
{
	TeBox b = getDataWorld ();

	if (!TeIntersects (b, poly.box ()))
		return;

	int		i, j, k, np;
 	QPoint	p;
	painter_.setBrush(polygonBrush_);
	painter_.setPen (polygonPen_);

	if(polygonTransparency_==100 || polygonBrush_.style() == Qt::NoBrush) // contour
	{
		TeLinearRing ring = poly[0];
 		np = ring.size();
 		QPointArray parray(np);

 		for ( i = 0 ; i < np; i++)
 		{
			p = mapWtoV (ring[i]);
 			parray.setPoint(i, p);
 		}
		painter_.drawPolyline( parray );

		for ( k = 1; k < (int)(poly.size()); k++ )
  		{
  			ring = poly[k];
  			np = ring.size();
  			QPointArray hole(np);

  			for ( i = 0 ; i < np; i++)
  			{
				p = mapWtoV (ring[i]);
				hole.setPoint(i, p);
  			}
  			painter_.drawPolyline( hole );
  		}
	}
	else if(poly.size() == 1 && // no holes
		polygonTransparency_ == 0  && // and is opaque
		polygonBrush_.style() != Qt::NoBrush)
	{
		TeLinearRing ring = poly[0];
 		np = ring.size();
 		QPointArray parray(np);
 		for ( i = 0 ; i < np; i++)
 		{
			p = mapWtoV (ring[i]);
 			parray.setPoint(i, p);
 		}
		painter_.drawPolygon( parray );
	}
	else
	{
		QRect viewRect = painter_.viewport();

		TeBox	box = poly.box();
		mapWtoV (box); // data coordinate to viewport coordinate
		if(box.width() <= 0 || box.height() <= 0)
			return;
		QRect polyRect((int)box.x1_, (int)box.y1_, (int)box.width(), (int)box.height());
		if(polyRect.intersects(viewRect) == false)
			return;
		QRect interRect = viewRect & polyRect;

		if(painter_.hasClipping())
		{
			QRegion interRegion = QRegion(interRect);
			QRegion pclip = painter_.clipRegion();
			interRegion = interRegion.intersect(pclip);
			interRect = interRegion.boundingRect();
		}

		QPoint pOffset = interRect.topLeft();
		int width = interRect.width();
		int height = interRect.height();

		int r = width%8;
		if(r)
			width += (8-r);
		r = height%8;
		if(r)
			height += (8-r);

		if(width == 0)
			width = 8;
		if(height == 0)
			height = 8;

		QBitmap	bm;
		bm.resize(width, height);
		//Fill bitmap with 0-bits: clipping region
		bm.fill(Qt::color0);
		QPainter maskPainter(&bm);

		TeLinearRing ring = poly[0];
 		np = ring.size();
 		QPointArray parray(np);

		// Draw polygon with 1-bits: drawing region
		QBrush bs(Qt::color1, polygonBrush_.style());
		maskPainter.setBrush(bs);
		QPen pen(Qt::color1, polygonPen_.width());
		maskPainter.setPen(pen);
		maskPainter.translate(-pOffset.x(), -pOffset.y());
		j = 0;
		parray.setPoint(j, mapWtoV (ring[0]));
 		for ( i = 1 ; i < np; i++)
 		{
			p = mapWtoV (ring[i]);
 			parray.setPoint(++j, p);
 		}
 		maskPainter.drawPolygon( parray );

  		vector<QPointArray> holeVec;
		if(poly.size() > 1)
		{
			// Draw holes with 0-bits: clipping region
			maskPainter.setBrush(Qt::color0);
			pen.setColor(Qt::color0);
			maskPainter.setPen(pen);
			for ( k = 1; k < (int)poly.size(); k++ )
  			{
  				ring = poly[k];
  				np = ring.size();
  				QPointArray hole(np);

  				for ( i = 0 ; i < np; i++)
  				{
					p = mapWtoV (ring[i]);
					hole.setPoint(i, p);
  				}
				holeVec.push_back(hole);
  				maskPainter.drawPolygon( hole );
  			}
		}
		maskPainter.end();

		QRegion clipRegion(bm);
		clipRegion.translate(pOffset.x(), pOffset.y());

		if(polygonTransparency_ == 0)
		{
			painter_.setClipRegion(clipRegion);
			painter_.drawPolygon( parray );
			painter_.setClipping(false);
			for(i = 0; i < (int)holeVec.size(); i++)
				painter_.drawPolyline( holeVec[i] );
		}
		else
		{
			int transp = 255 - (polygonTransparency_ * 255 / 100);
			painter_.setClipRegion(clipRegion);
			// restore background from pixmap2_
			painter_.drawPixmap(pOffset.x(), pOffset.y(), *pixmap2_, pOffset.x(),
				pOffset.y(), interRect.width(), interRect.height());

			// set alpha buffer and color
			QImage img(interRect.width(), interRect.height(), 32);
			unsigned int val = (transp << 24) | (polygonColor_.red() << 16) | (polygonColor_.green() << 8) | polygonColor_.blue();
			img.fill(val);
			img.setAlphaBuffer(true);

			// plot transparency
//			bitBlt(painter_.device(), pOffset.x(), pOffset.y(), &img);
			painter_.drawPixmap(pOffset.x(), pOffset.y(), img);

			// plot contours
			painter_.setClipping(false);
			painter_.drawPolyline( parray );
			for(i = 0; i< (int)holeVec.size(); i++)
				painter_.drawPolyline( holeVec[i] );
		}
	}
}


/*void TeQtBasicCanvas::plotPolygon (TePolygon &poly)
{
	TeBox b = getDataWorld ();

	if (!TeIntersects (b, poly.box (), 0.0))
		return;

	int		i, np;

	if(polygonTransparency_==0 &&
		polygonBrush_.style()!=Qt::NoBrush)
	{
		painter_->setBrush(polygonBrush_);
		painter_->setPen (polygonPen_);
		if (poly.size() == 1)
 		{
 			TeLinearRing ring = poly[0];
 			np = ring.size();
 			QPointArray a(np);
 			QPoint p;

		// Draw polygon with polygon brush and line pen
 			for ( i = 0 ; i < np; i++)
 			{
				p = mapWtoV (ring[i]);
 				a.setPoint(i,p);
 			}
			painter_->drawPolygon( a );
		}
		else
		{
			QRect rect = painter_->viewport();
			int	w = rect.width();
			int	h = rect.height();

			int r = w%8;
			if(r)
				w += (8-r);
			r = h%8;
			if(r)
				h += (8-r);

			QBitmap	bm;
			bm.resize(w, h);
			//Fill bitmap with 0-bits:background(or transparent)
			bm.fill(Qt::color0);
			QPainter qp(&bm);
 
			TeLinearRing ring = poly[0];
 			np = ring.size();
 			QPointArray a(np);
 			QPoint p;

			// Draw outer polygon with 1-bits:foreground(or opaque)
			qp.setBrush(Qt::color1);
			QPen pen(Qt::color1, polygonPen_.width());
			qp.setPen(pen);
 			for ( i = 0 ; i < np; i++)
 			{
				p = mapWtoV (ring[i]);
 				a.setPoint(i,p);
 			}
 			qp.drawPolygon( a );

			// Draw holes with 0-bits:background(or transparent)
			qp.setBrush(Qt::color0);
			pen.setColor(Qt::color0);
			qp.setPen(pen);
			unsigned int k;
			for ( k = 1; k < poly.size(); k++ )
  			{
  				ring = poly[k];
  				np = ring.size();
  				QPointArray hole(np);

  				for ( i = 0 ; i < np; i++)
  				{
					p = mapWtoV (ring[i]);
					hole.setPoint(i,p);
  				}
  				qp.drawPolygon( hole );
  			}
			
			QRegion region(bm);
			painter_->setClipRegion(region);
			painter_->drawPolygon( a );
			painter_->setClipping(false);
		}
	}
	else
	{
		Qt::BrushStyle bstyle = polygonBrush_.style();
		int	transp = polygonTransparency_;
		if(polygonBrush_.style()==Qt::NoBrush || polygonTransparency_==100)
		{
			polygonTransparency_ =  100;
			polygonBrush_.setStyle(Qt::SolidPattern);
		}

		QRect rect = painter_->viewport();

		int		j, xmin, ymin, xmax, ymax, height, width;
		TeBox	box = poly.box();
		QPoint	pll, pur;
		pll = mapWtoV (box.lowerLeft());
		pur = mapWtoV (box.upperRight());
		xmin = MAX (pll.x(), rect.left());
		xmax = MIN (pur.x(), rect.right());
		ymin = MAX (pur.y(), rect.top());
		ymax = MIN (pll.y(), rect.bottom());
		height = ymax - ymin;
		width = xmax - xmin;

		int	w = rect.width();
		int	h = rect.height();

		int r = w%8;
		if(r)
			w += (8-r);
		r = h%8;
		if(r)
			h += (8-r);

		QBitmap	bm;
		bm.resize(w, h);
		//Fill bitmap with 0-bits:background(or transparent)
		bm.fill(Qt::color0);
		QPainter qp(&bm);

		TeLinearRing ring = poly[0];
 		np = ring.size();
 		QPointArray a(np);
 		QPoint p;

		// Draw outer polygon with 1-bits: foreground(or opaque)
		qp.setBrush(Qt::color1);
		QPen pen(Qt::color1, polygonPen_.width());
		qp.setPen(pen);
 		for ( i = 0 ; i < np; i++)
 		{
			p = mapWtoV (ring[i]);
 			a.setPoint(i,p);
 		}
 		qp.drawPolygon( a );

		if(poly.size() > 1)
		{
			// Draw holes with 0-bits:background(or transparent)
			qp.setBrush(Qt::color0);
			pen.setColor(Qt::color0);
			qp.setPen(pen);
			unsigned int k;
			for ( k = 1; k < poly.size(); k++ )
  			{
  				ring = poly[k];
  				np = ring.size();
  				QPointArray hole(np);

  				for ( i = 0 ; i < np; i++)
  				{
					p = mapWtoV (ring[i]);
					hole.setPoint(i,p);
  				}
  				qp.drawPolygon( hole );
  			}
		}

		QRegion region(bm);
		qp.end();
		painter_->setClipRegion(region);

		QPixmap pixmap(rect.width(), rect.height());
		QPainter painter(&pixmap);
		painter.setBrush(polygonBrush_);
		if(bstyle == Qt::NoBrush)
			painter.setPen(polygonColor_);
		else
			painter.setPen(polygonPen_);
		painter.drawPolygon( a );

		QImage imap = pixmap.convertToImage();
		painter.end();

		if(qimage_.width()==0 || qimage_.height()==0)
		{
			qimage_.reset();
			qimage_ = pixmap2_->convertToImage();
		}

		QRgb polyRgb = polygonColor_.rgb() & 0x00ffffff;
//		QRgb contourRgb = polygonPen_.color().rgb() & 0x00ffffff; //contourRgb not used
		double alpha = (double)polygonTransparency_ / 100.;
		double beta = 1. - alpha;
		for(i=xmin; i<xmax; i++)
		{
			for(j=ymin; j<ymax; j++)
			{
				QRgb vp = imap.pixel(i, j);
				if(vp == polyRgb)
				{
					QRgb v = qimage_.pixel(i, j);
					int r = (int)((double)qRed(v) * alpha + (double)qRed(vp) * beta);
					int g = (int)((double)qGreen(v) * alpha + (double)qGreen(vp) * beta);
					int b = (int)((double)qBlue(v) * alpha + (double)qBlue(vp) * beta);
					QRgb t = qRgb(r, g, b);
					imap.setPixel(i, j, t);
				}
				else if(vp == 0)
					imap.setPixel(i, j, qimage_.pixel(i, j));
			}
		}
		pixmap.convertFromImage(imap);
		bitBlt (painter_->device(),xmin,ymin,&pixmap,xmin,ymin,width,height,Qt::CopyROP,true);
		painter_->setClipping(false);
		polygonBrush_.setStyle(bstyle);
		polygonTransparency_ = transp;
	}
}*/

QRect TeQtBasicCanvas::getLegendRect (QPoint p, const QPixmap* pix, string tx)
{
	QRect rect;
	TeVisual visual;
	TeColor cor(0, 0, 0);

	visual.fixedSize(true);
	visual.size(8);
	visual.bold(true);
	visual.color(cor);
	setTextColor (cor.red_, cor.green_, cor.blue_);
	string fam(visual.family());
	setTextStyle (fam, visual.size(), visual.bold(), visual.italic());

	if(pix)
	{
		int x = p.x()+pix->width()+3;
		int y = p.y()+13;
		QFontMetrics fm(textFont_);
		rect = fm.boundingRect(tx.c_str());
		rect.setRight(rect.right() + x);
		rect.setLeft(rect.left() + x);
		rect.setTop(rect.top() + y);
		rect.setBottom(rect.bottom() + y);
		QRect prect = pix->rect();
		prect.setRight(prect.right() + p.x());
		prect.setLeft(prect.left() + p.x());
		prect.setTop(prect.top() + p.y());
		prect.setBottom(prect.bottom() + p.y());
		rect |= prect;
	}
	else
	{
		int x = p.x()+1;
		int y = p.y()+13;
		QFontMetrics fm(textFont_);
		rect = fm.boundingRect(tx.c_str());
		rect = fm.boundingRect(tx.c_str());
		rect.setRight(rect.right() + x);
		rect.setLeft(rect.left() + x);
		rect.setTop(rect.top() + y);
		rect.setBottom(rect.bottom() + y);
	}
	return rect;
}

void TeQtBasicCanvas::plotLegend (QPoint p, const QPixmap* pix, string tx)
{
//	if(painter_ == 0)
//		return;

	TeVisual visual;
	TeColor cor(0, 0, 0);

	visual.fixedSize(true);
	visual.size(8);
	visual.bold(true);
	visual.color(cor);
	setTextColor (cor.red_, cor.green_, cor.blue_);
	string fam = visual.family();
	setTextStyle (fam, visual.size(), visual.bold(), visual.italic());
	painter_.setFont(textFont_);
	painter_.setPen(textPen_);

	if(pix)
	{
		int x = p.x()+pix->width()+3;
		int y = p.y()+13;
		bitBlt (painter_.device(), p.x(), p.y(), pix, 0, 0, pix->width(), pix->height(), Qt::CopyROP, true);
		painter_.drawText(x, y, tx.c_str());
	}
	else
	{
		int x = p.x()+1;
		int y = p.y()+13;
		painter_.drawText(x, y, tx.c_str());
	}
}

void TeQtBasicCanvas::plotLine (TeLine2D &line)
{
	TeBox b = getDataWorld ();
	if (!TeIntersects (b, line.box ()))
		return;
	int i,np;
	
	linePen_.setColor (lineColor_);
	painter_.setPen(linePen_);

	np = line.size();
	QPointArray a(np);
	QPoint p;
	for ( i = 0 ; i < np; i++)
	{
		p = mapWtoV (line[i]);
		a.setPoint(i,p);
	}
	painter_.drawPolyline( a );
}

void TeQtBasicCanvas::plotArc (TeArc &arc)
{
	TeBox b = getDataWorld ();
	if (!TeIntersects (b, arc.box ()))
		return;
	
	arcPen_.setColor (arcColor_);
	painter_.setPen(arcPen_);

	QPoint pfrom, pto;

	pfrom = mapWtoV (arc.fromNode().location());
	pto   = mapWtoV (arc.toNode().location());

	painter_.drawLine( pfrom, pto );

	double ang = atan2 (double(pfrom.y()-pto.y()), double (pfrom.x()-pto.x()));
	QPoint pm((int)((pto.x()+pfrom.x())/2.+0.5),(int)((pto.y()+pfrom.y())/2.+0.5));
	
	double ang1 = ang + 25.*TeCDR;
	double ang2 = ang - 25.*TeCDR;
	QPoint	p1((int)(10.*cos(ang1)+0.5),(int)(10.*sin(ang1)+.5)),
			p2((int)(10.*cos(ang2)+0.5),(int)(10.*sin(ang2)+.5));
	p1 += pm;
	p2 += pm;
	painter_.drawLine( pm, p1 );
	painter_.drawLine( pm, p2 );
	painter_.drawLine( p1, p2 );

}

void TeQtBasicCanvas::plotPie (double x, double y, double w, double h, double a, double alen)
{
	int	dx, dy, dw, dh, da, df;

	pieBrush_.setColor (pieColor_);
	painter_.setBrush(pieBrush_);
	painter_.setPen(linePen_);

	TeCoord2D p(x, y);
	QPoint	qp = mapWtoV(p);
	dx = qp.x();
	dy = qp.y();

	TeCoord2D pt(x+w, y+h);
	QPoint	qpt = mapWtoV(pt);
	dw = abs(qpt.x() - dx);
	dh = abs(qpt.y() - dy);

	da = (int)(a * 16);
	df = (int)(alen * 16);
	painter_.drawPie(dx, dy-dh, dw, dh, da, df);
}

void TeQtBasicCanvas::setPieColor (int r, int g, int b)
{
	pieColor_.setRgb(r, g, b);
	pieBrush_.setStyle(Qt::SolidPattern);
}

void TeQtBasicCanvas::plotRect (QRect& rect)
{
	rectBrush_.setColor (rectColor_);
	painter_.setBrush(Qt::NoBrush);
	painter_.setPen(linePen_);
	painter_.drawRect(rect);
}

void TeQtBasicCanvas::plotRect (double x, double y, double w, double h, int transp, bool legend)
{
	if(pixmap1_ == 0)
		return;

	int	dx, dy, dw, dh;

	TeCoord2D p(x, y);
	QPoint	qp;
	if(legend)
		qp = QPoint((int)p.x(), (int)p.y());
	else
		qp = mapWtoV(p);
	dx = qp.x();
	dy = qp.y();

	TeCoord2D pt(x+w, y+h);
	QPoint	qpt;
	if(legend)
		qpt = QPoint((int)pt.x(), (int)pt.y());
	else
		qpt = mapWtoV(pt);
	dw = abs(qpt.x() - dx);
	dh = abs(qpt.y() - dy);

	if(transp == 0)
	{
		rectBrush_.setColor (rectColor_);
		painter_.setBrush(rectBrush_);
		painter_.setPen(linePen_);
		painter_.drawRect(dx, dy-dh, dw, dh);
	}
	else
	{
		double alpha = transp / 100.;
		double beta = 1 - alpha;
		double red = rectColor_.red() * beta;
		double green = rectColor_.green() * beta;
		double blue = rectColor_.blue() * beta;

		int r = dw%8;
		if(r)
			dw += (8-r);
		r = dh%8;
		if(r)
			dh += (8-r);

//		QImage ima = pixmap1_->convertToImage();
		QImage ima = pixmap0_->convertToImage();
		QImage imap(dw, dh, 32);
		int i, j;
		int x = qp.x();
		int	y = qp.y() - dh;
		for(i=x; i-x<dw && i<ima.width(); i++)
		{
			if(i < 0)
				continue;
			for(j=y; j-y<dh && j<ima.height(); j++)
			{
				if(j < 0)
					continue;
				QRgb v = ima.pixel(i, j);
				int r = (int)((double)qRed(v) * alpha + red);
				int g = (int)((double)qGreen(v) * alpha + green);
				int b = (int)((double)qBlue(v) * alpha + blue);
				QRgb t = qRgb(r, g, b);
				imap.setPixel(i-x, j-y, t);
			}
		}

		QPixmap pm(dw, dh);
		pm.convertFromImage(imap);
		QRect rec = pm.rect();
		rec &= painter_.viewport();
		bitBlt (painter_.device(), x, y, &pm, 0, 0, rec.width(), rec.height(), Qt::CopyROP, true);
	}
}

void TeQtBasicCanvas::setRectColor (int r, int g, int b)
{
	rectColor_.setRgb(r, g, b);
	rectBrush_.setStyle(Qt::SolidPattern);
}

static inline int blendComponent( int back, int fore, int falpha )
{
	int balpha = 255 - falpha;
    int a = falpha + balpha -(falpha*balpha)/255;
    return (falpha*fore + balpha*back -(balpha*falpha*fore)/255)/a;  
}

int d2dms(double v)
{
    double av;
    int d, m, s;
    av = fabs(v);
    d = (int)av;
    av = av - d;
    av = av * 60;
    m = (int)av;
    av = av - m;
    av = av * 60;
    s = (int)av;
    int gms = d * 10000 + m * 100 + s ;
	return gms;
}

double dms2d (int vs)
{
	int v = abs(vs);
	int dd = v/10000;
	int dm = (v-dd*10000)/100;
	int ds = v - dd*10000 - dm*100;
	double d = dd + dm/60. + ds/3600.;
	if (vs < 0)
		d = -d;
	return d;
}

string buildMosaicName (int x, int y, int delta)
{
	int		lad,lam,las,lod,lom,los,dd,dm,ds;
	char	hemis,hemiss;
	int		lx1,ly1;

	hemis = 'S';
	if (y>=0)hemis = 'N';
	ly1 = abs(y);
	lad = ly1/3600;
	lam = (ly1-lad*3600)/60;
	las = ly1 - lad*3600 - lam*60;

	hemiss = 'O';
	if (x>=0)hemis = 'E';
	lx1 = abs(x);
	lod = lx1/3600;
	lom = (lx1-lod*3600)/60;
	los = lx1 - lod*3600 - lom*60;

	dd = delta/10000;
	dm = (delta-dd*10000)/100;
	ds = delta - dd*10000 - dm*100;

	char name[128];
	sprintf(name,"%c%02d%02d%02d%c%02d%02d%02d%02d%02d%02d",
		hemis,lad,lam,las,hemiss,lod,lom,los,dd,dm,ds);
	return string(name);
}

void TeQtBasicCanvas::copyPixmap0ToPrinter()
{
	if (pixmap0_)
	{
		QPrinter printer;
		if (printer.setup())
		{
			QPainter p(&printer);

			QPaintDevice* dev = painter_.device();
			if(dev == pixmap0_)
				painter_.end();

			p.drawPixmap(0, 0, *pixmap0_);
			p.flush();
			p.end();

			if(dev && painter_.device() == 0)
				painter_.begin(dev);
		}
	}
}

void TeQtBasicCanvas::copyPixmapToWindow(QPixmap* p, int ulx, int uly, int w, int h)
{
	bitBlt (viewport(),ulx,uly,p,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap0To(QPaintDevice* dev)
{
	if (pixmap0_)
	{
		bitBlt (dev,0,0,pixmap0_,0,0,width_,height_,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPixmap0ToWindow()
{
	if (pixmap0_)
	{
		bitBlt (viewport(),0,0,pixmap0_,0,0,width_,height_,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPixmap1ToWindow()
{
	if (pixmap1_)
	{
		bitBlt (viewport(),0,0,pixmap1_,0,0,width_,height_,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPixmap2ToWindow()
{
	if (pixmap2_)
	{
		bitBlt (viewport(),0,0,pixmap2_,0,0,width_,height_,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPixmap3ToWindow()
{
	if (pixmap3_)
	{
		bitBlt (viewport(),0,0,pixmap3_,0,0,width_,height_,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPixmap0ToWindow(int ulx, int uly, int w, int h)
{
	if (pixmap0_)
	{
		bitBlt (viewport(),ulx,uly,pixmap0_,ulx,uly,w,h,Qt::CopyROP,true);
	}
}

void TeQtBasicCanvas::copyPanArea(int x, int y)
{
	if (pixmap0_)
	{
		QPaintDevice* dev = painter_.device();
		if(dev)
			painter_.end();
		painter_.begin(viewport());
		int	w, h;
		QRect a, b;
		int hh = pixmap0_->height();
		int ww = pixmap0_->width();
		if(x <= 0 && y <= 0)
		{
			a = QRect(0, 0, ww, -y);
			b = QRect(0, -y, -x, hh+y);
			w = ww + x;
			h = hh + y;
			painter_.fillRect(a, painter_.backgroundColor());
			painter_.fillRect(b, painter_.backgroundColor());
			bitBlt (viewport(), -x, -y, pixmap1_, 0, 0, w, h, Qt::CopyROP, true);
		}
		else if(x >= 0 && y >= 0)
		{
			a = QRect(0, hh-y, ww, y);
			b = QRect(ww-x, 0, x, hh-y);
			w = ww - x;
			h = hh - y;
			painter_.fillRect(a, painter_.backgroundColor());
			painter_.fillRect(b, painter_.backgroundColor());
			bitBlt (viewport(), 0, 0, pixmap1_, x, y, w, h, Qt::CopyROP, true);
		}
		else if(x >= 0 && y <= 0)
		{
			a = QRect(0, 0, ww, -y);
			b = QRect(ww-x, -y, x, hh+y);
			w = ww - x;
			h = hh + y;
			painter_.fillRect(a, painter_.backgroundColor());
			painter_.fillRect(b, painter_.backgroundColor());
			bitBlt (viewport(), 0, -y, pixmap1_, x, 0, w, h, Qt::CopyROP, true);
		}
		else
		{
			a = QRect(0, hh-y, ww, y);
			b = QRect(0, 0, -x, hh-y);
			w = ww + x;
			h = hh - y;
			painter_.fillRect(a, painter_.backgroundColor());
			painter_.fillRect(b, painter_.backgroundColor());
			bitBlt (viewport(), -x, 0, pixmap1_, 0, y, w, h, Qt::CopyROP, true);
		}
		if(dev)
			painter_.begin(dev);
	}
}

void TeQtBasicCanvas::copyPixmap0ToPixmap1()
{
	if (pixmap1_ && pixmap0_)
		bitBlt (pixmap1_,0,0,pixmap0_,0,0,width_,height_,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap0ToPixmap1(int ulx, int uly, int w, int h)
{
	if (pixmap1_ && pixmap0_)
		bitBlt (pixmap1_,ulx,uly,pixmap0_,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap1ToPixmap0()
{
	if (pixmap1_ && pixmap0_)
		bitBlt (pixmap0_,0,0,pixmap1_,0,0,width_,height_,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap1ToPixmap0(int ulx, int uly, int w, int h)
{
	if (pixmap1_ && pixmap0_)
		bitBlt (pixmap0_,ulx,uly,pixmap1_,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap1ToPixmap2()
{
	if (pixmap1_ && pixmap2_)
	{
		bitBlt (pixmap2_,0,0,pixmap1_,0,0,width_,height_,Qt::CopyROP,true);
		qimage_.reset();
		qimage_ = pixmap2_->convertToImage();
	}
}

void TeQtBasicCanvas::copyPixmap1ToPixmap2(int ulx, int uly, int w, int h)
{
	if (pixmap1_ && pixmap2_)
	{
		bitBlt (pixmap2_,ulx,uly,pixmap1_,ulx,uly,w,h,Qt::CopyROP,true);
		qimage_.reset();
		qimage_ = pixmap2_->convertToImage();
	}
}

void TeQtBasicCanvas::copyPixmap2ToPixmap1()
{
	if (pixmap1_ && pixmap2_)
		bitBlt (pixmap1_,0,0,pixmap2_,0,0,width_,height_,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap2ToPixmap1(int ulx, int uly, int w, int h)
{
	if (pixmap1_ && pixmap2_)
		bitBlt (pixmap1_,ulx,uly,pixmap2_,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap2ToPixmap3()
{
	if (pixmap3_ && pixmap2_)
		bitBlt (pixmap3_,0,0,pixmap2_,0,0,width_,height_,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap2ToPixmap3(int ulx, int uly, int w, int h)
{
	if (pixmap3_ && pixmap2_)
		bitBlt (pixmap3_,ulx,uly,pixmap2_,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap3ToPixmap2()
{
	if (pixmap3_ && pixmap2_)
		bitBlt (pixmap2_,0,0,pixmap3_,0,0,width_,height_,Qt::CopyROP,true);
}

void TeQtBasicCanvas::copyPixmap3ToPixmap2(int ulx, int uly, int w, int h)
{
	if (pixmap3_ && pixmap2_)
		bitBlt (pixmap2_,ulx,uly,pixmap3_,ulx,uly,w,h,Qt::CopyROP,true);
}

void TeQtBasicCanvas::plotRaster(TeRaster* raster, TeRasterTransform* transf, TeQtProgress *progress)
{
	QPixmap* pix = (QPixmap*)(painter_.device());
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
	params_.fileName_ = raster->params().fileName_;

	if (buildRaster ())
	{
		// Calculates the box of input image that intersects the box of the canvas
		TeBox bboxBackRaster = backRaster_->params().boundingBox();		
		TeBox bboxSearched = TeRemapBox(bboxBackRaster, backRaster_->projection(), raster->projection());
		TeBox bboxIntersection;
		if (!TeIntersection (raster->params().boundingBox(),bboxSearched,bboxIntersection))
			return ;			// no intersection 

		// Create a remapping tool to back raster
		TeDecoderQtImage* decqt = reinterpret_cast<TeDecoderQtImage*>(backRaster_->decoder());
		TeRasterRemap remap;
		if (transf)
			remap.setTransformer(transf);
		remap.setOutput(backRaster_);

		// Calculates best resolution level to display the input image on this canvas
		TeBox box = params_.boundingBox();
		int nlines = params_.nlines_;
		int ncols = params_.ncols_;
		int res = raster->decoder()->bestResolution(box, nlines, ncols, params_.projection());

		// Check if raster blocks in best level of resolution that intersects the canvas box
		TeRasterParams parBlock;	
		if (raster->selectBlocks(bboxIntersection,res,parBlock))        
		{
			params_.resolution_ = res;	
			if (progress)
				progress->setTotalSteps(raster->numberOfSelectedBlocks());
			t2 = clock();
			t0 = t1 = t2;
			
			// Process each block as an independent raster decoded in memory
			TeRaster* block = new TeRaster;
			TeDecoderMemory* decMem = new TeDecoderMemory(parBlock);
			decMem->init();
			remap.setInput(block);

			// Portal of raster block selection behaves as portal of geometries
			// use the "bool flag - do - while" scheme
			int numBlockProcessed=0;
			bool flag = true;
			do
			{
				flag = raster->fetchRasterBlock(decMem);
				block->setDecoder(decMem);
				remap.apply();
				numBlockProcessed++;

				TeRasterParams par = decMem->params();
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					if((int)(t2-t0) > dt2)	// show progress and refresh painted area
					{
						if (progress)
						{
							if (progress->wasCancelled())
								break;
							else
								progress->setProgress(numBlockProcessed);
						}
						if(pix == pixmap0_ || pix == pixmap1_ || pix == pixmap2_ || pix == pixmap3_)
						{
							QPaintDevice* dev = painter_.device();
							if(dev)
								painter_.end();
							if(dev == pixmap0_)
							{
								pixmap0_->convertFromImage (*(decqt->getImage ()));
								copyPixmap0ToWindow();
							}
							else if(dev == pixmap1_)
							{
								pixmap1_->convertFromImage (*(decqt->getImage ()));
								copyPixmap1ToWindow();
							}
							else if(dev == pixmap2_)
							{
								pixmap2_->convertFromImage (*(decqt->getImage ()));
								copyPixmap2ToWindow();
							}
							else if(dev == pixmap3_)
							{
								pixmap3_->convertFromImage (*(decqt->getImage ()));
								copyPixmap3ToWindow();
							}
							if(dev)
								painter_.begin(dev);
						}
						else
							pix->convertFromImage (*(decqt->getImage ()));
					}
				}
			} while (flag);
			if (progress)
				progress->reset();
			decMem->clear();
			delete block;
			raster->clearBlockSelection();
		}
		else		// no blocks found try to remap the whole raster
		{
			remap.setInput(raster);
			remap.apply(true);	
		}

		if(pix == pixmap0_ || pix == pixmap1_ || pix == pixmap2_ || pix == pixmap3_)
		{
			QPaintDevice* dev = painter_.device();
			if(dev)
				painter_.end();

			if(dev == pixmap0_)
			{
				pixmap0_->convertFromImage (*(decqt->getImage ()));
				copyPixmap0ToWindow();
			}
			else if(dev == pixmap1_)
			{
				pixmap1_->convertFromImage (*(decqt->getImage ()));
				copyPixmap1ToWindow();
			}
			else if(dev == pixmap2_)
			{
				pixmap2_->convertFromImage (*(decqt->getImage ()));
				copyPixmap2ToWindow();
			}
			else if(dev == pixmap3_)
			{
				pixmap3_->convertFromImage (*(decqt->getImage ()));
				copyPixmap3ToWindow();
			}
			if(dev)
				painter_.begin(dev);
		}
		else
			pix->convertFromImage (*(decqt->getImage ()));
	}
}

bool TeQtBasicCanvas::buildRaster ()
{
/*	if (backRaster_)
	{
		TeRasterParams  rparams = backRaster_->params();
		if (!(rparams.box() == params_.box()) ||
			  rparams.ncols_ != params_.ncols_ ||
			  rparams.nlines_ != params_.nlines_)
		{  
			delete backRaster_;
			backRaster_ = 0;
		}
		else
		{
			if (rparams.fileName_ != params_.fileName_)
				return true;
			return false;
		}
	}
*/
	if (!backRaster_)
	{
		backRaster_ = new TeRaster();
		params_.mode_ = 'c';
		params_.useDummy_ = true;
		params_.setDummy(255);
		TeDecoderQtImage *dec = new TeDecoderQtImage(params_);
		backRaster_->setDecoder (dec);
		backRaster_->init();
	}
	return true;
}

void TeQtBasicCanvas::endPrinting()
{
	QPaintDevice* dev = painter_.device();
	if(dev)
		painter_.end();
	if(pixmap0_)
		painter_.begin(pixmap0_);
}

void TeQtBasicCanvas::plotGraphicScale(TeVisual& visual)
{
	TeColor	cor = visual.color();
	string	family = visual.family();
	int		size = visual.size();
	bool	bold = visual.bold();
	bool	italic = visual.italic();
	TeBox	box = getWorld();
	double	w = box.width();
	double	dx = w / 9.;
	long	idx = (long)dx;
	double	f;
	double	fa = 1.;
	int		fw, fh;

	if(box.isValid() == false)
		return;

	int conta = 1000;
	if(idx > 0)
	{
		while(--conta)
		{
			dx /= 10.;
			fa *= 10.;
			idx = (long)dx;
			if(idx == 0)
			{
				idx = (long)(dx * 10.);
				f = idx * fa / 10.;
				break;
			}
		}
	}
	else
	{
		while(--conta)
		{
			dx *= 10.;
			fa /= 10.;
			idx = (long)dx;
			if(idx > 0)
			{
				f = idx * fa;
				break;
			}
		}
	}
	if(conta == 0)
		return;

	double space = f * 3.;
	double hini = box.x1_ + (w - space) / 2.;

	QFont font(family.c_str(), size);
	font.setBold (bold);
	font.setItalic (italic);
	painter_.setFont(font);

	QFontMetrics fm(font);
	QRect rect;
	string unit = canvasProjection_->units();
	rect = fm.boundingRect(unit.c_str());
	fh = rect.height();
	double vini = box.y1_ + mapVtoW(fh+6);

	TeCoord2D wp(hini, vini);
	QPoint p1((int)((wp.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((wp.y() - ymin_)*f_ + 0.5));
	correctScrolling (p1);

	wp.x(wp.x() + f);
	QPoint p2((int)((wp.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((wp.y() - ymin_)*f_ + 0.5));
	correctScrolling (p2);

	wp.x(wp.x() + f);
	QPoint p3((int)((wp.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((wp.y() - ymin_)*f_ + 0.5));
	correctScrolling (p3);

	wp.x(wp.x() + f);
	QPoint p4((int)((wp.x() - xmin_)*f_ + 0.5)+x0_,
		height_ - y0_ - (int)((wp.y() - ymin_)*f_ + 0.5));
	correctScrolling (p4);

	QPen pen(QColor(cor.red_, cor.green_, cor.blue_));
	painter_.setPen(pen);

	if(bold == false)
	{
		QPoint pa = p2;
		QPoint pb = p3;
		painter_.drawLine(pa, pb);

		pa = p1;
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);

		pa = p2; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);

		pa = p3; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);

		pa = p4; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);

		pa = p1;
		pb = p4;
		pa.setY(pa.y() - 1);
		pb.setY(pb.y() - 1);
		painter_.drawLine(pa, pb);
		pa.setY(pa.y() + 2);
		pb.setY(pb.y() + 2);
		painter_.drawLine(pa, pb);
	}
	else
	{
		QPoint pa = p2;
		QPoint pb = p3;
		painter_.drawLine(pa, pb);
		pa.setY(pa.y()-1);
		pb.setY(pb.y()-1);
		painter_.drawLine(pa, pb);

		pa = p1;
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);
		pa.setX(pa.x()-1);
		pb.setX(pb.x()-1);
		painter_.drawLine(pa, pb);

		pa = p2; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);
		pa.setX(pa.x()-1);
		pb.setX(pb.x()-1);
		painter_.drawLine(pa, pb);

		pa = p3; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);
		pa.setX(pa.x()-1);
		pb.setX(pb.x()-1);
		painter_.drawLine(pa, pb);

		pa = p4; 
		pb = pa;
		pa.setY(pa.y() - 5);
		pb.setY(pb.y() + 4);
		painter_.drawLine(pa, pb);
		pa.setX(pa.x()-1);
		pb.setX(pb.x()-1);
		painter_.drawLine(pa, pb);

		pa = p1;
		pb = p4;
		pa.setY(pa.y() - 2);
		pb.setY(pb.y() - 2);
		painter_.drawLine(pa, pb);
		pa.setY(pa.y() + 3);
		pb.setY(pb.y() + 3);
		painter_.drawLine(pa, pb);
	}

	rect = fm.boundingRect("0");
	fw = rect.width();
	painter_.drawText (p1.x()-fw/2, p1.y()-8, "0");

	char buf[50];
	long n = (long)f;

	if(n > 0)
	{
		sprintf(buf, "%ld", n);
		QString s = buf;
		rect = fm.boundingRect(s);
		int fw1 = rect.width();

		sprintf(buf, "%ld", n*2);
		s = buf;
		rect = fm.boundingRect(s);
		int fw2 = rect.width();

		sprintf(buf, "%ld", n*3);
		s = buf;
		rect = fm.boundingRect(s);
		int fw3 = rect.width();

		if((fw1 + fw2)/2 < p3.x()-p2.x() &&
			(fw2 + fw3)/2 < p3.x()-p2.x())
		{
			sprintf(buf, "%ld", n);
			QString s = buf;
			rect = fm.boundingRect(s);
			fw = rect.width();
			painter_.drawText (p2.x()-fw/2, p2.y()-8, s);

			sprintf(buf, "%ld", n*2);
			s = buf;
			rect = fm.boundingRect(s);
			fw = rect.width();
			painter_.drawText (p3.x()-fw/2, p3.y()-8, s);
		}

		sprintf(buf, "%ld", n*3);
		s = buf;
		rect = fm.boundingRect(s);
		fw = rect.width();
		painter_.drawText (p4.x()-fw/2, p4.y()-8, s);
	}
	else
	{
		int nn=0;
		int fn = (int)fa;
		while(fn == 0)
		{
			fa *= 10;
			fn = (int)fa;
			nn++;
		}
		sprintf (buf, "%.*f", nn, f);
		QString s = buf;
		rect = fm.boundingRect(s);
		int fw1 = rect.width();

		sprintf (buf, "%.*f", nn, f*2.);
		s = buf;
		rect = fm.boundingRect(s);
		int fw2 = rect.width();

		sprintf (buf, "%.*f", nn, f*3.);
		s = buf;
		rect = fm.boundingRect(s);
		int fw3 = rect.width();

		if((fw1 + fw2)/2 < p3.x()-p2.x() &&
			(fw2 + fw3)/2 < p3.x()-p2.x())
		{
			sprintf (buf, "%.*f", nn, f);
			QString s = buf;
			rect = fm.boundingRect(s);
			fw = rect.width();
			painter_.drawText (p2.x()-fw/2, p2.y()-8, s);

			sprintf (buf, "%.*f", nn, f*2.);
			s = buf;
			rect = fm.boundingRect(s);
			fw = rect.width();
			painter_.drawText (p3.x()-fw/2, p3.y()-8, s);
		}

		sprintf (buf, "%.*f", nn, f*3.);
		s = buf;
		rect = fm.boundingRect(s);
		fw = rect.width();
		painter_.drawText (p4.x()-fw/2, p4.y()-8, s);
	}

	int sw = p4.x() - p1.x();
	rect = fm.boundingRect(unit.c_str());
	fw = rect.width();
	fh = rect.height();
	int suini = (sw - fw) / 2 + p1.x();
	painter_.drawText (suini, p1.y()+fh+2, unit.c_str());
}

void TeQtBasicCanvas::setClipRegion(int x, int y, int w, int h)
{
	QRegion region(x, y, w, h);
	if(painter_.device())
		painter_.setClipRegion(region);
}

void TeQtBasicCanvas::setClipRegion(QRegion region)
{
	if(painter_.device())
		painter_.setClipRegion(region);
}

void TeQtBasicCanvas::setClipping(bool enable)
{
	if(painter_.device())
		painter_.setClipping(enable);
}

void TeQtBasicCanvas::clearRaster()
{
	if (backRaster_)
		delete backRaster_;
	backRaster_ = 0;
}
