/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeException.h"
#include "TeDecoderASCIIGrid.h"
#include "TeAsciiFile.h"
#include "TeUtils.h"
#ifdef WIN32
#include <io.h>
#else
#include <unistd.h>
#endif

/*
Expected format for ESRI ASCII GRID Files (Copied from the ArcWorkstation 8.3 Help File):

<NCOLS xxx>
<NROWS xxx>
<XLLCENTER xxx | XLLCORNER xxx>
<YLLCENTER xxx | YLLCORNER xxx>
<CELLSIZE xxx>
{NODATA_VALUE xxx}
row 1
row 2
.
.
.
row n
*/

TeDecoderASCIIGrid::TeDecoderASCIIGrid (const TeRasterParams& par) : 
	TeDecoderMemory()
{	
	if (par.fileName_.empty())
		return;
	params_ = par;
	params_.decoderIdentifier_ = "ASCIIGRID";
	if (params_.mode_ == 'w' || params_.mode_ == 'r')
		readParameters();
}

TeDecoderASCIIGrid::~TeDecoderASCIIGrid ()
{
	if ( isModified_ && (params_.mode_ == 'w' || params_.mode_ == 'c'))
	{			// save contents to disk
		try
		{
			TeAsciiFile sFile(params_.fileName_,"w+");
			if (writeParameters(sFile))
				saveData(sFile);
		}
		catch (...)
		{
		}
	}
	TeDecoderMemory::clear();
}

bool
TeDecoderASCIIGrid::clear()
{
	if ( isModified_ && (params_.mode_ == 'w' || params_.mode_ == 'c'))
	{			// save contents to disk
		try
		{
			TeAsciiFile sFile(params_.fileName_,"w+");
			if (writeParameters(sFile))
				saveData(sFile);
		}
		catch (...)
		{
		}
		isModified_ = false;
	}
	TeDecoderMemory::clear();
	return true;
}

bool 
TeDecoderASCIIGrid::readParameters()
{
	try
	{
		TeAsciiFile	pFile (params_.fileName_);
		string name;
		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"NCOLS"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		params_.ncols_ = pFile.readInt();
		pFile.findNewLine();

		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"NROWS"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		params_.nlines_ = pFile.readInt();
		pFile.findNewLine();

		bool isCenter = false;
		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"XLLCENTER"))
			{
				isCenter = true; 
				break;
			}
			else if (TeStringCompare(name,"XLLCORNER"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		double llx = pFile.readFloat();
		pFile.findNewLine();

		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"YLLCENTER"))
			{
				isCenter = true; 
				break;
			}
			else if (TeStringCompare(name,"YLLCORNER"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		double lly = pFile.readFloat();	
		pFile.findNewLine();

		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"CELLSIZE"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		params_.resx_ = params_.resy_ = pFile.readFloat();		
		pFile.findNewLine();

		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (TeStringCompare(name,"NODATA_VALUE"))
				break;
		}
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		double dummy = pFile.readFloat();		
		params_.useDummy_ = true;
		params_.nBands(1);
		params_.setDummy(dummy);
		params_.nBands(1);
		params_.useDummy_ = true;
		params_.setDummy(dummy);
		params_.setDataType(TeDOUBLE);
		params_.setPhotometric(TeRASTERMULTIBAND);
		params_.lowerLeftResolutionSize(llx,lly,params_.resx_,params_.resy_,
				                      params_.ncols_,params_.nlines_,isCenter);
		TeProjection* pp = new TeNoProjection();
		params_.projection(pp);
		delete pp;
	}
	catch(...)
	{
		return false;
	}
	return true;
}

bool
TeDecoderASCIIGrid::readFile(const string& filename)
{
	string valstring;
	try 
	{
		TeAsciiFile	pFile (filename);
		string name;
		while (pFile.isNotAtEOF() && !TeStringCompare(name,"NODATA_VALUE"))
			name = pFile.readString();
		if (!pFile.isNotAtEOF())					// unexpected end of file
			return false;
		pFile.findNewLine();
		for (int lin = 0; lin < params_.nlines_; ++lin)
		{
			for (int col = 0; col < params_.ncols_; ++col)
			{
				setElement(col,lin,pFile.readFloat());
			}
		}
	}
	catch(...)
	{
		return false;
	}
	return true;
}

void
TeDecoderASCIIGrid::init()
{
	params_.status_= TeNOTREADY;
	int nb = params_.nBands();
	if (nb != 1 )
		return;

	TeDecoderMemory::init();			// try to allocate enough memory
	if (params_.status_ == TeNOTREADY)
		return;

	params_.status_= TeNOTREADY;
	if (params_.mode_ == 'c')	// creating a new file
	{
		if (access(params_.fileName_.c_str(),00) != -1) // remove existing file
		{
			if (unlink(params_.fileName_.c_str()) == -1)
				return;
		}
		try
		{
			TeAsciiFile sFile(params_.fileName_,"w+");
			if (!writeParameters(sFile) || !saveData(sFile))
				return;
			else
			{
				params_.status_ = TeREADYTOWRITE;
			}
		}
		catch (...)
		{
			return;
		}
	}
	else if (params_.mode_ == 'w')
	{
		if ((access(params_.fileName_.c_str(),06) == -1) || !readFile(params_.fileName_))
			return;
		params_.status_ = TeREADYTOWRITE;
	}
	else if (params_.mode_ == 'r')
	{
		if ((access(params_.fileName_.c_str(),04) == -1)|| !readFile(params_.fileName_))
			return;
		params_.status_ = TeREADYTOREAD;	
	}
}

bool 
TeDecoderASCIIGrid::writeParameters(TeAsciiFile& pFile)
{
	string name;
	try
	{
		TeBox box = params_.box();
		name = "NCOLS " + Te2String(params_.ncols_) + "\n";
		pFile.writeString(name);
		name = "NROWS " + Te2String(params_.nlines_) + "\n";
		pFile.writeString(name);
		name = "XLLCENTER " + Te2String(box.x1_,6) + "\n";
		pFile.writeString(name);
		name = "YLLCENTER " + Te2String(box.y2_,6) + "\n";
		pFile.writeString(name);
		name = "CELLSIZE " + Te2String(params_.resx_,6) + "\n";
		pFile.writeString(name);
		name = "NODATA_VALUE " + Te2String(params_.dummy_[0],6) + "\n";
		pFile.writeString(name);
	}
	catch (...)
	{
		return false;
	}
	return true;
}

bool 
TeDecoderASCIIGrid::saveData(TeAsciiFile& pFile)
{
	bool isi = (params_.dataType_[0] != TeDOUBLE) && (params_.dataType_[0] != TeFLOAT);

	FILE* fp = pFile.FilePtr();
	char fmt[100];
	if (isi)
		strcpy(fmt,"%.0f ");
	else
		strcpy(fmt,"%f ");
	try
	{
		double d;
		for (int l=0; l<params_.nlines_; ++l)
		{
			for (int c=0; c<params_.ncols_; ++c)
			{
				TeDecoderMemory::getElement(c,l,d);
				fprintf(fp,fmt,d);
			}
			pFile.writeNewLine();
		}
	}
	catch(...)
	{
		return false;
	}
	return true;
}
