/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include <TeDecoderDatabase.h>
#include <TeDecoderMemory.h>
#include <TeRasterRemap.h>
#include <zlib.h>
#ifdef WIN32
#include <jpeg.h>
#endif
#include <TeDataTypes.h>
#include "TeProgress.h"
#include "TeVectorRemap.h"

TeDecoderDatabase::TeDecoderDatabase(const TeRasterParams& par ):
	TeDecoderVirtualMemory(par),
	db_(par.database_),
	blockPortal_(0),
	memAux_(0),
	nSelectedBlocks_(0)
{
	params_ = par;
	params_.decoderIdentifier_ = "DB";
}

TeDecoderDatabase::~TeDecoderDatabase()
{
	clearBlockSelection();
	clear();
}

bool
TeDecoderDatabase::create()
{
	if (db_ == 0)
		return false;

	// Create raster table in database
  	if (!db_->createRasterGeometry (params_.fileName_))
		return false;
	return true;
}

void
TeDecoderDatabase::init()
{
	clearBlockSelection();
	clear();
	params_.status_= TeNOTREADY;
	if (db_ == 0)			// there is not valid database connection
		return;

	if (params_.mode_ == 'c')
	{
		if (db_->tableExist(params_.fileName_))
		{
			string sql = "DROP TABLE " + params_.fileName_;
			db_->execute(sql);
		}
  		if (!db_->createRasterGeometry (params_.fileName_))
			return;
		else params_.status_ = TeREADYTOWRITE;
	}
	else if (params_.mode_ == 'w')
	{
		if (db_->tableExist(params_.fileName_))
			params_.status_ = TeREADYTOWRITE;
	}
	else if (params_.mode_ == 'r')
	{
		if (db_->tableExist(params_.fileName_))
			params_.status_ = TeREADYTOREAD;		
	}
	TeDecoderVirtualMemory::init();
	return;
}

bool
TeDecoderDatabase::clear()
{
	if (db_ == 0)
		return false;
	TeDecoderVirtualMemory::clear();
	if (blockPortal_)
		delete blockPortal_;
	blockPortal_ = 0;
	if (memAux_)
		delete []memAux_;
	memAux_ = 0;
	params_.status_ = TeNOTREADY;
	return true;
}

string 
TeDecoderDatabase::codifyId(int col, int lin, int band, int /*res*/, int /*subb*/)
{
	char id[32];
	if (params_.tiling_type_ == TeExpansible)
	{
		TeCoord2D tmpCoord = TeCoord2D(col,lin);
		TeCoord2D xy = params_.index2Coord(tmpCoord);

		double bXSize = params_.blockWidth_*params_.resx_;
		double bYSize = params_.blockHeight_*params_.resy_;

		int magicX, magicY;
		if ( xy.x() < 0)
			magicX = (int)(xy.x()/bXSize - 1);
		else
			magicX = (int)(xy.x()/bXSize);

		if (xy.y() < 0)
			magicY = (int) (xy.y()/bYSize-1);
		else
			magicY = (int) (xy.y()/bYSize);

		sprintf(id,"X%dY%dB%dR%dS0",magicX,magicY,band,params_.resolution_);
	}
	else
		sprintf(id,"X%dY%dB%dR%dS0",
	    (int)(col/params_.blockWidth_),(int)(lin/params_.blockHeight_),band,params_.resolution_);

	return string(id);
}

void 
TeDecoderDatabase::decodifyId(const string& id, int& col,int& lin, int& band, int& res, int& subb)
{
	char lixo;
	int magicX, magicY;
	if (params_.tiling_type_ == TeExpansible)
	{
		sscanf(id.c_str(),"%1c%d%1c%d%1c%d%1c%d%1c%d",&lixo,&magicX,&lixo,&magicY,&lixo,&band,&lixo,&res,&lixo,&subb);

		double bXSize = params_.blockWidth_*params_.resx_;
		double bYSize = params_.blockHeight_*params_.resy_;
		
		TeCoord2D xy(magicX*bXSize+params_.resx_/2,magicY*bYSize+params_.resy_/2);

		xy = params_.coord2Index(xy);
		lin = TeRound(xy.y())-params_.blockHeight_+1;
		col = TeRound(xy.x());
	}
	else 
	{
		sscanf(id.c_str(),"%1c%d%1c%d%1c%d%1c%d%1c%d",&lixo,&magicX,&lixo,&magicY,&lixo,&band,&lixo,&res,&lixo,&subb);
		col = magicX*params_.blockWidth_;
		lin = magicY*params_.blockHeight_;
	}
}

bool 
TeDecoderDatabase::getRasterBlock(const string& index, void *block, int& ulCol, int& ulLin)
{

	TeDatabasePortal* portal = db_->getPortal();
	if (!portal) 
		return false;
	
	string q;					// try to get the tile from databatase
	q ="SELECT * FROM " + params_.fileName_ + " WHERE block_id='" + index + "'";

	if (!portal->query(q))		// error submting query 
	{
		delete portal;
		return 0;
	}

	int res, subb, band=0;
	if (!portal->fetchRow())	// tile is not in the database
	{
		decodifyId(index,ulCol,ulLin,band,res, subb);
		delete portal;
		return 0;
	}

	if (!memAux_)
	{
		long size = params_.blockHeight_ * params_.blockWidth_ * params_.nbitsperPixel_[0]/8;
		memAux_ = new unsigned char[size];
	}

	// tile is already in the database
	unsigned long blobLen;
	if (!portal->getRasterBlock(blobLen,memAux_))
	{
		delete portal;
		return false;
	}

	// decompress blob into a tile (previously allocated) 
	if (params_.compression_[band] == TeZLib)		// zlib compression
	{
		unsigned long blockLen;	// size after decompression
		uncompress (reinterpret_cast<unsigned char*>(block),&blockLen,memAux_,blobLen);
	}
#ifdef WIN32
	else if (params_.compression_[band] == TeJPEG) //jpeg compression
	{
		int nb;
		bool status = Jpeg::ReadImage(params_.blockWidth_,params_.blockHeight_,nb,memAux_,blobLen,reinterpret_cast<unsigned char*>(block));
	}
#endif
	else											// no compression
	{
		memcpy(block,memAux_,blobLen);
	}
	TeCoord2D xy(portal->getDouble("lower_x"),portal->getDouble("upper_y"));
	TeCoord2D ij = params_.coord2Index(xy);

	// This is the rounding algorithm for col/lin value:
	// Pixel 0 covers column and line indexes from [-0.5,0.5)
	ulCol = TeRoundRasterIndex(ij.x_);
	ulLin = TeRoundRasterIndex(ij.y_);
	delete portal;
	return true;
}

bool 
TeDecoderDatabase::putRasterBlock(const string& index, void *block, long bsize)
{

	if (db_ == 0 || index.empty())
		return false;

	// decodify tile parameters from index
	int res, subb, band, blin, bcol;
	decodifyId(index,bcol,blin,band,res, subb);

	// calculates the box coordinates of the tile
	TeCoord2D llt(bcol, blin+params_.blockHeight_-1);
	TeCoord2D urt(bcol+params_.blockWidth_-1, blin);
	TeCoord2D ll = params_.index2Coord(llt);
	TeCoord2D ur = params_.index2Coord(urt);

	TeBox blockbb(ll.x()-params_.resx_/2.,ll.y()-params_.resy_/2.,ur.x()+params_.resx_/2.,ur.y()+params_.resy_/2.);
	bool status;

	// verify compression option
	if (params_.compression_[band] == TeNoCompression)
	{
		status = db_->insertRasterBlock(params_.fileName_,index,blockbb.lowerLeft(),blockbb.upperRight(),reinterpret_cast<unsigned char*>(block),bsize,band,res,subb);
		return status;
	}

	unsigned long finalsize = bsize;  // final size after compression
	if (!memAux_)
	{
		long size = params_.blockHeight_ * params_.blockWidth_ * params_.elementSize();
		memAux_ = new unsigned char[int(size*1.1 + 12.)];
	}

	if (params_.compression_[band] == TeZLib)		// ZLib compression
	{
		// zlib needs some more space)
		finalsize = (unsigned long)(bsize*1.1 + 12.);
		compress (memAux_, &finalsize,reinterpret_cast<unsigned char*>(block),bsize);
	}
#ifdef WIN32
	else if (params_.compression_[band] == TeJPEG)	// JPEG compression
	{
		status = Jpeg::Compress(reinterpret_cast<unsigned char*>(block),memAux_,params_.blockWidth_,params_.blockHeight_,1,finalsize);
	}
#endif

	status = db_->insertRasterBlock(params_.fileName_,index,blockbb.lowerLeft(),blockbb.upperRight(),memAux_,finalsize,band,res,subb);
	return status;
}

bool
TeDecoderDatabase::saveLUTTable()
{
	if (params_.lutName_.empty())
		return false;
	if (!db_ || !db_->createLUTTable(params_.lutName_))
		return false;
	for (unsigned int i=0; i<params_.lutb_.size(); i++)
	{
		string sql = "INSERT INTO " + params_.lutName_ + " VALUES(";
		sql += Te2String(i) + ", ";
		sql += Te2String(params_.lutr_[i]) + ", ";
		sql += Te2String(params_.lutg_[i]) + ", ";
		sql += Te2String(params_.lutb_[i]) + ")";
		if (!db_->execute(sql))
			return false;
	}
	return true;
}

bool
TeDecoderDatabase::selectBlocks(TeBox& bb, int resFac, TeRasterParams& parBlock)  
{ 
	if (!db_)
		return false;

	if (blockPortal_)
		delete blockPortal_;

	blockPortal_ = db_->getPortal();
	if (!blockPortal_)
		return false;

	string sql;
	string sel = db_->getSQLBoxSelect(params_.fileName_, TeRASTER);
	string where = db_->getSQLBoxWhere (bb, TeRASTER);

	sql = "SELECT "+ sel;
	sql += " FROM " + params_.fileName_;
	sql += " WHERE " + where + " AND resolution_factor= " + Te2String(resFac);
	sql += " ORDER BY lower_x, lower_y, upper_x, upper_y, block_id";
	
	if (!blockPortal_->query(sql))
	{
		delete blockPortal_;
		blockPortal_ = 0;
		return false;
	}
	nSelectedBlocks_ = blockPortal_->numRows();
	parBlock.projection(params_.projection());
	parBlock.resx_ = params_.resx_*resFac;
	parBlock.resy_ = params_.resy_*resFac;
	parBlock.nlines_ = params_.blockHeight_;
	parBlock.ncols_= params_.blockWidth_;
	parBlock.nBands(params_.nBands());
	parBlock.dataType_ = params_.dataType_;
	parBlock.dummy_ = params_.dummy_;
	parBlock.useDummy_ = params_.useDummy_;
	parBlock.photometric_ = params_.photometric_;
	parBlock.vmax_ = params_.vmax_;
	parBlock.vmin_ = params_.vmin_;
	parBlock.lutr_ = params_.lutr_;
	parBlock.lutg_ = params_.lutg_;
	parBlock.lutb_ = params_.lutb_;
	parBlock.interleaving_ = TePerBand;
	return blockPortal_->fetchRow();
}

bool
TeDecoderDatabase::getSelectedRasterBlock(TeDecoderMemory* memDec) 
{ 
	if (!blockPortal_)
		return 0;
	unsigned long srcLen;
	int i = 0;
	if (!memAux_ )
	{
		long size = params_.blockHeight_ * params_.blockWidth_ * params_.elementSize();
		memAux_ = new unsigned char[size];
	}
	int res;
	TeBox bbBlock;
	string blockId;
	do
	{
		bbBlock = TeBox (blockPortal_->getDouble("lower_x"),blockPortal_->getDouble("lower_y"),
					   blockPortal_->getDouble("upper_x"),blockPortal_->getDouble("upper_y"));
		blockId = blockPortal_->getData("block_id");
		res = blockPortal_->getInt("resolution_factor");

		int banda;
		int l;
		this->decodifyId(blockId,l,l,banda,l,l);
		blockPortal_->getRasterBlock(srcLen,memAux_);

		if (params_.compression_[banda] == TeZLib)
		{
			unsigned long destLen;
			int status;
			status = uncompress (reinterpret_cast<unsigned char*>(memDec->data(banda)),&destLen,memAux_,srcLen);
		}
#ifdef WIN32
		else if (params_.compression_[banda] == TeJPEG)
		{
			int nbytes = params_.nbitsperPixel_[banda]/8;
			int nb = 1;
			bool status = Jpeg::ReadImage(params_.blockWidth_,params_.blockHeight_,nb,memAux_,srcLen,reinterpret_cast<unsigned char*>(memDec->data(banda)));
		}
#endif
		else
			memcpy(reinterpret_cast<unsigned char*>(memDec->data(banda)),reinterpret_cast<unsigned char*>(memAux_),srcLen);
		i++;
	}while (i<params_.nBands() && blockPortal_->fetchRow());
	
	memDec->params().boundingBoxResolution(bbBlock.x1_,bbBlock.y1_,bbBlock.x2_,bbBlock.y2_,
		                                   params_.resx_*res,params_.resy_*res);
	memDec->params().blockId_ = blockId;
	return blockPortal_->fetchRow();
}

void
TeDecoderDatabase::clearBlockSelection() 
{
	if (blockPortal_)
		delete blockPortal_;
	blockPortal_ = 0;
	nSelectedBlocks_ = 0;
}

int 
TeDecoderDatabase::bestResolution(TeBox& bb, int ncols, int nlines, TeProjection* proj)
{
	TeBox box = bb;
	if (proj)
		box = TeRemapBox(bb, proj, params_.projection());
	
	double resx = box.width()/ncols;
	double resy = box.height()/nlines;
	
	double auxx = resx/params_.resx_;
	double auxy = resy/params_.resy_;

	int aux = (int) (min(auxx,auxy));

	string sql = "SELECT resolution_factor FROM " + params_.fileName_;
	sql += " WHERE resolution_factor <= " + Te2String(aux) + " ORDER BY resolution_factor DESC";

	TeDatabasePortal* portal = params_.database_->getPortal();
	if (!portal)
		return 1;

	if (!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return 1;
	}

	int res = atoi(portal->getData(0));
	delete portal;
	return res;
}


