/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeRasterParams.h
    This file contains definitions about parameters of a raster structure
*/
#ifndef  __TERRALIB_INTERNAL_RASTERPARAMS_H
#define  __TERRALIB_INTERNAL_RASTERPARAMS_H

#include "TeProjection.h"
#include "TeBox.h"
#include "TeDataTypes.h"

#include <string>
#include <list>
#include <vector>

//! Pixels interleaving mode
enum TeInterLeaving
{ TePerPixel, TePerLine, TePerBand };

using namespace std;

#define TeRASTERUNKNOWN		0	//!< unknown type of raster (an intermediate state)		
#define TeRASTERPALETTE		1	//!< a sintetic image that has an associated LUT (pixel val -> RGB triple)	
#define TeRASTERMULTIBAND	2	//!< an image which 1 to n bands
#define TeRASTERRGB			3	//!< a particular case of multiband, with exactly 3 bands
#define TeRASTERTHEMATIC	4	//!< a temathic image
#define TeRASTEROBJECTID	5	//!< each pixel has an ID of the corresponding object
#define TeRASTERLDD			6	//!< each pixel is a local drain direction 
#define TeRASTERKERNEL		7	

#define	TeNOTREADY				0x0000		// Decoder is not read to read or read
#define	TeREADYTOREAD			0x0001		// Decoder is read to read
#define	TeREADYTOWRITE			0x0002		// Decoder is read to write


class TeDatabase;

//!  A class to handle the parameters set associated with a raster
/*! 
 \sa
  TeRaster
*/
class TeRasterParams
{
public:

	//! Default contructor
	TeRasterParams () :
		nlines_(0),
		ncols_(0),
		resx_(1),
		resy_(1),
		resolution_(1),
		subBand_(0),
		swap_(false),
		useDummy_(false),
		mode_('r'),
		status_(TeNOTREADY),
		offset_(0),
		tiling_type_(TeExpansible),
		blockHeight_(0),
		blockWidth_(0),
		nTilesInMemory_(0),
		database_ (0),
		interleaving_(TePerPixel),
		dxJ_(0.0),
		dyJ_(0.0),
		dxI_(0.0),
		dyI_(0.0),
		x0_(0.0),
		y0_(0.0),
		nbands_(1),
		box_(TeBox()),
		projection_(0)
	{
		nbitsperPixel_.resize(nbands_);
		vmin_.resize(nbands_);
		vmax_.resize(nbands_);
		bands_.resize(nbands_);
		photometric_.resize(nbands_);
		dataType_.resize(nbands_);
		dummy_.resize(nbands_);
		compression_.resize(nbands_);
		for (int i=0; i<nbands_; i++)
		{
			bands_[i] = i;
			vmin_[i] = TeMAXFLOAT;
			vmax_[i] = -TeMAXFLOAT;
			dataType_[i] = TeUNSIGNEDCHAR;
			photometric_[i] = TeRASTERUNKNOWN;
			dummy_[i] = 0.0;
		}
	}

	//! Copy constructor
	TeRasterParams(const TeRasterParams& other);

	//! Operator =
	TeRasterParams& operator=(const TeRasterParams& rhs);

	//! Destructor
	virtual ~TeRasterParams()
	{
		bands_.clear();
		vmin_.clear();
		vmax_.clear();
		lutr_.clear();
		lutg_.clear();
		lutb_.clear();
		photometric_.clear();
		dataType_.clear();
		dummy_.clear();
		compression_.clear();
		if (projection_)
			delete projection_;
	}

	//! Return the box based on the coordinates of  center of pixel
	TeBox box()
	{ return box_; }

	//! Returns the outline box 
	TeBox boundingBox ();

	//! Sets box parameters
	/*
		This methods recalculates the number of lines and columns and box accordingly to 
		a desired bounding box and resolution
		\param x1 bounding box lower left point X coordinate 
		\param y1 bounding box lower left point Y coordinate 
		\param x2 bounding box upper right point X coordinate
		\param y2 bounding box upper right point Y coordinate
		\param resx X resolution
		\param resy Y resolution
		\param lock corner of the bounding box that should be preserved
	*/
	void boundingBoxResolution (double x1, double y1, double x2, double y2, 
								double resx, double resy, TeBox::TeBoxCorner lock = TeBox::TeUPPERLEFT);

	//! Sets box parameters
	/*
		This methods recalculates the resolutions and box accordingly to 
		a desired bounding box and number of lines and columns
		\param x1 bounding box lower left point X coordinate 
		\param y1 bounding box lower left point Y coordinate 
		\param x2 bounding box upper right point X coordinate
		\param y2 bounding box upper right point Y coordinate
		\param nlines  number of lines
		\param ncols number of columns
		\param lock corner of the bounding box that should be preserved
	*/
	void boundingBoxLinesColumns (double x1, double y1, double x2, double y2, 
		                          int nlines, int ncols, TeBox::TeBoxCorner lock = TeBox::TeUPPERLEFT);

	//! Sets box parameters
	/*
		This methods recalculates the number of lines and columns and box accordingly to 
		a desired box and resolution
		\param x1 box lower left point X coordinate 
		\param y1 box lower left point Y coordinate 
		\param x2 box upper right point X coordinate
		\param y2 box upper right point Y coordinate
		\param resx X resolution
		\param resy Y resolution
		\param lock corner of the bounding box that should be preserved
	*/
	void boxResolution (double x1, double y1, double x2, double y2, 
		                double resx, double resy, TeBox::TeBoxCorner lock = TeBox::TeUPPERLEFT);

	//! Sets box parameters
	/*
		This methods recalculates the resolutions and box accordingly to 
		a desired box and number of lines and columns
		\param x1 box lower left point X coordinate 
		\param y1 box lower left point Y coordinate 
		\param x2 box upper right point X coordinate
		\param y2 box upper right point Y coordinate
		\param nlines  number of lines
		\param ncols number of columns
		\par Calculates X resolution and Y resolution 
		\param lock corner of the bounding box that should be preserved
	*/
	void boxLinesColumns (double x1, double y1, double x2, double y2, 
		                  int nlines, int ncols, TeBox::TeBoxCorner lock = TeBox::TeUPPERLEFT);

	//! Sets box parameters
	/*
		\param left lower left point X coordinate 
		\param top  upper right point Y coordinate
		\param resx X resolution
		\param resy Y resolution
		\param nlines  number of lines
		\param ncols number of columns
		\param coordIsCentrePixel flag to indicate that top left coordinate is centre of pixel
	*/
	void topLeftResolutionSize (double left, double top, double resx, double resy, int ncol, int nlin, bool coordIsCentrePixel=true);
		
	//! Sets box parameters
	/*
		\param left lower left point X coordinate 
		\param lower lower left point Y coordinate
		\param resx X resolution
		\param resy Y resolution
		\param nlines  number of lines
		\param ncols number of columns
		\param coordIsCentrePixel flag to indicate that top left coordinate is centre of pixel
	*/
	void lowerLeftResolutionSize (double left, double lower, double resx, double resy, int ncol, int nlin, bool coordIsCentrePixel=true);

	//! Resizes the raster bounding box to allow tiling operation
	/*! 
		\param bb an initial bounding box
		\param bWidth block width (in number of elements/pixels)
		\param bHeight block height (in number of elements/pixels)
		\note uses X an Y resolutions already defined
	*/ 
	void resizeToTiling(TeBox& bb, int bWidth, int bHeight);

	//! Sets the number of bands
	void nBands(int n);

	//! Sets the projection
	/*
		A new instance of projection is created according to the parameters described
		by proj.
	*/
	void projection(TeProjection* proj);

	//! Returns the projection
	TeProjection* projection()
	{	return projection_; }

	//! Transform a point from projection domain to line/column domain
	TeCoord2D coord2Index (TeCoord2D& pt);

	//! Transform a point from line/column domain to projection domain
	TeCoord2D index2Coord (TeCoord2D& pt);

	//----------------- MEMBERS ---------------------

	int nlines_;			//!< number of lines
	int ncols_;				//!< number of columns

	double	resx_,			//!< pixel horizontal resolution
			resy_;			//!< pixel vertical resolution	


	int resolution_;		//! resolution level (used in storing raster as wavelets)
	int subBand_;			//! sub band identification (used in storing rasters as wavelets)

	string	lutName_;		//!< name of a lut table associated to this raster
	bool swap_;				//!< a flag to indicate that raster has swapped values
	bool useDummy_;			//!< a flag to indicate that raster has dummy values

	vector<int>		nbitsperPixel_;		//!< number of bits per pixel in each band
	vector<double>	vmin_;				//!< minimum value per band
	vector<double>	vmax_;				//!< maximum value per band
	vector<int>		bands_;				//!< bands information 
	vector<double>	dummy_;				//!< no data value
	vector<short>   photometric_;		//!< raster type (palette, multiband, rgb, ...)
	vector<TeCompressionMode>	compression_;	//!< compression type of tiles as this raster is stored
	vector<TeDataType> dataType_;				//!< size of each pixel (char,int,unsigned char...), as it will be saved

	string	fileName_;		//!< source of the raster data (table or disk file)
	char	mode_;			//! Instantiating mode for the raster data 'r', 'w' or 'c'
	int status_;			//! Status for reading and writing to raster
	int offset_;			//!< offset 
	
	TeRasterTilingType	tiling_type_; //!< raster tiling scheme 
	int blockHeight_;		//!< block height of tiles as this raster is stored
	int blockWidth_;		//!< block width of tiles as this raster is stored

	string	decoderIdentifier_;	//!< decoder associated to his raster
	int nTilesInMemory_;		//!< number of tiles to be kept in virtual memory decoders

	TeDatabase		*database_;		//!< database associated to this driver

	string blockId_;				//!< block identification

	TeInterLeaving interleaving_;	//!< interleaving mode

	/** @name Raster pallete
	*  Look up table associated to a raster 
	*/
	//@{ 
	vector<unsigned short> lutr_;  //!<  Red pallete
	vector<unsigned short> lutg_;  //!<  Green pallete
	vector<unsigned short> lutb_;  //!<  Blue pallete
	//@}

	/** @name TFW parameters
	*  Parameters associated to TFW codification of georeferencing 
	*/
	//@{ 
	double dxJ_;	//!< X offset due to increase of one column position		       
	double dyJ_;	//!< Y offset due to increase of one column position		       
	double dxI_;	//!< X offset due to increase of one line position		       
	double dyI_;	//!< X offset due to increase of one line position		       
	double x0_;		//!< X coordinate of the upper left raster pixel
	double y0_;		//!< Y coordinate of the upper left raster pixel
	//@}

	//! Sets the size of raster element per in each band
	void setDataType(TeDataType type, int band=-1);

	//! Returns the size in bytes of each raster element
	int elementSize(int band=0);

	//! Sets the photometric type of each band
	void setPhotometric(short photom, int band=-1);

	//! Sets the compression mode of each band
	void setCompressionMode(TeCompressionMode cmode, int band=-1);

	//! Sets the dummy value in each band
	void setDummy(double dummy, int band=-1);

	//! Returns the number of bands of the raster
	int nBands() const
	{	return nbands_; }

	//! Sets the number of entries in the pallete table
	void setNumberPalleteEntries(int n)
	{
		lutr_.resize(n+1);
		lutg_.resize(n+1);
		lutb_.resize(n+1);
	}

	//! Returns the identifier of the decoder associated to the raster
	string decName()
	{ return decoderIdentifier_; }

	string lutName()
	{ return lutName_; }

private:
	int		nbands_;		//!< number of bands
	TeBox	box_;			//!< raster box in center of pixel coordinates
	TeProjection* projection_;	//!< raster projection
};

#endif

