/*  tiffep - Ti File Format Engine & Proxy
 *  Copyright (C) 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
  This unit contains functions for managing exchange of messages
  between the TiFFEP server and the libtiffep.
  This unit is built as a overlayer of the unix/win32_sock.c unit.
*/

#include <stdio.h>
#ifdef __WIN32__
# include <process.h>	// for getpid()
#elif defined(__LINUX__)
# include <unistd.h>
#endif

#include "hfiles.h"
#include "tiffep.h"

/*****************/
/* Send commands */
/*****************/


/*
   Clear the message content
   - msg [in/out]: the address of the message to clear
*/
int msg_clear(TiffepMsg *msg)
{
  msg->type = TP_UNKNOWN;
#if defined(__WIN32__)
  msg->id = _getpid();
#elif defined(__LINUX__)
  msg->id = getpid();
#endif
  msg->cmd = CMD_NONE;
  msg->ack = 0;
  msg->data = 0;
  strcpy(msg->str, "");

  return 0;
}


/*
   Send a message and wait until completion
   - s [in]: the shm
   - m [in]: the message to send
*/
int msg_send (TiffepShm s, TiffepMsg m)
{
  volatile TiffepMsg *msg = s.addr;

  msg->type = m.type;
  strcpy((char *)(msg->str), m.str);
  msg->data = m.data;
  msg->ack = m.ack;
  msg->id = m.id;
  msg->cmd = m.cmd;

  while(msg->cmd == m.cmd);

  return 0;
}


/*
   Wait a message
   - s [in]: the shm
   - m [out]: the address where the message can be stored
*/
TIEXPORT
int msg_recv (TiffepShm s, TiffepMsg *m)
{
  volatile TiffepMsg *msg = s.addr;

  while(msg->cmd == CMD_NONE);
  m->type = msg->type;
  m->id = msg->id;
  m->cmd = msg->cmd;
  m->ack = msg->ack;
  m->data = msg->data;
  strcpy((char *)(m->str), (char *)(msg->str));
  msg->cmd = CMD_NONE;

  return 0;
}


/*
   Check if a message is available
   - s [in]: the shm
   Return 1 if available, 0 otherwise
*/
TIEXPORT
int msg_check(TiffepShm s)
{
  volatile TiffepMsg *msg = s.addr;

  return ((msg->cmd == CMD_NONE) ? 0 : 1);
}


/*
   Remove message
   - s [in]: the shm
   Return always 0
*/
int msg_remove(TiffepShm s)
{
  volatile TiffepMsg *msg = s.addr;

  msg->cmd = CMD_NONE;

  return 0;
}
