/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id$
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "accountview.h"
#include "transactionwindow.h"
#include <qbanking/qbpickstartdate.h>
#include <qbanking/qbeditaccount.h>

#include <aqbanking/jobgetbalance.h>
#include <aqbanking/jobgettransactions.h>
#include <aqbanking/jobgetstandingorders.h>
#include <aqbanking/jobgetdatedtransfers.h>

#include <kbanking.h>

#include <qevent.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>

#include <gwenhywfar/debug.h>

#ifdef WIN32
# define strcasecmp stricmp
#endif


#define BUTTON_WIDTH 110


AccountView::AccountView(KBanking *kb,
                         QWidget* parent,
                         const char* name,
                         WFlags fl)
:AccountViewUi(parent, name, fl), _app(kb) {
  assert(kb);

  // Manually create and add layout here because the .ui-generated
  // QGroupBox doesn't have one.
  accountBox->setColumnLayout(0, Qt::Vertical );
  QBoxLayout *accountBoxLayout = new QHBoxLayout( accountBox->layout() );
  accountBoxLayout->setAlignment( Qt::AlignTop );

  _accList=new AccountListView(accountBox, name);
  accountBoxLayout->addWidget(_accList);

  QPopupMenu *updateMenu=new QPopupMenu(this);
  updateMenu->insertItem(tr("Normal"), this, SLOT(slotUpdate()));
  updateMenu->insertItem(tr("With Timespan"),
                         this, SLOT(slotUpdateWithTimeSpan()));
  updateButton->setPopup(updateMenu);

  QObject::connect(kb->flagStaff(), SIGNAL(signalAccountsUpdated()),
                   this, SLOT(slotUpdated()));

  QObject::connect(detailsButton, SIGNAL(clicked()),
                   this, SLOT(slotDetails()));
  QObject::connect(transactionsButton, SIGNAL(clicked()),
                   this, SLOT(slotTransactions()));
  QObject::connect(updateButton, SIGNAL(clicked()),
                   this, SLOT(slotUpdate()));

}



AccountView::~AccountView(){
}



bool AccountView::init(){
  GWEN_DB_NODE *dbConfig=NULL;
  int rv;

  rv=_app->loadAppSubConfig("gui/views/accountview/dynamic", &dbConfig);
  if (rv<0) {
    DBG_INFO(0, "here (%d)", rv);
  }
  else {
    int i, j;
    const char *p;

    p=GWEN_DB_GetCharValue(dbConfig, "sortOrder", 0, "ascending");
    if (p) {
      if (strcasecmp(p, "ascending")==0)
        _accList->setSortOrder(Qt::Ascending);
      else
        if (strcasecmp(p, "descending")==0)
          _accList->setSortOrder(Qt::Descending);
    }
    i=GWEN_DB_GetIntValue(dbConfig, "sortColumn", 0, -1);
    if (i!=-1)
      _accList->setSortColumn(i);

    /* found settings */
    for (i=0; i<_accList->columns(); i++) {
      _accList->setColumnWidthMode(i, QListView::Manual);
      j=GWEN_DB_GetIntValue(dbConfig, "columns", i, -1);
      if (j!=-1)
        _accList->setColumnWidth(i, j);
    } /* for */
    GWEN_DB_Group_free(dbConfig);
  } /* if settings */

  _accList->addAccounts(_app->getAppAccounts());

  return true;
}



bool AccountView::fini(){
  GWEN_DB_NODE *dbConfig;
  int rv;
  int i, j;

  dbConfig=GWEN_DB_Group_new("config");
  assert(dbConfig);

  switch(_accList->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(dbConfig, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(dbConfig, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_SetIntValue(dbConfig, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", _accList->sortColumn());

  for (i=0; i<_accList->columns(); i++) {
    j=_accList->columnWidth(i);
    GWEN_DB_SetIntValue(dbConfig, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  rv=_app->saveAppSubConfig("gui/views/accountview/dynamic", dbConfig);
  GWEN_DB_Group_free(dbConfig);
  if (rv<0) {
    DBG_INFO(0, "here (%d)", rv);
    return false;
  }

  return true;
}



void AccountView::slotDetails(){
  Account *acc;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(this,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), QString::null);
    return;
  }
  else {
    AB_ACCOUNT *ba;

    ba=_app->getAccount(acc->getBankingId());
    if (!ba) {
      DBG_ERROR(0, "Account not available");
      QMessageBox::critical(this,
                            tr("Account Not Available"),
                            tr("The account you requested is not available\n"
                               "with any backend."),
                            tr("Dismiss"), QString::null);
      return;
    }
    if (QBEditAccount::editAccount(_app, ba, this)) {
      DBG_DEBUG(0, "Accepted account changed");
    }
    else {
      DBG_DEBUG(0, "Rejected account changes");
    }
  }
}



void AccountView::slotTransactions(){
  Account *acc;
  TransactionWindow *tv;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(this,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), QString::null);
    return;
  }

  tv=new TransactionWindow(acc, 0, "TransactionView",
                           Qt::WDestructiveClose);
  tv->setCaption(tr("Transactions"));
  if (!tv->init()) {
    QMessageBox::critical(this,
                          tr("Internal Error"),
                          tr("Could not initialize transaction view."),
                          tr("Dismiss"), QString::null);
    delete tv;
    return;
  }
  tv->show();
}



void AccountView::slotUpdateWithTimeSpan() {
  _accountUpdate(true);
}



void AccountView::slotUpdate(){
  _accountUpdate(false);
}



void AccountView::_accountUpdate(bool forceTimeSpan){
  GWEN_DB_NODE *dbConfig=NULL;
  Account *acc;
  int rv;
  int oks=0;
  int tries=0;
  AB_ACCOUNT *ba;
  AB_JOB *job;
  const GWEN_TIME *ti;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(this,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), QString::null);
    return;
  }

  /* get corresponding AB_ACCOUNT */
  ba=_app->getAccount(acc->getBankingId());
  if (!ba) {
    DBG_ERROR(0, "Account not available");
    QMessageBox::critical(this,
			  tr("Account Not Available"),
			  tr("The account you requested is not available\n"
			     "with any backend."),
			  tr("Dismiss"), QString::null);
    GWEN_DB_Group_free(dbConfig);
    return;
  }

  /* update transactions */
  if (acc->getOptionUpdateTransactions()) {
    tries++;
    job=AB_JobGetTransactions_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetTransactions\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      ti=acc->getLastTransactionDate();
      if (ti && !forceTimeSpan) {
	AB_JobGetTransactions_SetFromTime(job, ti);
      }
      else {
	int days;
	int year, month, day;
	QDate qd, qd2;

	days=AB_JobGetTransactions_GetMaxStoreDays(job);
	if (days>0) {
	  GWEN_TIME *ti1;
	  GWEN_TIME *ti2;

	  ti1=GWEN_CurrentTime();
	  ti2=GWEN_Time_fromSeconds(GWEN_Time_Seconds(ti1)-(60*60*24*days));
	  GWEN_Time_free(ti1);
	  ti1=ti2;

	  if (GWEN_Time_GetBrokenDownDate(ti1, &day, &month, &year)) {
	    DBG_ERROR(0, "Bad date");
	    qd=QDate();
	  }
	  else
	    qd=QDate(year, month+1, day);
	  GWEN_Time_free(ti1);
	}

	if (ti) {
	  if (GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year)) {
	    DBG_ERROR(0, "Bad date");
	    qd2=QDate();
	  }
	  else
	    qd2=QDate(year, month+1, day);
	}

	QBPickStartDate psd(_app, qd, qd2, 3, this,
			    "PickStartDate", true);
	if (psd.exec()!=QDialog::Accepted) {
	  AB_Job_free(job);
	  GWEN_DB_Group_free(dbConfig);
	  return;
	}
	qd=psd.getDate();
	if (qd.isValid()) {
	  GWEN_TIME *ti1;

	  ti1=GWEN_Time_new(qd.year(), qd.month()-1, qd.day(), 0, 0, 0, 0);
	  AB_JobGetTransactions_SetFromTime(job, ti1);
	  GWEN_Time_free(ti1);
	}
      }

      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
	DBG_NOTICE(0, "Error %d", rv);
	GWEN_DB_Group_free(dbConfig);
	return;
      }
      oks++;
    }
  }

  /* update balance */
  if (acc->getOptionUpdateBalance()) {
    tries++;
    job=AB_JobGetBalance_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetBalance\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
	DBG_NOTICE(0, "Error %d", rv);
	return;
      }
      oks++;
    }
  }

  /* update standing orders */
  if (acc->getOptionUpdateStandingOrders()) {
    tries++;
    job=AB_JobGetStandingOrders_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetStandingOrders\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
	DBG_NOTICE(0, "Error %d", rv);
	GWEN_DB_Group_free(dbConfig);
	return;
      }
      oks++;
    }
  }

  /* update dated transfers */
  if (acc->getOptionUpdateDatedTransfers()) {
    tries++;
    job=AB_JobGetDatedTransfers_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetDatedTransfers\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
	DBG_NOTICE(0, "Error %d", rv);
	GWEN_DB_Group_free(dbConfig);
	return;
      }
      oks++;
    }
  }

  if (!oks) {
    if (!tries)
      QMessageBox::critical(this,
			    tr("No Job enqueued"),
			    tr("<qt>"
			       "No update jobs have been enqueued because "
			       "all those jobs are disabled with this "
			       "account.\n"
			       "</qt>"),
			    tr("Dismiss"), QString::null);
    else
      QMessageBox::critical(this,
			    tr("No Job enqueued"),
			    tr("<qt>"
			       "No update jobs is supported by the "
			       "bank/account/backend.\n"
			       "</qt>"),
			    tr("Dismiss"), QString::null);
    _app->statusMessage(tr("No jobs added to outbox"));
    GWEN_DB_Group_free(dbConfig);
    return;
  }

  _app->statusMessage(tr("Jobs added to outbox"));
}



void AccountView::slotUpdated() {
  DBG_DEBUG(0, "Account view updated");
  _accList->clear();
  _accList->addAccounts(_app->getAppAccounts());
}



