/* ==================================================== ======== ======= *
 *
 *  ucond.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ucond_hpp_
#define	_ucond_hpp_
//pragma ident	"@(#)ucond.hpp	ubit:03.05.03"

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** base class for Ubit conditions.
 *  Notes:
 * - this class (and its subclasses: UOn, UFlag, etc.)
 *   are NOT derived from UBrick (=> they can't be UBox children)
 * - !CAUTION: NEVER delete UCond (and subclasses) instances
 */
class UCond {
public:
  virtual bool verifies(const UContext*, const class UCtrl*) const = 0;
  virtual void setParentModes(UCtrl *parent) const {};
};

/* ==================================================== ======== ======= */

class UNotCond : public UCond {
public:
  UNotCond(const UCond&);
  virtual bool verifies(const UContext*, const class UCtrl*) const;

private:
  const UCond& cond;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** event conditions.
 * for specifying conditionnal objects and callback functions
 * - general usage:  condition / object
 * - callbacks:      condition / ucall(...)
 * - see also: ucall() template, UCall class, ucall.hh header.
 * - !CAUTION: NEVER delete UOn (and subclasses) instances
 */
class UOn : public UCond {
public:
  
  static UOn 
    /** no action event.
     * used for style specs. 
     */
  idle,
    
    /** high-level activation event.
     * event received by activable boxes (eg. UButton, UCheckbox, UTextbox).
     * Notes:
     * - Boxes can be made activable by setting their CAN_ACTION mode.
     * - This event is discarded if the box is <b>not enabled</b> 
     *   (see: UGroup::enable())
      */
    action,

    /** arm/disarm button events.
     * events received by armable boxes (eg. UButton, UCheckbox) when 
     * they are activated (event order is: arm / action / disarm)
     * Notes: 
     * - Boxes can be made armable by setting their CAN_ARM mode
     *   (see: UMode::canArm and UGroup::setCmodes())
     * - these events are discarded if the box is <b>not enabled</b> 
     *   (see: UGroup::enable())
  */
    arm, disarm,

    /** mouse events.
     * Details:
     * - mmove is received by the box where the mouse is currently located
     * - mdrag and mrelease are received by the box that got the mpress
     * - mrelax is similar to mrelease but it is received by the box
     *   where the mouse was released (instead of the box that got mpress)
     */
    enter, leave, 
    mpress, //mpress1, mpress2, mpress3, //mlongpress,
    mrelease, //mrelease1, mrelease2, mrelease3, 
    mdrag, mmove, mrelax,

    /** mouse clicks.
     * mclick is received when a mouse click is performed on a box, 
     * mbiclick when a double click is performed, etc.
     * Notes:
     * - a double click first produces a mclick then a mbiclick (and so on 
     *   for other multiple click events)
     * - mouse clicks are not emitted if the mouse has been moved more
     *   that a certain radius between the mouse press and release
     * - UOn::action is received after the mouse clicks
     */
    mclick, mbiclick, mtriclick, mquadclick,

    /** key events.
     * Notes:
     * - kpress and krelease are produced by raw key events
     * - ktype is a higher-level event that compose ASCII characters
     *   it is not activated by special or function keys (eg. LEFT, F1...)
     */
    ktype, kpress, krelease,

    /** item selection.
     *  events received by selectable boxes (eg. UCheckbox) when they are
     *  selected (resp. unselected) or by listboxes when one of their
     *  item is selected (resp. unselected)
     *
     *  Notes:
     * -  UEvent::getTarget(), getBoxTarget() return the listbox item
     *   that was selected or unselected
     * -  any box can be made selectable by setting its CAN_SELECT mode
     *   (see: UMode::canSelect and UGroup::setCmodes())
     */
    select, unselect,

    /** value change.
     *  events received by UProp, UElem, UStr objects when their value 
     *  is changed.
     * - Note:
     *   UEvent::getChangedProp(), getChangedElem(), getChangedStr()
     *   return the object whose value was changed
     */
    change,

    /** box child change.
     *  events received by boxes when the value of their UProp, UElem, UStr...
     *  children is changed.
     *
     * Notes: 
     * - elemChange and strChange are both emitted by UStr children (in this order)
     * - UEvent::getChangedProp(), getChangedElem(), getChangedStr()
     *   return the object whose value was changed
     */
    propChange, elemChange, strChange, //caretChange,

    /** adding and removing children.
     *  events received by the parent(s) of the objects that are being 
     *  added or removed.
     *  - Note:
     *  UEvent::getTarget(),getBoxTarget(),getBrickTarget() returns the child
     *  that was added or removed (which may have been destroyed in the latter case)
     */
    add, remove,
 
    /** object destruction.
     * event received by objects that are being destructed.
       */
    destruct,

    /** window closing.
     * event received by windows that are being closed.
     */
    close,
  
    /** drag and drop.
     */
    dragStart, dragDone, dropEnter, dropLeave, dropDone,

    /** View events.
     * These events are related to object' *views*
     * (and thus, may be produced several times for a single Box object
     *  if this object has several parents)
     */
    viewPaint, viewResize, viewMove,

    /** detects events that *will* be received by children.
     * intercepts events that will be received by the direct or indirect
     * children of an UBox (or subclass):
     * - the corresponding callbacks are called while traversing the
     *   instance graph (*before* the children are reached)
     * - children will then receive events as usual
     * - all UEvent methods (such as getSource(), getX(), getY() ...)
     *   are relative to the parent (as the Event has not yet reached 
     *   the destination child it can't be identified at this stage)
     * - note that many event conditions are meaningless at this stage
     *   (for instance, UOn:select is meaningless as the child has not
     *    yet be reached)
     * - this mechanism can be used to modify events dynamically
     *   (it is for instance used for transparent tools)
     * - this will only work with UBox(es) and subclasses : an UGroup 
     *   can't intercept the Events that are sent to its children
     * 
     * See also: 
     * - UBox::onPostChildEvent() which makes it possible to
     *     detect events that have already been received by children.
     * Note:
     * - another way to detect events thst would be sent to children
     *   is to use transparent glasses (all Ubit gadgets can be transparent)
     *   This technique makes it also possible to draw on the top
     *   of other objects.
     */
    preChildEvent,

    /** Input and Timer events.
     * these events are send (resp.) by Input sources and Timers.
     * See: UAppli:openInput(), UAppli::openTimer()
     */
    //input,
    //timeout,

    /** application and hard window events.
     * - umessage : ubit message sent by another Ubit appli
     * - notify   : notification events sent by the windowing system
     * - rawEvent : any X event (received before other events)
     */
    umessage,
    notifyEvent,
    rawEvent;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  enum ISTATE {   // interaction state
    ANY = -1, 
    IDLE = 0,			// must be 0 (index in style arrays)
    DISABLED, ENTERED, ARMED, ACTIONED, DRAGGED, DROP_ENTERED, 
    ACTION_COUNT			// (used in Style Arrays)
    // !! BEWARE must be coherent with typedef u_state in udefs.hh !!
  };

  UOn(u_state istate, u_modes bmodes, u_modes cmodes);
  //< !CAUTION: NEVER delete UOn instances.

  virtual bool verifies(const UContext*, const class UCtrl*) const;
  virtual void setParentModes(UCtrl *parent) const;

private:
  friend class UCtrl;
  friend class UGroup;
  u_modes bmodes, cmodes;
  u_state istate;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Flagging conditions.
 * for specifying conditionnal objets that are only active when this
 * condition is verified
 * - usage:       flag / object
 * - works with:  UFlagdef
 * - !CAUTION: NEVER delete UFlag instances
 */
class UFlag : public UCond {
public:
  static const UFlag none;
  
  UFlag(const char* name = null);
  friend UFlag& uflag(const char* name = null) {return *new UFlag(name);}
  ///< !CAUTION: NEVER delete UFlag instances.

  UCond& operator!() const;
  
  const char* getName() const {return name;}
  virtual bool verifies(const UContext*, const class UCtrl*) const;
  
protected:
  const char *name;
  mutable class UNotCond* not_cond;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Scaling conditions (for semantic zooming).
 * specify conditionnal objets that are only active when the scale 
 * condition is verified.
 * - usage:       scale_range / object
 * - works with:  UScale
 * - !CAUTION: NEVER delete USrange instances
 */
class USrange : public UCond {
public:
  USrange(int, int);
  friend USrange& usrange(int scale_min, int scale_max);
  ///< !CAUTION: NEVER delete USrange instances.

  int getMin() const {return scale_min;}
  int getMax() const {return scale_max;}
  
  virtual bool verifies(const UContext*, const class UCtrl*) const;
  
private:
    int scale_min, scale_max;
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif

