/* ==================================================== ======== ======= *
 *
 *  ugraph.hpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ugraph_hpp_
#define	_ugraph_hpp_
#include <ubit/uappli.hpp>
//pragma ident	"@(#)ugraph.hh	ubit:03.06.04"


/** client class for drawing graphics.
 * !Warning: client should not use the UWinGraph class 
 * (which is reserved for the Ubit lib)
 */
class UGraph {
public:
  UGraph();
  UGraph(class UView*);
  UGraph(class UEvent*);
  UGraph(class UEvent&);
  /**< 
   * creates a new UGraph for drawing in the UView given as an explicit
   * or implicit argument.
   * Function:
   * -  UGraph(UView*) allocates physical resources for drawing in this View
   * -  UGraph(UEvent*) allocates resources for drawing in the source View
   *    that is relative to this Event (see: UEvent::getView()) and sets
   *    the clipping area according to UEvent::redrawClip when applicable.
   *    This makes it possible to redraw just the subpart of the View
   *    that needs to be refreshed.
   * 
   * Typical usage:
   * <br> void callback_function(UEvent& e) {
   * <br>     UGraph g(e);
   * <br>     g.drawLine(...);
   * <br> }
   *
   * Notes:
   * - as UGraph objects allocates physical resources, they should be
   *   created in the stack (as in the previous example) in order to
   *   be released as soon as possible. Creating several UGraphs by using
   *   the 'new' primitive is not a good idea and can pose problem
   *   with certain implementations.
   *
   * - The Font and the foreground and background Colors are undefined 
   *   when the UGraph is created and must be set explicitely (when needed)
   *
   * - The default drawing mode (called 'PaintMode') overwrites pixels.
   *   method setXORMode() sets the 'XORMode' that alternates the current
   *   foreground and background Colors.
   */
  virtual ~UGraph();

  // ==== Related objects ==================================================

  UView *getView() const {return boxview;}
  ///< returns the current View where graphics are being drawn.

  UDisp* getDisp() const;
  ///< returns the logical Display of this View.

  UAppli* getAppli() const;
  ///< returns the Application context.

  // ==== Modes and Attributes =============================================

  void setColor(const UColor*);
  void setColor(const UColor&);
  /**< sets the Foreground color.
   * note: the color is undefined when the UGraph is created 
   */

  void setBgcolor(const UColor*);
  void setBgcolor(const UColor&);
  /**< sets the Background color.
   * note: the color is undefined when the UGraph is created 
   */
  
  void setFont(const UFont*);
  void setFont(const UFont&);
  void setFont(const UFontDesc&);
  /**< sets the current Font.
   * notes:
   * - the font is undefined when the UGraph is created
   * - UFontDesc is an internal representation of UFont (see file ufont.hh)
   */

  void setCursor(const UCursor*);
  void setCursor(const UCursor&);
  ///< changes the mouse Cursor.

  bool isXORMode();
  void setXORMode();
  void setPaintMode();
  /** these functions set the drawing mode. 
   * notes:
   * - PaintMode (the default) overwrites pixels with the foreground Color
   * - XORMode alternates the current foreground and background Colors
   */

  void setThickness(int = 0);
  int  getThickness();
  /**< changes line thickness when drawing lines, etc.
   *  the default value 0 sets thickness to 1 pixel and optimizes drawing
   */

  // ==== Clip and Flush ==================================================

  void setClip(const URegion&);
  void setClip(u_pos x, u_pos y, u_dim width, u_dim height);
  /**< set the clipping zone relatively to VIEW coordinates.
   * coordinates are relative to the origin of the VIEW 
   */

  void setWinClip(const URegion&);
  void setWinClip(u_pos x, u_pos y, u_dim width, u_dim height);
  /**< set the clipping zone relatively to WINDOW coordinates.
   * coordinates are relative to the origin of the WINDOW 
   */

  void flush();
  ///< flushes all drawing requests (when applicable).

  int grabPointer(class UCursor* = null, bool confine_to_window = false); 
  /**< starts a "physical grab" on the X server.
   * Warning: this function may lock the X server!
   */

  void ungrabPointer();
  ///< ends a "physical grab" on the X server.

  // ==== Coordinates ======================================================

  u_pos XToXwin(u_pos x_in_view);
  u_pos YToYwin(u_pos y_in_view);
  ///< conversion from local View coords to global Window coords.

  u_pos XwinToX(u_pos x_in_win);
  u_pos YwintoY(u_pos y_in_win);
  ///< conversion from global Window coords to local View coords.

  // ==== Clearing and Copying routines ====================================

  void clearRect(u_pos x, u_pos y, u_dim width, u_dim height,
		 bool repaint_view = false);
  /**< clears a rectangle.
   * Args:
   * - if 'repaint_view' is true : repaints the normal content of the view
   *   in the specified region
   * - otherwise: fills the area with the UGraph's background Color
   * - see also: UView::updatePaint() in uview.hh for more options
   * - Note: this function is not altered by the XOR mode
   */

  void clearView(bool repaint_view = false);
  ///< same as clearRect() but clears the entire view.

  void copyArea(u_pos x, u_pos y, u_dim width, u_dim height,
		u_dim delta_x, u_dim delta_y,
		bool generate_refresh_events_when_obscured);
  /** copies a rectangular area.
   *  copies a rectangular area of size (width,height) at location (x,y) 
   *  to a location shifted by delta_x, delta_y
   * - note: CopyArea can't copy obscured parts of the component
   *   but will generate refresh events for repainting the missing parts
   *   if the last argument is true 
   */

  // ==== Drawing routines =================================================
  // Note: these functions only use the Foreground Color in PaintMode
  //       they also use the Background Color in XORMode

  void drawArc(u_pos x, u_pos y, u_dim width, u_dim height, 
	       float from_angle, float arc_angle);
  void drawEllipse(u_pos x, u_pos y, u_dim width, u_dim height); 
  /**< draws an ellipse or an arc of an ellipse.
   * Args:
   * - x,y,width,height : top/left position and size of the enclosing rectangle
   * - angles are in degrees
   */

  void drawLine(u_pos x1, u_pos y1, u_pos x2, u_pos y2);
  ///< draws a line.

  void drawPoint(u_pos x, u_pos y);
  ///< draws a point.

  void drawPolyline(u_pos xpoints[], u_pos ypoints[], u_count card);
  void drawPolyline(std::vector<u_pos> xpoints, 
		    std::vector<u_pos> ypoints, 
		    u_count card);
  ///< draws a sequence of connected lines.

  void drawPolygon(u_pos xpoints[], u_pos ypoints[], u_count card);  
  void drawPolygon(std::vector<u_pos> xpoints, 
		   std::vector<u_pos> ypoints, 
		   u_count card);
  ///< draws a polygon (which is automatically closed).

  void drawRect(u_pos x, u_pos y, u_dim width, u_dim height);
  ///< draws a rectangle.

  void drawString(const UStr&, u_pos x, u_pos y);
  void drawString(const UStr*, u_pos x, u_pos y);
  void drawString(const char*, int str_length, u_pos x, u_pos y);
  /**< draw a character string.
   * Note: these functions use the current Font (see setFont()).
   */

  // ==== Filling routines =================================================
  // Note: these functions only use the Foreground Color in PaintMode
  //       they also use the Background Color in XORMode

  void fillArc(u_pos x, u_pos y, u_dim width, u_dim height,
	       float from_angle, float arc_angle);
  /**< fills an arc of a circle (or a complete circle).
   * Args:
   * - x,y,width,height : top/left position and size of the enclosing rectangle
   * - angles are in degrees
   */

  void fillPolygon(u_pos xpoints[], u_pos ypoints[], 
		   u_count card, int mode=0);
  void fillPolygon(std::vector<u_pos> xpoints, 
		   std::vector<u_pos> ypoints, 
		   u_count card, int mode=0);
  /**< fills a polygon (which is automatically closed).
   * argument 'mode' can be set for optimisation purpose: 
   * - by default, mode=0 : the polygon can have an arbitrary shape
   * - mode=2 assumes a Convex polygon. 
   * - mode=1 assumes a non Convex but non Intersecting polygon. 
   */

  void fillRect(u_pos x, u_pos y, u_dim width, u_dim height);
  ///< fills the area.

  void fillView(); 
  ///< fills the whole View.

  // ==== Images and Pixmaps =============================================

  //void drawIma(const class UIma*, u_pos x, u_pos y);
  void drawIma(const class UNatIma*, u_pos x, u_pos y);
  void drawIma(const class UNatPix*, u_pos x, u_pos y);

  static void getImaSize(class UNatIma*, u_dim &w, u_dim &h);
  static void getPixSize(class UNatPix*, u_dim &w, u_dim &h);

  // ==== Font Metrics ====================================================

  bool getCharDefaultSize(const UFont&, int& w, int& h) const;
  bool getCharDefaultSize(const UFontDesc&, int& w, int& h) const;

  void getTextSize(const UFont&, const UStr&, 
		   u_dim& width, u_dim& height) const;
  void getTextSize(const UFontDesc&, const char* str, int str_len, 
		   u_dim& width, u_dim& height) const;
  u_dim getTextHeight(const UFont&) const;
  u_dim getTextHeight(const UFontDesc&) const;
  u_dim getTextWidth(const UFont&, const UStr&) const;
  u_dim getTextWidth(const UFontDesc&, const char* str, int str_len) const;
  ///< returns font metrics.

  int getCharPos(const UFontDesc&, const char* str, int str_len, 
		 u_pos x) const;
  int getXPos(const UFontDesc&, const char* str, int str_len, 
	      int char_pos) const;
  ///< converts x pos to char pos and vice-versa.

  bool getClippedText(const char*s, int len, u_pos x0, u_pos y0,
		      int& charpos_begin, int& charpos_end,
		      u_pos& xpos_begin, u_pos& xpos_end) const;
  ///< returns the substring that is enclosed in the current clip.

  bool getSubTextSize(const UFontDesc&, const char* str, int str_len, 
		      u_dim& w, u_dim& h, u_dim available_width, 
		      int& sublen, int& change_line) const;
  /**< returns font metrics of a substring that verifies certain conditions.
   * Notes:
   * - the substring starts from the beginning of 'str' and ends at 
   *   the 1st \n  or 1st " " that is before 'available_width'.
   * - it is equal to 'str' otherwise.
   * - returns w and h that this substring occupies and its length 'sublen'
   * - change_line = 1 si line must be changed because of size
   *   and 2 if a \n was found
   */

  int getSubTextSize(const UFontDesc*, const char* str, int str_len, 
		     u_dim maxwidth, u_dim chw, u_dim *w, u_dim *h) const;
  ///< returns font metrics of a substring (obsolete version).

  
  // === Misc. ===========================================================

  class UNatWin* createBuffer(u_dim width, u_dim height);
  ///< creates an off-screen pixmap.

  void deleteBuffer(class UNatWin*);
  ///< deletes an off-screen pixmap.

  class UNatGraph* getNatGraph() const {return natgraph;}
  class UWinGraph* getWinGraph() const {return wg;}
  class UNatDisp*  getNatDisp()  const {return natdisp;}
  class UNatWin*   getNatWin()   const;
  /**< gives acces to implementation-dependent hidden objects.
   * Note: these "Native Objects" encapsulate platform-dependant data.
   * their implementation is subject to change and to vary depending on
   * the native platform.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
protected:
  friend class UGraphElem; 

  enum GType {NONE, WIN, CLIENT, SUBWIN, DOUBLEBUF, BLEND};

  class UView*       boxview;  // logical View where the graphics are drawn
  class UWinGraph*   wg;       // Window Graphics context
  class UNatGraph*   natgraph; // (hidden) Native Graphic Context
  class UNatDisp*    natdisp;  // (hidden) Native Display Context
  struct UGraphElem* gelem;    // pointer in graphics stack
  enum {OPAQUE, UNIFORM_BLEND, PSEUDO_BLEND, TRUE_BLEND} blend_mode;
  float alpha;

  void constructs(class UView*);
  void push(GType, class UNatWin* dest, const URegion &clip, 
	    u_pos _xwin, u_pos _ywin);
  void push(GType, bool alloc, const URegion &r);
  void pop();
  void popAll();

  UNatWin* allocBuffer(u_dim width, u_dim height);
  void releaseBuffer(UNatWin *buf);
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** internal Graphics context for drawing Ubit objects.
 * Notes:
 * - clients should use the UGraph (<b>not</b> UWinGraph that is reserved for 
 *   the Ubit lib)
 * - UWinGraph is an optimized version of UGraph for drawing directly
 *   in Ubit windows. Its methods use <b>absolute</b> Window coordinate
 *   instead of relative coordinates in the current View
 * - note that UWinGraph does not use the same internal X GCs than UGraph
 */
class UWinGraph : public UGraph {
public:
  UWinGraph(class UWin*);
  virtual ~UWinGraph();

  bool init(UDisp*, UView* window_view);
  /**< inits the WinGraph.
   * Note: init() does not create the X Window (this is done by UWin::realize)
   */

  bool isWinRealized() const;
  ///< is the X Window created ?.

  class UDisp*  getDisp()  const {return disp;}
  class UAppli* getAppli() const {return &disp->getAppli();}

  class UWin*   getHardwin() const {return hardwin;}
  ///< returns the "hard window" that contains the current View.

  class UNatWin*  getNatWin()  const {return natwin;}
  class UNatDisp* getNatDisp() const {return disp->getNatDisp();}

  void begin(const URegion& clip);
  /**< must be called before drawing (to allocate graphical resources).
   * the clip specificlip specifies where drawing will take place in the WINDOW.
   * a drawing sequence must start by begin() and finish by end()
   */

  void end();
  /**< must be called after drawing (to release graphical resources).
   * a drawing sequence must start by begin() and finish by end()
   */

  void beginDoublebuf(const URegion& clip);
  void beginBlend(const URegion& clip, float alpha);
  void beginSubwin(const URegion& clip, u_pos xwin, u_pos ywin);
  /**< variants of the begin() function.
   * a drawing sequence must always be terminated by end()
   */
  
  void endBlend();

  void blendColor(const URegion& clip, float alpha);
  /**< blends the 'clip' zone with current color.
   * 'alpha' is the value for alpha blending. the current color must
   * have been specified by calling setColor()
   */

  void setClip(const URegion&);
  void setClip(u_pos x, u_pos y, u_dim width, u_dim height);
  void setWinClip(const URegion &r);
  void setWinClip(u_pos x, u_pos y, u_dim width, u_dim height);
  /**< setClip and setWinClip are similar for this class:.
   * notes:
   * - coordinates are relative to the origin of the WINDOW in all cases
   * - the view is no taken into account
   * - see also: UGraph::setClip() and UGraph::setWinClip()
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
private:
  friend class UGraph;
  friend class UGraphElem;
  class UDisp*   disp;	     // logical Display
  class UWin*    hardwin;    // the corresponding hard window
  class UNatWin* natwin;     // pointer to the (hidden) Native Window

  void blendLayers(UGraphElem* layer1, UGraphElem* layer2);
#endif
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
