/***  HANDLERS.C: Contains the handling routines for standard signals and events  ***/

/* ########################################################################

   uwm - THE ude WINDOW MANAGER

   ########################################################################

   Copyright (c) : Christian Ruppert

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ######################################################################## */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/X.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/extensions/shape.h>

#include "uwm.h"
#include "init.h"
#include "windows.h"
#include "special.h"
#include "properties.h"
#include "nodes.h"
#include "handlers.h"

#include "move.h"
#include "resize.h"
#include "menu.h"
#include "uwmmenu.h"
#include "winmenu.h"
#include "workspaces.h"
#include "widgets.h"
#include "applications.h"

#include "MwmUtil.h"

extern UDEScreen TheScreen;
extern Display *disp;
extern XContext UWMContext;
extern UltimateContext *ActiveWin;
extern Atom WM_STATE_PROPERTY;
extern Atom WM_CHANGE_STATE;
extern Atom MOTIF_WM_HINTS;
extern InitStruct InitS;

int ShapeEvent;

/***  Contains actual event handler configuration  ***/

HandlerTable *Handle;


/***  possible event handler configurations  ***
 ***  are set up by InitHandlers()           ***
 ***  should remain private                  ***/

HandlerTable DefaultHandle[LASTEvent];
HandlerTable MoveHandle[LASTEvent];
HandlerTable ResizeHandle[LASTEvent];
HandlerTable WinMenuHandle[LASTEvent];
HandlerTable MenuHandle[LASTEvent];

/*** in case we catch a system term signal... ***/
void TermHandler(int dummy)
{
  SeeYa(1,"Term-Signal received");
}

/*** Handler for system Error ***/
void UWMErrorHandler(Display *disp, XErrorEvent *Err)
{
  char et[256];
  XGetErrorText(disp,Err->error_code,et,255);
  fprintf(TheScreen.errout,"An error #%d occured:\n%s\n",Err->error_code,et);
  fprintf(TheScreen.errout,"It was caused by command (Major-Minor): %d-%d\n",\
                                           Err->request_code,Err->minor_code);
  fprintf(TheScreen.errout,"Look up the meanings in X11/Xproto.h\n");
DBG( fprintf(TheScreen.errout,"The considered resource was: %d\n",\
                                                 Err->resourceid);)
}

/*** Handler invoked in case display connection broke down ***/
void ArmeageddonHandler(void)
{
  SeeYa(1,"Hmm, seems like everything is just about to break down...\nConnect to X-Server lost");
}

/*** in case another WM is running while we start: ***/
void RedirectErrorHandler(void)
{
#ifndef DEBG
  SeeYa(1,"Looks like there is another Window manager running.");
#else
  fprintf (TheScreen.errout,"Looks like there is another Window manager "
	   "running.\nBut since we're debuging we'll continue.");
#endif
}


/*** Handlers for X-EVENTS ***/
/* Have to be functions of type: void NAME(XEvent *event) */


/*** a new Window is born... ***/

void HandleCreate(XEvent *event)
{
  UltimateContext *uc;

DBG(  XWindowAttributes Attr;
      XGetWindowAttributes(disp,event->xcreatewindow.window,&Attr);
      fprintf(TheScreen.errout,"Creating Window %d: x=%d; y=%d; width=%d; height=%d; parent=%d\n",event->xcreatewindow.window,Attr.x,Attr.y,Attr.width,Attr.height,event->xcreatewindow.parent);)

  if(!event->xcreatewindow.override_redirect) {
    if(XFindContext(disp,event->xcreatewindow.window,UWMContext,(XPointer*)&uc))
      uc=UltimizeWin(event->xcreatewindow.window,event->xcreatewindow.parent);
      if(uc) SetWinMapState(uc,WithdrawnState);
  }
}

void HandleDestroy(XEvent *event)
{
  UltimateContext *uc;
  if(!XFindContext(disp,event->xdestroywindow.window,UWMContext,\
                                                (XPointer *)&uc))
    DeUltimizeWin(uc,False);
}

/*** just a change in focus ***/
void HandleEnter(XEvent *event)
{
  UltimateContext *uc;
  if(!XFindContext(disp,event->xcrossing.window,UWMContext,(XPointer *)&uc)){
    ActivateWin(uc);
    if((event->xcrossing.window==uc->title) && (InitS.BorderTitleFlags &\
                           BT_DODGY_TITLE)) XLowerWindow(disp,uc->title);
  }
  /*else
    XSetInputFocus(disp,event->xcrossing.window,RevertToPointerRoot,CurrentTime);*/
}

void HandleLeave(XEvent *event)
{ /* Lazy Focus... */
}

void HandleExpose(XEvent *event)
{
  UltimateContext *uc;

  if(event->xexpose.count) return;
  if(!XFindContext(disp,event->xcrossing.window,UWMContext,(XPointer *)&uc)){
    if(uc->border==event->xcrossing.window) DrawFrameBevel(uc,uc==ActiveWin);
    if(uc->title==event->xcrossing.window) DrawTitle(uc,uc==ActiveWin);
  }
  DBG(fprintf(TheScreen.errout,"Exposure Requested\n");)
}

void HandleMap(XEvent *event)
{
  UltimateContext *uc;
  Bool mapit;
  int format;
  unsigned long number,bytesafter;
  unsigned long *data;
  Atom type;

  if(XFindContext(disp,event->xmaprequest.window,UWMContext,(XPointer *)&uc)) {
    if(!(uc=UltimizeWin(event->xmaprequest.window,event->xmaprequest.parent))){
      XMapWindow(disp,event->xmaprequest.window);
      return;
    }
  }

  UpdateUWMContext(uc);

  if(Success==XGetWindowProperty(disp,uc->win,WM_STATE_PROPERTY,0,2,False,\
                                          WM_STATE_PROPERTY,&type,&format,\
                             &number,&bytesafter,(unsigned char **)&data)) {
    if(data && (data[0]==NormalState)) {
      XFree(data);
      return;
    }
    if(data) XFree(data);
  }

  mapit = True;

  if((uc->WMHints)&&(uc->WMHints->flags&StateHint)){
    switch(uc->WMHints->initial_state){
      case WithdrawnState:
        DBG(fprintf(TheScreen.errout,"WithdrawnState\n");)
        mapit=False;
        break;
      case NormalState:
        DBG(fprintf(TheScreen.errout,"NormalState\n");)
        break;
      case IconicState:
        DBG(fprintf(TheScreen.errout,"IconicState\n");)
        EnborderWin(uc);
        IconifyWin(uc);
        mapit=False;
        break;
    }
  }
  if(mapit) MapWin(uc,False);

DBG({ XWindowAttributes Attr;
    XGetWindowAttributes(disp,event->xmaprequest.window,&Attr);
    fprintf(TheScreen.errout,"Mapping Window %d: x=%d; y=%d; width=%d; height=%d\n",event->xmaprequest.window,Attr.x,Attr.y,Attr.width,Attr.height);})
}

void HandleMapNotify(XEvent *event)
{
  UltimateContext *uc;

  if(!XFindContext(disp,event->xmap.window,UWMContext,(XPointer *)&uc)){
    if((uc->frame==event->xmap.event)&&(uc->win==event->xmap.window))
      uc->status&=~REPARENTING;
    if(uc->title!=None){
      if(uc->name && (!(uc->status & SHAPED)) && (InitS.BorderTitleFlags &\
                          BT_INACTIVE_TITLE)) XRaiseWindow(disp,uc->title);
      else XLowerWindow(disp,uc->title); 
    }
  }
}

void HandleUnmap(XEvent *event)
{
  UltimateContext *uc;
  XEvent event2;

  DBG(fprintf(TheScreen.errout,"Unmap Requested: %d\n",event->xunmap.window);)

  if(!XFindContext(disp,event->xunmap.window,UWMContext,(XPointer *)&uc)){
    if(uc->status & REPARENTING) {
      uc->status &= ~REPARENTING;
      return;
    }

    if(event->xunmap.window==uc->win) {
      if(uc->status&KEEPMAPPED) uc->status &= ~KEEPMAPPED;
      else uc->status &= ~MAPPED;
    }

    if(uc==ActiveWin) ActivateWin(NULL);
    GrabServer();
    XSync(disp,False);
    if(XCheckTypedWindowEvent(disp,event->xunmap.event,DestroyNotify,&event2)&&\
                                       (event2.xdestroywindow.window==uc->win)){
                          /* window already destroyed, clean up */
      DeUltimizeWin(uc,False);
    } else if(uc->win==event->xunmap.window){
      if(uc->status&KEEPENBORDERED) {
        uc->status&=~KEEPENBORDERED;
        XUnmapWindow(disp,uc->frame);
      } else {
        DisenborderWin(uc,True);
        SetWinMapState(uc,WithdrawnState);
        XRemoveFromSaveSet(disp,uc->win);
      }
    }
    UngrabServer();
  }
}

void HandleButtonPress(XEvent *event)
{
  UltimateContext *uc;
  int dummy;
  Window rootret,childret;

DBG(fprintf(TheScreen.errout,"Button pressed: ");)

  XQueryPointer(disp,TheScreen.root,&rootret,&childret,&dummy,&dummy,\
                                                &dummy,&dummy,&dummy);
  if((event->xbutton.window==TheScreen.root)&&(childret==None)){
    switch(event->xbutton.button){
      case Button1: ShowMenu(0,event->xbutton.x,event->xbutton.y);
                    break;
      case Button2: ShowMenu(1,event->xbutton.x,event->xbutton.y); 
                    break;
      case Button3: ShowMenu(2,event->xbutton.x,event->xbutton.y);
                    break;
      case Button4: break;
      case Button5: break;
    }
  }
 
  if(!XFindContext(disp,event->xbutton.window,UWMContext,(XPointer *)&uc)){
    int x,y;

    if(uc->frame!=None) XQueryPointer(disp,uc->frame,&rootret,&childret,\
                                             &dummy,&dummy,&x,&y,&dummy);
    ActivateWin(uc);
    switch(event->xbutton.button){
      case Button1: BorderButton(0,uc,x,y,event->xbutton.x_root,\
                                          event->xbutton.y_root);
                    break;
      case Button2: BorderButton(1,uc,x,y,event->xbutton.x_root,\
                                          event->xbutton.y_root);
                    break;
      case Button3: BorderButton(2,uc,x,y,event->xbutton.x_root,\
                                          event->xbutton.y_root);
                    break;
      case Button4: DBG(fprintf(TheScreen.errout,"4\n");)break;
      case Button5: DBG(fprintf(TheScreen.errout,"5\n");)break;
    }
  }
}

void HandleButtonRelease(XEvent *event)
{
  DBG(fprintf(TheScreen.errout,"Button Released\n");)
}

void HandleMotion(XEvent *event)
{
  DBG(fprintf(TheScreen.errout,"Motion Requested\n");)
}


void HandleConfigure(XEvent *event)
{
  UltimateContext *uc;
  
  if(XFindContext(disp,event->xconfigurerequest.window,UWMContext,(XPointer *)\
                  &uc)) XConfigureWindow(disp,event->xconfigurerequest.window,\
                       event->xconfigurerequest.value_mask,(XWindowChanges *)&\
                                                 (event->xconfigurerequest.x));
  else {
    if(event->xconfigurerequest.value_mask&CWSibling){
      if(XFindContext(disp,event->xconfigurerequest.above,UWMContext,\
                        (XPointer *)&(uc->MySibling))) uc->MySibling=\
           UltimizeWin(event->xconfigurerequest.above,TheScreen.root);
        
      if(uc->MySibling)
        if(!NodeAppend(uc->MySibling->SiblingTo,uc))
	  SeeYa(1,"FATAL: out of memory!");
    }
    if(event->xconfigurerequest.value_mask&CWStackMode)
      uc->StackMode=event->xconfigurerequest.detail;
    if(uc->frame!=None){
      XWindowChanges ch;
      ch=*((XWindowChanges *)&(event->xconfigurerequest.x));
      ch.width+=2*uc->BorderWidth;
      ch.height+=2*uc->BorderWidth+TheScreen.TitleHeight;
      if(event->xconfigurerequest.value_mask&CWSibling){
        if(uc->MySibling->frame) ch.sibling=uc->MySibling->frame;
	else ch.sibling=uc->MySibling->win;
      }  
      XConfigureWindow(disp,uc->border,event->xconfigurerequest.value_mask&\
                                                    (CWWidth|CWHeight),&ch);
      XConfigureWindow(disp,uc->frame,event->xconfigurerequest.value_mask&\
                     (CWSibling|CWStackMode|CWX|CWY|CWWidth|CWHeight),&ch);
      XConfigureWindow(disp,uc->win,event->xconfigurerequest.value_mask&\
                                  (CWWidth|CWHeight),(XWindowChanges *)&\
                                           (event->xconfigurerequest.x));
      UpdateUWMContext(uc);
      if((uc->title!=None) && (InitS.BorderTitleFlags & BT_CENTER_TITLE) && \
                             (event->xconfigurerequest.value_mask & CWWidth))
        XMoveWindow(disp,uc->title,(uc->Attr.width-uc->TitleWidth)/2,\
             (uc->BorderWidth-TheScreen.desktop.FrameBevelWidth-1)/2+\
                                   TheScreen.desktop.FrameBevelWidth);
      if((event->xconfigurerequest.value_mask&CWStackMode)||(\
            (event->xconfigurerequest.value_mask&(CWX|CWY))&&\
            (!(event->xconfigurerequest.value_mask&(CWWidth|CWHeight))))) 
        SendConfigureEvent(uc);
    } else {
      XConfigureWindow(disp,event->xconfigurerequest.window,\
                        event->xconfigurerequest.value_mask,\
            (XWindowChanges *)&(event->xconfigurerequest.x));
      UpdateUWMContext(uc);
    }
  }
DBG(fprintf(TheScreen.errout,"Reconfigured: %d\n",event->xconfigurerequest.window);)

}

void HandleColormap(XEvent *event)
{
  DBG(fprintf(TheScreen.errout,"Colormap Requested\n");)
}

void HandleClientMsg(XEvent *event)
{
DBG(fprintf(TheScreen.errout,"Client sent Msg\n");)
  if((event->xclient.message_type == WM_CHANGE_STATE)&&\
             (event->xclient.data.l[0] == IconicState)){
    UltimateContext *uc;
    if(!XFindContext(disp,event->xclient.window,UWMContext,(XPointer *)&uc)){
      IconifyWin(uc);
    }
  }
}

void HandleKeyPress(XEvent *event)
{
/*  DBG(fprintf(TheScreen.errout,"Key pressed\n");)
*/}

void HandleKeyRelease(XEvent *event)
{
  if(event->xkey.state==(ControlMask|Mod1Mask)){
    Node *n,*n2;
    switch(XKeycodeToKeysym(disp,event->xkey.keycode,0)){
      case XK_Right: ChangeWS((TheScreen.desktop.ActiveWorkSpace +1)\
                                     % TheScreen.desktop.WorkSpaces);
                     break;
      case XK_Left:  ChangeWS((TheScreen.desktop.ActiveWorkSpace
                              + TheScreen.desktop.WorkSpaces
                              - 1) % TheScreen.desktop.WorkSpaces);
                     break;
      case XK_Up:    n=n2=InNodeList(TheScreen.UltimateList,ActiveWin);{
                       do {
                         if(n2=NodePrev(TheScreen.UltimateList,n2))
                           if(WinVisible(n2->data)) break;
                       } while(n!=n2);
                       if(n2) ActivateWin(n2->data); else ActivateWin(NULL);
                     }
                     break;
      case XK_Down:  n=n2=InNodeList(TheScreen.UltimateList,ActiveWin);{
                       do {
                         if(n2=NodeNext(TheScreen.UltimateList,n2))
                           if(WinVisible(n2->data)) break;
                       } while(n!=n2);
                       if(n2) ActivateWin(n2->data); else ActivateWin(NULL);
                     }
                     break;
      case XK_Page_Up: if(ActiveWin) RaiseWin(ActiveWin);
                     break;
      case XK_Page_Down: if(ActiveWin) LowerWin(ActiveWin);
                     break;
      case XK_End:   if(ActiveWin) IconifyWin(ActiveWin);
                     break;
    }
  }
}

void HandlePropertyChange(XEvent *event)
{
  UltimateContext *uc;
  if(!XFindContext(disp,event->xproperty.window,UWMContext,(XPointer *)&uc)){
    switch(event->xproperty.atom){
      case XA_WM_NORMAL_HINTS: Updatera(uc);break;
      case XA_WM_NAME: UpdateName(uc);break;
      case XA_WM_HINTS: UpdateWMHints(uc);break;
      default:
        if(event->xproperty.atom==MOTIF_WM_HINTS) UpdateMotifHints(uc);
    }
  }
}  

void HandleShape(XEvent *event)
{
  UltimateContext *uc;
  XShapeEvent *shev;
  shev=(XShapeEvent *)event;
  if(!XFindContext(disp,shev->window,UWMContext,(XPointer *)&uc)){
    ShapeFrame(uc);
  }
}  


/***********************/
void InitHandlers()
{
  int i;
  
  for(i=0; i < LASTEvent; i++)
    DefaultHandle[i]= NULL;
  DefaultHandle[Expose]=HandleExpose;
  DefaultHandle[CreateNotify]=HandleCreate;
  DefaultHandle[DestroyNotify]=HandleDestroy;
  DefaultHandle[UnmapNotify]=HandleUnmap;
  DefaultHandle[ClientMessage]=HandleClientMsg;
  DefaultHandle[ConfigureRequest]=HandleConfigure;
  DefaultHandle[ColormapNotify]=HandleColormap;
  DefaultHandle[MapRequest]=HandleMap;
  DefaultHandle[MapNotify]=HandleMapNotify;
  DefaultHandle[EnterNotify]=HandleEnter;
  DefaultHandle[LeaveNotify]=HandleLeave;
  DefaultHandle[ButtonPress]=HandleButtonPress;
  DefaultHandle[MotionNotify]=HandleMotion;
  DefaultHandle[ButtonRelease]=HandleButtonRelease;
  DefaultHandle[KeyPress]=HandleKeyPress;
  DefaultHandle[KeyRelease]=HandleKeyRelease;
  DefaultHandle[PropertyNotify]=HandlePropertyChange;

  XShapeQueryExtension(disp,&ShapeEvent,&i);
  ShapeEvent+=ShapeNotify;

  Handle=DefaultHandle;
  for(i=0;i<LASTEvent;i++) MoveHandle[i]=DefaultHandle[i];
  MoveHandle[EnterNotify]=NULL;
  MoveHandle[LeaveNotify]=NULL;
  MoveHandle[ButtonPress]=MoveButtonPress;
  MoveHandle[MotionNotify]=MoveMotion;
  MoveHandle[ButtonRelease]=MoveButtonRelease;
  MoveHandle[UnmapNotify]=MoveUnmap;
  for(i=0;i<LASTEvent;i++) ResizeHandle[i]=DefaultHandle[i];
  ResizeHandle[EnterNotify]=NULL;
  ResizeHandle[LeaveNotify]=NULL;
  ResizeHandle[ButtonPress]=ResizeButtonPress;
  ResizeHandle[MotionNotify]=ResizeMotion;
  ResizeHandle[ButtonRelease]=ResizeButtonRelease;
  for(i=0;i<LASTEvent;i++) WinMenuHandle[i]=DefaultHandle[i];
  WinMenuHandle[EnterNotify]=WinMenuEnterNotify;
  WinMenuHandle[LeaveNotify]=NULL;
  WinMenuHandle[ButtonPress]=WinMenuButtonPress;
  WinMenuHandle[ButtonRelease]=WinMenuButtonRelease;
  WinMenuHandle[VisibilityNotify]=WinMenuVisibility;
  for(i=0;i<LASTEvent;i++) MenuHandle[i]=DefaultHandle[i];
  MenuHandle[EnterNotify]=MenuEnterNotify;
  MenuHandle[LeaveNotify]=MenuLeaveNotify;
  MenuHandle[ButtonPress]=MenuButtonPress;
  MenuHandle[ButtonRelease]=MenuButtonRelease;
  MenuHandle[VisibilityNotify]=MenuVisibility;
}

/**********/

void ReinstallDefaultHandle()
{
  Handle=DefaultHandle;
}

void InstallMoveHandle()
{
  Handle=MoveHandle;
}

void InstallResizeHandle()
{
  Handle=ResizeHandle;
}

void InstallWinMenuHandle()
{
  Handle=WinMenuHandle;
}

void InstallMenuHandle()
{
  Handle=MenuHandle;
}
