/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <ug_settings_dialog.h>
#include <ug_utils.h>

#include <glib/gi18n.h>

// callback functions
static void	on_clipboard_monitor_toggled (GtkWidget* widget, UgSettingsDialog* dialog);
static void	on_launch_app_toggled (GtkWidget* widget, UgSettingsDialog* dialog);
static void	on_auto_save_toggled (GtkWidget* widget, UgSettingsDialog* dialog);

UgSettingsDialog*	ug_settings_dialog_new (const gchar* title, GtkWindow* parent)
{
	UgSettingsDialog*	dialog;
	GtkWidget*			widget;
	GtkWidget*			entry;
	GtkBox*				vbox;
	GtkBox*				hbox;

	dialog = g_malloc0 (sizeof (UgSettingsDialog));
	dialog->self = (GtkDialog*) gtk_dialog_new_with_buttons (title, parent,
	         (GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_DESTROY_WITH_PARENT),
	                  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                  GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                  NULL);
	gtk_dialog_set_default_response (dialog->self, GTK_RESPONSE_OK);

	dialog->notebook = (GtkNotebook*) gtk_notebook_new ();
	gtk_widget_set_size_request ((GtkWidget*) dialog->notebook, 365, 300);
	vbox = (GtkBox*) dialog->self->vbox;
	gtk_box_pack_start ((GtkBox*) vbox, (GtkWidget*) dialog->notebook, FALSE, FALSE, 0);

	// ------------------------------------------------------------------------
	// Clipboard settings page
	vbox = (GtkBox*) gtk_vbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
	gtk_notebook_append_page (dialog->notebook, (GtkWidget*)vbox, gtk_label_new (_("Clipboard")));
	// Monitor button
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	widget = gtk_check_button_new_with_mnemonic (_("_Monitor clipboard for specified file types:"));
	gtk_box_pack_start (hbox, widget, FALSE, FALSE, 2);
	g_signal_connect (widget, "toggled", G_CALLBACK (on_clipboard_monitor_toggled), dialog);
	dialog->monitor_button = widget;

	// file type pattern : label and entry
	entry = gtk_entry_new ();
	gtk_box_pack_start (vbox, entry, FALSE, FALSE, 2);
	dialog->pattern_entry = entry;

	// tips
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	gtk_box_pack_end (hbox, gtk_label_new (_("Separate the types with character '|'.")), FALSE, FALSE, 2);
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	gtk_box_pack_end (hbox, gtk_label_new (_("You can use regular expressions here.")), FALSE, FALSE, 2);
	// focus
	gtk_container_set_focus_child (GTK_CONTAINER (dialog->self), dialog->monitor_button);

	// ------------------------------------------------------------------------
	// Others settings page
	vbox = (GtkBox*) gtk_vbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
	gtk_notebook_append_page (dialog->notebook, (GtkWidget*)vbox, gtk_label_new (_("Others")));

	// Launch default application for specified file types.
	// launch app button
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	widget = gtk_check_button_new_with_mnemonic (_("_Launch default application for specified file types:"));
	gtk_box_pack_start (hbox, widget, FALSE, FALSE, 2);
	g_signal_connect (widget, "toggled", G_CALLBACK (on_launch_app_toggled), dialog);
	dialog->launch_app_button = widget;
	// launch app entry
	entry = gtk_entry_new ();
	gtk_box_pack_start (vbox, entry, FALSE, FALSE, 2);
	dialog->launch_app_entry = entry;
	// Launch app tips
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	gtk_box_pack_end (hbox, gtk_label_new (_("Separate the types with character '|'.")), FALSE, FALSE, 2);
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	gtk_box_pack_end (hbox, gtk_label_new (_("You can use regular expressions here.")), FALSE, FALSE, 2);
	// separator
	gtk_box_pack_start (vbox, gtk_hseparator_new (), FALSE, FALSE, 4);

	// ------------------------------------------------------------------------
	// auto save button
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	widget = gtk_check_button_new_with_mnemonic (_("_Auto save"));
	gtk_box_pack_start (hbox, widget, FALSE, FALSE, 2);
	g_signal_connect (widget, "toggled", G_CALLBACK (on_auto_save_toggled), dialog);
	dialog->auto_save_button = widget;
	// auto save spin & label (interval)
	widget = gtk_label_new_with_mnemonic (_("minutes"));
	gtk_box_pack_end (hbox, widget, FALSE, FALSE, 2);
	dialog->auto_save_minutes_label = widget;
	widget = gtk_spin_button_new_with_range (1.0, 120.0, 1.0);
	gtk_box_pack_end (hbox, widget, FALSE, FALSE, 2);
	dialog->auto_save_spin = widget;
	// auto save label
	widget = gtk_label_new_with_mnemonic (_("_Interval:"));
	gtk_box_pack_end (hbox, widget, FALSE, FALSE, 2);
	dialog->auto_save_label = widget;
	gtk_label_set_mnemonic_widget (GTK_LABEL(dialog->auto_save_label), dialog->auto_save_spin);
	// separator
	gtk_box_pack_start (vbox, gtk_hseparator_new (), FALSE, FALSE, 4);
	// confirmation
	widget = gtk_check_button_new_with_label (_("Show confirmation dialog on close"));
	gtk_box_pack_start (vbox, widget, FALSE, FALSE, 2);
	dialog->close_confirmation_button = widget;
	// start in tray
	widget = gtk_check_button_new_with_label (_("Minimize to tray on startup"));
	gtk_box_pack_start (vbox, widget, FALSE, FALSE, 2);
	dialog->start_in_tray_button = widget;

	gtk_widget_show_all ((GtkWidget*) dialog->notebook);
//	gtk_container_set_focus_child (GTK_CONTAINER (dialog->self), dialog->pattern_entry);
//	g_signal_connect (dialog->pattern_entry, "key-press-event", G_CALLBACK (on_key_press_event), dialog);

	return dialog;
}

void	ug_settings_dialog_destroy (UgSettingsDialog* dialog)
{
	gtk_widget_destroy ((GtkWidget*) dialog->self);
	g_free (dialog);
}

void	ug_settings_dialog_get (UgSettingsDialog* dialog, Uget* app)
{
	// clipboard
	app->clipboard_monitor = gtk_toggle_button_get_active ((GtkToggleButton*) dialog->monitor_button);
	gtk_check_menu_item_set_active ((GtkCheckMenuItem*) app->menubar.edit.clipboard_monitor, app->clipboard_monitor);

	g_free (app->clipboard_pattern);
	app->clipboard_pattern = g_strdup (gtk_entry_get_text ((GtkEntry*) dialog->pattern_entry));
	g_regex_unref (app->clipboard_regex);
	app->clipboard_regex = g_regex_new (app->clipboard_pattern, G_REGEX_CASELESS, 0, NULL);

	// launch app
	app->launch_app = gtk_toggle_button_get_active ((GtkToggleButton*) dialog->launch_app_button);
	g_free (app->launch_app_types);
	app->launch_app_types = g_strdup (gtk_entry_get_text ((GtkEntry*) dialog->launch_app_entry));
	g_regex_unref (app->launch_app_regex);
	app->launch_app_regex = g_regex_new (app->launch_app_types, G_REGEX_CASELESS, 0, NULL);

	// auto save
	app->auto_save = gtk_toggle_button_get_active ((GtkToggleButton*) dialog->auto_save_button);
	app->auto_save_interval = gtk_spin_button_get_value_as_int ((GtkSpinButton*) dialog->auto_save_spin);

	// confirmation
	app->window_close_confirmation = gtk_toggle_button_get_active ((GtkToggleButton*) dialog->close_confirmation_button);
	app->start_in_tray = gtk_toggle_button_get_active ((GtkToggleButton*) dialog->start_in_tray_button);
}

void	ug_settings_dialog_set (UgSettingsDialog* dialog, Uget* app)
{
	// clipboard
	gtk_toggle_button_set_active ((GtkToggleButton*) dialog->monitor_button, app->clipboard_monitor);
	gtk_entry_set_text ((GtkEntry*) dialog->pattern_entry, app->clipboard_pattern);
	on_clipboard_monitor_toggled (dialog->monitor_button, dialog);

	// launch app
	gtk_toggle_button_set_active ((GtkToggleButton*) dialog->launch_app_button, app->launch_app);
	gtk_entry_set_text ((GtkEntry*) dialog->launch_app_entry, app->launch_app_types);
	on_launch_app_toggled (dialog->launch_app_button, dialog);

	// auto save
	gtk_toggle_button_set_active ((GtkToggleButton*) dialog->auto_save_button, app->auto_save);
	gtk_spin_button_set_value ((GtkSpinButton*)dialog->auto_save_spin, (gdouble)app->auto_save_interval);
	on_auto_save_toggled (dialog->auto_save_button, dialog);

	// confirmation
	gtk_toggle_button_set_active ((GtkToggleButton*) dialog->close_confirmation_button, app->window_close_confirmation);
	gtk_toggle_button_set_active ((GtkToggleButton*) dialog->start_in_tray_button, app->start_in_tray);
}

// ----------------------------------------------------------------------------
// callback functions
static void	on_clipboard_monitor_toggled (GtkWidget* widget, UgSettingsDialog* dialog)
{
	gboolean	sensitive;

	sensitive = gtk_toggle_button_get_active ((GtkToggleButton*)dialog->monitor_button);
	gtk_widget_set_sensitive (dialog->pattern_entry, sensitive);
}

static void	on_launch_app_toggled (GtkWidget* widget, UgSettingsDialog* dialog)
{
	gboolean	sensitive;

	sensitive = gtk_toggle_button_get_active ((GtkToggleButton*)dialog->launch_app_button);
	gtk_widget_set_sensitive (dialog->launch_app_entry, sensitive);
}

static void	on_auto_save_toggled (GtkWidget* widget, UgSettingsDialog* dialog)
{
	gboolean	sensitive;

	sensitive = gtk_toggle_button_get_active ((GtkToggleButton*)dialog->auto_save_button);
	gtk_widget_set_sensitive (dialog->auto_save_label, sensitive);
	gtk_widget_set_sensitive (dialog->auto_save_spin, sensitive);
	gtk_widget_set_sensitive (dialog->auto_save_minutes_label, sensitive);
}

