/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 2001  Simon Baldwin (simonb@caldera.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * cw.c - Filter program to output morse code on the system console.
 *
 */

/* Include files. */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/param.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <limits.h>
#include <stdarg.h>
#include <errno.h>
#include <assert.h>

#if defined(HAVE_STRING_H)
#	include <string.h>
#endif /* HAVE_STRING_H */
#if defined(HAVE_STRINGS_H)
#	include <strings.h>
#endif /* HAVE_STRINGS_H */

#if defined(HAVE_GETOPT_H)
#	include <getopt.h>			/* Linux */
#endif /* HAVE_GETOPT_H */

#if !defined(MAXPATHLEN)
#	define	MAXPATHLEN	_POSIX_PATH_MAX	/* OpenServer */
#endif /* not MAXPATHLEN */

/* Include definitions of CW library functions, and control parameters. */
#include "cwlib.h"
#include "cw.h"

/* Macro definitions. */
#define	VERSION		"cw version 2.0"
#define	COPYRIGHT	"Copyright (C) 2001  Simon Baldwin\n"		\
"This program comes with ABSOLUTELY NO WARRANTY; for details\n"		\
"please see the file 'COPYING' supplied with the source code.\n"	\
"This is free software, and you are welcome to redistribute it\n"	\
"under certain conditions; again, see 'COPYING' for details.\n"		\
"This program is released under the GNU General Public License."

#define	ASC_NUL		'\0'		/* End of string */
#define	ASC_SPACE	' '		/* ASCII space char */
#define	ASC_FNS		'/'		/* ASCII filename sep char */

/* Other general macros. */
#define	MAXPRINTFCW	1024		/* Longest buffered cw message */
#define	ASC_WHITESPACE	" \t\n\r"	/* Characters treated as whitespace */

#define	MAXARGS		128		/* Args to exec */
#define	MAXOPTSTR	1024		/* Longest _OPTIONS env variable */
#define	ARGS_WHITESPACE	" \t"		/* Argument separators */
#define	ENV_OPTIONS	"CW_OPTIONS"	/* Env string holding options */

#define	INITIAL_SOUNDFD	STDOUT_FILENO	/* Initial sound on stdout */

/* True/False macros. */
#define	FALSE		(0)
#define	TRUE		(!FALSE)

/*
 * Program-specific state variables, settable from the command line, or from
 * embedded input stream commands.  The do* 'boolean' options are declared
 * here as ints to allow the embedded command parser to set them to values
 * other than strictly TRUE or FALSE; all non-zero values are equivalent to
 * TRUE.
 */
static int do_echo	= TRUE;		/* Initially echoing characters */
static int do_errors	= TRUE;		/* Print error messages to stderr */
static int do_commands	= TRUE;		/* Execute embedded commands */
static int do_combinations
			= TRUE;		/* Execute [...] combinations */
static int do_comments	= TRUE;		/* Allow {...} as comments */

/* Base name of the program, from argv[0]. */
static char *argv0	= NULL;


/*
 * print_usage()
 *
 * Print out a brief message directing the user to the help function.
 */
static void
print_usage ()
{
	assert (argv0 != NULL);

	fprintf (stderr,
#if defined(HAVE_GETOPT_LONG)
		"Try '%s --help' for more information.\n",
#else  /* not HAVE_GETOPT_LONG */
		"Try '%s -h' for more information.\n",
#endif /* not HAVE_GETOPT_LONG */
		argv0);
	exit (1);
}

/*
 * print_help()
 *
 * Print out a brief page of help information.
 */
static void
print_help ()
{
	int	min_speed, max_speed;		/* Cw library speed limits */
	int	min_frequency, max_frequency;	/* Cw library frequncy limits */
	int	min_gap, max_gap;		/* Cw library gap limits */
	assert (argv0 != NULL);

	/* Read the cw library limits on its operating parameters. */
	cw_get_speed_limits	(&min_speed, &max_speed);
	cw_get_frequency_limits	(&min_frequency, &max_frequency);
	cw_get_gap_limits	(&min_gap, &max_gap);

	printf (
#if defined(HAVE_GETOPT_LONG)
	"Usage: %s [options...]\n\n\
	-d, --device=DEVICE	use DEVICE for sound ioctl [default stdout]\n\
	-f, --file=FILE		read from FILE [default stdin]\n\
	-w, --wpm=WPM		set initial words per minute [default %d]\n\
				valid WPM values are between %d and %d\n\
	-t, --hz,--tone=HZ	set initial tone to HZ [default %d]\n\
				valid HZ values are between %d and %d\n\
	-g, --gap=GAP		set extra gap between letters [default %d]\n\
				valid GAP values are between %d and %d\n\
	-e, --noecho		don't echo sending to stdout [default echo]\n\
	-m, --nomessages	don't write messages to stderr [default msgs]\n\
	-c, --nocommands	don't execute embedded commands [default cmds]\n\
	-o, --nocombinations	don't allow [...] combinations [default combo]\n\
	-p, --nocomments	don't allow {...} comments [default comments]\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#else  /* not HAVE_GETOPT_LONG */
	"Usage: %s [options...]\n\n\
	-d DEVICE	use DEVICE for sound ioctl [default stdout]\n\
	-f FILE		read from FILE [default stdin]\n\
	-w WPM		set initial words per minute [default %d]\n\
			valid WPM values are between %d and %d\n\
	-t HZ		set initial tone to HZ [default %d]\n\
			valid HZ values are between %d and %d\n\
	-g GAP		set extra gap between letters [default %d]\n\
			valid GAP values are between %d and %d\n\
	-e		don't echo sending to stdout [default echo]\n\
	-m		don't write messages to stderr [default msgs]\n\
	-c		don't execute embedded commands [default cmds]\n\
	-o		don't allow [...] combinations [default combo]\n\
	-p		don't allow {...} comments [default comments]\n\
	-h		print this message\n\
	-v		output version information and exit\n\n",
#endif /* not HAVE_GETOPT_LONG */
	argv0,
	cw_get_send_speed (),	min_speed,	max_speed,
	cw_get_frequency (),	min_frequency,	max_frequency,
	cw_get_gap (),		min_gap,	max_gap);
	exit (0);
}


/*
 * parse_cmdline()
 *
 * Parse the command line options for initial values for the various
 * global and flag definitions.
 */
static void
parse_cmdline (int argc, char **argv)
{
	int	c;				/* Option character */
	int	sound_fd;			/* Output file descriptor */
	int	intarg;				/* General integer arg holder */
	int	argind;				/* Loop index */
	char	env_options[ MAXOPTSTR ];	/* Env options string */
	char	*sptr;				/* String pointer */
	char	*local_argv[ MAXARGS ];		/* Local argv array */
	int	local_argc = 0;			/* Local argc */
	char	device[ MAXPATHLEN ];		/* Path to device file */
#if defined(HAVE_GETOPT_LONG)
	int	option_index;			/* Option index */
	static const struct option long_options[] = {	/* Options table */
		{ "device",	1, 0, 'd' }, { "file",	1, 0, 'f' },
		{ "tone",	1, 0, 't' }, { "hz",	1, 0, 't' },
		{ "wpm",	1, 0, 'w' }, { "gap",	1, 0, 'g' },
		{ "noecho",	0, 0, 'e' }, { "nomessages",
						     	0, 0, 'm' },
		{ "nocommands",	0, 0, 'c' },
		{ "nocombinations",
				0, 0, 'o' },
		{ "nocomments",	0, 0, 'p' }, { "help",	0, 0, 'h' },
		{ "version",	0, 0, 'v' },
		{ 0, 0, 0, 0 }};
#endif /* HAVE_GETOPT_LONG */

	/* Set argv0 to be the basename part of the program name. */
	argv0 = argv[0] + strlen (argv[0]);
	while (*argv0 != ASC_FNS && argv0 > argv[0])
		argv0--;
	if (*argv0 == ASC_FNS)
		argv0++;

	/*
	 * Build a new view of argc and argv by first prepending
	 * the strings from ..._OPTIONS, if defined, then putting the
	 * command line args on (so they take precedence).
	 */
	local_argv[ local_argc++ ] = argv[0];
	if (getenv (ENV_OPTIONS) != NULL)
	    {
		strcpy (env_options, getenv (ENV_OPTIONS));
		sptr = env_options;
		while (local_argc < MAXARGS - 1)
		    {
			while (strchr (ARGS_WHITESPACE, *sptr) != NULL
					&& *sptr != ASC_NUL)
				sptr++;
			if ( *sptr == ASC_NUL )
				break;
			else
			    {
				local_argv[ local_argc++ ] = sptr;
				while (strchr (ARGS_WHITESPACE, *sptr)
						== NULL && *sptr != ASC_NUL)
					sptr++;
				if (strchr (ARGS_WHITESPACE, *sptr)
						!= NULL && *sptr != ASC_NUL)
				    {
					*sptr = ASC_NUL;
					sptr++;
				    }
			    }
		    }
	    }
	for (argind = 1; argind < argc; argind++)
	    {
		local_argv[ local_argc++ ] = argv[ argind ];
	    }

	/* Initialize the sound device to stdout. */
	sound_fd = dup (INITIAL_SOUNDFD);
	if (sound_fd == -1)
	    {
		perror ("dup");
		exit (1);
	    }
	strcpy (device, "stdout");

	/* Process every option. */
	while (TRUE)
	    {
#if defined(HAVE_GETOPT_LONG)
		c = getopt_long (local_argc, local_argv, "d:f:t:w:g:emcophv",
					long_options, &option_index);
#else  /* not HAVE_GETOPT_LONG */
		c = getopt (local_argc, local_argv, "d:f:t:w:g:emcophv");
#endif /* not HAVE_GETOPT_LONG */
		if (c == -1)
			break;

		switch (c)
		    {
			case 'd':
				close (sound_fd);

				sound_fd = open (optarg, O_RDWR);
				if (sound_fd == -1)
				    {
					fprintf (stderr,
					    "%s: error opening output device\n",
									argv0);
					perror (optarg);
					exit (1);
				    }
				strcpy (device, optarg);
				break;

			case 'f':
				if (freopen (optarg, "r", stdin) == NULL)
				    {
					fprintf (stderr,
					    "%s: error opening input file\n",
									argv0);
					perror (optarg);
					exit (1);
				    }
				break;

			case 't':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_frequency (intarg) != 0 )
				    {
					fprintf (stderr,
					    "%s: invalid tone value\n", argv0);
					exit (1);
				    }
				break;

			case 'w':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_send_speed (intarg) != 0 )
				    {
					fprintf (stderr,
					    "%s: invalid wpm value\n", argv0);
					exit (1);
				    }
				break;

			case 'g':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_gap (intarg) != 0 )
				    {
					fprintf (stderr,
					    "%s: invalid gap value\n", argv0);
					exit (1);
				    }
				break;

			case 'e':
				do_echo		= FALSE;
				break;

			case 'm':
				do_errors	= FALSE;
				break;

			case 'c':
				do_commands	= FALSE;
				break;

			case 'o':
				do_combinations	= FALSE;
				break;

			case 'p':
				do_comments	= FALSE;
				break;

			case 'h':
				print_help ();
				exit (0);

			case 'v':
				printf ("%s, ", VERSION);
				printf ("%s\n", COPYRIGHT);
				exit (0);

			case '?':
				print_usage ();

			default:
				fprintf (stderr,
					"%s: getopts returned '%c'\n",
								argv0, c);
				exit (1);
		    }
	    }
	if (optind != local_argc)
		print_usage ();

	/* Check that the output device, default or selected, does sound. */
	if (cw_set_file_descriptor (sound_fd) != 0)
	    {
		fprintf (stderr, "%s: output device won't do sound\n", argv0);
		perror (device);
		exit (1);
	    }
}


/*
 * fprintf_stdout()
 * fprintf_stderr()
 *
 * Local fprintf functions that suppress output to stdout or stderr if
 * the do_echo and/or do_errors flags are not set.
 */
static void
fprintf_stdout (const char *format, ...)
{
	va_list		ap;			/* Varargs list */

	/* Only call the fprintf if the output flag is set. */
	if (do_echo)
	    {
		va_start (ap, format);
		vfprintf (stdout, format, ap);
		fflush (stdout);
		va_end (ap);
	    }
}
static void
fprintf_stderr (const char *format, ...)
{
	va_list		ap;			/* Varargs list */

	/* Only call the fprintf if the output flag is set. */
	if (do_errors)
	    {
		va_start (ap, format);
		vfprintf (stderr, format, ap);
		fflush (stderr);
		va_end (ap);
	    }
}


/*
 * fprintf_cw()
 *
 * Fake fprintf function that allows us to conveniently 'print' to the
 * cw output 'stream'.  Like I said, it's a fake.
 */
static void
fprintf_cw (const char *format, ...)
{
	va_list		ap;			/* Varargs list */
	char		buffer[ MAXPRINTFCW ];	/* Sprintf buffer */

	/*
	 * Format the cw send buffer using vsprintf.  Formatted strings
	 * longer than the declared buffer will be truncated to the buffer
	 * length.
	 */
	va_start (ap, format);
	vsnprintf (buffer, sizeof (buffer), format, ap);
	va_end (ap);

	/* Sound out the buffer, and wait for the send to complete. */
	if (cw_send_string (buffer) != 0)
	    {
		perror ("cw_send_string");
		cw_flush_tone_queue ();
		exit (1);
	    }
	if (cw_tone_queue_wait () != 0)
	    {
		perror ("cw_tone_queue_wait");
		cw_flush_tone_queue ();
		exit (1);
	    }
}


/*
 * handle_stream_query()
 *
 * Handle a query received in the input stream.  The command escape
 * character and the query character have already been read and recognized.
 */
static void
handle_stream_query (FILE *stream)
{
	unsigned int	c;			/* Character from stdin */

	/* Get the specific query identifier. */
	c = toupper (getc (stream));
	switch (c)
	    {
		case EOF:
			return;
		default:
			fprintf_stderr
				("%c%c%c", CW_STATUS_ERR, CW_CMD_QUERY, c );
			break;
		case CW_CMDV_FREQUENCY:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c,
				 			cw_get_frequency ());
			break;
		case CW_CMDV_SPEED:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c,
				 			cw_get_send_speed ());
			break;
		case CW_CMDV_GAP:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, cw_get_gap ());
			break;
		case CW_CMDV_ECHO:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, do_echo);
			break;
		case CW_CMDV_ERRORS:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, do_errors);
			break;
		case CW_CMDV_COMMANDS:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, do_commands);
			break;
		case CW_CMDV_COMBINATIONS:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, do_combinations);
			break;
		case CW_CMDV_COMMENTS:
			fprintf_stderr
				("%c%c%d", CW_STATUS_OK, c, do_comments);
			break;
	    }
}


/*
 * handle_stream_cwquery()
 *
 * Handle a cwquery received in the input stream.  The command escape
 * character and the cwquery character have already been read and recognized.
 */
static void
handle_stream_cwquery (FILE *stream)
{
	unsigned int	c;			/* Character from stream */

	/* Get the specific query identifier. */
	c = toupper (getc (stream));
	switch (c)
	    {
		case EOF:
			return;
		default:
			fprintf_stderr
				("%c%c%c", CW_STATUS_ERR, CW_CMD_CWQUERY, c);
			break;
		case CW_CMDV_FREQUENCY:
			fprintf_cw
				("%d HZ ", cw_get_frequency ());
			break;
		case CW_CMDV_SPEED:
			fprintf_cw
				("%d WPM ", cw_get_send_speed ());
			break;
		case CW_CMDV_GAP:
			fprintf_cw
				("%d DOTS ", cw_get_gap ());
			break;
		case CW_CMDV_ECHO:
			fprintf_cw
				("ECHO %s ", do_echo ? "ON" : "OFF");
			break;
		case CW_CMDV_ERRORS:
			fprintf_cw
				("ERRORS %s ", do_errors ? "ON" : "OFF");
			break;
		case CW_CMDV_COMMANDS:
			fprintf_cw
				("COMMANDS %s ", do_commands ? "ON" : "OFF");
			break;
		case CW_CMDV_COMBINATIONS:
			fprintf_cw
				("COMBINATIONS %s ",
				 		do_combinations ? "ON" : "OFF");
			break;
		case CW_CMDV_COMMENTS:
			fprintf_cw
				("COMMENTS %s ", do_comments ? "ON" : "OFF");
			break;
	    }
}


/*
 * handle_stream_parameter()
 *
 * Handle a parameter setting command received in the input stream.
 * The command type character has already been read from the stream, and
 * is passed in as an argument.  The new value is also known and passed
 * in, so this routine needs to read nothing from the input stream.
 */
static void
handle_stream_parameter (FILE *stream, unsigned int c, int new_value)
{
	/* Handle the new value according to the command used. */
	switch (c)
	    {
		case EOF:
		default:
			return;

		case CW_CMDV_FREQUENCY:
			if (cw_set_frequency (new_value) != 0)
			    {
				fprintf_stderr ("%c%c", CW_STATUS_ERR, c);
				return;
			    }
			break;
		case CW_CMDV_SPEED:
			if (cw_set_send_speed (new_value) != 0)
			    {
				fprintf_stderr ("%c%c", CW_STATUS_ERR, c);
				return;
			    }
			break;
		case CW_CMDV_GAP:
			if (cw_set_gap (new_value) != 0)
			    {
				fprintf_stderr ("%c%c", CW_STATUS_ERR, c);
				return;
			    }
			break;
		case CW_CMDV_ECHO:
			do_echo		= new_value;
			break;
		case CW_CMDV_ERRORS:
			do_errors	= new_value;
			break;
		case CW_CMDV_COMMANDS:
			do_commands	= new_value;
			break;
		case CW_CMDV_COMBINATIONS:
			do_combinations	= new_value;
			break;
		case CW_CMDV_COMMENTS:
			do_comments	= new_value;
			break;
	    }

	/* Confirm the new value with a stderr message. */
	fprintf_stderr ("%c%c%d", CW_STATUS_OK, c, new_value);
}


/*
 * handle_stream_command()
 *
 * Handle a command received in the input stream.  The command escape
 * character has already been read and recognized.
 */
static void
handle_stream_command (FILE *stream)
{
	unsigned int	c;			/* Character from stream */
	int		new_value;		/* New parameter value */

	/* Get the specific command identifier. */
	c = toupper (getc (stream));
	switch (c)
	    {
		case EOF:
			return;
		default:
			fprintf_stderr
				("%c%c%c", CW_STATUS_ERR, CW_CMD_ESCAPE, c);
			break;

		case CW_CMDV_FREQUENCY:
		case CW_CMDV_SPEED:
		case CW_CMDV_GAP:
		case CW_CMDV_ECHO:
		case CW_CMDV_ERRORS:
		case CW_CMDV_COMMANDS:
		case CW_CMDV_COMBINATIONS:
		case CW_CMDV_COMMENTS:
			if (fscanf (stream, "%d;", &new_value) != 1)
			    {
				fprintf_stderr ("%c%c", CW_STATUS_ERR, c);
				return;
			    }
			handle_stream_parameter (stream, c, new_value);
			break;

		case CW_CMD_QUERY:
			handle_stream_query (stream);
			break;

		case CW_CMD_CWQUERY:
			handle_stream_cwquery (stream);
			break;

		case CW_CMDV_QUIT:
			cw_flush_tone_queue ();
			exit (0);
	    }
}


/*
 * handle_character_send()
 *
 * Sends the given character to the CW sender, and waits for it to complete
 * sounding the tones.  The character to send may be a partial or a complete
 * character.
 */
static void
handle_character_send (unsigned int c, int partial)
{
	int		error;			/* Cw call status code */

	/*
	 * Convert all whitespace into a single space.  Nul is not
	 * whitespace, but strchr 'finds' it, so make it a special case.
	 */
	if (c != ASC_NUL)
	    {
		if (strchr (ASC_WHITESPACE, c) != NULL)
			c = ASC_SPACE;
	    }

	/* Send the character to the cw sender. */
	if (partial)
		error = cw_send_character_partial (c);
	else
		error = cw_send_character (c);
	if (error)
	    {
	       if (errno != ENOENT)
		    {
			perror ("cw_send_character[_partial]");
			cw_flush_tone_queue ();
			exit (1);
		    }
		else
			fprintf_stderr ("%c%c", CW_STATUS_ERR, c);
	    }

	/* Echo the character while sending it. */
	fprintf_stdout ("%c", c);

	/* Wait for the character to complete. */
	if (cw_tone_queue_wait () == -1)
	    {
		perror ("cw_tone_queue_wait");
		cw_flush_tone_queue ();
		exit (1);
	    }
}


/*
 * parse_stream()
 *
 * Read characters from a file stream, and either sound them, or interpret
 * controls in them.  Returns on end of file.
 */
static void
parse_stream (FILE *stream)
{
	unsigned int	c;			/* Character from stream */
	enum {NORMAL,COMBINATION,COMMENT,COMBINATION_COMMENT}
			parse_state = NORMAL;	/* Parsing state */

	/* Read characters from the stream until there's nothing more. */
	while (TRUE)
	    {
		/* Get a character, and check for end of file. */
		c = getc (stream);
		if (c == EOF)
			return;

		/*
		 * Handle the character depending on the current parser
		 * state.
		 */
		switch (parse_state)
		    {
			/*
			 * Monitor for special characters, otherwise, just
			 * send the stuff we see.
			 */
			case NORMAL:
				/* Check for comments and combinations. */
				if (do_comments
						&& c == CW_COMMENT_START)
				    {
					parse_state = COMMENT;
					fprintf_stdout ("%c", c);
					break;
				    }
				if (do_combinations
						&& c == CW_COMBINATION_START)
				    {
					parse_state = COMBINATION;
					fprintf_stdout ("%c", c);
					break;
				    }

				/* Check for embedded command start. */
				if (do_commands
						&& c == CW_CMD_ESCAPE)
				    {
					handle_stream_command (stream);
					break;
				    }

				/* Send the character to the cw sender. */
				handle_character_send (c, FALSE);
				break;

			/*
			 * As above, but do all sending as partial characters.
			 * Look for combination end as a way out.
			 */
			case COMBINATION:
				/* Check for comments and combinations. */
				if (do_comments
						&& c == CW_COMMENT_START)
				    {
					parse_state = COMBINATION_COMMENT;
					fprintf_stdout ("%c", c);
					break;
				    }
				if (c == CW_COMBINATION_END)
				    {
					parse_state = NORMAL;
					fprintf_stdout ("%c", c);
					break;
				    }

				/* Check for embedded command start. */
				if (do_commands
						&& c == CW_CMD_ESCAPE)
				    {
					handle_stream_command (stream);
					break;
				    }

				/* Send the character to the cw sender. */
				handle_character_send (c, TRUE);
				break;

			/*
			 * Ignore everything except comment end.  Revert to
			 * normal state on comment end.
			 */
			case COMMENT:
				if (c == CW_COMMENT_END)
					parse_state = NORMAL;
				fprintf_stdout ("%c", c);
				break;

			/*
			 * Ignore everything except comment end.  Go to the
			 * combination state on comment end.
			 */
			case COMBINATION_COMMENT:
				if (c == CW_COMMENT_END)
					parse_state = COMBINATION;
				fprintf_stdout ("%c", c);
				break;
		    }
	    }
}


/*
 * signal_handler()
 *
 * Signal handler for signals, to stop tone before exit.
 */
static void
signal_handler (int sig)
{
	/* Stop any tone and exit. */
	cw_flush_tone_queue ();
	exit (0);
}


/*
 * main()
 *
 * Parse command line args, then produce cw output until end of file.
 */
int
main (int argc, char **argv)
{
	int			error;		/* Error return status */
	struct sigaction	action;		/* Sigaction structure */

	/* Parse the command line parameters and arguments. */
	parse_cmdline (argc, argv);

	/* Install the handler for signals. */
	action.sa_handler	= signal_handler;
	action.sa_flags		= 0;
	sigemptyset (&action.sa_mask);
	error  =  sigaction (SIGHUP,  &action, NULL);
		+ sigaction (SIGINT,  &action, NULL);
		+ sigaction (SIGQUIT, &action, NULL);
		+ sigaction (SIGPIPE, &action, NULL);
		+ sigaction (SIGTERM, &action, NULL);
	if (error != 0)
	    {
		perror ("sigaction");
		exit (1);
	    }

	/* Read data from stream until end of file. */
	parse_stream (stdin);

	/* All done. */
	return 0;
}
