/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "xsf.h"
#include <visu_tools.h>
#include <renderingMethods/renderingAtomic.h>
#include <renderingMethods/renderingAtomic_ascii.h>
#include <renderingMethods/renderingSpin.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolElements.h>


#define XSF_DESCRIPTION _("<span size=\"smaller\">" \
			  "This plug-in reads <b>XSF</b>"	\
			  "input files.</span>")
#define XSF_AUTHORS     "Caliste Damien"

/* Local methods */
static gboolean loadXsfIn(VisuData *data, const gchar* filename,
			  FileFormat *format _U_, int nSet, GError **error);
static gboolean loadXsfSpin(VisuData *data, const gchar* filename,
			    FileFormat *format _U_, int nSet _U_, GError **error);
static int read_xsf_file(VisuData *data, GIOChannel *flux, int nSet, GError **error);
static RenderingFormatLoad* xsfStructuralInit();
static RenderingFormatLoad* xsfSpinInit();
/* Local variables */
static gchar *iconPath;

/* Required methods for a loadable module. */
gboolean xsfInit()
{
  RenderingFormatLoad* meth, *spin;

  DBG_fprintf(stderr, "XSF: loading plug-in 'xsf'...\n");

  DBG_fprintf(stderr, "XSF: declare a new rendering load method.\n");
  meth = xsfStructuralInit();
  renderingAtomicAdd_loadMethod(meth);

  DBG_fprintf(stderr, "XSF: declare a new spin load method.\n");
  spin = xsfSpinInit();
  rspin_addLoadMethod(spin);

  iconPath = g_build_filename(v_sim_pixmaps_dir, "xsf.png", NULL);

  return TRUE;
}

const char* xsfGet_description()
{
  return XSF_DESCRIPTION;
}

const char* xsfGet_authors()
{
  return XSF_AUTHORS;
}

const char* xsfGet_icon()
{
  return iconPath;
}

static RenderingFormatLoad* xsfStructuralInit()
{
  char *type[] = {"*.xsf", "*.axsf", (char*)0};
  char *descr = _("XCrysDen Structure File format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "XCrysDen Structure File (XSF) format";
  meth->fmt = fileFormatNew(descr, type);
  if (!meth->fmt)
    {
      g_error("Can't initialize the XSF loading method, aborting...\n");
    }
  meth->priority = 90;
  meth->load = loadXsfIn;

  return meth;
}

static gboolean loadXsfIn(VisuData *data, const gchar* filename,
			  FileFormat *format _U_, int nSet, GError **error)
{
  int res;
  GIOChannel *readFrom;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = g_io_channel_new_file(filename, "r", error);
  if (!readFrom)
    return FALSE;

  res = read_xsf_file(data, readFrom, nSet, error);

  if (res < 0)
    /* The file is not a XSF file. */
    return FALSE;
  else if (res > 0)
    /* The file is a XSF file but some errors occured. */
    return TRUE;
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}

struct xsf_reader
{
  GString *line;
  GList   *lst;
  int ntype;
  GHashTable *elements;
  VisuElement **nodeTypes;
  int nNodes;
  float *coords;
  float *forces;
  GIOStatus status;
  GIOChannel *flux;
};

static void reader_free(struct xsf_reader *rd)
{
  GList *tmpLst;

  g_string_free(rd->line, TRUE);
  g_hash_table_destroy(rd->elements);
  if (rd->nodeTypes)
    g_free(rd->nodeTypes);
  if (rd->coords)
    g_free(rd->coords);
  if (rd->forces)
    g_free(rd->forces);
  
  if (rd->lst)
    {
      tmpLst = rd->lst;
      while (tmpLst)
	{
	  g_free(tmpLst->data);
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(rd->lst);
    }  

  if (g_io_channel_shutdown(rd->flux, FALSE, (GError**)0) != G_IO_STATUS_NORMAL)
    g_warning("XSF: can't close file.");
}

static gboolean skip_commentary(struct xsf_reader *rd, GError **error)
{
  gsize term;

  /* Eat blank or commentary lines. */
  do
    {
      rd->status = g_io_channel_read_line_string(rd->flux, rd->line, &term, error);
      if (rd->status != G_IO_STATUS_NORMAL && rd->status != G_IO_STATUS_EOF)
	return FALSE;
      g_strstrip(rd->line->str);
    }
  while (rd->status != G_IO_STATUS_EOF &&
	 (rd->line->str[0] == '#' ||
	  rd->line->str[0] == '!' ||
	  rd->line->str[0] == '\0'));
  return TRUE;
}

static gboolean read_flag(struct xsf_reader *rd, gboolean *found, const gchar *flag,
			  int *value, gboolean  mandatory, GError **error)
{
  size_t len;

  *found = FALSE;
  len = strlen(flag);
  g_strstrip(rd->line->str);
  if (!strncmp(rd->line->str, flag, len))
    {
      if (mandatory && sscanf(rd->line->str + len, "%d", value) != 1 && *value <= 0)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XSF format, '%s' flag has a"
				 " wrong value.\n"), flag);
	  return FALSE;
	}
      else if (!mandatory && sscanf(rd->line->str + len, "%d", value) == 1)
	{
	  if (*value <= 0)
	    {
	      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
				   _("Wrong XSF format, '%s' flag has a"
				     " wrong value.\n"), flag);
	      return FALSE;
	    }
	}
      *found = TRUE;
    }
  if (*found)
    return skip_commentary(rd, error);
  else
    return TRUE;
}

static gboolean read_box(struct xsf_reader *rd, float box[6], GError **error)
{
  int i;
  double cart[3][3];
  
  for (i = 0; i < 3; i++)
    {
      if (sscanf(rd->line->str, "%lf %lf %lf\n",
		 cart[i], cart[i] + 1, cart[i] + 2) != 3)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XSF format, missing float values"
				 " after tag '%s'.\n"), "PRIMVEC");
	  return FALSE;
	}
      if (!skip_commentary(rd, error))
	return FALSE;
    }
  DBG_fprintf(stderr, "XSF: read box %8g %8g %8g\n"
	      "              %8g %8g %8g\n"
	      "              %8g %8g %8g\n",
	      cart[0][0], cart[0][1], cart[0][2],
	      cart[1][0], cart[1][1], cart[1][2],
	      cart[2][0], cart[2][1], cart[2][2]);
  if (!matrix_reducePrimitiveVectors(box, cart))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("Wrong XSF format, primitive vectors are not 3D.\n"));
      return FALSE;
    }
  return TRUE;
}

static gboolean read_coords(struct xsf_reader *rd, GError **error)
{
  int i, res, zele, nb;
  float rcov, fx, fy, fz;
  gchar *name, *ptChar;
  VisuElement *type;
  struct dataAscii *infos;

  if (sscanf(rd->line->str, "%d\n", &rd->nNodes) != 1 && rd->nNodes <= 0)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("Wrong XSF format, missing or wrong integer values"
			     " after tag '%s'.\n"), "PRIMCOORD");
      return FALSE;
    }
  if (!skip_commentary(rd, error))
    return FALSE;
  rd->ntype     = 0;
  rd->nodeTypes = g_malloc(sizeof(VisuElement*) * rd->nNodes);
  rd->coords    = g_malloc(sizeof(float) * 3 * rd->nNodes);
  DBG_fprintf(stderr, "XSF: read coordinates.\n");
  for  (i = 0; i < rd->nNodes; i++)
    {
      name = g_strdup(rd->line->str);
      g_strstrip(name);
      ptChar = strchr(name, ' ');
      if (!ptChar)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XSF format, can't read coordinates.\n"));
	  return FALSE;
	}
      *ptChar = '\0';
      /* The first three values are coordinates.
	 Possible three others are forces. */
      nb = sscanf(ptChar + 1, "%f %f %f %f %f %f\n", rd->coords + 3 * i,
		  rd->coords + 3 * i + 1, rd->coords + 3 * i + 2, &fx, &fy, &fz);
      if (nb != 3 && nb != 6)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XSF format, can't read coordinates.\n"));
	  return FALSE;
	}
      if (nb == 6)
	{
	  /* If forces where not used, we allocate then now. */
	  if (!rd->forces)
	    rd->forces = g_malloc(sizeof(float) * 3 * rd->nNodes);
	  *(rd->forces + 3 * i    ) = fx;
	  *(rd->forces + 3 * i + 1) = fy;
	  *(rd->forces + 3 * i + 2) = fz;
	}
      /* Try to find a z number instead of a name. */
      rcov = -1.f;
      if (sscanf(name, "%d", &zele) == 1 &&
	  toolElementsGet_element(&ptChar, &rcov, zele))
	{
	  g_free(name);
	  name = g_strdup(ptChar);
	}
      /* adding nomloc to the hashtable */
      type = visuElementGet_fromName(name);
      if (!type)
	{
	  type = visuElementNew_withName(name);
	  if (!type)
	    {
	      g_warning("Cannot create a new type for '%s'.", name);
	      return FALSE;
	    }
	  res = visuElementAdd(type);
	  if (res)
	    {
	      g_warning("Cannot add new element, maximum has been reached.");
	      return FALSE;
	    }
	  if (rcov > 0.f)
	    renderingAtomicSet_radius(type, rcov * 0.52917720859);
	}
      rd->nodeTypes[i] = type;
      infos = (struct dataAscii*)g_hash_table_lookup(rd->elements,
						     (gconstpointer)type);
      if (!infos)
	{
	  infos = g_malloc(sizeof(struct dataAscii));
	  infos->ele = type;
	  infos->pos = rd->ntype;
	  infos->nbNodes = 1;
	  g_hash_table_insert(rd->elements, (gpointer)type, (gpointer)infos);
	  rd->ntype += 1;
	}
      else
	infos->nbNodes += 1;
      if (!skip_commentary(rd, error))
	return FALSE;
    }
  DBG_fprintf(stderr, " | read OK.\n");
  return TRUE;
}


static int read_xsf_file(VisuData *data, GIOChannel *flux, int nSet, GError **error)
{
  struct xsf_reader rd;
  gboolean found;
  int valInt;
  float box[6];
  int i;
  int res, nSets, iNodes;
  VisuElement **types;
  unsigned int *nattyp;

  /* We read every line that corresponds to this schema : "%s %f %f %f" */
  DBG_fprintf(stderr, "XSF: reading file as an xsf file.\n");

  rd.flux = flux;

  /* The storage for read line. */
  rd.line = g_string_new("");

  /* Storage of number of elements per types. */
  rd.ntype     = 0;
  rd.elements  = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
  rd.nodeTypes = (VisuElement**)0;

  rd.coords    = (float*)0;
  rd.forces    = (float*)0;
  rd.lst       = (GList*)0;

  /* We read the file completely to find the number of sets of points
     and we store only the one corresponding to @nSet. */
  nSets     = -1;
  if (!skip_commentary(&rd, error))
    {
      reader_free(&rd);
      return -1;
    }
  do
    {
      /* If ANIMSTEPS is found, we are in animated mode. */
      if (!read_flag(&rd, &found, "ANIMSTEPS", &valInt, TRUE, error))
	{
	  reader_free(&rd);
	  return 1;
	}
      if (found)
	{
	  DBG_fprintf(stderr, "XSF: found the 'ANIMSTEPS' flag (%d).\n", valInt);
	  if (nSets > 0)
	    {
	      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
				   _("Wrong XSF format, '%s' tag already"
				     " defined.\n"), "ANIMSTEPS");
	      reader_free(&rd);
	      return 1;
	    }
	  else
	    nSets = valInt;
	}
/*       fprintf(stderr, "'%s'\n", rd.line->str); */

      /* If PRIMVEC is found, we store the box. */
      valInt = -1;
      if (!read_flag(&rd, &found, "PRIMVEC", &valInt, FALSE, error))
	{
	  reader_free(&rd);
	  return 1;
	}
      if (found && (valInt < 0 || (valInt - 1) == nSet))
	{
	  DBG_fprintf(stderr, "XSF: found the 'PRIMVEC' flag (%d).\n", valInt);
	  /* We read the box. */
	  if (!read_box(&rd, box, error))
	    {
	      reader_free(&rd);
	      return 1;
	    }
	  /* We set nSets to 1 if not already set. */
	  if (nSets < 0)
	    nSets = 1;
	}
/*       fprintf(stderr, "'%s'\n", rd.line->str); */

      /* If PRIMCOORD is found, we store the coordinates. */
      valInt = -1;
      if (!read_flag(&rd, &found, "PRIMCOORD", &valInt, FALSE, error))
	{
	  reader_free(&rd);
	  return 1;
	}
      if (found && (valInt < 0 || (valInt - 1) == nSet))
	{
	  DBG_fprintf(stderr, "XSF: found the 'PRIMCOORD' flag (%d).\n", valInt);
	  /* We read the coords. */
	  if (!read_coords(&rd, error))
	    {
	      reader_free(&rd);
	      return 1;
	    }
	  /* We set nSets to 1 if not already set. */
	  if (nSets < 0)
	    nSets = 1;
	}
      if (!skip_commentary(&rd, error))
	{
	  reader_free(&rd);
	  return 1;
	}
/*       fprintf(stderr, "'%s'\n", rd.line->str); */
    }
  while(rd.status != G_IO_STATUS_EOF && rd.coords == (float*)0);
  
  DBG_fprintf(stderr, " | found %d types.\n", rd.ntype);
  if (rd.ntype == 0)
    {
      reader_free(&rd);
      return -1;
    }

  /* Allocate the space for the nodes. */
  types  = g_malloc(sizeof(VisuElement*) * rd.ntype);
  nattyp = g_malloc(sizeof(int) * rd.ntype);
  g_hash_table_foreach(rd.elements, (GHFunc)putValTypeInAsciiType, (gpointer)&types);
  g_hash_table_foreach(rd.elements, (GHFunc)putValNbInAsciiType, (gpointer)&nattyp);

  DBG_fprintf(stderr, " | begin to transfer data to VisuData.\n");
  /* Begin the storage into VisuData. */
  visuDataSet_nSet(data, nSets);

  res = visuDataSet_population(data, rd.ntype, nattyp, types);
  if (!res)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  DBG_fprintf(stderr, "XSF: there are %d types in this file.\n", rd.ntype);
  if (DEBUG)
    for (i = 0; i < rd.ntype; i++)
      fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
  g_free(nattyp);
  g_free(types);
         
  /* Store the coordinates */
  for(iNodes = 0; iNodes < rd.nNodes; iNodes++)
    visuDataAdd_nodeFromElement(data, rd.nodeTypes[iNodes], rd.coords + 3 * iNodes);
  visuDataSet_boxGeometry(data, box, TRUE);

  /* We store the forces as a property to be used later by the spin
     loading method. */
  if (rd.forces)
    {
      visuDataSet_propertyWithDestroyFunc(data, "XSF_forces",
					  (gpointer)rd.forces, g_free);
      /* We nullify the rd.forces pointer to avoid its deletion. */
      rd.forces = (float*)0;
    }

  /* Free the local data. */
  reader_free(&rd);
  
  return 0;
}

static void freeSpin(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * 3, obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopySpin(gconstpointer obj, gpointer data _U_)
{
  float *spinData;

#if GLIB_MINOR_VERSION > 9
  spinData = g_slice_alloc(sizeof(float) * 3);
#else
  spinData = g_malloc(sizeof(float) * 3);
#endif
  if (obj)
    memcpy(spinData, obj, sizeof(float) * 3);
  else
    memset(spinData, 0, sizeof(float) * 3);
    
  return (gpointer)spinData;
}

static RenderingFormatLoad* xsfSpinInit()
{
  char *type[] = {"*.xsf", "*.axsf", (char*)0};
  char *descr = _("XCrysDen Structure File format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "XCrysDen Structure File (XSF) format";
  meth->fmt = fileFormatNew(descr, type);
  if (!meth->fmt)
    {
      g_error("Can't initialize the XSF loading method, aborting...\n");
    }
  meth->priority = 40;
  meth->load = loadXsfSpin;

  return meth;
}

static gboolean loadXsfSpin(VisuData *data, const gchar* filename,
			    FileFormat *format _U_, int nSet _U_, GError **error)
{
  float *forces, *svgMaxSpinModulus, *svgSpinValues;
  float sph[3], vals[3];
  VisuNodeProperty *spin;
  VisuDataIter iter;
  GValue spinValue = {0, {{0}, {0}}};

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  forces = (float*)visuDataGet_property(data, "XSF_forces");
  if (!forces)
    return FALSE;

  /* We check that spin and position are the same. */
  /* TODO... */

  /* Create a storage for max values of spin modulus for each element. */
  svgMaxSpinModulus = g_malloc(sizeof(float) * data->ntype);
  memset(svgMaxSpinModulus, 0, sizeof(float) * data->ntype);
  visuDataSet_propertyWithDestroyFunc(data, SPINMAXMODULUS_ID,
				      (gpointer)svgMaxSpinModulus, g_free);
  spin = visuNodeNew_pointerProperty(visuDataGet_nodeArray(data), SPINVALUES_ID,
				     freeSpin, newOrCopySpin, (gpointer)0);

  g_value_init(&spinValue, G_TYPE_POINTER);
  visuDataIter_new(data, &iter);
  for(visuDataIter_startNumber(data, &iter); iter.node;
      visuDataIter_nextNodeNumber(data, &iter))
    {
      matrix_cartesianToSpherical(sph, forces + iter.node->number * 3);
      vals[SPIN_MODULUS] = sph[0];
      vals[SPIN_THETA]   = sph[1];
      vals[SPIN_PHI]     = sph[2];
      svgSpinValues = newOrCopySpin(vals, (gpointer)0);
      g_value_set_pointer(&spinValue, svgSpinValues);
      visuNodePropertySet_value(spin, iter.node, &spinValue);
      svgMaxSpinModulus[iter.iElement] = MAX(vals[SPIN_MODULUS],
					     svgMaxSpinModulus[iter.iElement]);
    }

  /* We kill the temporary forces property. */
  g_free(g_object_steal_data(G_OBJECT(data), "XSF_forces"));

  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}
