/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "interactive.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include "view.h"
#include "objectList.h"
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_extension.h>
#include <renderingBackend/visu_actionInterface.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <visu_pickMesure.h>
#include <coreTools/toolConfigFile.h>

#include <math.h>

/* To be removed. */
#include "openGLFunctions/text.h"


#define FLAG_PARAMETER_OBSERVE_METHOD "opengl_observe_method"
#define DESC_PARAMETER_OBSERVE_METHOD "Choose the observe method ; integer (0: constrained mode, 1: walker mode)"

struct _VisuInteractive
{
  /* Internal object gestion. */
  GObject parent;
  gboolean dispose_has_run;

  /* Private stored information. */
  GenericRenderingWindow window;
  VisuData   *data;
  PickMesure *pick;

  VisuInteractiveId id;

  /* The save camera positions are implemented as
     a circular list we a pointer on the current head. */
  GList *savedCameras, *lastCamera;
  /* Data needed in case of pick. */
  int xOrig, yOrig;
  int xPrev, yPrev;  
  /* Data for the move action. */
  gint singleNode;
};

enum
  {
    INTERACTIVE_OBSERVE_SIGNAL,
    INTERACTIVE_SELECTION_SIGNAL,
    INTERACTIVE_MOVE_SIGNAL,
    INTERACTIVE_STOP_SIGNAL,
/*     INTERACTIVE_EVENT_SIGNAL, */
    N_INTERACTIVE_SIGNALS
  };

struct _VisuInteractiveClass
{
  GObjectClass parent;

  VisuInteractiveMethod preferedObserveMethod;

  OpenGLExtension *moveAtomExtension;
  int moveAtomExtension_list;
};
static VisuInteractiveClass *local_class = NULL;

/* Internal variables. */
static guint interactive_signals[N_INTERACTIVE_SIGNALS] = { 0 };

/* Object gestion methods. */
static void visuInteractive_dispose (GObject* obj);
static void visuInteractive_finalize(GObject* obj);

/* When an atom is moved (see move()), it is associated
   to a specific extension list to avoid to recreate
   the complete allElement list. */
#define  moveAtomName        "MovedANode"
#define  moveAtomNameI18n    _("Moved a node")
#define  moveAtomDescription _("Draw the node that is displaced.")

/* Local methods. */
static void exportParameters(GString *data, VisuData *dataObj);
static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj, GError **error);
static gboolean observe(VisuInteractive *inter, SimplifiedEvents *ev);
static gboolean pick(VisuInteractive *inter, SimplifiedEvents *ev);
static gboolean move(VisuInteractive *inter, SimplifiedEvents *ev);
static gboolean mark(VisuInteractive *inter, SimplifiedEvents *ev);
static gboolean pickAndObserve(VisuInteractive *inter, SimplifiedEvents *ev);
static int getSelectElement(VisuData *data, int x, int y);
static GList* getSelectElementsRegion(VisuData *data, int x1, int y1,
				      int x2, int y2);
static void freeCameras(VisuInteractive *inter);

G_DEFINE_TYPE(VisuInteractive, visuInteractive, G_TYPE_OBJECT)

static void visuInteractive_class_init(VisuInteractiveClass *klass)
{
  GType paramPointer[1] = {G_TYPE_POINTER};
  GType paramBool[1] = {G_TYPE_BOOLEAN};
  VisuConfigFileEntry *resourceEntry;

  local_class = klass;

  DBG_fprintf(stderr, "visu Interactive: creating the class of the object.\n");

  DBG_fprintf(stderr, "                - adding new signals ;\n");
  /**
   * Plane::moved:
   * @plane: the object emitting the signal.
   *
   * This signal is emitted each time the plane position is changed
   * (either distance or normal).
   *
   * Since: 3.3
   */
  interactive_signals[INTERACTIVE_OBSERVE_SIGNAL] =
    g_signal_newv("observe", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__BOOLEAN,
		  G_TYPE_NONE, 1, paramBool);

  interactive_signals[INTERACTIVE_SELECTION_SIGNAL] =
    g_signal_newv("selection", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  interactive_signals[INTERACTIVE_MOVE_SIGNAL] =
    g_signal_newv("move", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  interactive_signals[INTERACTIVE_STOP_SIGNAL] =
    g_signal_newv("stop", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose  = visuInteractive_dispose;
  G_OBJECT_CLASS(klass)->finalize = visuInteractive_finalize;

  DBG_fprintf(stderr, "                - add the resources.\n");

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OBSERVE_METHOD,
					  DESC_PARAMETER_OBSERVE_METHOD,
					  1, readOpenGLObserveMethod);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);

  /* Set local variables to default. */
  klass->preferedObserveMethod = interactive_constrained;
  klass->moveAtomExtension = (OpenGLExtension*)0;
}

static void visuInteractive_init(VisuInteractive *obj)
{
  obj->dispose_has_run = FALSE;

  DBG_fprintf(stderr, "Visu Interactive: creating a new interactive session (%p).\n",
	      (gpointer)obj);
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visuInteractive_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Visu Interactive: dispose object %p.\n", (gpointer)obj);

  if (VISU_INTERACTIVE(obj)->dispose_has_run)
    return;

  VISU_INTERACTIVE(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(visuInteractive_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visuInteractive_finalize(GObject* obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Visu Interactive: finalize object %p.\n", (gpointer)obj);

  freeCameras(VISU_INTERACTIVE(obj));

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visuInteractive_parent_class)->finalize(obj);
}

VisuInteractive* visuInteractiveNew(GenericRenderingWindow window)
{
  VisuInteractive *inter;

  g_return_val_if_fail(window, (VisuInteractive*)0);

  inter = VISU_INTERACTIVE(g_object_new(VISU_INTERACTIVE_TYPE, NULL));
  g_return_val_if_fail(inter, (VisuInteractive*)0);

  DBG_fprintf(stderr, "Interactive: start new interactive session %p.\n",
	      (gpointer)inter);

  inter->window = window;

  return inter;
}

OpenGLCamera* visuInteractivePop_savedCamera(VisuInteractive *inter)
{
  OpenGLCamera *cur;

  g_return_val_if_fail(IS_VISU_INTERACTIVE(inter), (OpenGLCamera*)0);

  if (!inter->lastCamera)
    return (OpenGLCamera*)0;

  cur = (OpenGLCamera*)inter->lastCamera->data;

  inter->lastCamera = g_list_next(inter->lastCamera);
  if (!inter->lastCamera)
    inter->lastCamera = inter->savedCameras;

  DBG_fprintf(stderr, "Interactive: pop, pointing now at camera %d.\n",
	      g_list_position(inter->savedCameras, inter->lastCamera));

  return cur;
}
void visuInteractiveGet_savedCameras(VisuInteractive *inter,
				     GList **cameras, GList **head)
{
  g_return_if_fail(IS_VISU_INTERACTIVE(inter) && cameras && head);
  
  *cameras = inter->savedCameras;
  *head    = inter->lastCamera;
}
static void freeCameras(VisuInteractive *inter)
{
  /* We delete all the saved cameras. */
  for (inter->lastCamera = inter->savedCameras;
       inter->lastCamera;
       inter->lastCamera = g_list_next(inter->lastCamera))
    g_free(inter->lastCamera->data);
  if (inter->savedCameras)
    g_list_free(inter->savedCameras);
  inter->lastCamera   = (GList*)0;
  inter->savedCameras = (GList*)0;
}
static gboolean cmpCameras(OpenGLCamera *c1, OpenGLCamera *c2)
{
  return (c1 == c2) ||
    (c1->theta == c2->theta &&
     c1->phi == c2->phi &&
     c1->omega == c2->omega &&
     c1->xs == c2->xs &&
     c1->ys == c2->ys &&
     c1->gross == c2->gross &&
     c1->d_red == c2->d_red);
}
void visuInteractivePush_savedCamera(VisuInteractive *inter, OpenGLCamera *camera)
{
  OpenGLCamera *tmp;

  g_return_if_fail(IS_VISU_INTERACTIVE(inter) && camera);

  for (inter->lastCamera = inter->savedCameras ;
       inter->lastCamera &&
	 !cmpCameras((OpenGLCamera*)inter->lastCamera->data, camera);
       inter->lastCamera = g_list_next(inter->lastCamera));

  /* Case we don't find it, we add. */
  if (!inter->lastCamera || (OpenGLCamera*)inter->lastCamera->data != camera)
    {
      tmp = g_malloc(sizeof(OpenGLCamera));
      tmp->theta = camera->theta;
      tmp->phi   = camera->phi;
      tmp->omega = camera->omega;
      tmp->xs    = camera->xs;
      tmp->ys    = camera->ys;
      tmp->gross = camera->gross;
      tmp->d_red = camera->d_red;
      inter->savedCameras = g_list_prepend(inter->savedCameras, (gpointer)tmp);
    }
  inter->lastCamera = inter->savedCameras;
  DBG_fprintf(stderr, "Interactive: push, storing now %d cameras.\n",
	      g_list_length(inter->savedCameras));
}


void visuInteractiveSet_type(VisuInteractive *inter, VisuInteractiveId type)
{
  VisuInteractiveClass *klass;

  g_return_if_fail(IS_VISU_INTERACTIVE(inter));
  klass = VISU_INTERACTIVE_GET_CLASS(inter);

  DBG_fprintf(stderr, "Interactive: set session type to %d (%d).\n", type, inter->id);

  if (inter->id == type)
    return;
  inter->id = type;
  
  switch (type)
    {
    case interactive_observe:
    case interactive_pickAndObserve:
      setObserveEventListener(inter->window);
      break;
    case interactive_pick:
    case interactive_move:
    case interactive_mark:
      setPickEventListener(inter->window);
      break;
    case interactive_none:
      removeEventListener(inter->window);
      break;
    }
  if (type == interactive_move && !klass->moveAtomExtension)
    {
      klass->moveAtomExtension_list = openGLObjectList_new(1);
      klass->moveAtomExtension = OpenGLExtension_new(moveAtomName, moveAtomNameI18n,
						     moveAtomDescription,
						     klass->moveAtomExtension_list,
						     NULL);
      OpenGLExtensionSet_priority(klass->moveAtomExtension,
				  OPENGL_EXTENSION_PRIORITY_FIRST + 1);
      OpenGLExtensionRegister(klass->moveAtomExtension);
    }
}

void visuInteractiveHandle_event(VisuInteractive *inter, SimplifiedEvents *ev)
{
  gboolean stop;

  g_return_if_fail(IS_VISU_INTERACTIVE(inter));

  switch (inter->id)
    {
    case interactive_observe:
      stop = observe(inter, ev);
      break;
    case interactive_pickAndObserve:
      stop = pickAndObserve(inter, ev);
      break;
    case interactive_pick:
      stop = pick(inter, ev);
      break;
    case interactive_move:
      stop = move(inter, ev);
      break;
    case interactive_mark:
      stop = mark(inter, ev);
      break;
    default:
      stop = FALSE;
      break;
    }

  if (stop)
    g_signal_emit(G_OBJECT(inter),
		  interactive_signals[INTERACTIVE_STOP_SIGNAL], 0, NULL);
}

PickMesure* visuInteractiveGet_pickMesure(VisuInteractive *inter)
{
  g_return_val_if_fail(IS_VISU_INTERACTIVE(inter), (PickMesure*)0);

  return inter->pick;
}

void visuInteractiveSet_visuData(VisuInteractive *inter, VisuData *data)
{
/*   OpenGLView *view; */

  g_return_if_fail(IS_VISU_INTERACTIVE(inter));

  inter->data = data;
  if (data)
    inter->pick = g_object_get_data(G_OBJECT(inter->data), "pickMesure_data");
  else
    inter->pick = (PickMesure*)0;

/*   freeCameras(inter); */

/*   if (data) */
/*     { */
/*       view = visuDataGet_openGLView(data); */
/*       inter->savedCameras      = (GList*)0; */
/*       visuInteractivePush_savedCamera(inter, view->camera); */
/*       inter->lastCamera        = inter->savedCameras; */
/*     } */
}

/******************************************************************************/

static gboolean pickAndObserve(VisuInteractive *inter, SimplifiedEvents *ev)
{
  /* If button 1 or 2, use observe mode */
  if (ev->button != 3)
    return observe(inter, ev);
  /* If button 3, use pick mode with button 3 as button 1 */
  else
    {
      if (ev->shiftMod && !ev->controlMod)
	{
	  ev->button = 2;
	}
      else if (!ev->shiftMod && ev->controlMod)
	{
	  ev->button = 2;
	}
      else if (ev->shiftMod && ev->controlMod)
	{
	  ev->shiftMod = 0;
	  ev->controlMod = 1;
	  ev->button = 1;
	}
      else
	ev->button = 1;
      ev->motion = 0;
      
      return pick(inter, ev);
    }
}

static gboolean observe(VisuInteractive *inter, SimplifiedEvents *ev)
{
  int reDrawNeeded;
  int sign_theta;
  int dx, dy;
  OpenGLView *view;
  OpenGLCamera *camera;
  float angles[3], ratio;
  gboolean zoom;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3)
    return (ev->buttonType == BUTTON_TYPE_PRESS);
  /* If the realese event is triggered, we exit only if it's
     not a scroll event (button 4 and 5). */
  if (ev->buttonType == BUTTON_TYPE_RELEASE)
    if (ev->button != 4 && ev->button != 5 )
      {
	if (ev->button == 1)
	  g_signal_emit(G_OBJECT(inter),
			interactive_signals[INTERACTIVE_OBSERVE_SIGNAL],
			0 , FALSE, NULL);
	return FALSE;
      }

  view = visuDataGet_openGLView(inter->data);
  g_return_val_if_fail(view, FALSE);

/*   fprintf(stderr, "%d %d, %d\n", ev->x, ev->y, ev->button); */
/*   fprintf(stderr, "%d %d, %d\n", ev->shiftMod, ev->controlMod, ev->motion); */
  reDrawNeeded = 0;
  /* Support de la roulette en zoom et perspective. */
  if (ev->specialKey == Key_Page_Up || ev->specialKey == Key_Page_Down ||
      ev->button == 4 || ev->button == 5)
    {
      if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross * 1.1);
      else if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red * 1.1);
    }
  else if (ev->button && !ev->motion)
    {
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
      if (ev->button == 1)
	g_signal_emit(G_OBJECT(inter),
		      interactive_signals[INTERACTIVE_OBSERVE_SIGNAL],
		      0 , TRUE, NULL);
    }
  else if (ev->motion ||
	   ev->specialKey == Key_Arrow_Down  || ev->specialKey == Key_Arrow_Up ||
	   ev->specialKey == Key_Arrow_Right || ev->specialKey == Key_Arrow_Left)
    {
      if (ev->motion)
	{
	  dx =   ev->x - inter->xOrig;
	  dy = -(ev->y - inter->yOrig);
	}
      else
	{
	  dx = 0;
	  dy = 0;
	  if (ev->specialKey == Key_Arrow_Left)
	    dx = -10;
	  else if (ev->specialKey == Key_Arrow_Right)
	    dx = +10;
	  else if (ev->specialKey == Key_Arrow_Down)
	    dy = -10;
	  else if (ev->specialKey == Key_Arrow_Up)
	    dy = +10;
	}
      zoom = (ev->button == 2);
      if(!zoom && !ev->shiftMod && !ev->controlMod)
	{
	  if (local_class->preferedObserveMethod == interactive_constrained)
	    {
	      if(view->camera->theta > 0.0)
		sign_theta = 1;
	      else
		sign_theta = -1;
	      openGLViewRotate_box(view, dy * 180.0f / view->window->height,
				   -dx * 180.0f / view->window->width * sign_theta, angles);
	      reDrawNeeded = visuDataSet_angleOfView(inter->data, angles[0], angles[1],
						     0., MASK_THETA | MASK_PHI);
	    }
	  else if (local_class->preferedObserveMethod == interactive_walker)
	    {
	      openGLViewRotate_camera(view, dy * 180.0f / view->window->height,
				      -dx * 180.0f / view->window->width, angles);
	      reDrawNeeded = visuDataSet_angleOfView(inter->data, angles[0], angles[1], angles[2],
						     MASK_THETA | MASK_PHI | MASK_OMEGA);
	    }
	}
      else if(!zoom && ev->shiftMod && !ev->controlMod)
	{
	  ratio = 1. / MIN(view->window->width, view->window->height) /
	    view->camera->gross * (view->camera->d_red - 1.f) / view->camera->d_red;
	  reDrawNeeded =
	    visuDataSet_positionOfView(inter->data,
				       view->camera->xs + (float)dx * ratio,
				       view->camera->ys + (float)dy * ratio,
				       MASK_XS | MASK_YS);
	}
      else if(!zoom && ev->controlMod && !ev->shiftMod)
	{
	  if (abs(dx) > abs(dy))
	    reDrawNeeded = visuDataSet_angleOfView(inter->data, 0., 0.,
						   view->camera->omega + dx *
						   180.0f / view->window->width,
						   MASK_OMEGA);
	  else
	    reDrawNeeded = visuDataSet_angleOfView(inter->data, 0., 0.,
						   view->camera->omega + dy *
						   180.0f / view->window->height,
						   MASK_OMEGA);
	}
      else if(zoom && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross *
					      (1. + (float)dy *
					       3.0f / view->window->height));
      else if(zoom && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red *
						     (1. - (float)dy *
						      5.0f / view->window->height));
      if (ev->motion)
	{
	  inter->xOrig = ev->x;
	  inter->yOrig = ev->y;
	}
    }
  else if (ev->letter == 'r')
    {
      camera = visuInteractivePop_savedCamera(inter);
      if (camera)
	{
	  reDrawNeeded = visuDataSet_angleOfView(inter->data,
						 camera->theta,
						 camera->phi,
						 camera->omega,
						 MASK_THETA |
						 MASK_PHI | MASK_OMEGA) ||
	    reDrawNeeded;
	  reDrawNeeded = visuDataSet_positionOfView(inter->data,
						    camera->xs,
						    camera->ys,
						    MASK_XS | MASK_YS) ||
	    reDrawNeeded;
	  reDrawNeeded = visuDataSet_zoomOfView(inter->data,
						camera->gross) ||
	    reDrawNeeded;
	  reDrawNeeded = visuDataSet_perspectiveOfView(inter->data,
						       camera->d_red) ||
	    reDrawNeeded;
	}
    }
  else if (ev->letter == 's')
    visuInteractivePush_savedCamera(inter, view->camera);
  if (reDrawNeeded)
    {
      DBG_fprintf(stderr, "Visu Interactive: redraw on observe.\n");
      g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		      GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
    }
  return FALSE;
}
static void glDrawSelection(VisuInteractive *inter, int x, int y)
{
  int viewport[4];

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glGetIntegerv(GL_VIEWPORT, viewport);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, (float)viewport[2], 0., (float)viewport[3]);   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.f, 1.f, 1.f, 1.f);
  glLineWidth(2);
  glDrawBuffer(GL_FRONT);
  /* We erase the previous selecting rectangle. */
  glBegin(GL_LINE_LOOP);
  glVertex3f(inter->xOrig, (float)viewport[3] - inter->yOrig, 0.f);
  glVertex3f(inter->xPrev, (float)viewport[3] - inter->yOrig, 0.f);
  glVertex3f(inter->xPrev, (float)viewport[3] - inter->yPrev, 0.f);
  glVertex3f(inter->xOrig, (float)viewport[3] - inter->yPrev, 0.f);
  glEnd();
  glFlush();

  /* We draw the new selecting rectangle. */
  if (x > 0 && y > 0)
    {
      glBegin(GL_LINE_LOOP);
      glVertex3f(inter->xOrig, (float)viewport[3] - inter->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - inter->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - y, 0.f);
      glVertex3f(inter->xOrig, (float)viewport[3] - y, 0.f);
      glEnd();
      glFlush();
    }
  glDrawBuffer(GL_BACK);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopAttrib();
}

static gboolean pick(VisuInteractive *inter, SimplifiedEvents *ev)
{
  int nodeId;
  GList *nodes;

  g_return_val_if_fail(ev && inter, TRUE);
  /* We store the pickInfo into the VisuData on press,
     but we apply it only at release if there was no drag action. */
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }

  g_return_val_if_fail(inter->data, TRUE);

  if (ev->button == 1 && ev->motion)
    {
      /* We drag a selecting area. */
      /* A nodeInfo should have been stored, we retrieve it. */
      DBG_fprintf(stderr, "Interactive: pick, drag to %dx%d.\n", ev->x, ev->y);
      glDrawSelection(inter, ev->x, ev->y);
      inter->xPrev = ev->x;
      inter->yPrev = ev->y;
    }
  else if (ev->buttonType == BUTTON_TYPE_PRESS)
    {
      /* We create and store a nodeInfo data. */
      DBG_fprintf(stderr, "Interactive: pick, press at %dx%d.\n", ev->x, ev->y);
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
      inter->xPrev = ev->x;
      inter->yPrev = ev->y;
    }
  else if (ev->buttonType == BUTTON_TYPE_RELEASE)
    {
      /* If no drag action, we select the node and compute the pick mesure. */
      if (inter->xOrig == inter->xPrev && inter->yOrig == inter->yPrev)
	{
	  nodeId = getSelectElement(inter->data, ev->x, ev->y);

	  DBG_fprintf(stderr, "Interactive: set selection (single %d %d).\n",
		      ev->shiftMod, ev->controlMod);
	  if (ev->button == 1 && !ev->controlMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_SELECTED);
	  else if (ev->button == 1 && ev->controlMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_HIGHLIGHT);
	  else if (ev->button == 2 && ev->shiftMod && !ev->controlMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_REFERENCE_1);
	  else if (ev->button == 2 && !ev->shiftMod && ev->controlMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_REFERENCE_2);
	  else if (ev->button == 2 && !ev->shiftMod && !ev->controlMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_INFORMATION);
	  else
	    return FALSE;
	  DBG_fprintf(stderr, " | OK.\n");
	}
      /* If drag action, we compute all the nodes in the rectangle
	 and we erase it. */
      else
	{
	  /* We unset the pickMesure. */
/* 	  mesureData = pickMesureSet_selection(nodeInfo); */
	  /* We get the list of selected nodes. */
	  glDrawSelection(inter, -1, -1);
	  /* Get all nodes in the region. */
	  nodes = getSelectElementsRegion(inter->data,
					  inter->xOrig, inter->yOrig,
					  ev->x, ev->y);
	  DBG_fprintf(stderr, "Interactive: set selection (drag).\n");
	  pickMesureSet_pickRegion(inter->pick, nodes);
	  g_list_free(nodes);
	  DBG_fprintf(stderr, " | OK.\n");
	}
      DBG_fprintf(stderr, "Interactive: call callback routine on pick.\n");
      g_signal_emit(G_OBJECT(inter),
		    interactive_signals[INTERACTIVE_SELECTION_SIGNAL],
		    0 , inter->pick, NULL);
    }
  return FALSE;
}

static gboolean move(VisuInteractive *inter, SimplifiedEvents *ev)
{
  int dx, dy;
  float ratio, delta[3], z, xyz[3];
  OpenGLView *view;
  VisuNode *node;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }
  if (ev->button != 1 &&
      ev->specialKey != Key_Arrow_Left && ev->specialKey != Key_Arrow_Right &&
      ev->specialKey != Key_Arrow_Up && ev->specialKey != Key_Arrow_Down)
    return FALSE;

  g_return_val_if_fail(inter->data, TRUE);

  if (ev->motion == 1)
    {
      DBG_fprintf(stderr, "Interactive: drag action (%dx%d).\n", ev->x, ev->y);
      dx =   ev->x - inter->xOrig;
      dy = -(ev->y - inter->yOrig);
      DBG_fprintf(stderr, " | dx x dy : %d x %d\n", dx, dy);

      /* Get the camera orientation. */
      view = visuDataGet_openGLView(inter->data);      
      if (!ev->shiftMod && !ev->controlMod)
	{
	  node = pickMesureGet_selectedNode(inter->pick);
	  if (node && inter->singleNode == (gint)node->number)
	    {
	      visuDataGet_nodePosition(inter->data, node, xyz);
	      z = openGLViewGet_zCoordinate(view, xyz);
	    }
	  else
	    {
	      if (node)
		{
		  visuDataGet_nodePosition(inter->data, node, xyz);
		  z = openGLViewGet_zCoordinate(view, xyz);
		}
	      else
		z = 0.5f;
	      openGLViewGet_realCoordinates(view, xyz,
					    (float)inter->xOrig,
					    (float)inter->yOrig, z, TRUE);
	    }
	  openGLViewGet_realCoordinates(view, delta, (float)ev->x,
					(float)ev->y, z, TRUE);
	  delta[0] -= xyz[0];
	  delta[1] -= xyz[1];
	  delta[2] -= xyz[2];
	}
      else
	{
	  ratio = OpenGLViewGet_fileUnitPerPixel(view);
	  if (ev->shiftMod)
	    {
	      delta[0] = ratio * dx;
	      delta[1] = ratio * dy;
	      delta[2] = 0.f;
	    }
	  else if (ev->controlMod)
	    {
	      delta[0] = 0.f;
	      delta[1] = 0.f;
	      delta[2] = ratio * dy;
	    }
	}
      pickMesureSet_dragMove(inter->pick, delta[0], delta[1], delta[2]);

      /* Update stored position for drag info. */
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
    }
/*   else if (ev->specialKey != Key_None) */
/*     { */
/*       DBG_fprintf(stderr, "Interactive: keyboard drag action.\n"); */
      
/*       dx = dy = 0; */
/*       switch (ev->specialKey) */
/* 	{ */
/* 	case Key_Arrow_Left: */
/* 	  dx = -1; dy =  0; break; */
/* 	case Key_Arrow_Right: */
/* 	  dx = +1; dy =  0; break; */
/* 	case Key_Arrow_Up: */
/* 	  dx =  0; dy = +1; break; */
/* 	case Key_Arrow_Down: */
/* 	  dx =  0; dy = -1; break; */
/* 	default: */
/* 	  g_error("Key not handled for drag."); */
/* 	} */

/*       view = visuDataGet_openGLView(inter->data); */
/*       for (i = 0; i < 3; i++) */
/* 	{ */
/* 	  xAxis[i] = 0.; */
/* 	  yAxis[i] = 0.; */
/* 	  zAxis[i] = 0.; */
/* 	} */
/*       if (!ev->shiftMod && !ev->controlMod) */
/* 	OpenGLViewGet_screenAxes(view, xAxis, yAxis); */
/*       else */
/* 	{ */
/* 	  if (ev->shiftMod) */
/* 	    { */
/* 	      xAxis[0] = 1.; */
/* 	      yAxis[1] = 1.; */
/* 	    } */
/* 	  else if (ev->controlMod) */
/* 	    zAxis[2] = 1.; */
/* 	} */
/*       ratio = OpenGLViewGet_fileUnitPerPixel(view); */
/*       pickMesureSet_dragMove(inter->pick, */
/* 			     ratio * (dx * xAxis[0] + dy * yAxis[0] + dy * zAxis[0]), */
/* 			     ratio * (dx * xAxis[1] + dy * yAxis[1] + dy * zAxis[1]), */
/* 			     ratio * (dx * xAxis[2] + dy * yAxis[2] + dy * zAxis[2])); */
/*     } */
  else if (ev->button == 1 && ev->buttonType == BUTTON_TYPE_PRESS)
    {
      pickMesureSet_dragStart(inter->pick,
			      getSelectElement(inter->data, ev->x, ev->y));

      /* Store the position to find the drag values. */
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
    }
  else if (ev->button == 1 && ev->buttonType == BUTTON_TYPE_RELEASE)
    pickMesureSet_dragStop(inter->pick);

  DBG_fprintf(stderr, "Visu Interactive: emit the 'move' signal.\n");
  g_signal_emit(G_OBJECT(inter),
		interactive_signals[INTERACTIVE_MOVE_SIGNAL],
		0 , inter->pick, NULL);

  return FALSE;
}
void visuInteractiveStart_move(VisuInteractive *inter, GList *nodes)
{
  GList *tmpLst;

  DBG_fprintf(stderr, "Interactive: start to drag a list of %d nodes.\n",
	      g_list_length(nodes));
  /* Hide all nodes. */
  tmpLst = nodes;
  while (tmpLst)
    {
      visuNodeSet_visibility((VisuNode*)tmpLst->data, FALSE);
      tmpLst = g_list_next(tmpLst);
    }
  /* If we have only one element, we recreate the list of only this
     element, otherwise, we recreate all the elements. */
  if (nodes && !nodes->next)
    visuData_createNodes(inter->data, inter->data->fromIntToVisuElement[((VisuNode*)nodes->data)->posElement]);
  else
    visuData_createAllNodes(inter->data);
  inter->singleNode = (nodes && !nodes->next)?
    (gint)((VisuNode*)nodes->data)->number:-1;

  /* Show again the selected nodes and add it in
     the specific moveAtom list. */
  glNewList(local_class->moveAtomExtension_list, GL_COMPILE);
  tmpLst = nodes;
  while (tmpLst)
    {
      visuNodeSet_visibility((VisuNode*)tmpLst->data, TRUE);
      visuData_createNode(inter->data, (VisuNode*)tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  glEndList();
  local_class->moveAtomExtension->used = 1;

  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
}
void visuInteractiveStop_move(VisuInteractive *inter, GList *nodes)
{
  DBG_fprintf(stderr, "Interactive: stop dragging a list of %d nodes.\n",
	      g_list_length(nodes));
  /* Show again the selected element in the allElement list. */
  if (nodes && !nodes->next)
    visuData_createNodes(inter->data, inter->data->fromIntToVisuElement[((VisuNode*)nodes->data)->posElement]);
  else
    visuData_createAllNodes(inter->data);

  /* Stop the move extension. */
  local_class->moveAtomExtension->used = 0;

  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
  DBG_fprintf(stderr, "Restore standard allElement after move.\n");
}
void visuInteractiveMove(VisuInteractive *inter, GList *nodes, float drag[3])
{
  OpenGLView *view;
  VisuNode *node;

  DBG_fprintf(stderr, "Interactive: drag a list of %d nodes of %gx%gx%g.\n",
	      g_list_length(nodes), drag[0], drag[1], drag[2]);
  view = visuDataGet_openGLView(inter->data);
  glNewList(local_class->moveAtomExtension_list, GL_COMPILE);
  while (nodes)
    {
      node = (VisuNode*)nodes->data;
      node->xyz[0] += drag[0];
      node->xyz[1] += drag[1];
      node->xyz[2] += drag[2];
      visuData_createNode(inter->data, node);
      nodes = g_list_next(nodes);
    }
  glEndList();
  visuDataEmit_nodePositionChanged(inter->data);
	  
  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
}

static gboolean mark(VisuInteractive *inter, SimplifiedEvents *ev)
{
  int nodeId;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3 && ev->buttonType == BUTTON_TYPE_PRESS)
    return TRUE;
  if (ev->buttonType == BUTTON_TYPE_RELEASE)
    return FALSE;

  g_return_val_if_fail(inter->data, TRUE);

  nodeId = getSelectElement(inter->data, ev->x, ev->y);
  if (nodeId < 0)
    return FALSE;

  pickMesureSet_pickNode(inter->pick, nodeId, PICK_HIGHLIGHT);

  DBG_fprintf(stderr, "Interactive: call callback routine on mark.\n");
  g_signal_emit(G_OBJECT(inter),
		interactive_signals[INTERACTIVE_SELECTION_SIGNAL],
		0 , inter->pick, NULL);

  return FALSE;
}

static GList* getSelectElementsRegion(VisuData *dataObj, int x1, int y1,
				      int x2, int y2)
{
   GLsizei bufsize;
   GLuint *select_buf;
   GLint viewport[4] = {0, 0, 0, 0};
   int hits, names, ptr, i;
   OpenGLView *view;
   GList *lst;
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), (GList*)0);
   
   DBG_fprintf(stderr, "Interactive: get elements in region %dx%d - %dx%d.\n",
	       x1, x2, y1, y2);

   if ((x1 == x2) || (y1 == y2))
     return (GList*)0;

   view = visuDataGet_openGLView(dataObj);

   bufsize = visuDataGet_nodeArray(dataObj)->nbOfAllStoredNodes * 4;
   select_buf = g_malloc(sizeof(GLuint) * bufsize);
   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
   
   lst = (GList*)0;
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;

   glNewList(10, GL_COMPILE);
   gluPickMatrix(0.5f * (x1 + x2) , (float)view->window->height - 0.5f * (y1 + y2),
		 (float)ABS(x2 - x1), (float)ABS(y2 - y1), viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 
   glPushMatrix();
   glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

   glCallList(visuDataGet_objectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "%d elements are on the z buffer %dx%d - %dx%d.\n", hits,
	       x1, y1, x2, y2);
   ptr = 0;

   /* return the buffer to normal */
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   for(i=0; i<hits; i++)
     {
       names = select_buf[ptr];
       if (names != 1)
	 {
	   g_warning("OpenGL picking is not working???\n");
	   return (GList*)0;
	 }
       ptr = ptr + 3;
       lst = g_list_prepend(lst, GINT_TO_POINTER((int)select_buf[ptr]));
       ptr = ptr + 1;
     }
   g_free(select_buf);

   return lst;
}

static int getSelectElement(VisuData *dataObj, int x, int y)
{
#define bufsize 512
   GLuint select_buf[bufsize];
   GLint viewport[4] = {0, 0, 0, 0};
#define wpck 2.0
#define hpck 2.0
   int hits, names, ptr, i;
   unsigned int z1;
   unsigned int z1_sauve = UINT_MAX;
   int number;
   int found;
   OpenGLView *view;
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0);

   view = visuDataGet_openGLView(dataObj);

   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
      
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;
   glNewList(10, GL_COMPILE);
      gluPickMatrix(1.0*x, 1.0*(view->window->height-y), wpck, hpck, viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   DBG_fprintf(stderr, "Interactive: frustum %f %f.\n",
	       view->window->near, view->window->far);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 
   glPushMatrix();
   glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
   DBG_fprintf(stderr, "Interactive: translated %f %f %f.\n",
	       -view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

   glCallList(visuDataGet_objectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "Interactive: %d elements are on the z buffer.\n", hits);
   ptr = 0;

   /* return the buffer to normal */
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   found = 0;
   number = -1;
   for(i=0; i<hits; i++) {
      names = select_buf[ptr];
      if (names != 1)
	{
	  g_warning("OpenGL picking is not working???\n");
	  return -1;
	}
      ptr = ptr + 1;
      z1 = select_buf[ptr];
      DBG_fprintf(stderr, " | z position %f for %d\n", (float)z1/0x7fffffff,
		  (int)select_buf[ptr + 2]);
      ptr = ptr + 2;
      if (z1 < z1_sauve) {
         z1_sauve = z1;
	 number = (int)select_buf[ptr];
	 found = 1;
      }
      ptr = ptr + 1;
   }
   if (found && number >= 0)
     return number;
   else
     return -1;
}

static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj _U_, GError **error)
{
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_integer(lines[0], position, &val, 1, error))
    return FALSE;
  if (val != interactive_constrained && val != interactive_walker)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, 500);
      return FALSE;
    }
  visuInteractiveClassSet_preferedObserveMethod(val);

  return TRUE;
}
static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OBSERVE_METHOD);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_OBSERVE_METHOD,
			 visuInteractiveClassGet_preferedObserveMethod());
}
void visuInteractiveClassSet_preferedObserveMethod(VisuInteractiveMethod method)
{
  g_return_if_fail(method == interactive_constrained ||
		   method == interactive_walker);

  if (!local_class)
    visuInteractive_get_type();

  local_class->preferedObserveMethod = method;
}
VisuInteractiveMethod visuInteractiveClassGet_preferedObserveMethod()
{
  if (!local_class)
    visuInteractive_get_type();

  return local_class->preferedObserveMethod;
}

/* From an array of nodes ids, we specify their 2D coordinates */
void getNodes2DCoordinates(VisuData *dataObj, unsigned int *nodeIds,
    	unsigned int nNodes, GLfloat *coordinates2D, unsigned int *size)
{
  int i,j;
  unsigned int k;
  VisuNode *node;
  float xyz[3];
  GLfloat *coord;
  GLint nValues;
  OpenGLView *view;
  view = visuDataGet_openGLView(dataObj);
  coord = g_malloc(sizeof(GLfloat) * nNodes * (1 + 2));
  glFeedbackBuffer(nNodes * (2 + 1), GL_2D, coord);
  glRenderMode(GL_FEEDBACK);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  /* Get nodes from ids and store their 2D coordinates */
  glBegin(GL_POINTS);
  for(k = 0; k < nNodes; k++){
    node = visuDataGet_nodeFromNumber(dataObj, nodeIds[k]);
    if(node != NULL) 
    {
      visuDataGet_nodePosition(dataObj, node, xyz);
      glVertex3fv(xyz);
    }
  }
  glEnd();
  glPopMatrix();
  nValues = glRenderMode(GL_RENDER);
  i = 0;
  j = 0;
  /* Keep only the coordinates */
  while (i < nValues)
  {
    if (coord[i] == GL_POINT_TOKEN)
    {
      coordinates2D[j] = coord[i + 1];
      coordinates2D[j+1] = coord[i + 2];
      i += 3;
      j += 2;
    }
    else
      i++;
  }
  *size = j;
}

/* Calculate the node vertices coordinates (contained in the
array nodevertices), from node coordinates (xn,yn) */
void defineNodeVertices(int nVert, double radius, double xn, double yn, double *nodeVertices) {
	int i;
	for (i = 0; i < nVert; i++) {
		nodeVertices[2 * i] = (xn + radius * (cos (((2 * G_PI * i) / nVert))));
		nodeVertices[2 * i + 1] = (yn + radius * (sin (((2 * G_PI * i) / nVert))));
	}
}

/* Add node vertices to vertices array, from position i+1 */
void addVerticesToGlobalArray(int nVert, double *nodeVertices, double *vertices, int i) {
	int j;
	int k = 0;
	for (j = (i * 2) * nVert; j < (i * 2) * nVert + (nVert * 2); j += 2) {
		vertices[j] = nodeVertices[k];
		vertices[j + 1] = nodeVertices[k + 1];
		k += 2;
	}
}
